/*
*+
*  Name:
*     channel.h

*  Type:
*     C include file.

*  Purpose:
*     Define the interface to the Channel class.

*  Invocation:
*     #include "channel.h"

*  Description:
*     This include file defines the interface to the Channel class and
*     provides the type definitions, function prototypes and macros,
*     etc. needed to use this class.
*
*     A Channel is the basic form of AST I/O channel, through which
*     Objects may be written and later read back. It causes I/O to
*     take place using a textual format via standard input and
*     standard output.
*
*     Writing to a Channel will result in a textual representation of
*     an Object being produced on standard output. Reading from a
*     Channel will causes a textual description of an Object to be
*     read from standard input, and that Object to be
*     re-created. Channel I/O is stream based, and multiple objects
*     may be written or read in succession through the same Channel. A
*     null Object pointer is returned if there is no more input to
*     read.

*  Inheritance:
*     The Channel class inherits from the Object class.

*  Attributes Over-Ridden:
*     None.

*  New Attributes Defined:
*     Comment (integer)
*        A boolean value (0 or 1) which controls whether comments are
*        to be included in textual output generated by a Channel. If
*        this value is non-zero (the default), then comments will be
*        included. If it is zero, comments will be omitted.
*     Full (integer)
*        A three-state flag (taking values -1, 0 or +1) which controls
*        the amount of information included in textual output
*        generated by a Channel. If this value is zero (the default),
*        then a modest amount of non-essential but useful information
*        will be included along with the output. If Full is negative,
*        all non-essential information will be suppressed, while if it
*        is positive, the output will include the maximum amount of
*        information about the Object being written.
*     Skip (integer)
*        A boolean value which indicates whether the Objects being
*        read through a Channel are inter-mixed with other external
*        data. If this value is zero (the default), then the source of
*        input data is expected to contain descriptions of AST Objects
*        and comments and nothing else (if anything else is read, an
*        error will result). If Skip is non-zero, then any non-Object
*        data encountered between Objects will simply be skipped over
*        in order to reach the next Object.

*  Methods Over-Ridden:
*     Public:
*        None.
*
*     Protected:
*        astClearAttrib
*           Clear an attribute value for a Mapping.
*        astGetAttrib
*           Get an attribute value for a Mapping.
*        astSetAttrib
*           Set an attribute value for a Mapping.
*        astTestAttrib
*           Test if an attribute value has been set for a Mapping.

*  New Methods Defined:
*     Public:
*        astRead
*           Read an Object from a Channel.
*        astWrite
*           Write an Object to a Channel.
*
*     Protected:
*        astClearComment
*           Clear the Comment attribute for a Channel.
*        astClearFull
*           Clear the Full attribute for a Channel.
*        astClearSkip
*           Clear the Skip attribute for a Channel.
*        astGetComment
*           Get the value of the Comment attribute for a Channel.
*        astGetFull
*           Get the value of the Full attribute for a Channel.
*        astGetNextData
*           Read the next item of data from a data source.
*        astGetNextText
*           Read the next line of input text from a data source.
*        astGetSkip
*           Get the value of the Skip attribute for a Channel.
*        astPutNextText
*           Write a line of output text to a data sink.
*        astReadClassData
*           Read values from a data source for a class loader.
*        astReadDouble
*           Read a double value as part of loading a class.
*        astReadInt
*           Read an int value as part of loading a class.
*        astReadObject
*           Read a (sub)Object as part of loading a class.
*        astReadString
*           Read a string value as part of loading a class.
*        astSetComment
*           Set the value of the Comment attribute for a Channel.
*        astSetFull
*           Set the value of the Full attribute for a Channel.
*        astSetSkip
*           Set the value of the Skip attribute for a Channel.
*        astTestComment
*           Test whether a value has been set for the Comment attribute of a
*           Channel.
*        astTestFull
*           Test whether a value has been set for the Full attribute of a
*           Channel.
*        astTestSkip
*           Test whether a value has been set for the Skip attribute of a
*           Channel.
*        astWriteBegin
*           Write a "Begin" data item to a data sink.
*        astWriteDouble
*           Write a double value to a data sink.
*        astWriteEnd
*           Write an "End" data item to a data sink.
*        astWriteInt
*           Write an integer value to a data sink.
*        astWriteIsA
*           Write an "IsA" data item to a data sink.
*        astWriteObject
*           Write an Object as a value to a data sink.
*        astWriteString
*           Write a string value to a data sink.

*  Other Class Functions:
*     Public:
*        astChannel
*           Create a Channel.
*        astChannelFor
*           Create a Channel from a foreign language interface.
*        astIsAChannel
*           Test class membership.
*
*     Protected:
*        astCheckChannel
*           Validate class membership.
*        astInitChannel
*           Initialise a Channel.
*        astInitChannelVtab
*           Initialise the virtual function table for the Channel class.
*        astLoadChannel
*           Load a Channel.

*  Type Definitions:
*     Public:
*        AstChannel
*           Channel object type.
*
*     Protected:
*        AstChannelVtab
*           Channel virtual function table type.

*  Feature Test Macros:
*     astCLASS
*        If the astCLASS macro is undefined, only public symbols are
*        made available, otherwise protected symbols (for use in other
*        class implementations) are defined. This macro also affects
*        the reporting of error context information, which is only
*        provided for external calls to the AST library.

*  Copyright:
*     Copyright (C) 2004 Central Laboratory of the Research Councils

*  Authors:
*     RFWS: R.F. Warren-Smith (Starlink)

*  History:
*     12-AUG-1996 (RFWS):
*        Original version.
*     12-DEC-1996 (RFWS):
*        Added the astChannelFor function.
*     11-NOV-2002 (DSB):
*        Added astWriteInvocations.
*     8-JAN-2003 (DSB):
*        Added protected astInitAxisVtab method.
*-
*/

/* Include files. */
/* ============== */
/* Interface definitions. */
/* ---------------------- */
#include "object.h"              /* Base Object class */

/* Note that the usual setting of the CHANNEL_INCLUDED flag, which
   prevents this file being included more than once, must be deferred
   until after including the "object.h" file. This is because
   "object.h" needs to include the present interface definition (as a
   form of "forward reference") in order to have access to I/O
   Channels itself. */
#if !defined( CHANNEL_INCLUDED )
#define CHANNEL_INCLUDED

/* C header files. */
/* --------------- */
#include <stddef.h>

/* Type Definitions. */
/* ================= */
/* Channel structure. */
/* ------------------ */
/* This structure contains all information that is unique to each
   object in the class (e.g. its instance variables). */
typedef struct AstChannel {

/* Attributes inherited from the parent class. */
   AstObject object;             /* Parent class structure */

/* Attributes specific to objects in this class. */
   const char *(* source)( void ); /* Pointer to source function */
   char *(* source_wrap)( const char *(*)( void ) );
                                 /* Source wrapper function pointer */
   void (* sink)( const char * ); /* Pointer to sink function */
   void (* sink_wrap)( void (*)( const char * ), const char * );
                                 /* Sink wrapper function pointer */
   int comment;                  /* Output comments? */
   int full;                     /* Set max/normal/min information level */
   int skip;                     /* Skip data between Objects? */
} AstChannel;

/* Virtual function table. */
/* ----------------------- */
/* This table contains all information that is the same for all
   objects in the class (e.g. pointers to its virtual functions). */
#if defined(astCLASS)            /* Protected */
typedef struct AstChannelVtab {

/* Properties (e.g. methods) inherited from the parent class. */
   AstObjectVtab object_vtab;    /* Parent class virtual function table */

/* Unique flag value to determine class membership. */
   int *check;                   /* Check value */

/* Properties (e.g. methods) specific to this class. */
   AstObject *(* Read)( AstChannel * );
   AstObject *(* ReadObject)( AstChannel *, const char *, AstObject * );
   char *(* GetNextText)( AstChannel * );
   char *(* ReadString)( AstChannel *, const char *, const char * );
   double (* ReadDouble)( AstChannel *, const char *, double );
   int (* GetComment)( AstChannel * );
   int (* GetFull)( AstChannel * );
   int (* GetSkip)( AstChannel * );
   int (* ReadInt)( AstChannel *, const char *, int );
   int (* TestComment)( AstChannel * );
   int (* TestFull)( AstChannel * );
   int (* TestSkip)( AstChannel * );
   int (* Write)( AstChannel *, AstObject * );
   void (* ClearComment)( AstChannel * );
   void (* ClearFull)( AstChannel * );
   void (* ClearSkip)( AstChannel * );
   void (* GetNextData)( AstChannel *, int, char **, char ** );
   void (* PutNextText)( AstChannel *, const char * );
   void (* ReadClassData)( AstChannel *, const char * );
   void (* SetComment)( AstChannel *, int );
   void (* SetFull)( AstChannel *, int );
   void (* SetSkip)( AstChannel *, int );
   void (* WriteBegin)( AstChannel *, const char *, const char * );
   void (* WriteDouble)( AstChannel *, const char *, int, int, double, const char * );
   void (* WriteEnd)( AstChannel *, const char * );
   void (* WriteInt)( AstChannel *, const char *, int, int, int, const char * );
   void (* WriteIsA)( AstChannel *, const char *, const char * );
   void (* WriteObject)( AstChannel *, const char *, int, int, AstObject *, const char * );
   void (* WriteString)( AstChannel *, const char *, int, int, const char *, const char * );
} AstChannelVtab;
#endif

/* Function prototypes. */
/* ==================== */
/* Prototypes for standard class functions. */
/* ---------------------------------------- */
astPROTO_CHECK(Channel)          /* Check class membership */
astPROTO_ISA(Channel)            /* Test class membership */

/* Constructor. */
#if defined(astCLASS)            /* Protected. */
AstChannel *astChannel_( const char *(*)( void ), void (*)( const char * ),
                         const char *, ... );
#else
AstChannel *astChannelId_( const char *(*)( void ), void (*)( const char * ),
                           const char *, ... );
AstChannel *astChannelForId_( const char *(*)( void ),
                              char *(*)( const char *(*)( void ) ),
                              void (*)( const char * ),
                              void (*)( void (*)( const char * ),
                                        const char * ),
                              const char *, ... );
#endif


#if defined(astCLASS)            /* Protected */

/* Initialiser. */
AstChannel *astInitChannel_( void *, size_t, int, AstChannelVtab *,
                             const char *, const char *(*)( void ), 
                             char *(*)( const char *(*)( void ) ), 
                             void (*)( const char * ), 
                             void (*)( void (*)( const char * ), 
                             const char * ) );


/* Vtab initialiser. */
void astInitChannelVtab_( AstChannelVtab *, const char * );

/* Loader. */
AstChannel *astLoadChannel_( void *, size_t, AstChannelVtab *,
                             const char *, AstChannel *channel );
#endif

/* Prototypes for member functions. */
/* -------------------------------- */
AstObject *astRead_( AstChannel * );
int astWrite_( AstChannel *, AstObject * );

char *astSourceWrap_( const char *(*)( void ) );
void astSinkWrap_( void (*)( const char * ), const char * );

# if defined(astCLASS)           /* Protected */
AstObject *astReadObject_( AstChannel *, const char *, AstObject * );
char *astGetNextText_( AstChannel * );
char *astReadString_( AstChannel *, const char *, const char * );
double astReadDouble_( AstChannel *, const char *, double );
int astGetComment_( AstChannel * );
int astGetFull_( AstChannel * );
int astGetSkip_( AstChannel * );
int astReadInt_( AstChannel *, const char *, int );
int astTestComment_( AstChannel * );
int astTestFull_( AstChannel * );
int astTestSkip_( AstChannel * );
void astClearComment_( AstChannel * );
void astClearFull_( AstChannel * );
void astClearSkip_( AstChannel * );
void astGetNextData_( AstChannel *, int, char **, char ** );
void astPutNextText_( AstChannel *, const char * );
void astReadClassData_( AstChannel *, const char * );
void astSetComment_( AstChannel *, int );
void astSetFull_( AstChannel *, int );
void astSetSkip_( AstChannel *, int );
void astWriteBegin_( AstChannel *, const char *, const char * );
void astWriteDouble_( AstChannel *, const char *, int, int, double, const char * );
void astWriteEnd_( AstChannel *, const char * );
void astWriteInt_( AstChannel *, const char *, int, int, int, const char * );
int astWriteInvocations_( void );
void astWriteIsA_( AstChannel *, const char *, const char * );
void astWriteObject_( AstChannel *, const char *, int, int, AstObject *, const char * );
void astWriteString_( AstChannel *, const char *, int, int, const char *, const char * );
#endif

/* Function interfaces. */
/* ==================== */
/* These macros are wrap-ups for the functions defined by this class
   to make them easier to invoke (e.g. to avoid type mis-matches when
   passing pointers to objects from derived classes). */

/* Interfaces to standard class functions. */
/* --------------------------------------- */
/* Some of these functions provide validation, so we cannot use them to
   validate their own arguments. We must use a cast when passing object
   pointers (so that they can accept objects from derived classes). */

/* Check class membership. */
#define astCheckChannel(this) astINVOKE_CHECK(Channel,this)

/* Test class membership. */
#define astIsAChannel(this) astINVOKE_ISA(Channel,this)

/* Constructor. */
#if defined(astCLASS)            /* Protected. */
#define astChannel astINVOKE(F,astChannel_)
#else
#define astChannel astINVOKE(F,astChannelId_)
#define astChannelFor astINVOKE(F,astChannelForId_)
#endif

#if defined(astCLASS)            /* Protected */

/* Initialiser. */
#define astInitChannel(mem,size,init,vtab,name,source,source_wrap,sink,sink_wrap) \
astINVOKE(O,astInitChannel_(mem,size,init,vtab,name,source,source_wrap,sink,sink_wrap))

/* Vtab Initialiser. */
#define astInitChannelVtab(vtab,name) astINVOKE(V,astInitChannelVtab_(vtab,name))
/* Loader. */
#define astLoadChannel(mem,size,vtab,name,channel) \
astINVOKE(O,astLoadChannel_(mem,size,vtab,name,astCheckChannel(channel)))
#endif

/* Interfaces to member functions. */
/* ------------------------------- */
/* Here we make use of astCheckChannel to validate Channel pointers
   before use.  This provides a contextual error report if a pointer
   to the wrong sort of Object is supplied. */

#define astRead(this) astINVOKE(O,astRead_(astCheckChannel(this)))
#define astWrite(this,object) \
astINVOKE(V,astWrite_(astCheckChannel(this),astCheckObject(object)))

#define astSourceWrap astSourceWrap_
#define astSinkWrap astSinkWrap_

#if defined(astCLASS)            /* Protected */
#define astClearComment(this) \
astINVOKE(V,astClearComment_(astCheckChannel(this)))
#define astClearFull(this) \
astINVOKE(V,astClearFull_(astCheckChannel(this)))
#define astClearSkip(this) \
astINVOKE(V,astClearSkip_(astCheckChannel(this)))
#define astGetComment(this) \
astINVOKE(V,astGetComment_(astCheckChannel(this)))
#define astGetFull(this) \
astINVOKE(V,astGetFull_(astCheckChannel(this)))
#define astGetNextData(this,begin,name,val) \
astINVOKE(V,astGetNextData_(astCheckChannel(this),begin,name,val))
#define astGetNextText(this) \
astINVOKE(V,astGetNextText_(astCheckChannel(this)))
#define astGetSkip(this) \
astINVOKE(V,astGetSkip_(astCheckChannel(this)))
#define astPutNextText(this,line) \
astINVOKE(V,astPutNextText_(astCheckChannel(this),line))
#define astReadClassData(this,class) \
astINVOKE(V,astReadClassData_(astCheckChannel(this),class))
#define astReadDouble(this,name,def) \
astINVOKE(V,astReadDouble_(astCheckChannel(this),name,def))
#define astReadInt(this,name,def) \
astINVOKE(V,astReadInt_(astCheckChannel(this),name,def))
#define astReadObject(this,name,def) \
astINVOKE(O,astReadObject_(astCheckChannel(this),name,(def)?astCheckObject(def):NULL))
#define astReadString(this,name,def) \
astINVOKE(V,astReadString_(astCheckChannel(this),name,def))
#define astSetComment(this,value) \
astINVOKE(V,astSetComment_(astCheckChannel(this),value))
#define astSetFull(this,value) \
astINVOKE(V,astSetFull_(astCheckChannel(this),value))
#define astSetSkip(this,value) \
astINVOKE(V,astSetSkip_(astCheckChannel(this),value))
#define astTestComment(this) \
astINVOKE(V,astTestComment_(astCheckChannel(this)))
#define astTestFull(this) \
astINVOKE(V,astTestFull_(astCheckChannel(this)))
#define astTestSkip(this) \
astINVOKE(V,astTestSkip_(astCheckChannel(this)))
#define astWriteBegin(this,class,comment) \
astINVOKE(V,astWriteBegin_(astCheckChannel(this),class,comment))
#define astWriteDouble(this,name,set,helpful,value,comment) \
astINVOKE(V,astWriteDouble_(astCheckChannel(this),name,set,helpful,value,comment))
#define astWriteEnd(this,class) \
astINVOKE(V,astWriteEnd_(astCheckChannel(this),class))
#define astWriteInt(this,name,set,helpful,value,comment) \
astINVOKE(V,astWriteInt_(astCheckChannel(this),name,set,helpful,value,comment))
#define astWriteIsA(this,class,comment) \
astINVOKE(V,astWriteIsA_(astCheckChannel(this),class,comment))
#define astWriteObject(this,name,set,helpful,value,comment) \
astINVOKE(V,astWriteObject_(astCheckChannel(this),name,set,helpful,astCheckObject(value),comment))
#define astWriteString(this,name,set,helpful,value,comment) \
astINVOKE(V,astWriteString_(astCheckChannel(this),name,set,helpful,value,comment))

#define astWriteInvocations astWriteInvocations_()

#endif
#endif
