// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "marker.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

static int markerSeqID = 1;

// Tag

Tag::Tag(const Tag& t)
{
  tag_ = dupstr(t.tag_);
}

Tag::Tag(const char* t)
{
  tag_ = dupstr(t);
}

Tag& Tag::operator=(const Tag& t) 
{
  tag_ = dupstr(t.tag_);
  return *this;
}

Tag::~Tag()
{
  if (tag_)
    delete [] tag_;
}

void Tag::set(const char* t)
{
  if (tag_)
    delete tag_;

  tag_ = dupstr(t);
}

// Marker Members Public

Marker::Marker(const Marker& a)
{
  parent = a.parent;
  center = a.center;
  properties = a.properties;

  id = a.id;
  strcpy(type, a.type);

  bbox = a.bbox;
  allBBox = a.allBBox;
  angle = a.angle;

  numHandle = a.numHandle;
  if (numHandle) {
    handle = new Vector[numHandle];
    for (int i=0; i<numHandle; i++)
      handle[i] = a.handle[i];
  }
  else
    handle = NULL;

  colorName = dupstr(a.colorName);
  color = a.color;
  lineWidth = a.lineWidth;
  selected = a.selected;

  display = a.display;
  gc = a.gc;

  text = dupstr(a.text);
  font = Tk_GetFont(parent->interp, parent->tkwin, Tk_NameOfFont(a.font));
  comment = dupstr(a.comment);

  tags = a.tags;

  // disable Callbacks by default
  doCB = 0;

  selectCB = a.selectCB;
  unselectCB = a.unselectCB;
  moveBeginCB = a.moveBeginCB;
  moveCB = a.moveCB;
  moveEndCB = a.moveEndCB;
  editBeginCB = a.editBeginCB;
  editCB = a.editCB;
  editEndCB = a.editEndCB;
  rotateBeginCB = a.rotateBeginCB;
  rotateCB = a.rotateCB;
  rotateEndCB = a.rotateEndCB;
  deleteCB = a.deleteCB;
  textCB = a.textCB;
  colorCB = a.colorCB;
  lineWidthCB = a.lineWidthCB;
  propertyCB = a.propertyCB;
  fontCB = a.fontCB;
  updateCB = a.updateCB;

  previous_ = NULL;
  next_ = NULL;
}

Marker::Marker(FrameBase* p, const Vector& ctr, const char* clr, int w,
	       const char* f, const char* t, unsigned short prop, 
	       const char* c, const List<Tag>& tg)
{
  parent = p;
  center = ctr;
  properties = prop;

  id = markerSeqID++;
  type[0] = '\0';

  handle = NULL;
  numHandle = 0;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);
  lineWidth = w;
  selected = 0;

  display = parent->display;
  gc = parent->markerGC;

  text = dupstr(t);
  font = Tk_GetFont(parent->interp, parent->tkwin, f);
  comment = dupstr(c);

  tags = tg;

  doCB = 1;

  previous_ = NULL;
  next_ = NULL;
}

Marker::~Marker()
{
  if (colorName)
    delete [] colorName;

  if (text)
    delete [] text;

  if (comment)
    delete [] comment;

  if (font)
    Tk_FreeFont(font);

  if (handle)
    delete [] handle;

  doCallBack(&deleteCB);
}

void Marker::clearCB()
{
  selectCB.deleteAll();
  unselectCB.deleteAll();
  moveBeginCB.deleteAll();
  moveCB.deleteAll();
  moveEndCB.deleteAll();
  editBeginCB.deleteAll();
  editCB.deleteAll();
  editEndCB.deleteAll();
  rotateBeginCB.deleteAll();
  rotateCB.deleteAll();
  rotateEndCB.deleteAll();
  deleteCB.deleteAll();
  textCB.deleteAll();
  colorCB.deleteAll();
  lineWidthCB.deleteAll();
  propertyCB.deleteAll();
  fontCB.deleteAll();
  updateCB.deleteAll();
}

void Marker::newIdentity()
{
  id = markerSeqID++;

  doCB = 1;
  clearCB();
  updateBBox();
}

void Marker::updateCoords(const Matrix& m)
{
  center*=m;
  updateBBox();
}

void Marker::draw(Drawable drawable)
{
  renderHandles(drawable);
  renderText(drawable, parent->refToWidget, SRC);
  render(drawable, parent->refToWidget, parent->getZoom(), SRC);
}

void Marker::drawXOR()
{
  renderText(Tk_WindowId(parent->tkwin), parent->refToWindow, XOR);
  render(Tk_WindowId(parent->tkwin), parent->refToWindow, 
	 parent->getZoom(), XOR);
}

void Marker::drawMagnifier(const Matrix& m)
{
  renderText(parent->magnifierPixmap, m, SRC);
  render(parent->magnifierPixmap, m, parent->getMagnifierZoom(), SRC);
}

void Marker::moveTo(const Vector& v)
{
  center=v;
  updateBBox();
  doCallBack(&moveCB);
}

void Marker::moveBegin()
{
  doCallBack(&moveBeginCB);
}

void Marker::move(const Vector& v)
{
  center+=v;
  updateBBox();
  doCallBack(&moveCB);
}

void Marker::moveEnd()
{
  doCallBack(&moveEndCB);
}

void Marker::editBegin(int)
{
  doCallBack(&editBeginCB);
}

void Marker::edit(const Vector& v, int h)
{
  doCallBack(&editCB);
}

void Marker::editEnd()
{
  doCallBack(&editEndCB);
}

void Marker::rotateBegin()
{
  doCallBack(&rotateBeginCB);
}

void Marker::rotate(const Vector& v, int h)
{
  // v is in ref coords
  // handles are in canvas coords

  Vector a = v - center;
  Vector b = (handle[h-1] * parent->canvasToRef) - center;

  if (parent->isIIS())
    angle -= a.angle() - b.angle();
  else
    angle += a.angle() - b.angle();

  updateBBox();
  doCallBack(&rotateCB);
}

void Marker::rotateEnd()
{
  doCallBack(&rotateEndCB);
}

#if __GNUC__ >= 3
void Marker::ps(int mode)
{
  if (properties & SOURCE)
    psLineNoDash();
  else
    psLineDash();

  // set color

  psColor(mode, colorName);

  // do text

  if (text && font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);

    ostringstream str;
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
      
    Tcl_DStringFree(&psFont);

    Vector bbc = bbox.center();
    Vector t =  Vector(bbc[0], bbox.ll[1]);

    str << t.TkCanvasPs(parent->canvas) << " moveto" << endl
	<< '(' << psQuote(text) << ')' << endl
	<< "dup stringwidth pop 2 div 0 exch sub " 
	<< metrics.descent << " rmoveto show" << endl
	<< ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}
#else
void Marker::ps(int mode)
{
  if (properties & SOURCE)
    psLineNoDash();
  else
    psLineDash();

  // set color

  psColor(mode, colorName);

  // do text

  if (text && font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);

    char buf[512];
    ostrstream str(buf,512);
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;
      
    Tcl_DStringFree(&psFont);

    Vector bbc = bbox.center();
    Vector t =  Vector(bbc[0], bbox.ll[1]);

    str << t.TkCanvasPs(parent->canvas) << " moveto" << endl
	<< '(' << psQuote(text) << ')' << endl
	<< "dup stringwidth pop 2 div 0 exch sub " 
	<< metrics.descent << " rmoveto show" << endl
	<< ends;

    Tcl_AppendResult(parent->interp, buf, NULL);
  }
}
#endif

void Marker::select() {
  // only call the CB if not already selected
  if (!selected)
    doCallBack(&selectCB);
  selected = 1;
}

void Marker::unselect() {
  // only call the CB if already selected
  if (selected)
    doCallBack(&unselectCB);
  selected = 0;
}

void Marker::toggleSelect() {
  selected = !selected;
  if (selected)
    doCallBack(&selectCB);
  else
    doCallBack(&unselectCB);
}

void Marker::setAngle(double a)
{
  angle = a;
  updateBBox();

  doCallBack(&rotateCB);
}

void Marker::setColor(const char* clr)
{
  if (colorName)
    delete [] colorName;

  colorName = dupstr(clr);
  color = parent->getColor(colorName);

  doCallBack(&colorCB);
}

void Marker::setLineWidth(int w)
{
  lineWidth = w;

  doCallBack(&lineWidthCB);
}

void Marker::setFont(const char* f)
{
  if (font)
    Tk_FreeFont(font);

  if (f)
    font = Tk_GetFont(parent->interp, parent->tkwin, f);
  else
    font = Tk_GetFont(parent->interp, parent->tkwin, "helvetica");

  updateBBox();
  doCallBack(&fontCB);
}

const char* Marker::getFont()
{
  if (font)
    return Tk_NameOfFont(font);
  else
    return NULL;
}

void Marker::addTag(const char* tg)
{
  Tag* t = new Tag(tg);
  tags.append(t);
}

void Marker::editTag(const char* from, const char* to)
{
  // change any tags
  {
    Tag* t = tags.head();
    while (t) {
      if (!strcmp(t->tag(),from)) {
	t->set(to);
      }
      t=t->next();
    }
  }

  // now, remove duplicates
  {
    Tag* t = tags.head();
    while (t) {
      Tag* tt=t->next();
      while (tt) {
	if (!strcmp(t->tag(),tt->tag())) {
	  Tag* ntt = tags.extractNext(tt);
	  delete tt;
	  tt = ntt;
	}
	else
	  tt=tt->next();
      }
      t=t->next();
    }
  }
}

void Marker::deleteTags()
{
  tags.deleteAll();
}

void Marker::deleteTag(const char* tg)
{
  Tag* t = tags.head();
  while (t) {
    if (!strcmp(t->tag(),tg)) {
      tags.extractNext(t);
      delete t;
      return;
    }
    t = t->next();
  }
}

void Marker::deleteTag(int w)
{
  Tag* t = tags.head();
  for (int i=0; i<w; i++)
    if (t)
      t = t->next();
    else
      break;

  if (t) {
    tags.extractNext(t);
    delete t;
  }
}

const char* Marker::getTag()
{
  Tag* t = tags.head();
  if (t)
    return t->tag();
  else
    return NULL;
}

const char* Marker::getNextTag()
{
  Tag* t = tags.next();
  if (t)
    return t->tag();
  else
    return NULL;
}

const char* Marker::getTag(int w)
{
  Tag* t = tags.head();
  for (int i=0; i<w; i++)
    if (t)
      t = t->next();
    else
      break;

  if (t)
    return t->tag();
  else
    return NULL;
}

int Marker::hasTag(const char* tg)
{
  Tag* t = tags.head();
  while (t) {
    if (!strcmp(t->tag(),tg))
      return 1;
    t = t->next();
  }
  return 0;
}

int Marker::onHandle(const Vector& v)
{
  // return handle number
  // work last to first for annuli

  for (int i=numHandle-1; i>=0; i--) {
    BBox bb(handle[i]);
    bb.expand(2);
    if (bb.isIn(v))
      return i+1;
  }
  return 0;
}

int Marker::getProperty(unsigned short which)
{
  return (properties & which) ? 1 : 0;
}

void Marker::setText(const char* str)
{
  if (text)
    delete [] text;
  text = dupstr(str);

  updateBBox();
  doCallBack(&textCB);
}

void Marker::setProperty(unsigned short prop, int value)
{
  if (value)
    properties |= prop;
  else
    properties &= ~prop;

  if (prop == FIXED) // bbox will change
    updateBBox();

  doCallBack(&propertyCB);
}

int Marker::setCallBack(Callback cbl, const char* proc, const char* arg)
{
  CallBack* cb = new CallBack(parent->interp, proc, arg);

  switch (cbl) {
  case SELECTCB:
    selectCB.append(cb);
    return TCL_OK;
  case UNSELECTCB:
    unselectCB.append(cb);
    return TCL_OK;
  case MOVEBEGINCB:
    moveBeginCB.append(cb);
    return TCL_OK;
  case MOVECB:
    moveCB.append(cb);
    return TCL_OK;
  case MOVEENDCB:
    moveEndCB.append(cb);
    return TCL_OK;
  case EDITBEGINCB:
    editBeginCB.append(cb);
    return TCL_OK;
  case EDITCB:
    editCB.append(cb);
    return TCL_OK;
  case EDITENDCB:
    editEndCB.append(cb);
    return TCL_OK;
  case ROTATEBEGINCB:
    rotateBeginCB.append(cb);
    return TCL_OK;
  case ROTATECB:
    rotateCB.append(cb);
    return TCL_OK;
  case ROTATEENDCB:
    rotateEndCB.append(cb);
    return TCL_OK;
  case DELETECB:
    deleteCB.append(cb);
    return TCL_OK;
  case TEXTCB:
    textCB.append(cb);
    return TCL_OK;
  case COLORCB:
    colorCB.append(cb);
    return TCL_OK;
  case LINEWIDTHCB:
    lineWidthCB.append(cb);
    return TCL_OK;
  case PROPERTYCB:
    propertyCB.append(cb);
    return TCL_OK;
  case FONTCB:
    fontCB.append(cb);
    return TCL_OK;
  case UPDATECB:
    updateCB.append(cb);
    return TCL_OK;
  }

  return TCL_ERROR;
}

void Marker::deleteAllCallBack(Callback cbl)
{
  List<CallBack>* list = NULL;

  switch (cbl) {
  case SELECTCB:
    list = &selectCB;
    break;
  case UNSELECTCB:
    list = &unselectCB;
    break;
  case MOVEBEGINCB:
    list = &moveBeginCB;
    break;
  case MOVECB:
    list = &moveCB;
    break;
  case MOVEENDCB:
    list = &moveEndCB;
    break;
  case EDITBEGINCB:
    list = &editBeginCB;
    break;
  case EDITCB:
    list = &editCB;
    break;
  case EDITENDCB:
    list = &editEndCB;
    break;
  case ROTATEBEGINCB:
    list = &rotateBeginCB;
    break;
  case ROTATECB:
    list = &rotateCB;
    break;
  case ROTATEENDCB:
    list = &rotateEndCB;
    break;
  case DELETECB:
    list = &deleteCB;
    break;
  case TEXTCB:
    list = &textCB;
    break;
  case COLORCB:
    list = &colorCB;
    break;
  case LINEWIDTHCB:
    list = &lineWidthCB;
    break;
  case PROPERTYCB:
    list = &propertyCB;
    break;
  case FONTCB:
    list = &fontCB;
    break;
  case UPDATECB:
    list = &updateCB;
    break;
  }

  if (list)
    list->deleteAll();
}

int Marker::deleteCallBack(Callback cbl, const char* proc)
{
  List<CallBack>* list = NULL;

  switch (cbl) {
  case SELECTCB:
    list = &selectCB;
    break;
  case UNSELECTCB:
    list = &unselectCB;
    break;
  case MOVEBEGINCB:
    list = &moveBeginCB;
    break;
  case MOVECB:
    list = &moveCB;
    break;
  case MOVEENDCB:
    list = &moveEndCB;
    break;
  case EDITBEGINCB:
    list = &editBeginCB;
    break;
  case EDITCB:
    list = &editCB;
    break;
  case EDITENDCB:
    list = &editEndCB;
    break;
  case ROTATEBEGINCB:
    list = &rotateBeginCB;
    break;
  case ROTATECB:
    list = &rotateCB;
    break;
  case ROTATEENDCB:
    list = &rotateEndCB;
    break;
  case DELETECB:
    list = &deleteCB;
    break;
  case TEXTCB:
    list = &textCB;
    break;
  case COLORCB:
    list = &colorCB;
    break;
  case LINEWIDTHCB:
    list = &lineWidthCB;
    break;
  case PROPERTYCB:
    list = &propertyCB;
    break;
  case FONTCB:
    list = &fontCB;
    break;
  case UPDATECB:
    list = &updateCB;
    break;
  }

  if (list) {
    CallBack* cb = list->head();

    while (cb) {
      if (!strcmp(cb->proc(), proc)) {
	list->extractNext(cb);
	delete cb;
	return TCL_OK;
      }
      else
	cb = cb->next();
    }
  }
  
  return TCL_ERROR;
}

void Marker::doCallBack(Callback cbl)
{
  switch (cbl) {
  case SELECTCB:
    doCallBack(&selectCB);
    return;
  case UNSELECTCB:
    doCallBack(&unselectCB);
    return;
  case MOVEBEGINCB:
    doCallBack(&moveBeginCB);
    return;
  case MOVECB:
    doCallBack(&moveCB);
    return;
  case MOVEENDCB:
    doCallBack(&moveEndCB);
    return;
  case EDITBEGINCB:
    doCallBack(&editBeginCB);
    return;
  case EDITCB:
    doCallBack(&editCB);
    return;
  case EDITENDCB:
    doCallBack(&editEndCB);
    return;
  case ROTATEBEGINCB:
    doCallBack(&rotateBeginCB);
    return;
  case ROTATECB:
    doCallBack(&rotateCB);
    return;
  case ROTATEENDCB:
    doCallBack(&rotateEndCB);
    return;
  case DELETECB:
    doCallBack(&deleteCB);
    return;
  case TEXTCB:
    doCallBack(&textCB);
    return;
  case COLORCB:
    doCallBack(&colorCB);
    return;
  case LINEWIDTHCB:
    doCallBack(&lineWidthCB);
    return;
  case PROPERTYCB:
    doCallBack(&propertyCB);
    return;
  case FONTCB:
    doCallBack(&fontCB);
    return;
  case UPDATECB:
    doCallBack(&updateCB);
    return;
  }
}

int Marker::isVisible(const BBox& b)
{
  // assume visible, prove otherwise
  // all coords are in canvas coords

  BBox bb(b);
  return 
    !((allBBox.ur[0] < bb.ll[0]) ||
    (allBBox.ll[0] > bb.ur[0]) ||
    (allBBox.ur[1] < bb.ll[1]) ||
    (allBBox.ll[1] > bb.ur[1]));
}

// Protected

#if __GNUC__ >= 3
void Marker::doCallBack(List<CallBack>* list)
{
  if (!doCB)
    return;

  ostringstream str;
  str << id << ends;

  if (list->head())
    do
      if ((list->current())->eval(str.str().c_str()))
	cerr << "Marker Internal Error: Unable to eval Callback "
	     << (list->current())->proc() << endl << "  :" 
	     << parent->interp->result << endl;

    while (list->next());
}
#else
void Marker::doCallBack(List<CallBack>* list)
{
  if (!doCB)
    return;

  char buf[8];
  ostrstream str(buf,8);
  str << id << ends;

  if (list->head())
    do
      if ((list->current())->eval(buf))
	cerr << "Marker Internal Error: Unable to eval Callback "
	     << (list->current())->proc() << endl << "  :" 
	     << parent->interp->result << endl;

    while (list->next());
}
#endif

#if __GNUC__ >= 3
void Marker::psColor(int mode, const char* cn)
{
  ostringstream str;
  switch ((FrameBase::PSColorSpace)mode) {
  case FrameBase::BW:
  case FrameBase::GRAY:
    psColorGray(cn, str);
    str << " setgray";
    break;
  case FrameBase::RGB:
    psColorRGB(cn, str);
    str << " setrgbcolor";
    break;
  case FrameBase::CMYK:
    psColorCMYK(cn, str);
    str << " setcmykcolor";
    break;
  }
  str << endl << ends;

  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}
#else
void Marker::psColor(int mode, const char* cn)
{

  char buf[256];
  ostrstream str(buf,256);
  switch ((FrameBase::PSColorSpace)mode) {
  case FrameBase::BW:
  case FrameBase::GRAY:
    psColorGray(cn, str);
    str << " setgray";
    break;
  case FrameBase::RGB:
    psColorRGB(cn, str);
    str << " setrgbcolor";
    break;
  case FrameBase::CMYK:
    psColorCMYK(cn, str);
    str << " setcmykcolor";
    break;
  }
  str << endl << ends;

  Tcl_AppendResult(parent->interp, buf, NULL);
}
#endif

#if __GNUC__ >= 3
void Marker::psLineNoDash()
{
  ostringstream str;
  str << lineWidth << " setlinewidth" << endl;
  str << "[] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}
#else
void Marker::psLineNoDash()
{
  char buf[256];
  ostrstream str(buf,256);
  str << lineWidth << " setlinewidth" << endl;
  str << "[] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);
}
#endif

#if __GNUC__ >= 3
void Marker::psLineDash()
{
  ostringstream str;
  str << lineWidth << " setlinewidth" << endl;
  str << "[8 3] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}
#else
void Marker::psLineDash()
{
  char buf[256];
  ostrstream str(buf,256);
  str << lineWidth << " setlinewidth" << endl;
  str << "[8 3] 0 setdash" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);
}
#endif

void Marker::calcAllBBox()
{
  allBBox = bbox;

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    Vector bbc = bbox.center();
    Vector ll =  Vector(bbc[0], bbox.ll[1]) * Translate(-width/2., 0);
    Vector ur = ll * Translate(width, -(metrics.linespace));

    allBBox.bound(ll);
    allBBox.bound(ur);
  }
}

void Marker::renderHandles(Drawable drawable)
{
  XSetForeground(display, gc, color);
  setLineNoDash();

  // handles are of length 5

  if (selected && canHighlite())
    for (int i=0; i<numHandle; i++) {
      Vector v = (handle[i]*parent->canvasToWidget - Vector(2,2)).round();
      XFillRectangle(display, drawable, gc, (int)v[0], (int)v[1], 5, 5);
    }
}

void Marker::renderText(Drawable drawable, const Matrix& m, RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }
  setLineNoDash();

  if (font)
    XSetFont(display, gc, Tk_FontId(font));

  if (text && font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);
    int width = Tk_TextWidth(font, text, strlen(text));

    BBox bb = bbox * parent->canvasToRef * m;
    Vector bbc = bb.center();
    Vector t =  Vector(bbc[0], bb.ll[1]) * 
      Translate(-width/2., -metrics.descent);
    XDrawString(display, drawable, gc, (int)t[0], (int)t[1], 
		text, strlen(text));
  }
}

void Marker::setGC(RenderMode mode)
{
  switch (mode) {
  case SRC:
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    if (properties & SOURCE)
      setLineNoDash();
    else
      setLineDash();
    break;
  case XOR:
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    setLineDash();
    break;
  }
}

void Marker::setLineDash()
{
  char dlist[] = {8,3};
  XSetLineAttributes(display, gc, lineWidth, LineOnOffDash, CapButt,JoinMiter);
  XSetDashes(display, gc, 0, dlist, 2);
}

void Marker::setLineNoDash()
{
  XSetLineAttributes(display, gc, lineWidth, LineSolid, CapButt, JoinMiter);
}

Vector* Marker::arrow(const Vector& p, const Vector& dd)
{
  Vector d = dd;      // stupid, however, no const warnings

  const int tip = 6;  // length from end of line to tip of arrow
  const int tail = 2; // length from end of line to tails of arrow
  const int wc = 2;   // width of arrow at end of line
  const int wt = 3;   // width of arrow at tails

  Vector* a = new Vector[6];
  a[0] = Vector(0, tip);
  a[1] = Vector(-wc, 0);
  a[2] = Vector(-wt, -tail);
  a[3] = Vector(0, 0);
  a[4] = Vector(wt, -tail);
  a[5] = Vector(wc, 0);

  Matrix m = Translate(0,-tip) * 
    Scale(1.5) * 
    Rotate(-M_PI/2) * 
    Rotate(-d.angle()) * 
    Translate(p);

  for (int i=0; i<6; i++) {
    a[i] = (a[i] * m).round();
  }

  return a;
}

void Marker::renderArrow(Drawable drawable, const Vector& p, const Vector& d)
{
  Vector* a = arrow(p,d);
  XPoint aa[6];
  for (int i=0; i<6; i++) {
    aa[i].x = (short)a[i][0];
    aa[i].y = (short)a[i][1];
  }

  XFILLPOLYGON(display, drawable, gc, aa, 6, Nonconvex, CoordModeOrigin);
  delete [] a;
}

#if __GNUC__ >= 3
void Marker::psArrow(const Vector& p, const Vector& d)
{
  Vector* a = arrow(p,d);
  for (int i=0; i<6; i++) {
    ostringstream str;
    if (i==0)
      str << "newpath " << endl
	  << a[i].TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
    else
      str << a[i].TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  str << "closepath fill" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  delete [] a;
}
#else
void Marker::psArrow(const Vector& p, const Vector& d)
{
  char buf[256];
  Vector* a = arrow(p,d);
  for (int i=0; i<6; i++) {
  ostrstream str(buf,256);
    if (i==0)
      str << "newpath " << endl
	  << a[i].TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
    else
      str << a[i].TkCanvasPs(parent->canvas) << " lineto" << endl << ends;

    Tcl_AppendResult(parent->interp, buf, NULL);
  }

  ostrstream str(buf,256);
  str << "closepath fill" << endl << ends;
  Tcl_AppendResult(parent->interp, buf, NULL);

  delete [] a;
}
#endif

double Marker::mapLenFromRef(double d, CoordSystem sys, SkyFormat format)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
      return ptr->mapLenFromRef(d,sys);
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys))
	return ptr->mapLenFromRef(d,sys,format);
      else
	return ptr->mapLenFromRef(d,sys);
    }
  }
  return 0;
}

void Marker::listProperties(ostream& str, char delim)
{
  // no props for semicolons
  if (delim == ';')
    return;

  if (strcmp("green",colorName) ||
      strcmp("helvetica 10 normal",getFont()) ||
      text[0] ||
      !(properties&SELECT) ||
      !(properties&EDIT) ||
      !(properties&MOVE) ||
      !(properties&ROTATE) ||
      !(properties&DELETE) ||
      !(properties&HIGHLITE) ||
      (properties&FIXED) ||
      !(properties&SOURCE) ||
      (tags.count() > 0) ||
      (comment && *comment)) {

    str << " #";
    listProps(str);
  }
}

void Marker::listProps(ostream& str)
{
  if (strcmp("green",colorName))
    str << " color=" << colorName;

  if (lineWidth != 1)
    str << " width=" << lineWidth;

  if (strcmp("helvetica 10 normal",getFont()))
    str << " font=\"" << getFont() << '"';

  if (text && *text) // only list text if there is something to list
    str << " text={" << text << '}';

  if (!(properties&SELECT))
    str << " select=0";

  if (!(properties&EDIT))
    str << " edit=0";

  if (!(properties&MOVE))
    str << " move=0";

  if (!(properties&ROTATE))
    str << " rotate=0";

  if (!(properties&DELETE))
    str << " delete=0";

  if (!(properties&HIGHLITE))
    str << " highlite=0";

  if (properties&FIXED)
    str << " fixed=1";

  if (!(properties&SOURCE))
    str << " background";

  // tags
  Tag* t = tags.head();
  while (t) {
    str << " tag={" << t->tag() << '}';
    t = t->next();
  }

  if (comment && *comment) 
    str << ' ' << comment;
}

void Marker::listCoordSystem(ostream& str, CoordSystem sys, SkyFrame sky, 
			     FitsImage* ptr)
{
  switch (sys) {
  case IMAGE:
    str << "image";
    return;
  case PHYSICAL:
    str << "physical";
    return;
  case DETECTOR:
    str << "detector";
    return;
  case AMPLIFIER:
    str << "amplifier";
    return;
  default:
    if (ptr->hasWCSEqu(sys)) {
      switch (sky) {
      case FK4:
	str << "fk4";
	return;
      case FK5:
	str << "fk5";
	return;
      case ICRS:
	str << "icrs";
	return;
      case GALACTIC:
	str << "galactic";
	return;
      case ECLIPTIC:
	str << "ecliptic";
	return;
      }
    } 
    else {
      str << "linear";
      return;
    }
  }
}

void Marker::listProsCoordSystem(ostream& str, CoordSystem sys, SkyFrame sky)
{
  switch (sys) {
  case IMAGE:
    str << "logical";
    return;
  case PHYSICAL:
    str << "physical";
    return;
  case DETECTOR:
    str << "detector";
    return;
  case AMPLIFIER:
    str << "amplifier";
    return;
  default:
    switch (sky) {
    case FK4:
      str << "b1950";
      return;
    case FK5:
      str << "j2000";
      return;
    case ICRS:
      str << "icrs";
      return;
    case GALACTIC:
      str << "galactic";
      return;
    case ECLIPTIC:
      str << "ecliptic";
      return;
    default:
      str << "linear";
      return;
    }
  }
}

void Marker::listSAOtngPre(ostream& str, char delim)
{
  if (delim != ';' && text[0])
    str << '#' << text << delim;

  if (properties&INCLUDE)
    str << '+';
  else
    str << '-';
}

void Marker::listSAOtngPost(ostream& str, char delim)
{
  if (delim != ';') {
    str << " # ";
    if (comment && *comment)
      str << comment;
    else if (!(properties&SOURCE))
      str << "background";
    else
      str << colorName;
  }

  str << delim;
}

void Marker::listXY(ostream& str, CoordSystem sys, SkyFrame sky,
		    SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    str << setprecision(8) << ptr->mapFromRef(center,sys);
    break;
  default:
    if (ptr->hasWCS(sys)) {
      if (ptr->hasWCSEqu(sys)) {
	switch (format) {
	case DEGREES:
	  str << setprecision(8) << ptr->mapFromRef(center,sys,sky);
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << ra << ' ' << dec;
	  }
	  break;
	}
      }
      else {
	str << setprecision(8) << ptr->mapFromRef(center,sys);
      }
    }
    break;
  }

  str << delim;
}

void Marker::listPre(ostream& str, CoordSystem sys, SkyFrame sky, 
		     FitsImage* ptr)
{
  if (parent->hasFitsMosaic())
    str << "tile " << parent->findFits(ptr) << ';';

  listCoordSystem(str,sys,sky,ptr);
  str << ';';

  if (!(properties&INCLUDE))
    str << '-';
}

void Marker::listPre2(ostream& str, CoordSystem sys, SkyFrame sky, 
		      FitsImage* ptr)
{
  if (parent->hasFitsMosaic())
    str << "tile " << parent->findFits(ptr) << ';';

  listCoordSystem(str,sys,sky,ptr);
  str << ';';
}

void Marker::listPost(ostream& str, char delim)
{
  listProperties(str, delim);
  str << delim;
}

void Marker::listCiaoPre(ostream& str)
{
  if (!(properties&INCLUDE))
    str << '-';
}

