/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_config_agent.cpp,v 1.4 2004/07/02 14:06:36 suzhe Exp $
 *
 */

#define Uses_STL_IOSTREAM
#define Uses_C_LOCALE
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_CONFIG_PATH

#include "scim_private.h"
#include <scim.h>
#include <stdlib.h>

using namespace scim;
using std::cout;
using std::cerr;
using std::endl;

enum DataType
{
    DATA_TYPE_STRING,
    DATA_TYPE_INT,
    DATA_TYPE_DOUBLE,
    DATA_TYPE_BOOL,
    DATA_TYPE_STRING_LIST,
    DATA_TYPE_INT_LIST
};

enum Command
{
    DO_NOTHING,
    GET_DATA,
    SET_DATA,
    DEL_KEY
};

String get_param_portion (const String &str)
{
    String ret = str;
    return (ret.erase (ret.find_first_of (" \t\n\v="), ret.length() - 1));
}

String get_value_portion (const String &str)
{
    String ret = str;
    ret.erase (0, ret.find_first_of ("=") + 1);
    ret.erase (0, ret.find_first_not_of(" \n\t\v"));
    return (ret.erase (ret.find_last_not_of(" \t\n\v") + 1));
}


int main (int argc, char *argv [])
{
    //FIXME
    //If it's not static, then it will be crash with gconf config module.
    static ConfigModule  config_module;

    ConfigPointer        config;
    std::vector<String>  config_list;
    String               def_config;
    String               key;
    String               value;

    DataType             type = DATA_TYPE_STRING;
    Command              cmd = DO_NOTHING;
    String               app = String ("scim");

    int                  i;

    //get modules list
    scim_get_config_module_list (config_list);

    //Use simple Config module as default if available.
    if (config_list.size ()) {
        def_config = scim_global_config_read (SCIM_GLOBAL_CONFIG_DEFAULT_CONFIG_MODULE, String ("simple"));
        if (std::find (config_list.begin (),
                       config_list.end (),
                       def_config) == config_list.end ())
            def_config = config_list [0];
    } else {
        cerr << "No config module found.\n";
        return -1;
    }

    // parse command options
    i = 0;
    while (i<argc) {
        if (++i >= argc) break;

        if (String ("-h") == argv [i] ||
            String ("--help") == argv [i]) {
            cout << "Usage: " << argv [0] << " <option>...\n\n"
                 << "The options are:\n"
                 << "  -g, --get key        Get the value of this key.\n"
                 << "  -s, --set key=value  Set the value of this key.\n"
                 << "  -d, --del key        Delete the key and its data\n"
                 << "                       from user config file.\n"
                 << "  -t, --type type      The key value type, valid types are:\n"
                 << "                       string, int, double, bool, string-list,\n"
                 << "                       int-list. The default type is string.\n"
                 << "  -c, --config name    Use specified Config module,\n"
                 << "  -a, --app name       The application's name, default is scim.\n"
                 << "                       use simple module by default.\n"
                 << "  -h, --help           Show this help.\n";
            return 0;
        }

        if (String ("-g") == argv [i] ||
            String ("--get") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            if (cmd != DO_NOTHING) {
                cerr << "You can only do one thing at the same time!\n";
                return -1;
            }
            key = String (argv [i]);
            cmd = GET_DATA;
            continue;
        }

        if (String ("-s") == argv [i] ||
            String ("--set") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            if (cmd != DO_NOTHING) {
                cerr << "You can only do one thing at the same time!\n";
                return -1;
            }

            key = get_param_portion (String (argv [i]));
            value = get_value_portion (String (argv [i]));

            if (!key.length () || !value.length ()) {
                cerr << "Bad argument for option " << argv [i-1] << endl;
                return -1;
            }

            cmd = SET_DATA;
            continue;
        }

        if (String ("-d") == argv [i] ||
            String ("--del") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            if (cmd != DO_NOTHING) {
                cerr << "You can only do one thing at the same time!\n";
                return -1;
            }

            key = String (argv [i]);
            cmd = DEL_KEY;
            continue;
        }

        if (String ("-t") == argv [i] ||
            String ("--type") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            if (String (argv [i]) == "string")
                type = DATA_TYPE_STRING;
            else if (String (argv [i]) == "int")
                type = DATA_TYPE_INT;
            else if (String (argv [i]) == "double")
                type = DATA_TYPE_DOUBLE;
            else if (String (argv [i]) == "bool")
                type = DATA_TYPE_BOOL;
            else if (String (argv [i]) == "string-list")
                type = DATA_TYPE_STRING_LIST;
            else if (String (argv [i]) == "int-list")
                type = DATA_TYPE_STRING_LIST;
            else {
                cerr << "Bad argument for option " << argv [i-1] << endl;
                return -1;
            }
            continue;
        }

        if (String ("-c") == argv [i] ||
            String ("--config") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            def_config = String (argv [i]);
            continue;
        }

        if (String ("-a") == argv [i] ||
            String ("--app") == argv [i]) {
            if (++i >= argc) {
                cerr << "No argument for option " << argv [i-1] << endl;
                return -1;
            }
            app = String (argv [i]);
            continue;
        }

        cerr << "Unknown option " << argv [i] << endl;
        return -1;
    }

    if (cmd == DO_NOTHING || !key.length ()) {
        cerr << "What do you want to do?\n";
        return -1;
    }

    if (!config_module.load (def_config)) {
        cerr << "Failed to load config module " << def_config << endl;
        return -1;
    }

    config = config_module.create_config (app);

    if (config.null ()) {
        cerr << "Failed to create config object.\n";
        return -1;
    }

    // Get data
    if (cmd == GET_DATA) {
        bool ok = false;
        if (type == DATA_TYPE_STRING) {
            if ((ok = config->read (key, &value)) == true)
                cout << value << endl;
        } else if (type == DATA_TYPE_INT) {
            int intval;
            if ((ok = config->read (key, &intval)) == true)
                cout << intval << endl;
        } else if (type == DATA_TYPE_DOUBLE) {
            double doubleval;
            if ((ok = config->read (key, &doubleval)) == true)
                cout << doubleval << endl;
        } else if (type == DATA_TYPE_BOOL) {
            bool boolval;
            if ((ok = config->read (key, &boolval)) == true)
                cout << (boolval ? "true" : "false") << endl;
        } else if (type == DATA_TYPE_STRING_LIST) {
            std::vector <String> strlistval;
            if ((ok = config->read (key, &strlistval)) == true)
                cout << scim_combine_string_list (strlistval, ',') << endl;
        } else if (type == DATA_TYPE_INT_LIST) {
            std::vector <int> intlistval;
            if ((ok = config->read (key, &intlistval)) == true) {
                for (size_t i = 0; i<intlistval.size (); ++i) {
                    cout << intlistval [i];
                    if (i < intlistval.size () - 1)
                        cout << ",";
                }
                cout << endl;
            }
        }

        if (!ok) {
            cerr << "Failed to get key value.\n";
            return -1;
        }
    }
 
    // Set data
    else if (cmd == SET_DATA) {
        bool ok = true;
        if (type == DATA_TYPE_STRING) {
            ok = config->write (key, value);
        } else if (type == DATA_TYPE_INT) {
            int intval = strtol (value.c_str (), 0, 10);
            ok = config->write (key, intval);
        } else if (type == DATA_TYPE_DOUBLE) {
            double doubleval = strtod (value.c_str (), 0);
            ok = config->write (key, doubleval);
        } else if (type == DATA_TYPE_BOOL) {
            bool boolval = false;
            if (value == "true" || value == "True" || value == "TRUE" || value == "1")
                boolval = true;
            ok = config->write (key, boolval);
        } else if (type == DATA_TYPE_STRING_LIST) {
            std::vector <String> strlistval;
            scim_split_string_list (strlistval, value, ',');
            ok = config->write (key, strlistval);
        } else if (type == DATA_TYPE_INT_LIST) {
            std::vector <int> intlistval;
            std::vector <String> strlist;
            scim_split_string_list (strlist, value, ',');
            for (size_t i = 0; i<strlist.size (); ++i)
                intlistval.push_back (strtol (strlist[i].c_str (), 0, 10));
            ok = config->write (key, intlistval);
        }

        if (!ok) {
            cerr << "Failed to get key value.\n";
            return -1;
        } else {
            cout << "Set data success.\n";
            config->flush ();
        }
    }

    // Delete key
    else if (cmd == DEL_KEY) {
        if (config->erase (key)) {
            cout << "Delete key success.\n";
            config->flush ();
        } else {
            cerr << "Failed to delete the key.\n";
            return -1;
        }
    }

    return 0;
}

/*
vi:ts=4:ai:nowrap:expandtab
*/
