=head1 Cron.pm

The module responsible for op=cron, and the cron admin. When it's called, it
deals with running all of the cron jobs that need to be, and updates the
database concerning when it was last run. This way, one real cron job can run
all of the ones for Scoop.

=head1 Functions

=over 4

=cut

package Scoop;
use strict;
my $DEBUG = 0;

=item * get_crons()

Gets all the cron info out of the database and returns it as a hash with key of
name, value of hash ref of func, run_every, last_run, and enabled.

=cut

sub get_crons {
	my $S = shift;

	my ($rv, $sth) = $S->db_select({
		WHAT => 'name, func, run_every, last_run, enabled, is_box',
		FROM => 'cron'
	});
	my $ret = {};
	while (my $row = $sth->fetchrow_arrayref) {
		$ret->{$row->[0]} = {
			func      => $row->[1],
			run_every => $row->[2],
			last_run  => $row->[3],
			enabled   => $row->[4],
			is_box    => $row->[5]
		};
	}
	$sth->finish;

	return $ret;
}

=item * _cron_to_run()

Gets the cron info, then figures out which of them need to be run. Returns the
info for all of the runs that should be run.

=cut

sub _cron_to_run {
	my $S = shift;

	require Time::Local;

	my $crons = $S->get_crons();
	my $to_run = {};
	my $now = time();
	while (my($k, $v) = each %{$crons}) {
		next unless $v->{enabled};
		my $secs;
		if ($v->{last_run} && ($v->{last_run} ne '0000-00-00 00:00:00')) {
			$v->{last_run} =~ /(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/;
			# seconds, minutes, hours, days, months, years
			my @date = ($6, $5, $4, $3, $2, $1);
			$date[4] -= 1;     # month
			$secs = Time::Local::timelocal(@date);
		} else {
			$secs = 0;
		}

		if (($secs + $v->{run_every}) <= $now) {
			$to_run->{$k} = [$v->{func}, $v->{is_box}];
			warn "<<CRON>> running $k\n" if $DEBUG;
		}
	}

	return $to_run;
}

=item * _cron_run(to run)

Given a hash ref (the very same that C<_cron_to_run> returns), runs each of the
cron jobs and returns 1 if everything goes okay. Otherwise, returns a hash ref
with keys of cron jobs that failed, and values the error they failed with.

=cut

sub _cron_run {
	my $S = shift;
	my $to_run = shift || return;

	my $errors = {};
	my $did_error = 0;
	while (my($k, $v) = each %{$to_run}) {
		warn "[cron] Running $v->[0]\n" if ($DEBUG);

		my $ret;
		# check if it's a box or a function
		if ($v->[1]) {
			$ret = $S->run_box($v->[0]);
		} else {
			my $func = $v->[0];
			$ret = $S->$func();  # this works... cool
		}

		if ($ret != 1) {
			$errors->{$k} = $ret;
			$did_error = 1;
		}
	}

	return 1 unless $did_error;
	return $errors;
}

=item * cron()

Handles the cron op. Mainly dispatches to other methods.

=cut

sub cron {
	my $S = shift;

	my $now = time();
	my $to_run = $S->_cron_to_run();
	$S->cron_update_last_run($now, keys %{$to_run});
	my $errors = $S->_cron_run($to_run);

	if ($errors == 1) {
		my $ran = join(", ", keys %{$to_run});
		$S->{UI}->{BLOCKS}->{CONTENT} = "Cron finished\nRan: $ran";
	} else {
		my $content = "Errors:\n";;
		foreach (keys %{$errors}) {
			$content .= "$_: $errors->{$_}\n";
		}

		$content =~ s/</&lt;/g;
		$content =~ s/>/&gt;/g;
		$S->{UI}->{BLOCKS}->{CONTENT} = $content;
	}
}

=item * cron_update_last_run(time, crons)

For each of the crons passed to it, updates the last_run time to B<time>.

=cut

sub cron_update_last_run {
	my $S = shift;
	my $now = $S->_cron_date(shift);

	my $update = {};
	foreach (@_) {
		$update->{$_} = { last_run => $now };
	}

	return $S->save_cron($update);
}

sub _cron_date {
	my $S = shift;
	my $time = shift || time();

	my @now = localtime($time);
	$now[5] += 1900;   # year
	$now[4]++;         # month
	
	return sprintf("%04d-%02d-%02d %02d:%02d:%02d", $now[5], $now[4], $now[3], $now[2], $now[1], $now[0]);
}

=item * save_cron(crons)

Crons is a hash ref, with keys cron names, and each value is a hash ref of
fields to update for that one, and their values.

=cut

sub save_cron {
	my $S = shift;
	my $crons = shift || return;

	while (my($cron, $change) = each %{$crons}) {
		my $set;
		foreach (keys %{$change}) {
			$set .= ' AND ' if $set;   # don't add first first one
			$change->{$_} =~ s/'/''/g;
			$set .= qq|$_ = '$change->{$_}'|;
		}

		my ($rv, $sth) = $S->db_update({
			WHAT  => 'cron',
			SET   => $set,
			WHERE => "name = '$cron'"
		});
		$sth->finish;
	}

	return 1;
}

=item * add_cron({name, is_box, function, run_every, enabled})

Adds a new cron to the database using the args passed as a hash.

=cut

sub add_cron {
	my $S = shift;
	my %fields = (ref($_[0]) eq 'HASH') ? %{ $_[0] } : @_;

	my $vals;
	foreach my $f (qw(name is_box function run_every enabled)) {
		if ($f eq 'is_box' || $f eq 'enabled') {
			$fields{$f} = $fields{$f} ? 1 : 0;
		} elsif ($f eq 'run_every') {
			$fields{$f} = $S->time_relative_to_seconds($fields{$f});
		}

		$vals .= ', ' if $vals;
		$vals .= $S->{DBH}->quote($fields{$f});
	}

	my ($rv,$sth) = $S->db_insert({
		INTO   => 'cron',
		COLS   => 'name, is_box, func, run_every, enabled',
		VALUES => $vals
	});
	$sth->finish;

	return "Error: $DBI::errstr" unless $rv;
	return 1;
}

=item * rem_cron(cron, [...])

Given a list of crons, it removes all of them.

=cut

sub rem_cron {
	my $S = shift;
	my @crons = @_;

	my $where;
	foreach my $c (@crons) {
		$where .= ' OR ' if $where;
		$where .= 'name = ' . $S->{DBH}->quote($c);
	}

	my ($rv, $sth) = $S->db_delete({
		FROM  => 'cron',
		WHERE => $where
	});
	$sth->finish;

	return unless $rv;
	return 1;
}

=item * cron_change_enabled(cron, value)

Changes the enabled value of B<cron> to B<value>.

=cut

sub cron_change_enabled {
	my $S = shift;
	my $cron = shift || return;
	my $newval = shift;

	return unless defined($newval);

	my ($rv, $sth) = $S->db_update({
		WHAT => 'cron',
		SET  => "enabled = $newval",
		WHERE => "name = '$cron'"
	});
	$sth->finish;
}	

=item * edit_cron()

The basis of the cron admin tool, this function is called by Admin.pm, and then
serves only to call some other functions and return their result.

=cut

sub edit_cron {
	my $S = shift;
	my $msg = $S->_write_cron();
	my $form = $S->_get_cron_form($msg);
	return $form;
}

sub _write_cron {
	my $S = shift;

	return unless $S->{CGI}->param('write');
	return "Permission Denied" unless $S->have_perm('cron_admin');

	my $action = $S->{CGI}->param('action');

	if ($action eq 'add') {
		my $err;
		my %fields = (
			new_cron => 'name', new_is_box => 'is_box', new_func => 'function',
			new_re   => 'run_every', new_enabled => 'enabled'
		);
		my %to_pass;
		foreach my $f (qw(new_cron new_is_box new_func new_re new_enabled)) {
			if (my $v = $S->{CGI}->param($f)) {
				$to_pass{ $fields{$f} } = $v;
			} else {
				$err .= $err ? ', ' : 'The following fields need to be filled in: ';
				$err .= $fields{$f};
			}
		}
		return $err if $err;

		my $ret = $S->add_cron(%to_pass);

		return $ret unless $ret == 1;
		return "Successfully added $to_pass{name}";
	}

	my @which  = $S->{CGI}->param('which');
	my $crons  = $S->get_crons();

	my $error;
	my $to_run = {};
	foreach my $c (@which) {
		unless ($crons->{$c}) {
			$error .= ", " if $error;
			$error .= "cron '$c' unknown";
		}

		if ($action eq 'toggle_enabled') {
			my $changeto = ($crons->{$c}->{enabled}) ? 0 : 1;
			$S->cron_change_enabled($c, $changeto);
		} elsif ($action eq 'edit_run_every') {
			my $newval = $S->{CGI}->param($c . "_re");
			next unless $newval;
			$newval = $S->time_relative_to_seconds($newval);
			$S->save_cron({$c => { run_every => $newval }});
		} elsif ($action eq 'clear_last_run') {
			$S->save_cron({$c => { last_run => 0 }});
		} elsif ($action eq 'run') {
			$to_run->{$c} = [$crons->{$c}->{func}, $crons->{$c}->{is_box}];
		} elsif ($action eq 'remove') {
			$S->rem_cron($c);
		} else {
			$error = "action '$action' is unknown";
			last;
		}
	}

	if ($action eq 'run') {
		my $now = time();
		$error .= '.' if $error;
		my $run_errs = $S->_cron_run($to_run);
		unless ($run_errs == 1) {
			$error .= "<br>\n";
			foreach (keys %{$run_errs}) {
				$error .= "$_: $run_errs->{$_}<br>\n";
			}
		}
		$S->cron_update_last_run($now, keys %{$to_run});
	}

	return "Error doing $action: $error" if $error;
	return "Finished $action of selected crons";
}

sub _get_cron_form {
	my $S = shift;
	my $msg = shift;

	my $page = qq|
	<FORM ACTION="%%rootdir%%/" METHOD="POST">
	<INPUT TYPE="hidden" NAME="op" VALUE="admin">
	<INPUT TYPE="hidden" NAME="tool" VALUE="cron">
	<TABLE WIDTH="100%" BORDER="0" CELLPADDING="0" CELLSPACING="0">
		<TR BGCOLOR="%%title_bgcolor%%">
			<TD>%%title_font%%Edit Crons%%title_font_end%%</TD>
		</TR>|;
	$page .= qq|
		<TR>
			<TD>%%title_font%%<FONT COLOR="#ff0000">$msg</FONT>%%title_font_end%%</TD>
		</TR>| if $msg;
	$page .= qq|
		<TR>
			<TD>
			<TABLE WIDTH="100%" BORDER="0" CELLPADDING="0" CELLSPACING="0">
			<TR>
				<TD>&nbsp;</TD>
				<TD>%%norm_font%%<B>Cron</B>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<B>Box?</B>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<B>Function</B>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<B>Run Every</B>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<B>Last Run</B>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<B>Enabled</B>%%norm_font_end%%</TD>
			</TR>|;
	my $crons = $S->get_crons();
	foreach my $c (sort keys %{$crons}) {
		my $v = $crons->{$c};
		$v->{run_every} = $S->time_seconds_to_relative($v->{run_every});
		my $is_box_str  = $v->{is_box}  ? 'Yes' : 'No';
		my $enabled_str = $v->{enabled} ? 'Yes' : 'No';
		$page .= qq|
			<TR>
				<TD><INPUT TYPE="checkbox" NAME="which" VALUE="$c"></TD>
				<TD>%%norm_font%%$c%%norm_font_end%%</TD>
				<TD>%%norm_font%%$is_box_str%%norm_font_end%%</TD>
				<TD>%%norm_font%%$v->{func}%%norm_font_end%%</TD>
				<TD>%%norm_font%%<INPUT TYPE="text" NAME="${c}_re" VALUE="$v->{run_every}" SIZE="10">%%norm_font_end%%</TD>
				<TD>%%norm_font%%$v->{last_run}%%norm_font_end%%</TD>
				<TD>%%norm_font%%$enabled_str%%norm_font_end%%</TD>
			</TR>|;
	}
	$page .= qq|
			<TR>
				<TD>&nbsp;</TD>
				<TD>%%norm_font%%<INPUT TYPE="text" NAME="new_cron">%%norm_font_end%%</TD>
				<TD>%%norm_font%%<INPUT TYPE="checkbox" NAME="new_is_box" CHECKED>%%norm_font_end%%</TD>
				<TD>%%norm_font%%<INPUT TYPE="text" NAME="new_func">%%norm_font_end%%</TD>
				<TD>%%norm_font%%<INPUT TYPE="text" NAME="new_re" SIZE="10">%%norm_font_end%%</TD>
				<TD>&nbsp;</TD>
				<TD>%%norm_font%%<INPUT TYPE="checkbox" NAME="new_enabled" CHECKED>%%norm_font_end%%</TD>
			</TR>
			</TABLE>
			</TD>
		</TR>
		<TR>
			<TD>%%norm_font%%<B><BR>Action:</B>
			<INPUT TYPE="radio" NAME="action" VALUE="run">Force Run 
			<INPUT TYPE="radio" NAME="action" VALUE="toggle_enabled">Toggle Enabled 
			<INPUT TYPE="radio" NAME="action" VALUE="edit_run_every">Change Run Every 
			<INPUT TYPE="radio" NAME="action" VALUE="clear_last_run">Clear Last Run
			<INPUT TYPE="radio" NAME="action" VALUE="add">Add Cron
			<INPUT TYPE="radio" NAME="action" VALUE="remove">Remove Cron
			%%norm_font_end%%</TD>
		</TR>
		<TR>
			<TD>%%norm_font%%<BR><A HREF="%%rootdir%%/?op=cron">Run Cron Now</A>%%norm_font_end%%</TD>
		</TR>
		<TR>
			<TD>%%norm_font%%<BR>
			<INPUT TYPE="submit" NAME="write" VALUE="Save crons"> 
			<INPUT TYPE="reset" VALUE="Reset">
			%%norm_font_end%%</TD>
		</TR>
	</TABLE>
	</FORM>|;

	return $page;
}

=back

=head1 Cron Jobs

After this point, it's all code for cron jobs.

=over 4

=item * cron_rdf

Job to generate an RDF file for the site.

Vars used: rdf_file, rdf_image, rdf_days_to_show, rdf_max_stories,
rdf_copyright, max_rdf_intro, slogan, sitename, rdf_creator, rdf_publisher

=cut

sub cron_rdf {
	my $S = shift;

	use XML::RSS;

	my $rss = XML::RSS->new(encoding => $S->{UI}->{VARS}->{charset});

	my $url = "$S->{UI}->{VARS}->{site_url}$S->{UI}->{VARS}->{rootdir}/";
	$rss->channel(
		title => $S->{UI}->{VARS}->{sitename},
		link  => $url,
		description => $S->{UI}->{VARS}->{slogan},
		dc => {
#			date      => scalar localtime,
			date      => $S->_rss_datetime(),
			creator   => $S->{UI}->{VARS}->{rdf_creator}   || $S->{UI}->{VARS}->{sitename},
			publisher => $S->{UI}->{VARS}->{rdf_publisher} || $S->{UI}->{VARS}->{sitename},
			rights    => $S->{UI}->{VARS}->{rdf_copyright},
			language  => 'en-us'
		}
	);

	$rss->image(
		title => $S->{UI}->{VARS}->{sitename},
		url   => $S->{UI}->{VARS}->{rdf_image},
		link  => $url
	);

	$rss->textinput(
		title => "Search $S->{UI}->{VARS}->{sitename}",
		name  => "string",
		link  => $url . 'search/'
	);

	# really should use getstories or something here, but right now it's easier
	# to just directly fetch it (easier to port, that is)
	my $excl_sect_sql = ' AND ' . $S->get_disallowed_sect_sql('norm_read_stories', 'Anonymous');
	$excl_sect_sql = '' if( $excl_sect_sql eq ' AND ' );
	my $days_to_show = $S->{UI}->{VARS}->{rdf_days_to_show};
	my $max_stories  = $S->{UI}->{VARS}->{rdf_max_stories};
	my $ad_section = $S->{UI}->{VARS}->{ad_story_section} || 'advertisements';
	$ad_section = $S->dbh->quote($ad_section);
	my $where = "TO_DAYS(NOW()) - TO_DAYS(time) <= $days_to_show AND displaystatus >= 0 AND section != 'Diary' AND section != $ad_section $excl_sect_sql";
	my ($rv, $sth) = $S->db_select({
		WHAT     => 'title, dept, sid, introtext',
		FROM     => 'stories',
		WHERE    => $where,
		ORDER_BY => 'time DESC',
		LIMIT    => $max_stories
	});

	while (my $story = $sth->fetchrow_hashref) {
		$story->{introtext} =~ s/[\n\r]/ /g;
		foreach (qw(title introtext)) {
			# (crudely) remove HTML
			$story->{$_} =~ s/<.*?>//g;
			# unfilter &lt; and &gt;, so that we don't turn them into &amp;lt;
			$story->{$_} =~ s/&lt;/</g;
			$story->{$_} =~ s/&gt;/>/g;
			# filter &
			$story->{$_} =~ s/&/&amp;/g;
			# (re-)filter < and >
			$story->{$_} =~ s/</&lt;/g;
			$story->{$_} =~ s/>/&gt;/g;
		}

		my $max_intro = $S->{UI}->{VARS}->{rdf_max_intro};
		if ($max_intro) {
			my @intro = split(' ', $story->{introtext});
			@intro = splice(@intro, 0, $max_intro);
			$story->{introtext} = join(' ', @intro) . '...';
		}

		my $link = $url . "story/$story->{sid}";
		$rss->add_item(
			title => $story->{title},
			link  => $link,
			description => $story->{introtext}
		);
	}

	$sth->finish;

	$rss->strict(1);
	eval { $rss->save($S->{UI}->{VARS}->{rdf_file}) };
	if ($@) {
		my $error = $@;
		chomp($error);
		return $error;
	}

	return 1;
}

sub _rss_datetime {
	my ($s,$m,$h,$d,$mo,$y) = (gmtime(time))[0..5];

	$y  += 1900;
	$mo += 1;
	return sprintf("%04d-%02d-%02dT%02d:%02d:%02dZ", $y, $mo, $d, $h, $m, $s);
}

=item * cron_rdf_fetch()

Job to fetch RDF's from other sites, and put them in the db.

Vars used: use_rdf_feeds, rdf_http_proxy

=cut

sub cron_rdf_fetch {
	my $S = shift;

	return "RDF feeds not enabled" unless $S->{UI}->{VARS}->{use_rdf_feeds};

	my $channels = $S->rdf_channels();

	my $errored;
	foreach my $c (@{$channels}) {
		next unless $c->{enabled};

		my $ret = $S->rdf_fetch_and_store($c->{rdf_link}, $c->{rid});
		$errored .= "$c->{title} ($c->{rid}), " unless $ret == 1;
	}
	$errored =~ s/, $//;

	return 1 unless $errored;
	return "Couldn't fetch $errored";
}

=item * cron_sessionreap()

Job to clean up old sessions in the db, and remove them.

Vars used: keep_sessions_for (in format "<time> <unit>")

=cut

sub cron_sessionreap {
	my $S = shift;

	$S->{UI}->{VARS}->{keep_sessions_for} =~ /(\d+)\s*(.+)/;
	my ($length, $unit) = ($1, $2);

	my ($rv, $sth) = $S->db_delete({
		FROM  => 'sessions',
		WHERE => "last_accessed < DATE_SUB(NOW(), Interval $length $unit)"
	});
	my $ret = $rv ? 1 : "couldn't cleanup sessions";
	$sth->finish;

	return $ret;
}

=item * cron_digest()

Job to send out digest mailings.

Vars used: enable_story_digests, digest_subject, local_email

Blocks used: digest_storyformat, digest_headerfooter

=cut

# this is basically a direct port from cron.pl style to web-based style cron,
# so it may be a little rough.

sub cron_digest {
	my $S = shift;

	return "can't run because enable_story_digests is false"
		unless $S->{UI}->{VARS}->{enable_story_digests};

	my @date = localtime();

	my $users = $S->_cron_digest_fetch_userlist(\@date);
	my $data  = $S->_cron_digest_fetch_email_conts(\@date);

	my $errors;
	foreach my $user (@{$users}) {
		my $ret = $S->_cron_digest_send_email($user, $data);
		if ($ret != 1) {
			$errors .= ", " if $errors;
			$errors .= $ret;
		}
	}

	return $errors if $errors;
	return 1;
}

sub _cron_digest_fetch_userlist {
	my $S = shift;
	my $date = shift;

	my $where = "userprefs.prefname = 'digest' AND userprefs.uid = users.uid AND userprefs.prefvalue != 'Never' AND userprefs.prefvalue != ''";
	# don't get weekly digests if today isn't Sunday
	$where .= " AND userprefs.prefvalue != 'Weekly'" if $date->[6] != 0;
	# don't get monthly digests if it's not the first of the month
	$where .= " AND userprefs.prefvalue != 'Monthly'" if $date->[3] != 1;

	my ($rv, $sth) = $S->db_select({
		WHAT  => "userprefs.uid, userprefs.prefvalue, users.realemail",
		FROM  => "userprefs, users",
		WHERE => $where
	});

	my @users;
	while (my($uid, $prefval, $email) = $sth->fetchrow()) {
		push(@users, {uid => $uid, email => $email, freq => $prefval});
	}
	$sth->finish();

	return \@users;
}

sub _cron_digest_fetch_email_conts {
	my $S = shift;
	my $date = shift;
	my $data = {};

	$data->{Daily}   = $S->_cron_digest_getdata('Daily');
	$data->{Weekly}  = $S->_cron_digest_getdata('Weekly');
	$data->{Monthly} = $S->_cron_digest_getdata('Monthly');

	return $data;
}

sub _cron_digest_getdata {
	my $S = shift;
	my $frequency = shift;

	# Populate $rollback with the user preferences for digest frequency. Timed
	# in minutes! days ends lots of redundant stuff
	my $rollback;
	if   ($frequency eq 'Daily')   { $rollback = 60 * 24;      }
	elsif($frequency eq 'Weekly')  { $rollback = 60 * 24 * 7;  }
	elsif($frequency eq 'Monthly') { $rollback = 60 * 24 * 30; }

	# Get topic and section names
	# NOTE: I think there's a func to do this, but I don't know of it as I do
	# this, so I'll just port the SQL   -kas
	my ($rv, $sth) = $S->db_select({
		FROM => 'topics',
		WHAT => 'tid, alttext'
	});
	my $topics = {};
	while (my($tid, $text) = $sth->fetchrow()) {
		$topics->{$tid} = $text;
	}
	$sth->finish();

	# NOTE: same here  -kas
	($rv, $sth) = $S->db_select({
		FROM => 'sections',
		WHAT => 'section, title',
	});
	my $sections = {};
	while (my($sec, $title) = $sth->fetchrow()) {
		$sections->{$sec} = $title;
	}
	$sth->finish();

	my $data = "";

	my $ad_section = $S->{UI}->{VARS}->{ad_story_section} || 'advertisements';
	$ad_section = $S->dbh->quote($ad_section);
	($rv, $sth) = $S->db_select({
		FROM  => 'stories',
		WHAT  => 'sid, tid, aid, time, title, dept, introtext, section',
		WHERE => "displaystatus >= 0 AND section != $ad_section AND section != 'Diary' AND time >= DATE_SUB(NOW(), INTERVAL $rollback minute)",
		ORDER_BY => 'time desc'
	});
	my $count = 0;
	while (my $storydata = $sth->fetchrow_hashref()) {
		$count = 1;
		$data .= $S->_cron_digest_format_stories($storydata, $sections, $topics);
	}
	$sth->finish();

	if ($count) {
		return $data;
	} else {
		return undef;
	}
}

sub _cron_digest_format_stories {
	my $S = shift;
	my ($story, $sections, $topics) = @_;

	my $story_template = $S->{UI}->{BLOCKS}->{digest_storyformat};
	my $url = "$S->{UI}->{VARS}->{site_url}$S->{UI}->{VARS}->{rootdir}/";
	$story->{url} = "${url}story/$story->{sid}";

	$story->{tid} = $topics->{ $story->{tid} };
	$story->{section} = $sections->{ $story->{section} };

	# fix up the introtext
	# Replace hrefs with plaintext
	$story->{introtext} =~ s/<A\s+HREF\s*=\s*['"]([^'"]*?)['"]\s*>\s*(.*?)\s*<\/A>/$2 [$1]/gi;

	$story->{introtext} =~ s/[\n\r]/ /g;
	$story->{introtext} =~ s/<P>/\n\n/g;
	$story->{introtext} =~ s/<BR>/\n/g;
	$story->{introtext} =~ s/<.*?>//g;

	require Text::Wrap;
	$Text::Wrap::columns = 75;
	$story->{introtext} = Text::Wrap::wrap('', '', $story->{introtext});

	foreach my $key (keys %{$story}) {
		my $find = "__${key}__";
		$story_template =~ s/$find/$story->{$key}/g;
	}

	return $story_template;
}

sub _cron_digest_send_email {
	my $S = shift;
	my ($user, $data) = @_;
	my $nick = $S->get_nick_from_uid($user->{uid});
	
	my $email_header_footer = $S->{UI}->{BLOCKS}->{digest_headerfooter};

	$email_header_footer =~ s/__FREQUENCY__/$user->{freq}/g;
	$email_header_footer =~ s/__USERID__/$user->{uid}/g;
	$email_header_footer =~ s/__NICKNAME__/$nick/g;


	return unless $data->{ $user->{freq} };

	my $mail = join("", $email_header_footer, $data->{ $user->{freq} }, "\n\n", $email_header_footer);
	$mail =~ s/\\n/\n/g;

	if ($mail) {  # if the body is empty, for some reason, don't send
		my $ret = $S->mail($user->{email}, $S->{UI}->{VARS}->{digest_subject}, $mail);
		return 1 if $ret == 1;
		return "couldn't send digest for $user->{email}";
	}
}

=back

=cut

1;
