=head1 Utility.pm

This file contains general utilities that are of use to more than one module in
scoop. Thus they have been put in here for easy maintenance.

=cut

package Scoop;
use strict;
use vars qw(%Escapes $Escapes_built);

my $DEBUG = 0;

=over 4

=item *
save_file_upload($advertiser_id,$tmpl)

This will get a file though the Apache::Request object, and save it
to the directory $S->{UI}->{VARS}->{ad_files_base}/$advertiser_id/
It returns the filename that it saved it as, the file size in bytes,
and an error message.  The error message will only contain something
if there was an error.

$tmpl is the name of the ad template that this file will display
under (used to check max_file_size)

=cut

sub save_file_upload {
	my $S = shift;
	my $adver_id = shift;
	my $tmpl = shift;

	my $base_path = $S->{UI}->{VARS}->{ad_files_base};
	$base_path =~ s/\/$//;
	my ( $filename, $size, $errmsg );

	# first check to see if the output directory exists, if not, create it.
	unless( -d "$base_path/$adver_id" ) {
		return('','',"Couldn't create directory to save upload") unless( $S->make_ad_path($adver_id) );
	}

	my $upload = $S->{APR}->upload;

	# don't use their filename!  bad bad bad!
	my $random_fn = $S->rand_stuff(7);
	($filename, $size) = ($upload->filename, $upload->size);
	$filename =~ /\.(\w+)$/;
	$random_fn = "${random_fn}.$1";
	my $abs_filename = "$base_path/$adver_id/$random_fn";
	warn "abs_filename is $abs_filename";

	if( $size eq 0 ) {
		warn "returning because of no file to upload";
		return( '', 0, '' );
	}

	warn "name : " . $upload->name if $DEBUG;
	warn "filename : " . $random_fn if $DEBUG;
	warn "size : " . $size if $DEBUG;
	warn "type : " . $upload->type if $DEBUG;

	# we need to check that they don't upload a file that is too big,
	# so get the max kbyte count, and start counting chars
	# within the while() loop.  If they get too big, break, remove
	# the file, and throw an error
	my $tmpl_info = $S->get_ad_tmpl_info($tmpl);
	my $max_bytes = $tmpl_info->{max_file_size} * 1024;
	if( $size > $max_bytes ) {
		return( '', '', "Sorry, but you can't upload files larger than <b>$max_bytes</b> bytes");
	}

	unless( open( OUTFH, ">$abs_filename" ) ) {
		warn "Couldn't open file $abs_filename: $!";
		return ('','',"Couldn't create file to save upload");
	}

	warn "Saving new file as $abs_filename" if $DEBUG;
	my $uploadfh = $upload->fh;
	while( my $line = <$uploadfh> ) {
		print OUTFH $line;
	}
	close( OUTFH ) or warn "Couldn't close file $abs_filename : $!";
	close( $uploadfh ) or warn "Couldn't close upload temporary file: $!";

	chmod( 0644, $abs_filename );

	warn "Finished saving file $abs_filename" if $DEBUG;

	return( $random_fn, $size, $errmsg );
}


=item *
make_ad_path($adver_id)

This makes all of the directories necessary to store advertising files

=cut

sub make_ad_path {
	my $S = shift;
	my $adver_id = shift;

	my $path = $S->{UI}->{VARS}->{ad_files_base};
	$path =~ s/\/$//;
	$path .= "/$adver_id";

	warn "Making directory $path\n" if $DEBUG;
	unless( mkdir ($path, 0755) ) {
		warn "Can't create directory $path: $!";
		return 0;
	}

	return 1;
}

=item * time_absolute_to_seconds(string)

Converts a string in the form "yyyy-mm-dd hh:mm:ss" to second since the epoch
and returns that.

=cut

sub time_absolute_to_seconds {
	my $S = shift;
	my $str = shift;

	$str =~ /(\d{4})-(\d{2})-(\d{2}) (\d{2}):(\d{2}):(\d{2})/;

	my @date = ($6, $5, $4, $3, $2, $1);
	$date[4] -= 1;  # month

	require Time::Local;
	return Time::Local::timelocal(@date);
}

=item * time_relative_to_seconds(string)

Takes a string which represents an interval of time and changes it into
seconds. Example: "1h30m25s" would be 1 hour, 30 minutes, and 25 seconds from
now, or 5425 seconds.

The useable characters are s (seconds), m (minutes), h (hours), d (days), w
(weeks), M (months), and Y (years).

When parsing, spaces are removed, a trailing plus sign (if any) is removed, and
an 's' is added to the end if the last character is a number.

=cut

sub time_relative_to_seconds {
	my $S = shift;
	my $str = shift || return;

	$str =~ s/\s//g;
	$str =~ s/^\+//;
	$str .= 's' if $str =~ /\d$/;

	my $parts = $S->_time_parts_hash();
	my $suffs = join('', '[', keys %{$parts}, ']');

	my $secs = 0;
	while ($str =~ /(\d+)($suffs)/go) {
		$secs += $1 * $parts->{$2};
	}

	return $secs;
}

=item time_seconds_to_relative(integer)

Takes a number of seconds and converts it into the relative form as used by
C<time_relative_to_seconds>. The individual parts will be seperated by spaces.

=cut

sub time_seconds_to_relative {
	my $S = shift;
	my $secs = shift || return;

	my ($divs, $suffix) = $S->_time_parts_array();

	my $idx = 0;
	my @string_parts;

	do {
		my $newval = int($secs/$divs->[$idx]);
		push(@string_parts, ' ', $newval, $suffix->[$idx]) if $newval > 0;
		$secs -= $newval * $divs->[$idx];
		$idx++;
	} while ($secs > 0);

	shift(@string_parts);

	return join('', @string_parts);
}

sub _time_parts_hash {
	my $S = shift;

	my $parts = {};
	my ($divs, $suffs) = $S->_time_parts_array();

	for (my $idx = 0; $idx < scalar(@{$suffs}); $idx++) {
		$parts->{ $suffs->[$idx] } = $divs->[$idx];
	}

	return $parts;
}

sub _time_parts_array {
	return (
		[ 60*60*24*365, 60*60*24*30, 60*60*24*7, 60*60*24, 60*60, 60, 1 ],
		[ qw( Y M w d h m s ) ]
	);
}

=item *
count_words($string)

This returns the number of words in a string passed to it.  Useful for checking the
number of words in a post or story.  Disregards all html, not the best way 
though, it should probably use HTML::Parser, but for now this is fine.

=cut

sub count_words {
	my $S = shift;
	my $string = shift;

	# First filter HTML for validity.
	$string = $S->filter_comment($string);

	# note that the html checker can result in errors, but we ignore them
	
	# get rid of all html... not very well. someday use HTML::Parser instead
	# the /s is so that . can span newlines
	$string =~ s/<[^>]*?>//sg;
	
	# get all of the words into an array
	my $word_c = my @word_a = split(/ /, $string );

	# return the size of the array
	return $word_c;
}

=item *
count_chars($string)

Returns the number of characters in the string, disregarding html.  Same html parsing problem
as above.

=cut

sub count_chars {
	my $S = shift;
	my $string = shift;
	
	# First filter HTML for validity.
	$string = $S->filter_comment($string);

	# note that the html checker can have errors, but we ignore them

	# get rid of all html 
	# the /s is so that . can span newlines
	$string =~ s/<[^>]*?>//sg;
	
	# get all of the characters into an array
	my $char_c = my @char_a = split( //, $string );

	# return the size of the array
	return $char_c;
}


=item *
make_url($op, $what)

This makes a url, it appears to be unfinished, since there are scalars that are unused.
odd.  We'll see if we can finish it up...   -A 2.23.00

=cut

sub make_url {
    my $S = shift;
    my $op = shift;
    my $what= shift;
    
    my %easy_ops = ("section"		=> 1,
					"topic"			=> 1,
					"displaystory"	=> 1 );

    my %tool_names = (	"admin"			=> "tool",
						"user"			=> "tool",
						"displaystory"	=> "sid",
						"special"		=> "page",
						"section"		=> "section");
		       
    my $path = '';
    my $query = '';

	# doesn't look like this will ever happen...
    if( $S->{UI}->{VARS}->{use_easy_urls} && $easy_ops{$op} ) {

		# this is a technically unnecessary kludge
		$op =~ s/displaystory/story/;
		#ok, done

		$path = "$op/$what";

    } else {

		$query .= sprintf( 'op=%s', $op );
		$query .= sprintf( ';%s=%s;', $tool_names{$op}, $what ) if( $what );

	}
   
	$query .= join( ';', @_ );
	$query = '?' . $query if $query;

	my $url = qq|%%rootdir%%/$path$query|;

	return $url;
}


=item *
make_anchor($op, $what)

This just calls make_url, and encapsulates its return value in <A HREF="">

=cut

sub make_anchor {
    my $S = shift;
    my $url = $S->make_url( @_ );

    return qq|<A HREF="$url">|;
}

=item *
filter_url( $string )

Filters a url for display.  Filters the same as for comment
subjects but ignores the & -> &amp; conversion

=cut

sub filter_url {
	my $S = shift;
	my $url = shift;

	$url = $S->filter_subject($url);
	$url =~ s/&amp;/&/g;

	return $url;
}

=item *
urlify( $string )

This escapes a string, useful for providing links when some of the args in
contain non-alphanumeric characters. Escapes everything except letters,
numbers, underscore, period, and dash.

=cut

sub urlify {
	my $S = shift;
	my $string = shift;

	$S->_build_url_escapes() unless $Escapes_built;

	$string =~ s/([^A-Za-z0-9_.-])/$Escapes{$1}/g;

	return $string;
}

sub _build_url_escapes {
	foreach (0..255) {
		$Escapes{chr($_)} = sprintf("%%%02X", $_);
	}

	$Escapes_built = 1;
}

=item * deurlify( $string )

Does the opposite of C<urlify>, changing hex-encoded characters into their
actual characters.

=cut

sub deurlify {
	my $S = shift;
	my $string = shift;

	$string =~ tr/+/ /;
	$string =~ s/%([a-fA-F0-9]{2})/chr(hex($1))/eg;

	return $string;
}

sub make_formkey {
	my $S = shift;
	
	my $new_key = $S->{SESSION_KEY}.':'.$S->{REMOTE_IP}.':';
	$new_key .= $S->_random_pass();
	
	return $new_key;
}

sub set_new_formkey {
	my $S = shift;

	my $keys = $S->session('formkeys') || [];

	unshift(@{$keys}, [$S->make_formkey(), time()]);

	$S->session('formkeys', $keys);
	
	return;
}

sub check_formkey {
	my $S = shift;

	# Get the key sent in by the form
	my $input_key = $S->{CGI}->param('formkey');
	return 1 unless $input_key;

	# get the list of formkeys
	my $keys = $S->session('formkeys') || [];

	# look at all of the formkeys, checking for both the specified formkey, and
	# for expired formkeys
	my $found = 0;
	my $time = time();
	my $timeout = $S->{UI}->{VARS}->{formkey_timeout};
	foreach my $k (0..$#{$keys}) {
		if (!$found && ($keys->[$k]->[0] eq $input_key)) {
			$found = 1;
			splice(@{$keys}, $k, 1);
		} elsif (($time - $keys->[$k]->[1]) > $timeout) {
			splice(@{$keys}, $k, 1);
		}
	}
	return 0 unless $found;

	# save the (most likely) changed list of keys
	$S->session('formkeys', $keys);
	
	return 1;
}


sub get_formkey_element {
	my $S = shift;
	
	# Make sure there's a current formkey!
	my $keys = $S->session('formkeys') || [];

	# create a new one if there isn't
	unless (@{$keys}) {
		$S->set_new_formkey();
		$keys = $S->session('formkeys');
	}
	
	my $element = qq|<INPUT TYPE="hidden" NAME="formkey" VALUE="$keys->[0]->[0]">|;

	# make a new formkey, since this one is now used
	$S->set_new_formkey();
	
	return $element;
}

sub make_blowfish_formkey {
	my $S = shift;
	my $now = time();
	my $cipher = new Crypt::CBC($S->{CONFIG}->{site_key}, 'Crypt::Blowfish');
	
	my $key = $cipher->encrypt_hex("$S->{REMOTE_IP}:$now");
	
	return $key;
}

sub check_blowfish_formkey {
	my $S = shift;
	my $key = shift;
	
	my $cipher = new Crypt::CBC($S->{CONFIG}->{site_key}, 'Crypt::Blowfish');
	
	my $d_key = $cipher->decrypt_hex($key);
	my $now = time();
	
	my ($ip, $stamp) = split /:/, $d_key;
	my $time_diff = $now - $stamp;
	
	return 1 if (($ip eq $S->{REMOTE_IP}) && ($time_diff < 1800));
	return 0;
}

=item * plaintext_format( $comment, $noescapetags )

Performs filtering for plain text posting mode on the given comment/story
text. If $noescapetags evaluates to true, then HTML special characters
will not be touched

=cut

sub plaintext_format {
	my $S = shift;
	local $_ = shift;
	my $noescapetags = shift || 0;

	# Remove excess newlines from the front and end of the text
	s#^\n\s*\n##gs;
	s#\n\s*\n$##gs;

	# Perform standard plain-old-text conversions
	unless($noescapetags) {
	s#&#&amp;#g;
	s#"#&quot;#g;
	s#<#&lt;#g;
	s#>#&gt;#g;
	}
	s#\r##gs;
	s#\n\s*\n#<p>\n#gs;
	s#(?<!<p>)\n#<br>\n#gs;
	s#\t#&nbsp;&nbsp;&nbsp; #g;
	s#\xA0#&nbsp;#g;
	s#^ #&nbsp;#gm;
	s#  # &nbsp;#g;
	s#[\x00-\x08\x0B-\x1F]##g; # Nuke control characters
	# Change remaining non-ASCII chars to entities
	s!([^\n\t\x20-\x7E])!'&#'.ord($1).';'!ge unless $noescapetags;
	return $_;
}

=item * auto_format( $comment )

Performs plaintext formatting, and then calls the auto formatting routines
(for bold/italics, links, lists, etc.)

=cut

sub auto_format {
	my $S = shift;
	local $_ = shift;

	# Since we'll be using high-bit characters to mark escaped characters,
	# turn existing ones into entities here.
	s!([\x80-\xFF])!'&#'.ord($1).';'!ge;
	# Escape significant characters preceded by a backslash
	s#\\\\#\xDC#g; # Escape double backslashes first
	s#\\<#\x81#g;
	s#\\>#\x82#g;
	s#\\&#\x83#g;
	s#\\"#\x84#g;
	s#\\(\S)#chr(ord($1)|0x80)#ge; # Mark the high bit
	# Change non-HTML-involved &< into entities
	s/&(?![A-Za-z0-9#]+;)/&amp;/g;
	s#<(?![A-Za-z/])#&lt;#g;
	# Perform plaintext formatting
	$_ = $S->plaintext_format($_, 1);
	# Run the URL linkifier here so that clean_html's word breaking doesn't
	# mess things up
	$_ = $S->_auto_linkify_urls($_);
	# Clean up HTML tags
	my $comment_ref = $S->html_checker->clean_html(\$_);
	$_ = $$comment_ref;

	# Make non-HTML-involved <>&" easier to sniff out.
	s#&lt;#\x01#g; # \x01 == < for now
	s#&gt;#\x02#g; # \x02 == > for now
	s#&amp;#\x03#g; # \x03 == & for now
	s#&quot;#\x04#g; # \x04 == " for now

	$_ = $S->_auto_bold_italic($_);
	$_ = $S->_auto_create_ul($_);
	$_ = $S->_auto_create_ol($_);

	# Switch back the marked characters
	s#([\x80-\xFF])#chr(ord($1)&0x7F)#ge;
	s#\x01#&lt;#g;
	s#\x02#&gt;#g;
	s#\x03#&amp;#g;
	s#\x04#&quot;#g;

	return $_;
}

=item * _auto_linkify_urls( $comment )

Turns URLs into HTML links for plaintext mode.

=cut

sub _auto_linkify_urls {
	my $S = shift;
	local $_ = shift;
	my $url_regex = '(?:http|ftp|file)://[^\s<]+?';
	my $url_regexg = '(?:http|ftp|file)://[^\s<]+'; # greedy
	
	# Mark URLs that are already in HTML attrs or links so we don't linkify
	# them
	s#(<[^>]+="[^">]*)($url_regex)#$1\x00$2#gso;
	s#(<a\s[^>]*href=[^>]*>[^<]*)($url_regex)#$1\x00$2#gso;

	# Grab expressions in brackets ('[]', '{}', or '<>', not '()')
	# and if they end in a URL, linkify them.
	s#\[([^\[][^]]+?)(?:\s|&nbsp;)*?($url_regex)\]#<a href="\x00$2">$1</a>#gmsio;
	s#{([^{][^}]+?)(?:\s|&nbsp;)*($url_regex)}#<a href="\x00$2">$1</a>#gmsio;

	s#\[($url_regexg)(?:\s|&nbsp;)+([^\[][^]]+?)\]#<a href="\x00$1">$2</a>#gmsio;
	s#{($url_regexg)(?:\s|&nbsp;)+([^{][^}]+?)}#<a href="\x00$1">$2</a>#gmsio;

	# Linkify all the remaining naked URLs
	s#([^\x00]|^)($url_regex)(?=[.!?_*=]?[\s\n<()\[\]{}\x01\x02]|$)#$1<a href="$2">$2</a>#gmio;
	# Remove placeholder chars
	s#\x00##gs;
	return $_;
}

=item * _auto_bold_italic( $comment )

Makes bold, code, and italic HTML tags for text between asterisks, equal signs
and underscores, respectively.

=cut

sub _auto_bold_italic {
	my $S = shift;
	local $_ = shift;

	# Only match [*=_/] when there's no space between them and the affected
	# text
	s#(?<![A-Za-z0-9])\*(\S|\S.*?\S)\*(?=[^A-Za-z0-9]|<br>|<p>|$)#<strong>$1</strong>#gs;
	s#(?<![A-Za-z0-9])=(\S|\S.*?\S)=(?=[^A-Za-z0-9]|<br>|<p>|$)#<code>$1</code>#gs;
	s#(?<![A-Za-z0-9])_(\S|\S.*?\S)_(?=[^A-Za-z0-9]|<br>|<p>|$)#<em>$1</em>#gs;
	# We have to make sure we don't match the / in closing tags or URLs
	s#(?<![A-Za-z0-9/:<])/(\S|\S.*?[^\s<:/])/(?=[^A-Za-z0-9]|<br>|<p>|$)#<em>$1</em>#gs;

	return $_;
}

=item * _auto_create_ul( $comment )

Creates bulleted lists from series of lines beginning in '* '.

=cut

sub _auto_create_ul {
	my $S = shift;
	local $_ = shift;

	# Match series of lines (at least two) beginning in '* ' or '- ',
	# eating up any <br> or <p> tags inserted by plaintext mode
	s#((?:<(?:br|p)>\n)?(?:^[*\-+]\s+.*(?:\n|$)){2,})#
		my $a = $1;
		$a =~ s!^[*\-+]!<li>!gm;
		$a =~ s!<br>(<|$)!$1!gm;
		$a =~ s!<p>(<|$)!$1!gm;
		"\n<ul>\n".$a."\n</ul>\n"
	#gme;
	return $_;
}

=item * _auto_create_ol( $comment )

Creates numbered lists from series of lines beginning in 'n.' or 'n)', where
n is a number of decimal radix.

=cut

sub _auto_create_ol {
	my $S = shift;
	local $_ = shift;

	# Match series of lines (at least two) beginning with numbers,
	# eating up any <br> or <p> tags inserted by plaintext mode
	s#((?:<(?:br|p)>\n)?(?:^[0-9]+[.)]?\s*.*(?:\n|$)){2,})#
		my $a = $1;
		$a =~ s!^([0-9]+)[.)]?!<li value="$1">!gm;
		$a =~ s!<br>(<|$)!$1!gm;
		$a =~ s!<p>(<|$)!$1!gm;
		"\n<ol>\n".$a."\n</ol>\n"
	#gme;
	return $_;
}

=back

=cut

=item * js_quote( $text )

Quotes the given text to make it safe for interpolation into a javascript
string.

=cut

sub js_quote {
	my $S = shift;
	my $string = shift;

	# Interpolate the string here, so that %%replaced%% bits get escaped
	# as well
	$string = $S->interpolate($string, $S->{UI}->{BLOCKS});
	$string = $S->interpolate($string, $S->{UI}->{VARS});

	$string =~ s#\\#\\\\#g;
	$string =~ s#\n#\\n#g;
	$string =~ s#\r#\\r#g;
	$string =~ s#\t#\\t#g;
	$string =~ s#'#\\'#g;
	$string =~ s#"#\\"#g;
	$string =~ s#</SCRIPT>#&lt;/SCRIPT&gt;#g;
	$string =~ s#([^\x20-\x7E])#'\\x'.sprintf("%02x",ord($1))#ge;
	return $string;
}

1;
