/***************************************************************************
                          editor.h  -  description
                             -------------------
    begin                : Sat Sep 30 2000
    copyright            : (C) 2000 by Juan Sebastian Linietsky
    email                : reduz@anime.com.ar
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef EDITOR_H
#define EDITOR_H

#include "song.h"
#include "midioutdevicemanager.h"

/**Here's the editor class!
all editing goes throught this.
  *@author Juan Sebastian Linietsky
  */

#define CHARACTERS_PER_COLUMN 17
#define SEPARATOR_WIDTH font_width
#define ROW_NUMBER_DECIMALS 10
#define ROW_SEPARATOR_PIXELS 1
#define CLIPBOARD_MAX_COLUMNS 64

#define TRACKER_REDRAW_ROW 1
#define TRACKER_REDRAW_COLUMN 2
#define TRACKER_REDRAW_TOP 4
#define TRACKER_REDRAW_PLAYING_ROW 8
#define TRACKER_REDRAW_ALL 16


class Pattern_Editor {
private:

	struct Note_Event {

		char note;
		char velocity;
		int column;
	};


	int column_offset,row_offset;

	int visible_columns,visible_rows;
	int window_width,window_height;
	int cursor_x,cursor_y,cursor_field;
	int playing_row,playing_row_old;
	int font_height,font_width;
	bool playing_new_row;
	bool follow_song;
	bool show_global_row;

	bool playing;

        // cursor and data insert locking

	bool locked_to_pattern;

	// internal note fields entering

	bool insert_note_at_cursor(char keynote);
	void insert_octave_at_cursor(Uint8 octave);
	void insert_volume_at_cursor_field1(Uint8 volume);
	void insert_volume_at_cursor_field2(Uint8 volume);
	void insert_command_at_cursor(char command);
	void insert_parameter_at_cursor_field1(Uint8 parameter);
	void insert_parameter_at_cursor_field2(Uint8 parameter);
	void insert_controller_set_at_cursor(char controller_command,int field);
	void insert_controller_set_at_cursor_field2(char controller_command);
	void insert_controller_set_at_cursor_field3(char controller_command);
	void insert_controller_value_at_cursor_field1(Uint8 value);
	void insert_controller_value_at_cursor_field2(Uint8 value);	

	// editing variables

	int default_octave;
	
	int last_volume;
	int last_command;
	int last_parameter;
	int last_controller_set;
	int last_controller_value;
	int last_note;

	bool volume_mask;
	bool command_parameter_mask;
	bool controller_mask;

	int last_volume_scale;

	int cursor_spacing;

	// orderlist editing

	int orderlist_cursor_x,orderlist_cursor_y;
	//selection

	Pattern *clipboard;

	int selection_begin_x,selection_begin_y,selection_end_x,selection_end_y;
	bool selection_active;
	int shift_selection_begin_x,shift_selection_begin_y,shift_selection_end_x,shift_selection_end_y;
	bool shift_selection_in_progress;
        // drawiong optimization

	char redraw_flags;

	//keyboard internationalization.
	char noteincr[29];

	int keyboard_type;
	int GiveNoteIncrement(char key);

	vector<Note_Event> midi_input;

public:

	MidiOutDeviceManager *midiout;
	Song *song;
        //string retrieving
	string get_note_string(int column,int row);
	string get_number_string(int row);
	Note get_note_from_string(string p_note_str);	

	//sizes retrieving
	int get_column_width() { return font_width*CHARACTERS_PER_COLUMN+SEPARATOR_WIDTH; }
	int get_row_height() { return get_font_height()+ROW_SEPARATOR_PIXELS; }
	int get_separator_width() { return SEPARATOR_WIDTH; }
	int left_numbers_width() { return (show_global_row?ROW_NUMBER_DECIMALS:4)*font_width+SEPARATOR_WIDTH; }
	int top_data_height() { return get_row_height()*2; }

	// flags/requests to the interface to speedup enditing

	bool interface_request_redrawatcursor;
	bool interface_request_redraw_all;


	//appareance

	int get_visible_columns();
	int get_visible_rows();
	void set_font_height(int value) { font_height=value; };
	int get_font_height() { return font_height; }
	void set_font_width(int value) { font_width=value; };
	int get_font_width() { return font_width; }
	void set_window_width(int value) { window_width=value; };
	int get_window_width() { return window_width; }
	void set_window_height(int value) { window_height=value; };
	int get_window_height() { return window_height; }
	void set_column_offset(int value) { column_offset=value; };
	int get_column_offset() { return column_offset; }
	void set_row_offset(int value) { row_offset=value; };
	int get_row_offset() { return row_offset; };

	int get_cursor_x() { return cursor_x; };
	int get_cursor_y() { return cursor_y; };

	int get_cursor_pattern();
	int get_cursor_track();
	int get_cursor_column();
	int get_cursor_row();

	int get_cursor_field() { return cursor_field; };

	bool get_volume_mask() { return volume_mask; };
	bool get_command_parameter_mask() { return command_parameter_mask; };
	bool get_controller_mask() { return controller_mask; };

	void default_octave_raise();
	void default_octave_lower();
	char get_default_octave();

	void add_column_to_current_track();
	int remove_column_to_current_track(); // -1 is normal / 0+ means track must be removed (no columns left)

	char get_note_from_key(int key);
	//editing


	int GetUpperLimit();
	int GetLowerLimit();
	int GetInsertLimit();

	void lock_to_pattern(bool p_lock) { locked_to_pattern=p_lock;};
	bool is_locked_to_pattern() { return locked_to_pattern;}
	//cursor
	void cursor_step_up(int amount=1);
	void cursor_step_down(int amount=1);
	void cursor_move_up(int amount);
	void cursor_move_left(int amount);
	void cursor_move_down(int amount);
	void cursor_move_right(int amount);
	void cursor_move_field_right();
	void cursor_move_field_left();
	void cursor_page_up();
	void cursor_page_down();
	void cursor_home();
	void cursor_end();
	void cursor_tab();	
	void cursor_back_tab();
	void cursor_insert();
	void cursor_delete();
	void toggle_mask_at_cursor();
	void clear_field_at_cursor();
	void cursor_next_pattern();
	void cursor_prev_pattern();

	void adjust_view_to_cursor();

	//selection
	void normalize_selection();
	void mark_block_begining();
	void mark_block_end();
	void selection_release();
	void mark_column_all();

	int get_selection_begin_x () { return selection_begin_x; };
	int get_selection_begin_y () { return selection_begin_y; };
	int get_selection_end_x () { return selection_end_x; };
	int get_selection_end_y () { return selection_end_y; };
	bool is_selection_active () { return selection_active; };
	
	void shift_selection_begin();
	void shift_selection_end();
	void shift_selection_update();

	//selection clipboard

	void selection_copy();
	void selection_paste_overwrite();
	void selection_paste_insert();
	void selection_paste_mix();
	//selection modifiers
	void selection_raise();
	void selection_lower();
	void selection_zap();
	void selection_controller_ramp();
	void selection_volume_ramp();
	void selection_parameter_ramp();
	void selection_set_volume_mask();
	void selection_wipe_stray_volumes();
	void selection_volume_scale(int p_percent);
	int get_last_volume_scale() { return last_volume_scale; };
	
	//misc

	void pattern_expand();
	void pattern_shrink();

	/* ORDERLIST KEY PRESS HANDLERS */

	void orderlist_move_up();
	void orderlist_move_down();
	void orderlist_move_left();
	void orderlist_move_right();
	void orderlist_page_up();
	void orderlist_page_down();

	void orderlist_insert_value(int number);
	void orderlist_add_separator();
	void orderlist_clear_order();
	void orderlist_insert();
	void orderlist_delete();
	void orderlist_increase_order();

	/* ORDERLIST INFO RETRIEVING */

	int orderlist_get_cursor_x() { return orderlist_cursor_x; };
	int orderlist_get_cursor_y() { return orderlist_cursor_y; };
	string orderlist_get_cursor_string(int p_order);
        //inserting
        bool press_key_at_cursor(int key); // A-Z (caps) / 0-9
	void press_note_off_at_cursor();
	void perform_raise_at_cursor();
	void perform_lower_at_cursor();

        //drawing optimization

	bool flag_redraw_row() { return (redraw_flags & TRACKER_REDRAW_ROW); };
	bool flag_redraw_all() { return (redraw_flags==0) || (redraw_flags & TRACKER_REDRAW_ALL); };
	bool flag_redraw_all_explicit() { return (redraw_flags & TRACKER_REDRAW_ALL); };
	bool flag_redraw_top() { return (redraw_flags & TRACKER_REDRAW_TOP); };
	bool flag_redraw_playing_row() { return (redraw_flags & TRACKER_REDRAW_PLAYING_ROW); };

	void clear_redraw_flags() { redraw_flags=0; };

	void set_flag_redraw_row() { redraw_flags|=TRACKER_REDRAW_ROW; };
	void set_flag_redraw_all() { redraw_flags|=TRACKER_REDRAW_ALL; };
	void set_flag_redraw_top() { redraw_flags|=TRACKER_REDRAW_TOP; };
	void set_flag_redraw_playing_row() { redraw_flags|=TRACKER_REDRAW_PLAYING_ROW; };	


	void notify_set_playing_row(int p_row);

	int get_playing_row() { return playing_row; };
	int get_old_playing_row() { return playing_row_old; };

	bool is_playing_new_row() { return playing_new_row; };

	bool is_player_active () { return playing; };

	int previous_cursor_x;
	int previous_cursor_y;

	void set_follow_song(bool p_follow) { follow_song=p_follow;};
	bool is_following_song() { return follow_song;}

	//misc & validation

	void validate_track_midi_instruments();
	void validate_devices_midiout();


	int set_midi_note_on_at_cursor(char p_note,char p_velocity); //return the column to play, -1 if no play!
	void set_midi_note_off_at_cursor(char p_note,char p_velocity); //return the column to play, -1 if no play!
//        void set_midi_note_realtime(char p_note,char p_velocity,bool p_noteoff);
	

	void set_pattern_properties(int pattern,int hlmaj,int hlmin,int hlrepeat,int patlength,int patrepeat);

	void link_to_song(Song *song_ptr) {

		song=song_ptr;
	}

	Pattern_Editor();
	~Pattern_Editor();
};

#endif
