/* 
   sitecopy rsh/rcp protocol driver module
   Copyright (C) 2000, Joe Orton <joe@manyfish.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* TODO: 
 * - Reimplement using 'rcmd' etc natively (need to re-imp rcp protocol
 *   under GPL.) 
 * - At least fork/exec to save a sh process
 * - Do something with stdout
 * - Lee M says ssh will prompt for passwords on stdout, so we'd have to 
 *   parse the output and frig appropriately.
 * - Dave K: actually it prompts on stderr
 */

#include <config.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_SNPRINTF_H
#include "snprintf.h"
#endif

#include <ne_alloc.h>

#include "protocol.h"

typedef struct {
    struct site *site;
    const char *rsh_cmd, *rcp_cmd;
    char buf[BUFSIZ];
    char err[BUFSIZ];
} rsh_session;

static int run_rsh(rsh_session *sess, const char *template, ...) 
#ifdef __GNUC__
                __attribute__ ((format (printf, 2, 3)))
#endif /* __GNUC__ */
    ;

static int run_rcp(rsh_session *sess, const char *template, ...) 
#ifdef __GNUC__
                __attribute__ ((format (printf, 2, 3)))
#endif /* __GNUC__ */
    ;

static int run_rsh(rsh_session *sess, const char *template, ...) 
{
    va_list params;
    char *pnt;
    size_t len;
    int written;
    
    if (sess->site->server.username) {
	len = 1 + strlen(sess->rsh_cmd) + 
	    strlen(sess->site->server.username) +
	    strlen(sess->site->server.hostname) +
	    strlen(" -l   \"");
	if (len>=BUFSIZ)
	    return SITE_FAILED;
	snprintf(sess->buf, len, "%s -l %s %s \"", 
		 sess->rsh_cmd,
		 sess->site->server.username,
		 sess->site->server.hostname);
    } else {
	len = 1 + strlen(sess->rsh_cmd) + 
	    strlen(sess->site->server.hostname) +
	    strlen("  \"");
	if (len>=BUFSIZ)
	    return SITE_FAILED;
	snprintf(sess->buf, len, "%s %s \"", 
		 sess->rsh_cmd,
		 sess->site->server.hostname);
    }
    pnt = sess->buf + len - 1;
    
    va_start(params, template);
#ifdef HAVE_VSNPRINTF
    written = vfsnprintf(pnt, BUFSIZ - len, template, params);
#else
    written = vsprintf(pnt, template, params);
#endif
    va_end(params);

    pnt += written;
    *pnt++ = '"';
    *pnt = '\0';
    
    if (system(sess->buf) == 0) {
	return SITE_OK;
    } else {
	return SITE_FAILED;
    }
}

static int run_rcp(rsh_session *sess, const char *template, ...) 
{
    va_list params;
    char *pnt;
    size_t len;
    
    strcpy(sess->buf, sess->rcp_cmd);
    len = strlen(sess->rcp_cmd);
    pnt = sess->buf + len;
    
    *pnt++ = ' ';

    va_start(params, template);
#ifdef HAVE_VSNPRINTF
    vfsnprintf(pnt, BUFSIZ - len, template, params);
#else
    vsprintf(pnt, template, params);
#endif
    va_end(params);

    if (system(sess->buf) == 0) {
	return SITE_OK;
    } else {
	return SITE_FAILED;
    }
}

static int 
init(void **session, struct site *site)
{
    rsh_session *sess = ne_calloc(sizeof *sess);
    *session = sess;
    if (site->rsh_cmd) {
	sess->rsh_cmd = site->rsh_cmd;
    } else {
	sess->rsh_cmd = "rsh";
    }
    if (site->rcp_cmd) {
	sess->rcp_cmd = site->rcp_cmd;
    } else {
	sess->rcp_cmd = "rcp";
    }
    sess->site = site;
    return SITE_OK;
}    

static void finish(void *session) {
    rsh_session *sess = session;
    free(sess);
}

static int file_move(void *session, const char *from, const char *to) {
    rsh_session *sess = session;
    return run_rsh(sess, "mv %s %s", from, to);
}

static int file_upload(void *session, const char *local, const char *remote, 
			int ascii) {
    rsh_session *sess = session;

    if (sess->site->server.username) {
	return run_rcp(sess, "%s %s@%s:%s",
		       local, sess->site->server.username,
		       sess->site->server.hostname, remote);
    }
    else {
	return run_rcp(sess, "%s %s:%s", local, 
		       sess->site->server.hostname, remote);
    }
}

static int file_upload_cond(void *session,
			    const char *local, const char *remote,
			    int ascii, time_t t) {
    return SITE_UNSUPPORTED;
}

static int file_get_modtime(void *sess, const char *remote, time_t *modtime)
{
    return SITE_UNSUPPORTED;
}
    
static int file_download(void *sess, const char *local, const char *remote,
			 int ascii)
{
    return SITE_UNSUPPORTED;
}

static int file_read(void *sess, const char *remote,
		     sock_block_reader reader, void *userdata) {
    return SITE_UNSUPPORTED;
}

static int file_delete(void *session, const char *filename) {
    rsh_session *sess = session;
    return run_rsh(sess, "rm %s", filename);
}

static int file_chmod(void *session, const char *fname, const mode_t mode) {
    rsh_session *sess = session;
    return run_rsh(sess, "chmod %03o %s", mode, fname);
}

static int dir_create(void *session, const char *dirname) {
    rsh_session *sess = session;
    return run_rsh(sess, "mkdir %s", dirname);
}

static int dir_remove(void *session, const char *dirname) {
    rsh_session *sess = session;
    return run_rsh(sess, "rmdir %s", dirname);
}

static int fetch_list(void *sess, const char *dirname, int need_modtimes,
		       struct proto_file **files) {
    return SITE_UNSUPPORTED;
}

static const char *error(void *session) {
    return "An error occurred.";
}

static int get_dummy_port(struct site *site)
{
    return 0;
}

/* The protocol drivers */
const struct proto_driver rsh_driver = {
    init,
    finish,
    file_move,
    file_upload,
    file_upload_cond,
    file_get_modtime,
    file_download,
    file_read,
    file_delete,
    file_chmod,
    dir_create,
    dir_remove,
    NULL, /* create link */
    NULL, /* change link target */
    NULL, /* delete link */
    fetch_list,
    error,
    get_dummy_port,
    get_dummy_port,
    "rsh/rcp"
};
