/* A generic pretty printer. */

/*
 * A pretty printer
 * John D. Ramsdell -- December 2002
 * See pprint.h for a description of the interface.
 */

/*
 * The alogithm is by Lawrence C. Paulson, who simplified an algorithm
 * by Derek C. Oppen.
 *
 * Derek C. Oppen, Prettyprinting, ACM Transactions on Programming
 * Languages and Systems, Vol 2, No. 4, October 1980, Pages
 * 465-483.
 */

/* A pretty printer based on ML programs with the following copyright

(**** ML Programs from Chapter 8 of

  ML for the Working Programmer, 2nd edition
  by Lawrence C. Paulson, Computer Laboratory, University of Cambridge.
  (Cambridge University Press, 1996)

Copyright (C) 1996 by Cambridge University Press.
Permission to copy without fee is granted provided that this copyright
notice and the DISCLAIMER OF WARRANTY are included in any copy.

DISCLAIMER OF WARRANTY.  These programs are provided `as is' without
warranty of any kind.  We make no warranties, express or implied, that the
programs are free of error, or are consistent with any particular standard
of merchantability, or that they will meet your requirements for any
particular application.  They should not be relied upon for solving a
problem whose incorrect solution could result in injury to a person or loss
of property.  If you do use the programs or functions in such a manner, it
is at your own risk.  The author and publisher disclaim all liability for
direct, incidental or consequential damages resulting from your use of
these programs or functions.
****)

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <obstack.h>
#include "xmalloc.h"
#include "pprint.h"

typedef struct env {
  int space;
  int margin;
  FILE *out;
} *env_t;

struct pretty
{
  int length;
  int (*break_it)(env_t, pretty_t, int);
  void (*print_it)(env_t, pretty_t, int, int, int);
  union {
    const char *string;
    struct {
      int indent;
      pretty_t p;
    } blk;
  } u;
  pretty_t next;
};

static struct pretty error_marker[1];

/* Allocation using obstack */

#define obstack_chunk_alloc xmalloc
#define obstack_chunk_free free

static struct obstack stack[1];

void
pprint_init()
{
  obstack_init(stack);
}

static pretty_t
alloc()
{
  return (pretty_t)obstack_alloc(stack, sizeof(struct pretty));
}

/* Helpers */

static void
blanks(env_t e, int n)
{
  for (; n > 0; n--) {
    fputc(' ', e->out);
    e->space--;
  }
}

static void
newline(env_t e)
{
  fputc('\n', e->out);
  e->space = e->margin;
}

static int
break_dist(env_t e, pretty_t p, int after)
{
  if (p)
    return p->break_it(e, p, after);
  else
    return after;
}

static void
printing(env_t e, pretty_t p, int block_space,
	 int after, int force_breaks)
{
  for (; p; p = p->next)
    p->print_it(e, p, block_space, after, force_breaks);
}

/* Strings */

static int
str_break_it(env_t e, pretty_t p, int after)
{
  return p->length + break_dist(e, p->next, after);
}

static void
str_print_it(env_t e, pretty_t p, int block_space,
	     int after, int force_breaks)
{
  fputs(p->u.string, e->out);
  e->space -= p->length;
}

pretty_t
pstring(const char *string, pretty_t next)
{
  if (string && next != error_marker) {
    pretty_t p = alloc();
    if (!p)
      return error_marker;
    p->length = strlen(string);
    p->break_it = str_break_it;
    p->print_it = str_print_it;
    p->u.string = string;
    p->next = next;
    return p;
  }
  else
    return error_marker;
}

/* Unbreakable space */

static void
spc_print_it(env_t e, pretty_t p, int block_space,
	     int after, int force_breaks)
{
  int n = p->length;
  e->space -= n;
  while (n-- > 0)
    fputc(' ', e->out);
}

pretty_t
pspace(int fill, pretty_t next)
{
  if (fill >= 0 && next != error_marker) {
    pretty_t p = alloc();
    if (!p)
      return error_marker;
    p->length = fill;
    p->break_it = str_break_it;
    p->print_it = spc_print_it;
    p->next = next;
    return p;
  }
  else
    return error_marker;
}

/* Breaks */

static int
brk_break_it(env_t e, pretty_t p, int after)
{
  return 0;
}

static void
brk_print_it(env_t e, pretty_t p, int block_space,
	     int after, int force_breaks)
{
  int len = p->length;
  if (!force_breaks && len + break_dist(e, p->next, after) <= e->space)
    blanks(e, len);
  else {
    newline(e);
    blanks(e, e->margin - block_space);
  }
}

pretty_t
pbreak(int skip, pretty_t next)
{
  if (skip >= 0 && next != error_marker) {
    pretty_t p = alloc();
    if (!p)
      return error_marker;
    p->length = skip;
    p->break_it = brk_break_it;
    p->print_it = brk_print_it;
    p->next = next;
    return p;
  }
  else
    return error_marker;
}

/* Blocks */

static int
sum_length(pretty_t p)
{
  int sum = 0;
  for (; p; p = p->next)
    sum += p->length;
  return sum;
}

static void
blk_print_it(env_t e, pretty_t p, int block_space,
	     int after, int force_breaks)
{
  int dist = break_dist(e, p->next, after);
  printing(e, p->u.blk.p, e->space - p->u.blk.indent, dist, 0);
}

pretty_t
pblock(int indent, pretty_t p, pretty_t next)
{
  if (p != error_marker && next != error_marker) {
    pretty_t r = alloc();
    if (!r)
      return error_marker;
    r->length = sum_length(p);
    r->break_it = str_break_it;
    r->print_it = blk_print_it;
    r->u.blk.indent = indent;
    r->u.blk.p = p;
    r->next = next;
    return r;
  }
  else
    return error_marker;
}

/* Groups */

static void
grp_print_it(env_t e, pretty_t p, int block_space,
	     int after, int force_breaks)
{
  int dist = break_dist(e, p->next, after);
  force_breaks = p->length + dist > e->space;
  printing(e, p->u.blk.p, e->space - p->u.blk.indent,
	   dist, force_breaks);
}

pretty_t
pgroup(int indent, pretty_t p, pretty_t next)
{
  if (p != error_marker && next != error_marker) {
    pretty_t r = alloc();
    if (!r)
      return error_marker;
    r->length = sum_length(p);
    r->break_it = str_break_it;
    r->print_it = grp_print_it;
    r->u.blk.indent = indent;
    r->u.blk.p = p;
    r->next = next;
    return r;
  }
  else
    return error_marker;
}

/* the pretty printer */

int
pprint(FILE *out, pretty_t pretty, int margin)
{
  if (pretty == error_marker) {
    obstack_free(stack, 0);
    pprint_init();
    return -1;
  }
  else {
    struct env e = { margin, margin, out };
    printing(&e, pretty, margin, 0, 0);
    obstack_free(stack, 0);
    pprint_init();
    return 0;
  }
}
