/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


// Qt includes
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

// KDE includes
#include <kmessagebox.h>
#include <klocale.h>
#include <kapplication.h>

// system includes
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>

// application specific includes
#include "smb4kfileio.h"
#include "smb4kdefs.h"

// static varibles
static Smb4KFileIO::FileItem sudoers_item;


Smb4KFileIO::Smb4KFileIO( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_proc = new KProcess( this, "FileIOProcess" );
  m_proc->setUseShell( true );
  
  m_state = Idle;

  connect( m_proc, SIGNAL( receivedStderr( KProcess *, char *, int ) ), SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_proc, SIGNAL( processExited( KProcess * ) ), SLOT( slotProcessExited( KProcess * ) ) );
}


Smb4KFileIO::~Smb4KFileIO()
{
}


/****************************************************************************
   TEMPORARY FUNCTION: Converts the super.tab file to the new style.
****************************************************************************/

void Smb4KFileIO::convertSuperTab()
{
  if ( !QFile::exists( "/tmp/smb4k.lock" ) )
  {
    // First generate the lock file.
    QFile lock_file( "/tmp/smb4k.lock" );
    lock_file.open( IO_WriteOnly );
    lock_file.close();
    
    m_config->setGroup( "Super User Privileges" );
    bool run_suid = m_config->readBoolEntry( "Run SUID", false );
    bool force_umount = m_config->readBoolEntry( "Force Unmount", false );
    
    if ( run_suid || force_umount )
    {
      FileItem item = readConfigFile( "super.tab" );
      
      if ( item.exists() )
      {
        QStringList contents = item.contents();
        
        if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 && contents.grep( ":define Smb4KUsers" ).count() == 0 )
        {
          KMessageBox::information( new QWidget(), i18n( "As of version 0.5.0, the entries in the super.tab file are stored in a different format. Smb4K is now going to convert the old ones." ) );
          
          QStringList::Iterator insert_point;
          QString users;
          bool ok = false;
          int entries = 0;
          m_config->setGroup( "Programs" );
          
          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "# Entries for Smb4K users." ) )
            {
              ok = true;
              continue;
            }
            else if ( ok && ((*it).startsWith( "kill" ) ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_kill" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_kill\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              insert_point = it;
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "umount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_umount" ), true );
              users = (*it).section( "\t", 2, 2 ).stripWhiteSpace();
              (*it).replace( users, "$(Smb4KUsers)", true );
              QString tmp = "smb4k_umount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "mount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_mount" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_mount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace()+"\tenv=PASSWD,USER" );
              entries++;
              continue;
            }
            else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
            {
              if ( entries < 3 )
              {
#ifndef __FreeBSD__
                contents.insert( it, "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
                contents.insert( it, "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif                            
              }
  
              break;
            }
            else
            {
              continue;
            }
          }
          
          contents.insert( insert_point, ":define Smb4KUsers "+users );
          
          item.replaceContents( contents );
          
          writeFile( &item, "0644" );      
        }
        else if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 && contents.grep( ":define Smb4KUsers" ).count() != 0 
                  && contents.grep( "smb4k_mount" ).first().contains( "env" ) == 0 )
        {
          KMessageBox::information( new QWidget(), i18n( "Smb4K is going to modify the entries in the super.tab file in order to work properly." ) );
          
          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "smb4k_mount" ) )
            {
              (*it).append( "\tenv=PASSWD,USER" );
              break;
            }
          }
          
          item.replaceContents( contents );
          
          writeFile( &item, "0644" );            
        }
      }
      else
        emit error( ERROR_FILE_NOT_FOUND, "super.tab" );
    }
    
    lock_file.remove();
  }
}


/****************************************************************************
   Writes entries to the super.tab file.
****************************************************************************/

bool Smb4KFileIO::writeSuperUserEntries( const QString &program )
{
  if ( !QFile::exists( "/tmp/smb4k.lock" ) )
  {
    QFile file( "/tmp/smb4k.lock" );
    file.open( IO_WriteOnly );
    file.close();
    
    m_config->setGroup( "Programs" );
    
    QStringList contents;
    bool write = true;
    
    if ( program == "super" )
    {
      FileItem item = readConfigFile( "super.tab" );
      
      if ( !item.exists() )
      {
        emit no_suid_program();
        emit finished_suid_writing();
        
        file.remove();
        return false;
      }
      
      QStringList contents = item.contents();
        
      if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 )
      {
        // Create the new entries:
        contents.append( "# Entries for Smb4K users." );
        contents.append( "# Generated by Smb4K. Please do not modify!" );
        contents.append( ":define Smb4KUsers "+QString( "%1" ).arg( getenv( "USER" ) ) );
#ifndef __FreeBSD__          
        contents.append( "smb4k_kill\t"+m_config->readPathEntry( "smb4k_kill" )+"\t$(Smb4KUsers)\tuid=root\tgid=root" );
        contents.append( "smb4k_umount\t"+m_config->readPathEntry( "smb4k_umount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root" );
        contents.append( "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
        contents.append( "smb4k_kill\t"+m_config->readPathEntry( "smb4k_kill" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel" );
        contents.append( "smb4k_umount\t"+m_config->readPathEntry( "smb4k_umount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel" );
        contents.append( "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif      
        contents.append( "# End of Smb4K user entries." );       
      }
      else
      {
        if ( contents.grep( ":define Smb4KUsers" ).count() != 0 )
        {
          // Append the user name, if necessary.
          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).contains( ":define Smb4KUsers" ) != 0 )
            {
              QString users = (*it).section( "Smb4KUsers", 1, 1 ).stripWhiteSpace();
              QStringList user_list = QStringList::split( ",", users, false );
                
              if ( user_list.find( QString( "%1" ).arg( getenv( "USER" ) ) ) == user_list.end() )
              {
                user_list.append( QString( "%1" ).arg( getenv( "USER" ) ) );
                (*it).replace( users, user_list.join( "," ) );
              }
              else
                write = false;
               
              break;
            }
            else
              continue;
          }
        }
        else
        {
          // Convert the old entries.
          QStringList::Iterator insert_point;
          QString users;
          bool ok = false;
          int entries = 0;
          m_config->setGroup( "Programs" );  
          
          for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
          {
            if ( (*it).startsWith( "# Entries for Smb4K users." ) )
            {
              ok = true;
              continue;
            }
            else if ( ok && ((*it).startsWith( "kill" ) ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_kill" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_kill\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              insert_point = it;
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "umount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_umount" ), true );
              users = (*it).section( "\t", 2, 2 ).stripWhiteSpace();
              (*it).replace( users, "$(Smb4KUsers)", true );
              QString tmp = "smb4k_umount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace() );
              entries++;
              continue;
            }
            else if ( ok && (*it).startsWith( "mount" ) )
            {
              (*it).replace( (*it).section( "\t", 1, 1 ).stripWhiteSpace(), m_config->readPathEntry( "smb4k_mount" ), true );
              (*it).replace( (*it).section( "\t", 2, 2 ).stripWhiteSpace(), "$(Smb4KUsers)", true );
              QString tmp = "smb4k_mount\t";
              *it = tmp.append( (*it).section( "\t", 1, -1 ).stripWhiteSpace()+"\tenv=PASSWD,USER" );
              entries++;
              continue;
            }
            else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
            {
              if ( entries < 3 )
              {
#ifndef __FreeBSD__
                contents.insert( it, "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=root\tenv=PASSWD,USER" );
#else
                contents.insert( it, "smb4k_mount\t"+m_config->readPathEntry( "smb4k_mount" )+"\t$(Smb4KUsers)\tuid=root\tgid=wheel\tsetenv=HOME=$CALLER_HOME\tenv=PASSWD,USER" );
#endif                            
              }
    
              break;
            }
            else
            {
              continue;
            }
          }
            
          QStringList user_list = QStringList::split( ",", users, false );

          if ( user_list.find( QString( "%1" ).arg( getenv( "USER" ) ) ) == user_list.end() )
            user_list.append( QString( "%1" ).arg( getenv( "USER" ) ) );
            
          contents.insert( insert_point, ":define Smb4KUsers "+user_list.join( "," ) );                        
        }
      }
      
      if ( write )
      {
        item.replaceContents( contents );
        
        if ( !writeFile( &item, "0644" ) )
        {
          emit no_suid_program();
        }
      }
      else
      {
        emit finished_suid_writing();
      }
      
      file.remove();
    }
    else if ( program == "sudo" )
    {
      sudoers_item = readConfigFile( "sudoers", false, true );
      
      if ( !sudoers_item.exists() )
      {
        emit no_suid_program();
        emit finished_suid_writing();
        
        file.remove();
        return false;
      }
      
      m_todo_sudoers = Add;
      m_state = copySudoers;
      
      *m_proc << "kdesu "+KProcess::quote( "cp "+sudoers_item.path()+" /tmp && chown "+QString( "%1:%2" ).arg( getuid() ).arg( getgid() )+" /tmp/sudoers && chmod 0666 /tmp/sudoers" );
      m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );
    }
  }
  else
  {
    emit error( ERROR_LOCKED, QString::null );
    emit no_suid_program();
    emit finished_suid_writing();
            
    return false;
  }
  
  return true;
}


/****************************************************************************
   Removes all Smb4K specific entries of the user from /etc/super.tab.
****************************************************************************/

bool Smb4KFileIO::removeSuperUserEntries()
{
  if ( !QFile::exists( "/tmp/smb4k.lock" ) )
  {
    QFile lock_file( "/tmp/smb4k.lock" );
    lock_file.open( IO_WriteOnly );
    lock_file.close();
    
    m_config->setGroup( "Super User Privileges" );
    QString program = m_config->readEntry( "SUID Program", QString::null );
    
    if ( program == "super" )
    {    
      FileItem item = readConfigFile( "super.tab" );
      
      if ( item.exists() )
      {
        QStringList contents = item.contents();
        QStringList::Iterator j;
        
        for ( QStringList::Iterator it = contents.begin(); it != contents.end(); ++it )
        {
          if ( (*it).startsWith( "# Entries for Smb4K users." ) )
          {
            j = it;
          }
          else if ( (*it).startsWith( ":define Smb4KUsers" ) )
          {
            QString users = (*it).section( "Smb4KUsers", 1, 1 ).stripWhiteSpace();
            
            if ( users.contains( "," ) == 0 )
            {
              while ( j != contents.end() )
              {
                if ( !(*j).startsWith( "# End of Smb4K user entries." ) )
                {                  
                  j = contents.remove( j );
                }
                else
                {
                  contents.remove( j );
                  break;
                }
              }
            }
            else
            {
              QStringList list = QStringList::split( ",", users, false );
              list.remove( QString( "%1" ).arg( getenv( "USER" ) ) );
              
              (*it).replace( users, list.join( "," ) );
            }
            
            break;
          }
          else
          {
            continue;
          }
        }
        
        item.replaceContents( contents );
        
        if ( !writeFile( &item, "0644" ) )
        {
          emit no_suid_program();
        }
      }
      else
      {
        emit no_suid_program();    
        return false;
      }
      
      lock_file.remove();
    }
    else if ( program == "sudo" )
    {
      sudoers_item = readConfigFile( "sudoers", false, true );
      
      if ( !sudoers_item.exists() )
      {
        emit no_suid_program();
        emit finished_suid_writing();
        return false;
      }
      
      m_todo_sudoers = Remove;
      m_state = copySudoers;
      
      *m_proc << "kdesu "+KProcess::quote( "cp "+sudoers_item.path()+" /tmp && chown "+QString( "%1:%2" ).arg( getuid() ).arg( getgid() )+" /tmp/sudoers && chmod 0666 /tmp/sudoers" );
      m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );    
    }
  }
  else
  {
    emit error( ERROR_LOCKED, QString::null );
    return false;
  }
  
  return true;
}


/****************************************************************************
   Extracts the non-system users from the /etc/passwd and /etc/group
****************************************************************************/


const QValueList<Smb4KUser *> Smb4KFileIO::getUsers()
{
  FileItem item = readConfigFile( "passwd" );
  QStringList contents = item.contents();
  
  QValueList<Smb4KUser *> users;
      
  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    if ( !(*it).stripWhiteSpace().startsWith( "#" ) && ( (*it).section( ":", 2, 2 ).toInt() >= 500 || (*it).section( ":", 2, 2 ).toInt() == (int)getuid() ) )
    {
      users.append( new Smb4KUser( (*it).section( ":", 2, 2 ).toInt(), (*it).section( ":", 3, 3 ).toInt() ) );    
    }
    else
    {
      continue;
    }
  }
  
  return users;
}


/****************************************************************************
   Detects all avaiable (global) options in the smb.conf
****************************************************************************/

const QMap<QString, QString> Smb4KFileIO::readSMBOptions()
{
  QMap<QString, QString> map;

  FileItem item = readConfigFile( "smb.conf", true );
  QStringList contents = item.contents();
  
  // Clean the list from everything that is not needed. Especially all
  // non-global sections will be removed.
  // REMARK: We assume that the global entries are not scattered through
  // the whole config file but are written en bloc.
  for ( QStringList::Iterator it = contents.erase( contents.begin(), ++(contents.find( "[global]" )) ); it != contents.end(); ++it )
  {
    if ( (*it).startsWith( "#" ) )
    {
      *it = QString::null;
    }
    else if ( (*it).startsWith( "[" ) )
    {
      contents.erase( it, contents.end() );
      break;
    }
  }

  contents.remove( QString::null );

  // Write all options into the map:
  for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
  {
    QString key = (*it).section( "=", 0, 0 ).stripWhiteSpace().lower();
    map[key] = QString( (*it).section( "=", 1, 1 ).stripWhiteSpace().upper() );
  }

  // Post-processing. Some values should be entered with their defaults, if they are
  // not already present.
  if ( !map.contains( "netbios name" ) )
  {
    char *hostname;
    size_t hostnamelen = 255;
    hostname = (char *)malloc( hostnamelen );

    if ( gethostname( hostname, hostnamelen ) == 0 )
      map["netbios name"] = ( QString( "%1" ).arg( hostname ) ).upper();
    else
      emit error( ERROR_GETTING_HOSTNAME, QString::null );

    free(hostname);
  }
  
  return map;
}


/****************************************************************************
   Returns the paper size defined on the system.
****************************************************************************/

const QString Smb4KFileIO::getPaperSize()
{
  FileItem item = readConfigFile( "papersize" );
  QStringList contents = item.contents();
  
  return contents.isEmpty() ? QString( "a4" ) : contents.join( " " ).stripWhiteSpace();
}


/****************************************************************************
   Locates a system config file and returns its path
****************************************************************************/

Smb4KFileIO::FileItem Smb4KFileIO::readConfigFile( const QString &fileName, bool strip, bool quiet )
{
  QStringList paths;
  paths << "/etc";
  paths << "/etc/samba";
  paths << "/usr/local/etc";
  paths << "/usr/local/etc/samba";
  
  QStringList contents;
  bool exists = false;
  bool read_failed = false;
  
  QString path;
  
  for ( QStringList::ConstIterator it = paths.begin(); it != paths.end(); ++it )
  {
    QDir::setCurrent( *it );
    if ( QFile::exists( fileName ) )
    {
      exists = true;
      path = QDir::currentDirPath();
      
      QFile file( fileName );
      
      if ( file.open( IO_ReadOnly ) )
      {
        QTextStream ts( &file );
        
        while ( !ts.atEnd() )
        {
          if ( !strip )
          {
            contents.append( ts.readLine() );
          }
          else
          {
            contents.append( ts.readLine().stripWhiteSpace() );
          }
        }

        file.close();
      }
      else
      {
        if ( !quiet )
        {
          read_failed = true;
          emit error( ERROR_READING_FILE, fileName );
        }
      }
      
      break;    
    }
    else
      continue;
  }
  
  if ( !exists )
  {
    emit error( ERROR_FILE_NOT_FOUND, fileName );
  }
    
  if ( read_failed )
  {
    exists = false;    // FIXME: That's an ugly way...
  }
 
  return Smb4KFileIO::FileItem( QString( "%1/%2" ).arg( path, fileName ), contents, exists );
}


/****************************************************************************
   Writes a file to its destination
****************************************************************************/

bool Smb4KFileIO::writeFile( Smb4KFileIO::FileItem *item, const QString &perm )
{
  QStringList contents = item->contents();
  
  QFile tmp_file( "smb4k.tmp" );
  QDir::setCurrent( "/tmp" );
  
  if ( tmp_file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &tmp_file );

    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
    {
      ts << *it << endl;
    }

    tmp_file.close();
  }
  else
  {
    emit error( ERROR_WRITING_FILE, tmp_file.name() );
    return false;
  }
  
  // Now move it to the right location.
  if ( tmp_file.exists() )
  {
    m_state = writeSU;
    QString tmp_path = QDir::currentDirPath()+"/"+tmp_file.name();
    
#ifndef __FreeBSD__
    QString command( "chown root:root "+tmp_path+" && chmod "+perm+" "+tmp_path+" && mv "+tmp_path+" "+item->path() );
#else
    QString command( "chown root:wheel "+tmp_path+" && chmod "+perm+" "+tmp_path+" && mv "+tmp_path+" "+item->path() );
#endif

    *m_proc << QString( "kdesu %1" ).arg( KProcess::quote( command ) );
    m_proc->start( KProcess::NotifyOnExit, KProcess::Stderr );    
  }
  else
  {
    emit error( ERROR_FILE_NOT_FOUND, tmp_file.name() );
    return false;
  }
  
  return true;
}


/****************************************************************************
   Process the sudoers file.
****************************************************************************/


void Smb4KFileIO::processSudoers()
{
  QDir::setCurrent( "/tmp" );
  QFile file( "sudoers" );
  QStringList contents;
  
  // In case the user canceled or did not provide the right
  // password.
  if ( !file.exists() )
  {
    file.setName( "smb4k.lock" );
    file.remove();
    return; 
  }
      
  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
        
    while ( !ts.atEnd() )
    {
      contents.append( ts.readLine() );
    }

    file.close();
  }
  else
  {
    emit error( ERROR_READING_FILE, file.name() );
    return;
  }
  
  file.remove();
  
  bool write = true;
 
  if ( m_todo_sudoers == Add )
  { 
    char *hn;
    hn = (char *)malloc( 200*sizeof(char) );
        
    if ( gethostname( hn, 200*sizeof(char) ) != 0 )
    {
      emit error( ERROR_GETTING_HOSTNAME, QString::null );
      return; 
    }
        
    QString hostname( hn );
        
    free( hn );
    
    m_config->setGroup( "Programs" );
  
    if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 )
    {
      contents.append( "# Entries for Smb4K users." );
      contents.append( "# Generated by Smb4K. Please do not modify!" );
      contents.append( "User_Alias\tSMB4KUSERS = "+QString( "%1" ).arg( getenv( "USER" ) ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+m_config->readPathEntry( "smb4k_kill" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+m_config->readPathEntry( "smb4k_umount" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "+m_config->readPathEntry( "smb4k_mount" ) ); 
      contents.append( "# End of Smb4K user entries." );          
    }
    else
    {
      for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
      {
        if ( (*it).startsWith( "User_Alias\tSMB4KUSERS" ) )
        {
          if ( (*it).contains( getenv( "USER" ), true ) == 0 ) 
          {
            (*it).append( ","+QString( getenv( "USER") ) );
          }
          else
          {
            write = false;
          }
          break;
        }
        else
        {
          continue;
        }
      }
    }            
  }
  else if ( m_todo_sudoers == Remove )
  {
    QStringList::Iterator j;
    bool remove_all = false;
    
    for ( QStringList::Iterator it = contents.find( "# Entries for Smb4K users." ); it != contents.end(); ++it )
    {
      if ( (*it).startsWith( "User_Alias" ) && (*it).contains( "SMB4KUSERS" ) != 0 )
      {
        j = it;
        
        if ( (*it).contains( getenv( "USER" ), true ) != 0 ) 
        {
          QStringList users = QStringList::split( ",", (*it).section( "=", 1, 1 ).stripWhiteSpace(), false );
          users.remove( QString( "%1" ).arg( getenv( "USER" ) ) );
          
          if ( users.isEmpty() )
          {
            remove_all = true;
          }
          else
          {
            (*it).replace( (*it).section( "=", 1, 1 ).stripWhiteSpace(), users.join( "," ), true );
          }
        }
        else
        {
          write = false;
        }
        
        break;
      }
      else if ( (*it).startsWith( "# End of Smb4K user entries." ) )
      {
        break;
      }
      else
      {
        continue;
      }
    }
    
    if ( remove_all )
    {
      while ( j != contents.end() )
      {
        if ( !(*j).startsWith( "# End of Smb4K user entries." ) )
        {
          j = contents.remove( j );
        }
        else
        {
          contents.remove( j );
          break;
        }
      }
    }
  }
  
  if ( write )
  {
    sudoers_item.replaceContents( contents );
        
    if ( !writeFile( &sudoers_item, "0440" ) )
    {
      emit no_suid_program();
    }
  }
  else
  {
    emit finished_suid_writing();
  }
         
  file.setName( "smb4k.lock" );
  file.remove();
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

void Smb4KFileIO::slotReceivedStderr( KProcess *, char *buf, int len )
{
  QString buffer( QString::fromLocal8Bit( buf, len ) );
  
  if ( buffer.contains( "KIconLoader" ) == 0 && buffer.contains( "(kdelibs)" ) == 0 )
  {
    emit error( ERROR_UNKNOWN, buffer );
  }
}


void Smb4KFileIO::slotProcessExited( KProcess * )
{
  switch ( m_state )
  {
    case writeSU:
      m_state = Idle;
      m_proc->clearArguments();
      emit finished_suid_writing();
      break;
    case copySudoers:
      m_state = Idle;
      m_proc->clearArguments();
      processSudoers();
      break;
    default:
      m_state = Idle;
      m_proc->clearArguments();
      break;
  }
}

#include "smb4kfileio.moc"
