/***************************************************************************
    smb4kpasswdreader.cpp  -  Reads passwords. Used by several classes.
                             -------------------
    begin                : Mit Mai 14 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qlayout.h>
#include <qlabel.h>
#include <qcheckbox.h>
#include <qgroupbox.h>

#ifdef __FreeBSD__
#include <qdir.h>
#include <qfile.h>
#include <qstringlist.h>
#include <qurlinfo.h>
#include <qmap.h>
#endif

// KDE includes
#include <kapplication.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kurl.h>
#include <klineedit.h>
#include <kcombobox.h>

// system includes
#include <cstdlib>

// application specific includes
#include "smb4kpasswdreader.h"
#include "smb4kcore.h"


Smb4KPasswdReader::Smb4KPasswdReader( QObject *parent, const char *name ) : QObject( parent, name )
{
  // First we need the directory.
  KStandardDirs *stddir = new KStandardDirs();
  QString dir = locateLocal( "appdata", QString::null, KGlobal::instance() );
  
  if ( !stddir->exists( dir ) )
    stddir->makeDir( dir );
  
  delete stddir;
    
  // Do not use the KConfig object, that is provided by kapp->config().
  // We want a separate one for the passwords.
  m_config = new KConfig( "passwords", false, true, "appdata" );
  // Give it strict permissions:
  m_config->setFileWriteMode( 00600 );

  // Get or create the key and convert the not obfuscated passwords,
  // if there are any:
  KConfig *global_config = kapp->config();
  global_config->setGroup( "Authentication" );

  m_key = global_config->readNumEntry( "Random Number", rand() );
  global_config->writeEntry( "Random Number", m_key );

  read();
  
#ifdef __FreeBSD__
  writeToSMBConfFile();
#endif
}


Smb4KPasswdReader::~Smb4KPasswdReader()
{
  m_list.clear();
}


/****************************************************************************
   Functions that read the authentication info.
****************************************************************************/

Smb4KAuthInfo *Smb4KPasswdReader::getAuth( const QString &workgroup, const QString &host, const QString &share )
{
  Smb4KAuthInfo *auth = new Smb4KAuthInfo( workgroup, host, share ); 
  
  if ( !m_list.isEmpty() )
  {
    for ( QValueList<Smb4KAuthInfo *>::ConstIterator it = m_list.begin(); it != m_list.end(); ++it )
    {
      if ( !workgroup.isEmpty() )
      {
        if ( (*it)->workgroup().upper() == workgroup.upper() && (*it)->host().upper() == host.upper() )
        {
          if ( !share.isEmpty() )
          {
            if ( (*it)->share().upper() == share.upper() )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
          
              break;
            }
            else if ( (*it)->share() == "*" && auth->user().isEmpty() )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
              
              continue;            
            }
            else
            {
              continue;
            }
          }
          else
          {
            if ( (*it)->share() == "*" )
            {
              auth->setUser( (*it)->user() );
              auth->setPassword( (*it)->password() );
              
              break;
            }
            else
            {
              continue;
            }
          }
        }
        else
        {
          continue;
        }
      }
      else
      {
        if ( (*it)->host().upper() == host.upper() )
        {
          auth->setUser( (*it)->user() );
          auth->setPassword( (*it)->password() );
          
          if ( (*it)->share() == "*" )
          {
            break;
          }
          else
          {
            continue;
          }
        }
        else
        {
          continue;
        }
      }
    }
  }

  if ( auth->user().isEmpty() )
  {
    kapp->config()->setGroup( "Authentication" );
    
    if ( kapp->config()->readBoolEntry( "Default Authentication", false ) )
    {
      m_config->setGroup( "Default" );
      QString pass = obfuscate( m_config->readEntry( "Password", QString::null ), false );
      auth->setUser( m_config->readEntry( "User", QString::null ) );
      auth->setPassword( pass );
    }
    else
    {
      auth->setUser( QString::null );
      auth->setPassword( QString::null );
    }
  }
  
  return auth; 
}


/****************************************************************************
   Reads the list of all authentication entries that are saved in the
   password file.
****************************************************************************/

void Smb4KPasswdReader::read()
{
  if ( m_config->hasGroup( "Logins" ) )
  {
    if ( !m_list.isEmpty() )
    {
      m_list.clear();
    }
  
    QStringList loginData;
    m_config->setGroup( "Logins" );
    
    for ( int pos = 0; ; pos++ )
    {
      loginData = m_config->readListEntry( QString( "%1" ).arg( pos, 0, 10 ).stripWhiteSpace(), ',' );

      // "Decrypt" the password.
      if ( !loginData.isEmpty() )
      {
        m_list.append( new Smb4KAuthInfo( loginData[0], loginData[1], loginData[2], loginData[3], obfuscate( loginData[4], false ) ) );
        
        loginData.clear();
      }
      else
        break;
    }
  }
}


/****************************************************************************
   Reads the default auth data.
****************************************************************************/

Smb4KAuthInfo *Smb4KPasswdReader::readDefaultAuthData()
{
  m_config->setGroup( "Default" );
  Smb4KAuthInfo *auth = new Smb4KAuthInfo( QString::null, QString::null, QString::null, m_config->readEntry( "User", QString::null ), obfuscate( m_config->readEntry( "Password", QString::null ), false ) );
  
  return auth;  
}


/****************************************************************************
   Adds authentication information to the existing list or modifies 
   existing ones.
****************************************************************************/

void Smb4KPasswdReader::commit( Smb4KAuthInfo *auth )
{
  bool exists = false;
  
  for ( QValueList<Smb4KAuthInfo *>::Iterator it = m_list.begin(); it != m_list.end(); ++it )
  {
    if ( (*it)->workgroup() == auth->workgroup() && (*it)->host() == auth->host() && (*it)->share() == auth->share() )
    {
      (*it)->setUser( auth->user() );
      (*it)->setPassword( auth->password() );
      
      exists = true;
      
      break;
    }
    else
      continue;
  }
  
  if ( !exists )
    m_list.append( new Smb4KAuthInfo( auth->workgroup(), auth->host(), auth->share(), auth->user(), auth->password() ) );
  
  write();
}


/****************************************************************************
   Writes the list of authentication data to the password file.
****************************************************************************/

void Smb4KPasswdReader::write()
{
  // Clear all entries.
  if ( m_config->hasGroup( "Logins" ) )
  {
    m_config->deleteGroup( "Logins", true, false );
    m_config->sync();
  }

  m_config->setGroup( "Logins" );
  
  int index = 0;

  QStringList item;
    
  for ( QValueList<Smb4KAuthInfo *>::ConstIterator it = m_list.begin(); it != m_list.end(); ++it )
  {
    item.append( (*it)->workgroup() );
    item.append( (*it)->host() );
    item.append( (*it)->share() );
    item.append( (*it)->user() );
    item.append(  obfuscate( (*it)->password(), true ) );
    
    m_config->writeEntry( QString( "%1" ).arg( index++ ), item, ',' );
    
    item.clear();
  }
  
  m_config->sync();
  
#ifdef __FreeBSD__
  writeToSMBConfFile();
#endif
}


/****************************************************************************
   Writes the default auth data that is passed by the configuration dialog
   on exit.
****************************************************************************/

void Smb4KPasswdReader::writeDefaultAuthData( Smb4KAuthInfo *auth )
{
  // Obfuscate the password:
  QString pass = obfuscate( auth->password(), true );
  
  // Now write the data:
  m_config->setGroup( "Default" );
  m_config->writeEntry( "User", auth->user() );
  m_config->writeEntry( "Password", pass );
  
  m_config->sync();
}


/****************************************************************************
   This function shows the askpass dialog.
****************************************************************************/

bool Smb4KPasswdReader::askpass( const QString &workgroup, const QString &host, const QString &share, int state )
{
  //
  // Build the widget.
  //
  passwdDlg = new KDialogBase( KDialogBase::Plain, i18n( "Authentication" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, new QWidget(), "PasswordDlg", false, true );

  QFrame *frame = passwdDlg->plainPage();
  QGridLayout *layout = new QGridLayout( frame );
  layout->setSpacing( 10 );
  layout->setMargin( 0 );
  
  QString message;
  
  switch ( state )
  {
    case NewData:
      break;
    case AccessDenied:
      message = i18n( "The access was denied. " );
      break;
    case BadPasswd:
      message = i18n( "The password is not correct. " );
      break;
    case PermDenied:
      message = i18n( "The permission was denied. " );
      break;
    case AuthError:
      message = i18n( "An authentication error occurred. " );
      break;
    case LogonFailure:
      message = i18n( "The logon failed. " );
      break;
    default:
      break;
  }    

  message.append( i18n( "Please enter authentication data for %1." ).arg( share.stripWhiteSpace().isEmpty() ? host : "//"+host+"/"+share ) );
  
  QGroupBox *message_box = new QGroupBox( 1, Qt::Horizontal, i18n( "Information" ), frame, "AskPassInfoBox" );
  QLabel *message_label = new QLabel( message_box );
  message_label->setText( message.stripWhiteSpace() );
  message_label->setTextFormat( Qt::RichText );
  
  layout->addWidget( message_box, 0, 0, 0 ); 
  
  QGroupBox *input_box = new QGroupBox( 1, Qt::Horizontal, i18n( "Input" ), frame, "AskPassInputBox" );
  QWidget *input_widget = new QWidget( input_box );
  QGridLayout *iw_layout = new QGridLayout( input_widget );
  iw_layout->setSpacing( 10 );
  iw_layout->setMargin( 0 );
  
  QLabel *userLabel = new QLabel( i18n( "User:" ), input_widget );
  iw_layout->addWidget( userLabel, 0, 0, 0 );
  
  KLineEdit *userEdit = NULL;
  KComboBox *userCombo = NULL;

  if ( share != "homes" )
  {
    delete userCombo;
    userEdit = new KLineEdit( input_widget );
    iw_layout->addWidget( userEdit, 0, 1, 0 );
  }
  else
  {
    delete userEdit;
    userCombo = new KComboBox( input_widget );
    userCombo->setEditable( true );
    iw_layout->addWidget( userCombo, 0, 1, 0 );
  }
  
  QLabel *passwdLabel = new QLabel( i18n( "Password:" ), input_widget );
  iw_layout->addWidget( passwdLabel, 1, 0, 0 );
  
  m_passwdEdit = new KLineEdit( input_widget );
  m_passwdEdit->setEchoMode( KLineEdit::Password );
  iw_layout->addWidget( m_passwdEdit, 1, 1, 0 );
  
  QCheckBox *use = new QCheckBox( i18n( "Use authentication for all shares." ), input_box );
  
  layout->addWidget( input_box, 1, 0, 0 );
  
  message_box->setFixedWidth( input_box->sizeHint().width() );  // Do not change this! You'll get layout probems otherwise.
  
  passwdDlg->setMainWidget( frame );
  
  //
  // Get the authentication info needed.
  //
  Smb4KAuthInfo *newAuth = new Smb4KAuthInfo( workgroup, host, share );
  
  QStringList user_names;
  
  if ( share != "homes" )
  {
    Smb4KAuthInfo *auth = getAuth( workgroup, host, share );

    if ( auth->user().isEmpty() && auth->password().isEmpty() )
    {
      if ( kapp->config()->hasGroup( "Homes Shares" ) )
      {
        kapp->config()->setGroup( "Homes Shares" );
        QString name = *( kapp->config()->readListEntry( host, ',' ).find( share ) );
        
        if ( !name.isEmpty() )
        {
          auth->setUser( name );
        }
      }
    }

    if ( !auth->user().isEmpty() )
    {
      userEdit->setText( auth->user() );
    }
     
    if ( !auth->password().isEmpty() )
    {
      m_passwdEdit->setText( auth->password() );
    }

    if ( userEdit->text().stripWhiteSpace().isEmpty() || ( !userEdit->text().stripWhiteSpace().isEmpty() && !m_passwdEdit->text().stripWhiteSpace().isEmpty() ) )
    {
      userEdit->setFocus();
    }
    else
    {
      m_passwdEdit->setFocus();
    }
  }
  else
  {
    if ( kapp->config()->hasGroup( "Homes Shares" ) )
    {
      kapp->config()->setGroup( "Homes Shares" );

      if ( kapp->config()->hasKey( host ) )
      {
        user_names = kapp->config()->readListEntry( host, ',' );
      }
    }

    if ( !user_names.isEmpty() )
    {
      userCombo->insertStringList( user_names, 0 );
    }

    userCombo->setCurrentText( QString::null );
    userCombo->setFocus();
   
    connect( userCombo, SIGNAL( activated( const QString & ) ), SLOT( slotGetPassword( const QString & ) ) );
  }
  
  //
  // Show the dialog and process the user supplied authentication
  // data.
  //
  bool new_password = false;
  passwdDlg->setInitialSize( passwdDlg->sizeHint(), true );
  
  if ( passwdDlg->exec() == KDialogBase::Accepted )
  {
    if ( share != "homes" )
    {
      if ( share.isEmpty() || use->isChecked() )
      {
        newAuth->setShare( "*" );
      }
      else
      {
        newAuth->setShare( share );
      }

      // Since we want to allow the user to provide an empty password,
      // only the user name is checked to be not empty.
      if ( !userEdit->text().stripWhiteSpace().isEmpty() )
      {
        newAuth->setUser( userEdit->text().stripWhiteSpace() );
        newAuth->setPassword( m_passwdEdit->text().stripWhiteSpace() );
        commit( newAuth );     
      }
    }
    else
    {
      if ( !userCombo->currentText().stripWhiteSpace().isEmpty() )
      {
        // Save the list of users to the config file.
        kapp->config()->setGroup( "Homes Shares" );
   
        if ( user_names.contains( userCombo->currentText().stripWhiteSpace() ) == 0 )
        {
          user_names.append( userCombo->currentText().stripWhiteSpace() );
        }   

        kapp->config()->writeEntry( host, user_names, ',' );
     
        // Put the share in the authentication data list:
        newAuth->setShare( userCombo->currentText().stripWhiteSpace() );
         
        // Save the authentication data:
        newAuth->setUser( userCombo->currentText().stripWhiteSpace() );
        newAuth->setPassword( m_passwdEdit->text().stripWhiteSpace() );
        commit( newAuth );
      }
    }
    new_password = true;
  }
  
  return new_password;
}


/****************************************************************************
   Updates the authentication information list.
****************************************************************************/

void Smb4KPasswdReader::updateAuthList( const QValueList<Smb4KAuthInfo *> &list )
{
  m_list.clear();
  m_list = list;
  
  write();
}


/****************************************************************************
   Obfuscation of the passwords.
****************************************************************************/

const QString Smb4KPasswdReader::obfuscate( const QString &password, bool encrypt )
{
  QString passwd;
  
  if ( encrypt )
  {
    uint index( 0 );
    while ( index < password.length() )
    {
      passwd[index] = (char)( (int)password[index]+m_key%62 );
      index++;
    }
  }
  else if ( !encrypt )
  {
    uint index( 0 );
    while ( index < password.length() )
    {
      passwd[index] = (char)( (int)password[index]-m_key%62 );
      index++;
    }
  }
    
  return passwd;
}


#ifdef __FreeBSD__

/****************************************************************************
   Write to the ~/.nsmbrc file
****************************************************************************/

void Smb4KPasswdReader::writeToSMBConfFile()
{
  QDir::setCurrent( QDir::homeDirPath() );
  
  QString wins = ((Smb4KCore *)parent())->getWINSServer();
  QMap<QString, QString> options = ((Smb4KCore *)parent())->getGlobalSambaOptions();
  
  QFile file( ".nsmbrc" );
  
  QStringList contents;
  
  if ( file.exists() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );

      while ( !ts.atEnd() )
        contents.append( ts.readLine().stripWhiteSpace() );

      file.close();
    }
     
    if ( contents.grep( "[default]" ).count() == 0 )
    {
      contents.append( "[default]" );
      
      if ( !wins.isEmpty() )
      {
        contents.append( "nbns="+wins );
      }
      
      if ( !options["workgroup"].isEmpty() )
      {
        contents.append( "workgroup="+options["workgroup"] );      
      }
        
      contents.append( QString::null );
    }    
  }
  else
  {
    contents.append( "[default]" );
    
    if ( !wins.isEmpty() )
    {
      contents.append( "nbns="+wins );
    }
      
    if ( !options["workgroup"].isEmpty() )
    {
      contents.append( "workgroup="+options["workgroup"] );
    }
      
    contents.append( QString::null );
  }
  
  // Now write the new data to ~/.nsmbrc. Passwords will be stored in plain text, but I DON'T 
  // CARE! I tried several hours to make encryption work with smbutil in Smb4KShellIO, but 
  // I got only crashes.
  
  for ( QValueListIterator<Smb4KAuthInfo *> it = m_list.begin(); it != m_list.end(); ++it )
  {
    if ( (*it)->share().isEmpty() )
    {
      continue;
    }
    else
    {
      if ( contents.grep( "["+(*it)->host()+":"+(*it)->user()+":"+(*it)->share()+"]", false ).count() == 0 )
      {
        contents.append( "["+(*it)->host().upper()+":"+(*it)->user().upper()+":"+(*it)->share().upper()+"]" );
        
        if ( !(*it)->workgroup().isEmpty() )
        {
          contents.append( "workgroup="+(*it)->workgroup().upper() );
        }
        
        contents.append( "password="+(*it)->password() );
        contents.append( QString::null );      
      }
      else
      {
        QStringList::Iterator i = contents.find( contents.grep( "["+(*it)->host()+":"+(*it)->user()+":"+(*it)->share()+"]", false ).first() );
        ++i;
        
        while ( !(*i).stripWhiteSpace().isEmpty() && !(*i).stripWhiteSpace().startsWith( "[" ) )
        {
          if( (*i).startsWith( "password", false ) )
          {
            QString pwd = (*i).section( "=", 1, 1 ).stripWhiteSpace();
            
            if ( pwd != (*it)->password() )
	    {
              (*i).replace( pwd, (*it)->password() );
	    }
          }
          
          ++i;
        }
      }
    }
  }
  
  QDir::setCurrent( QDir::homeDirPath() );
  
  // Write to the file.
  if ( file.open( IO_WriteOnly ) )
  {
    QTextStream ts( &file );
    
    for ( QStringList::ConstIterator it = contents.begin(); it != contents.end(); ++it )
      ts << *it << endl;
    
    file.close();
  }  
    
  // Get minimal security: Fix permissions.
  QString path( QDir::homeDirPath()+"/"+file.name() );
  chmod( path.ascii(), 00600 ) != 0;    
}

#endif


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////


/****************************************************************************
   Returns the password for a given user. This is mainly used by the 
   askpass dialog.
****************************************************************************/

void Smb4KPasswdReader::slotGetPassword( const QString &user )
{
  if ( passwdDlg )
  {
    QString wg = ((QLabel *)passwdDlg->child( "Workgroup", "QLabel", true ))->text();
    QString host = ((QLabel *)passwdDlg->child( "Host", "QLabel", true ))->text();
    Smb4KAuthInfo *auth = getAuth( wg, host, user );
    
    m_passwdEdit->clear();
    
    if ( !auth->password().isEmpty() )
      m_passwdEdit->setText( auth->password() );
  }
}


#include "smb4kpasswdreader.moc"
