/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif

#ifdef ENABLE_CDAUDIO

#include "../../player.h"
#include "../../output.h"
#include "../../other.h"
#include "../../util.h"
#include "cdaudio.h"

#ifdef ENABLE_GTK
#include "../../gtk/statusbar.h"
#endif

int16_t cdaudio_swap16 (int16_t);

void
cdaudio_run (void)
{
  int track = atoi (playing_file() + strlen ("cdrom://"));
  register int i;
  pthread_t buf;
  struct stat st;

  cdrom_drive *d;
  cdrom_paranoia *p;

  format_data_cdaudio *cdaudio =
    (format_data_cdaudio *) play->format_current->data;

  if (!cdaudio)
    return;

  d = cdda_find_a_cdrom (CDDA_MESSAGE_PRINTIT, NULL);

  if(!d) {
    msg_error (_("Cdaudio no present."));
    return;
  }

  if(!d && !lstat (play->cd, &st))
    d = cdda_identify (play->cd, CDDA_MESSAGE_PRINTIT, NULL);

  if (!d) {
    msg_error (_("Cdaudio error."));
    return;
  }

  cdda_verbose_set (d, CDDA_MESSAGE_PRINTIT, CDDA_MESSAGE_PRINTIT);

  if (cdda_open (d)) {
    msg_error (_("Cdaudio error."));
    return;
  }

  cdaudio->first = cdda_track_firstsector (d, track);
  cdaudio->last = cdda_track_lastsector (d, track);
  cdaudio->current = cdaudio->first;

  msg (_("Read: Track %d of %d; Frames %d to %d"), track, d->tracks,
       cdaudio->first, cdaudio->last);

  p = paranoia_init (d);
  paranoia_modeset (p, PARANOIA_MODE_FULL ^ PARANOIA_MODE_NEVERSKIP);
  paranoia_seek (p, cdaudio->first, SEEK_SET);

  cdaudio->channels = cdda_track_channels (d, track);

  if (cdaudio->channels > 2)
    {
      msg (_("No > 2 channels."));
      return;
    }

  msg (_("Play file audio: %s\n"), playing_name());

  output_open (44100, cdaudio->channels,16);
  output_info ();

  cdaudio->id_tail = 0;
  cdaudio->id_head = 0;
  cdaudio->go = 0;

  if (pthread_mutex_init (&cdaudio->mutex, NULL))
    fatal (_("Mutex error."));

  while (cdaudio->current <= cdaudio->last)
    {
      int16_t *readbuf = paranoia_read_limited (p, NULL, 20);
      char *err = cdda_errors (d);
      char *mes = cdda_messages (d);

      if (mes)
	{
	  fprintf (stderr, "%s\n", mes);
	  free (mes);
	}

      if (err)
	{
	  fprintf (stderr, "%s\n", err);
	  free (err);
	}

      if (big_endian ())
	for (i = 0; i < CD_FRAMESIZE_RAW / 2; i++)
	  readbuf[i] = cdaudio_swap16 (readbuf[i]);

      while (cdaudio->id_head == (cdaudio->id_tail + 1)
	     || (!cdaudio->id_head
		 && cdaudio->id_tail == CDAUDIO_CODE_MAX - 1))
	{
	  if (events.quit || events.skip)
	    break;

	  if (!play->play)
	    {
	      fprintf (stderr, _("* Buffering: %4d/%d frames \n\n"),
		       CDAUDIO_CODE_MAX, CDAUDIO_CODE_MAX);

#ifdef ENABLE_GTK
	      if (play->graphic)
		msg_statusbar (_("Buffering: %4d/%d frames"),
			       CDAUDIO_CODE_MAX, CDAUDIO_CODE_MAX);
#endif

	      if (pthread_create (&buf, NULL, cdaudio_play, NULL))
		fatal (_("Thread error."));

	      play->play = 1;
	    }

	  sched_yield ();
	}

      if (!play->play)
	fprintf (stderr, _("* Buffering: %4d/%d frames \r"),
		 cdaudio->id_tail + 1, CDAUDIO_CODE_MAX);

#ifdef ENABLE_GTK
      if (play->graphic)
	msg_statusbar (_("Buffering: %4d/%d frames"), cdaudio->id_tail + 1,
		       CDAUDIO_CODE_MAX);
#endif

      if (events.quit || events.skip)
	break;

      pthread_mutex_lock (&cdaudio->mutex);

      memcpy (cdaudio->buffer + (cdaudio->id_tail * CD_FRAMESIZE_RAW),
	      readbuf, CD_FRAMESIZE_RAW);

      if (cdaudio->id_tail == CDAUDIO_CODE_MAX - 1)
	cdaudio->id_tail = 0;
      else
	cdaudio->id_tail++;

      pthread_mutex_unlock (&cdaudio->mutex);

      cdaudio->current++;
    }

  cdaudio->go = 1;

  if(play->play)
    pthread_join (buf, NULL);

  paranoia_free (p);

  cdda_close (d);

  output_close ();

  pthread_mutex_destroy (&cdaudio->mutex);

}

void *
cdaudio_play (void *arg)
{
  format_data_cdaudio *cdaudio =
    (format_data_cdaudio *) play->format_current->data;

  if (!cdaudio)
    pthread_exit (NULL);

  while (1)
    {

      if (events.quit || events.skip)
	pthread_exit (NULL);

      if (events.pause)
	{
	  sched_yield ();
	  continue;
	}

      if (cdaudio->id_tail < 0)
	{
	  sched_yield ();
	  continue;
	}

      while (cdaudio->id_head == cdaudio->id_tail && !cdaudio->go)
	sched_yield ();

      if (cdaudio->go)
	break;

      pthread_mutex_lock (&cdaudio->mutex);

      output_write (cdaudio->buffer + (cdaudio->id_head * CD_FRAMESIZE_RAW),
		    CD_FRAMESIZE_RAW);

      if (cdaudio->id_head == CDAUDIO_CODE_MAX - 1)
	cdaudio->id_head = 0;
      else
	cdaudio->id_head++;

      pthread_mutex_unlock (&cdaudio->mutex);

      sched_yield ();
    }

  while (cdaudio->id_head != cdaudio->id_tail)
    {
      output_write (cdaudio->buffer + (cdaudio->id_head * CD_FRAMESIZE_RAW),
		    CD_FRAMESIZE_RAW);

      if (cdaudio->id_head == CDAUDIO_CODE_MAX - 1)
	cdaudio->id_head = 0;
      else
	cdaudio->id_head++;
    }

  play->play = 0;

  pthread_exit (NULL);
}

int16_t
cdaudio_swap16 (int16_t x)
{
  return ((((u_int16_t) x & 0x00ffU) << 8) |
	  (((u_int16_t) x & 0xff00U) >> 8));
}

char *
cdaudio_get_time (void)
{
  long offset_1, offset_2;
  static char output[100];
  format_data_cdaudio *cdaudio =
    (format_data_cdaudio *) play->format_current->data;

  if (!cdaudio)
    return NULL;

  offset_1 = cdaudio->current - cdaudio->first;
  offset_2 = cdaudio->last - cdaudio->current;

  if (cdaudio->id_head == (cdaudio->id_tail + 1)
      || (!cdaudio->id_head && cdaudio->id_tail == CDAUDIO_CODE_MAX - 1))
    {
      offset_1 -= CDAUDIO_CODE_MAX;
      offset_2 += CDAUDIO_CODE_MAX;
    }
  else if (cdaudio->id_head < cdaudio->id_tail)
    {
      offset_1 -= cdaudio->id_tail - cdaudio->id_head;
      offset_2 += cdaudio->id_tail - cdaudio->id_head;
    }
  else if (cdaudio->id_head > cdaudio->id_tail)
    {
      offset_1 -=
	(CDAUDIO_CODE_MAX - cdaudio->id_head) + 1 + cdaudio->id_tail;
      offset_2 +=
	(CDAUDIO_CODE_MAX - cdaudio->id_head) + 1 + cdaudio->id_tail;
    }

  sprintf (output, "T: %02d:%02d.%02d [%02d:%02d.%02d] ",
	   (int) (offset_1 / (60 * 75)), (int) ((offset_1 / 75) % 60),
	   (int) (offset_1 % 75), (int) (offset_2 / (60 * 75)),
	   (int) ((offset_2 / 75) % 60), (int) (offset_2 % 75));

  return output;
}

int
cdaudio_get_buffer (void)
{
  format_data_cdaudio *cdaudio =
    (format_data_cdaudio *) play->format_current->data;

  if (!cdaudio)
    return 0;

  if (cdaudio->id_head == cdaudio->id_tail)
    return 0;

  if (cdaudio->id_head == (cdaudio->id_tail + 1)
      || (!cdaudio->id_head && cdaudio->id_tail == CDAUDIO_CODE_MAX - 1))
    return 100;

  if (cdaudio->id_head < cdaudio->id_tail)
    return (cdaudio->id_tail - cdaudio->id_head) * 100 / CDAUDIO_CODE_MAX;

  return ((CDAUDIO_CODE_MAX - cdaudio->id_head) + 1 +
	  cdaudio->id_tail) * 100 / CDAUDIO_CODE_MAX;
}
#endif
