/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_MP3

#include "../../player.h"
#include "../../other.h"
#include "../../buffer.h"
#include "../../output.h"
#include "../../file.h"
#include "../../util.h"
#include "mp3.h"

#define PRNG(x) ((x * 0x0019660dL + 0x3c6ef35fL) & 0xffffffffL)

char *type_layer (enum mad_layer);
char *type_channel (enum mad_mode);

enum mad_flow
read_file (void *data, struct mad_stream *stream)
{
  int bytes = stream->bufend - stream->next_frame;
  int len = 0;
  char *rptr=(char *)data;

  if (events.skip || events.quit)
    return MAD_FLOW_STOP;

  if (bytes)
    memmove (data, stream->next_frame, bytes);

  if (bytes != SIZE_BUFFER) {

  if (!(len = buffer_pop (rptr + bytes, SIZE_BUFFER - bytes)))
    {
      events.skip = 1;
      return MAD_FLOW_STOP;
    }

  mad_stream_buffer (stream, data, bytes + len);
  } 
  
  /* Debug note: try to play this frame. Other mode is: bytes=0 and re read
   * data */
  else
    mad_stream_buffer (stream, data, bytes);

  return MAD_FLOW_CONTINUE;
}

enum mad_flow
read_header (void *data, struct mad_header const *header)
{
  format_data_mp3 *mp3 = (format_data_mp3 *) play->format_current->data;

  if (!mp3)
    return MAD_FLOW_STOP;

  if (events.skip || events.quit)
    return MAD_FLOW_STOP;

  if (!mp3->frame && !remote_file (playing_file()))
    {
      double time = (play->length * 8.0) / (header->bitrate);
      double timefrac = (double) time - ((long) (time));
      long nsamples = 32 * MAD_NSBSAMPLES (header);

      mp3->n_frames = (long) (time * header->samplerate / nsamples);

      mad_timer_set (&mp3->mad_duration, (long) time,
		     (long) (timefrac * 100), 100);
    }

  mp3->frame++;
  if (mp3->frame > 999999)
    mp3->frame = 1;

  mad_timer_add (&mp3->mad_time, header->duration);

  if (play->done == INIT)
    {
      fprintf (stderr, _("\tLayer %s - %d Hz - %ld kbit/s - %s\n"),
	       type_layer (header->layer), header->samplerate,
	       header->bitrate / 1000, type_channel (header->mode));

      fprintf (stderr, _("\tChannels %d - " "Copyright %s - " "Original %s - "
			 "Protection %s\n"), MAD_NCHANNELS (header),
	       header->flags & MAD_FLAG_COPYRIGHT ? "yes" : "no",
	       header->flags & MAD_FLAG_ORIGINAL ? "yes" : "no",
	       header->flags & MAD_FLAG_PROTECTION ? "yes" : "no");

      fprintf (stderr,
	       _("\tMode Extension %d - " "BPF %ld - " "Enphasis %d - "
		 "Extension %d\n\n"), header->mode_extension,
	       (header->bitrate / 100) * mad_timer_count (header->duration,
							  MAD_UNITS_CENTISECONDS),
	       header->emphasis, header->mode_extension);

      msg (_("Play file audio: %s\n"), playing_name());

      play->done = PLAY;

    }
  else
    play->play = 1;

  return MAD_FLOW_CONTINUE;
}

char *
type_channel (enum mad_mode mode)
{
  switch (mode)
    {
    case MAD_MODE_SINGLE_CHANNEL:
      return _("Single-Channel");
    case MAD_MODE_DUAL_CHANNEL:
      return _("Dual-Channel");
    case MAD_MODE_JOINT_STEREO:
      return _("Joint-Stereo");
    case MAD_MODE_STEREO:
      return _("Stereo");
    default:
      return _("Unknow");
    }
}


char *
type_layer (enum mad_layer layer)
{
  switch (layer)
    {
    case MAD_LAYER_I:
      return "I";
    case MAD_LAYER_II:
      return "II";
    case MAD_LAYER_III:
      return "III";
    default:
      return _("Unknow");
    }
}

enum mad_flow
mad_write (void *data, struct mad_header const *header, struct mad_pcm *pcm)
{
  register int pcmlength = pcm->length;
  mad_fixed_t const *left_ch = pcm->samples[0], *right_ch = pcm->samples[1];

  static unsigned char stream[1152 * 4];
  static unsigned int rate = 0;
  static int channels = 0;
  static struct audio_dither dither;

  unsigned char *ptr = stream;
  int sample;

  if (!play->play)
    {
      channels = 0;
      rate = 0;
    }

  if (channels > 2)
    {
      fprintf (stderr, _("No support channels > 2."));
      events.skip = 1;
      return MAD_FLOW_STOP;
    }

  if (!rate || !channels)
    {
      channels = MAD_NCHANNELS (header);
      rate = header->samplerate;
      output_open (header->samplerate, ((header)->mode ? 2 : 1), 16);
      output_info ();
    }

/* Can rate and channels change? I must check it?
 * channels != MAD_NCHANNELS (header) || rate != header->samplerate 
 */

  else if (!output_check ())
    {
      output_close ();
      channels = MAD_NCHANNELS (header);
      rate = header->samplerate;
      output_open (header->samplerate, ((header)->mode ? 2 : 1), 16);
      rate = header->samplerate;
    }

  if (pcm->channels == 2)
    {
      while (pcmlength--)
	{
	  sample = mad_dither (16, *left_ch++, &dither);

	  if (!big_endian ())
	    {
	      *ptr++ = (unsigned char) (sample >> 0);
	      *ptr++ = (unsigned char) (sample >> 8);
	    }
	  else
	    {
	      *ptr++ = (unsigned char) (sample >> 8);
	      *ptr++ = (unsigned char) (sample >> 0);
	    }

	  sample = mad_dither (16, *right_ch++, &dither);

	  if (!big_endian ())
	    {
	      *ptr++ = (unsigned char) (sample >> 0);
	      *ptr++ = (unsigned char) (sample >> 8);
	    }
	  else
	    {
	      *ptr++ = (unsigned char) (sample >> 8);
	      *ptr++ = (unsigned char) (sample >> 0);
	    }
	}

      output_write (stream, pcm->length * 4);
    }

  else
    {
      while (pcmlength--)
	{
	  sample = mad_dither (16, *left_ch++, &dither);

	  if (!big_endian ())
	    {
	      *ptr++ = (unsigned char) (sample >> 0);
	      *ptr++ = (unsigned char) (sample >> 8);
	    }
	  else
	    {
	      *ptr++ = (unsigned char) (sample >> 8);
	      *ptr++ = (unsigned char) (sample >> 0);
	    }
	}

      output_write (stream, pcm->length * 2);
    }

  return MAD_FLOW_CONTINUE;
}

long
mad_dither (unsigned int bits, mad_fixed_t sample,
	    struct audio_dither *dither)
{
  unsigned int scalebits;
  mad_fixed_t output, mask, random;

  /* noise shape */
  sample += dither->error[0] - dither->error[1] + dither->error[2];

  dither->error[2] = dither->error[1];
  dither->error[1] = dither->error[0] / 2;

  /* bias */
  output = sample + (1L << (MAD_F_FRACBITS + 1 - bits - 1));

  scalebits = MAD_F_FRACBITS + 1 - bits;
  mask = (1L << scalebits) - 1;

  /* dither */
  random = PRNG (dither->random);
  output += (random & mask) - (dither->random & mask);

  dither->random = random;

  /* clip */
  if (output > MAD_F_ONE - 1)
    {
      output = MAD_F_ONE - 1;

      if (sample > MAD_F_ONE - 1)
	sample = MAD_F_ONE - 1;
    }
  else if (output < -MAD_F_ONE)
    {
      output = -MAD_F_ONE;

      if (sample < -MAD_F_ONE)
	sample = -MAD_F_ONE;
    }

  /* quantize */
  output &= ~mask;


  /* error feedback */
  dither->error[0] = sample - output;

  /* scale */
  return output >> scalebits;
}

#endif
