/* ==================================================== ======== ======= *
 *
 *  ubrick.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ubrick_hpp_
#define	_ubrick_hpp_
//pragma ident	"@(#)ubrick.hpp		ubit:03.06.04"
#include <typeinfo>
#include <stdlib.h>  // for size_t, new(), delete()
#include <ubit/udefs.hpp>
#include <ubit/ubrickImpl.hpp>
#include <ubit/uerror.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Ubit Brick: base class for objects that can be added to the
*  instance graph (abstract class).
*
*  IMPORTANT NOTES:
*
*  1. When an object is destroyed, its direct and indirect children are
*  also destroyed except if:
*  - they have other (undestroyed) parents
*  - they are pointed by uptr's (Ubit "smart pointers", see below)
*
*  2. C++ pointers and references are unsafe and should only be used for
*  temporary variables. Ubit provides "smart pointers" that manage memory
*  in a simple and safe way (see template uptr<>).
*
*  Objects pointed by smart pointers:
*  - can't be destroyed by 'delete' (an error will be issued in this case)
*  - are automatically deleted when :
*    #- they are not part of the instance graph, and
*    #- no smart pointer points to them any longer
*
*  Hence, these objects pointed are automatically destroyed when needed:
*  <pre>
*     uptr<UBox> box = new UBox();
*     uptr<UButton> btn = ubutton("my button");
*     box->add(*btn);
*     btn = null;    // 'btn' does not point any longer to the UButton
*     box = null;    // destroys the UBox and its UButton child
*                    // if they do no have other parents and are not
*                    // pointed by other smart pointers
*   </pre>
*
*  3. Alternatively, C++ variables can also be "plain objects" instead
*  of pointers. Objects created in this way are destroyed automatically.
*  Note that:
*  - when an object is destroyed, its member variables that are "plain
*    objects" are automatically destroyed
*  - when a function returns, its local variables that are "plain
*    objects" are automatically destroyed
*
*  <pre>
*  void foo() {
*     UStr str ="abcde";
*  }  // the UStr is destroyed
*  </pre>
*
*  4. The = operator copies the content of C++ objects. UBrick and most
*  of its subclasses forbid this operator (the set() method must be
*  used instead)
*
* See also: UBrick::~UBrick and uptr<>
*/
class UBrick {  
  UBrick(const UBrick&);
  UBrick& operator=(const UBrick&);
  ///< assigment is forbidden (except for certain subclasses).

public:

  UBrick(u_modes b_modes = 0);
  /**<
    * standard constructor; see class UBrick and UBrick::~UBrick.
    * argument 'b_mode' is used internally and should not be set by clients.
    */

  virtual ~UBrick() {destructs();}
  /**<
   * see <b>important notes</b> on <b>recursive deletion</b>.
   * <p>
   * All destructors <b>destroys the children</b> and descendants
   * of the object being deleted if:
   * - they do not have other (undeleted) parents, and
   * - they are not referenced by an uptr (Ubit smart pointer), and
   * - they have been created in the heap (ie. by using the 'new' primivite
   *   or a 'creator shortcut' such as ugroup())
   *
   * <b>Advices:</b>
   * - avoid using the <b>delete</b> primivite and prefer the
   *   UGroup::remove() or UGroup::removeAll() functions
   *
   * - 'delete' does not update graphics and may produce unexpected 
   *   results or generate errors because objects:
   *   -# can be shared (= they can have several parents)
   *   -# can be created in the static memory or in the stack
   *   -# can be referenced by uptr's (Ubit smart pointers)
   *  deleting objects is illegal in the two later cases and will 
   *  generate an error or an exception.
   *
   * - See also: UGroup::remove(), UGroup::removeAll(), and uptr<>
   */

  static const char* getUbitVersion();
  ///< returns the version of the Ubit package.

  virtual const char* getClassName() const {return typeid(*this).name();}
  ///< returns the name of the class of <i>this</i> object.

  void setAutoUpdate(bool);
  ///< specifies whether update() is invoked when set(), add() or remove() are called.

  bool isAutoUpdate() const;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // access to parents

  virtual int getParentCount() const;
  /**< returns the number of parents. */

  virtual UGroup** getParents() const;
  virtual UGroup** getParents(int &parent_count) const;
  /**< 
   * returns a copy of the object's parent list.
   * args:
   * - returns null if the object has no parent and a copy of the
   *   parent list otherwise.
   * - the returned list must be destroyed by using: delete[]
   * notes:
   * - remember that bricks can have several parents! 
   */

  virtual int getParents(std::vector<UGroup*>& parent_vect) const;
  /**<
   * returns a copy of the object's parent list (and the parent count).
   * args:
   * - 'parent_vect' will contain a copy of the parent list. 
   *   note that this vector is not cleared: parents are appended to
   *   its current elements if it is not empty.
   * - return value: the number of parents.
   * notes:
   * - remember that bricks can have several parents! 
   */

  virtual UGroup* getParent(int pos) const;
  /**< 
   * returns the nth parent.
   * args:
   * - 'pos' = 0 means "first parent" and 'pos' = -1 means "last parent"
   * - returns null and issues a warning message if 'pos' is out of range
   * notes:
   * - remember that bricks can have several parents! 
   * - use getParents() is you need to access several parents:
   *   this will be more efficient than multiple calls to getParent()
   */

  virtual bool isChildOf(class UGroup *possible_parent, bool indirect) const;
  /**<
   * is this object a child of 'possible_parent'?.
   * args:
   * - second argument specifies if we accept indirect childhood (indirect' 
   * is true) or if we just consider direct children ('indirect' is false)
   */

  template<class CC>
    static bool isInstance(const UBrick*_obj) {return dynamic_cast<const CC*>(_obj);}
  /**<
    * is 'obj' an instance of 'CC'.
    */
  
  virtual void removeFromParents(bool update_parents = true);
  /**< 
   * removes the brick from all parents.
   * notes:
   * 1) Ubit objects can have several parents!
   * 2) this function is potentially dangerous: the UGroup::remove() 
   *    and UGroup::removeAll() methods should be prefered when possible.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // conditionnal objects

  friend class ULink& operator/(const UCond&, UBrick&);
  friend class ULink& operator/(const UCond&, UBrick*);
  /**< 
   * conditional operator (for callback functions and conditionnal objects).
   * this operator is used for specifying call-back functions or
   * conditionnal objects (see classes UCall, UOn and UFlag).
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // errors and warnings

  static void error(const char* id, const char* msg);
  static void error(const char* id, const char* msg, long arg);
  static void error(const char* id, const char* msg, const char* arg);
  static void error(const char* id, const char* msg, const UStr& arg);
  static void error(const char* id, const char* msg, const std::string& arg);
  /**< throws an error (and prints an error message).
   * arguments:
   * - 'id' is in the form "class::function" or "type@class::function"
   *        with type being one of: warning, fatal, internal 
   * - 'msg' is the error message
   * See also: UError::error()
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  bool isBmode(u_modes some_Bmodes) const 
  {return ((bmodes & some_Bmodes) != 0);}
  ///< [impl] is ONE of these Brick modes verified ? (see class UMode for details).

  u_modes getBmodes() const {return bmodes;}
  ///< [impl] returns current Brick mode bitmask.

  void setBmodes(u_modes bmodes, bool on_off);
  ///< [impl] sets Brick modes to ON (resp OFF) if last arg is true (resp false). 

  void* operator new(size_t);
  void  operator delete(void*);
  void  addRef() {refcount++;}
  void  removeRef();
  u_count getRefCount() const {return refcount;}
  ///< [impl] implementation: memory management.

  virtual void fire(class UEvent&, const class UOn&) const;
  ///< [impl] calls callback functions.

  virtual class UProp*  propCast()   {return null;}
  virtual class UElem*  elemCast()   {return null;}
  virtual class UStr*   strCast()    {return null;}
  virtual class UGroup* groupCast()  {return null;}
  virtual class UBox*   boxCast()    {return null;}
  virtual class UWin*   winCast()    {return null;}
  virtual class UGroup* getSubGroup(){return null;}
  ///< [impl] implementation: simulates dynamic cast.

  virtual void addingTo(class ULink *selflink, class UGroup *parent);
  virtual void removingFrom(class ULink *selflink, class UGroup *parent);
  virtual void destructs();
  /**< 
   * [impl] see <b>important note</b> on <b>subclassing</b>.
   * destructs() unlinks the object from its parents and destroys children 
   * (when applicable). 
   * Any class that redefines removingFrom() MUST HAVE A DESTRUCTOR 
   * that calls destructs().
   */

#ifndef NO_DOC // - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual const char* cname() const {return getClassName();}
  ///< [impl] shortcut: same as getClassName().

  virtual class ULink* makeLink() {return new ULink(this);}
  const class ULLChain& getParentList() const {return parents;}
  bool hasValidParent() const;
  bool hasValidParent(bool& fake_or_deleted_par) const;

  void addToCache(UBrick&);
  void addChangeCall(UBrick&);
  void addCall(ULink&);

protected:
  friend class ULink;
  friend class UArgs;
  class ULLChain parents;
  class UChain*  cache;  // attributes and callbacks
  u_modes  bmodes;
  u_count refcount;
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/* uptr template : Ubit smart pointers.
 *
 * Ubit smart pointers make it possible to handle objects created in the
 * heap (= in dynamic memory) in a safe way. uptr(s) should be used for 
 * pointing objects created by 'new' or by 'creator shortcuts' (such as
 * ubutton(), ugroup(), ustr(), etc.) 
 *
 * Ubit smart pointers are somewhat similar to Java references. 
 * Objects pointed by uptr(s) are <b>automatically destroyed</b> when:
 * - they are not pointed by any uptr any longer, AND
 * - they do not have parents in the GUI graph, AND
 * - they have been allocated in the heap.
 *
 * !CAUTION: 'delete' can not be used to destroy object pointed by uptr(s)
 * (this would produce an error and the object would not be destroyed).
 * To destroy an object : set all uptrs that point to it to null (and 
 * remove this object from the GUI graph).
 *
 *  Examples:
 *  <pre><tt>
 *    uptr<UStr> p1 = new UStr("111");   // p1 points to a new string
 *    uptr<UStr> p2 = ustr("222");       // ustr(..) == *new UStr(...)
 *
 *    *p1 = *p2;        // the CONTENT of string "222" is copied into
 *                      // string "111" (but p1 and p2 still point to
 *                      // different objects)
 *
 *    p1 = p2;          // p1 now points to the same string as p2
 *                      // the string that was pointed by p1 is deleted
 *                      // as there is no other refernce to it
 *
 *    p1->append("xyz") // the content of the string pointed by p1 is changed
 *
 *    cerr << *p1;      // writes the content of the pointed UStr: 222xyz
 *    cerr << &p1;      // writes the address of the pointed UStr
 *
 *    p1 = null;        // p1 points to null. the UStr is deleted
 */
template <class CC>
class uptr {
  CC* obj;
  CC& operator[](int);  ///< [] meaningless and forbidden.

public:
  uptr()         {obj = null;}
  uptr(CC& _obj) {obj = &_obj; obj->addRef();}
  uptr(CC* _obj) {obj = _obj; if(obj) obj->addRef();}
  ///< constructors:  uptr<ObjClass> ptr = obj;

  uptr(const uptr<CC>& ref2) {obj = ref2.obj; if(obj) obj->addRef();}
  ///< constructor:  uptr<ObjClass> ptr = p2;

  ~uptr() {if (obj) obj->removeRef();}
  ///< destructor: deletes pointed object (if not null and not pointed elsewhere).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  uptr<CC>& operator=(CC& _obj) {
    if(obj) obj->removeRef(); obj = &_obj; if(obj) obj->addRef(); return *this;
  }
  uptr<CC>& operator=(CC* _obj) {
    if(obj) obj->removeRef(); obj = _obj; if(obj) obj->addRef(); return *this;
  }
  ///< assignment: ptr = obj; (obj can be null).

  uptr<CC>& operator=(const uptr<CC>& p2) {
    if(obj) obj->removeRef(); obj = p2.obj; if(obj) obj->addRef(); return *this;
  }
  ///< assignment: ptr = ptr2;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    
  operator CC* () const {return obj;}
  ///< type conversion: allows for:  ObjClass* obj = ptr.

  CC* operator&() const {return obj;}
  ///< returns pointed object: if 'ptr' points to 'obj' then &ptr return obj.

  CC& operator*() const {
    if (!obj) UError::error("uptr::operator*", "attempt to derefence a uptr (Ubit smart pointer) that points to null; ref:", (long)this);
    return *obj;
  }
  ///< dereferencing: if 'ptr' points to 'obj' then *ptr return *obj.

  CC* operator->() const {
    if (!obj) UError::error("uptr::operator->", "attempt to derefence a uptr (Ubit  smart pointer) that points to null; ref:", (long)this);
    return obj;
  }
  ///< dereferencing: if 'ptr' points to 'obj' then ptr->foo() calls obj->foo().
};

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif
