/* ==================================================== ======== ======= *
 *
 *  uuwin.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uuwin.cpp	ubit:03.06.04"
#include <ubrick.hpp>
#include <uerror.hpp>
#include <ucall.hpp>
#include <uprop.hpp>
#include <ustr.hpp>
#include <uctrl.hpp>
#include <ubox.hpp>
#include <uboxImpl.hpp>
#include <uwin.hpp>
#include <uwinImpl.hpp>
#include <uview.hpp>
#include <uviewImpl.hpp>
#include <uevent.hpp>
#include <ustyle.hpp>
#include <ucolor.hpp>
#include <uborder.hpp>
#include <ugraph.hpp>
#include <unatwin.hpp>
#include <unatdisp.hpp>
#include <uappli.hpp>
#include <update.hpp>
#include <ugadgets.hpp>
#include <iostream>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle *UWin::style = null;
const UStyle& UWin::makeStyle() {
  if (!style) {
    style = new UStyle();
    style->textSeparator  = &ustr("\n");
    style->orient         = UOrient::VERTICAL;
    style->local.halign   = UHalign::FLEX;
    //NEW 30dec: dialogs a la fois hflex et vflex
    style->local.valign   = UValign::FLEX;
    style->local.hspacing = 1;
    style->local.vspacing = 1;
    style->local.padding.set(0, 0);
    style->font     = &UFont::standard;
    style->fgcolors = UStyle::makeColors(&UColor::black,&UColor::white); 
    style->bgcolors = UStyle::makeColors(&UBgcolor::grey, &UBgcolor::black);
  }
  return *style;
}

/* ==================================================== ======== ======= */

UWin::UWin(const UArgs& a) : UBox(a) {
  // par defaut les fenetres ne sont pas affichees (sauf Frame)
  setCmodes(UMode::CAN_SHOW | UMode::BOX, false);
  setCmodes(UMode::WIN_AUTO_OPEN_MODE, true);
  is_hardwin = ! UAppli::getDefaults().softwins;
  winviews = null;
  impl.initialized = null; // impl NOT initialized
}

USoftwinImpl::USoftwinImpl(UWin& win) {
  //NOTE: SOFTWIN => FLOATING
  // (c'est fait implicitement via l'ajout d'un UPos aux softwins)
  pos.set(0, 0);
  win.insertAttr(0, pos, false);   // dans cache
  current_winview = null;
}

UHardwinImpl::UHardwinImpl(UWin& win) :
  wingraph(*new UWinGraph(&win)),
  sub_softwins(null) {
}

// ==================================================== ======== =======
// !ATTENTION: les appels aux fcts virtuelles ne marchent pas normalement
// dans les constructeurs et les DESTRUCTEURS (l'appel est fait avec
// la classe du destructeur, pas la classe effective)
// ==> toute redefinition de 'removingFrom' necessite un DESTRUCTEUR !!

UWin::~UWin() {
  //NB: removeFromParents et removeAll fait par ~UBox ou subclass

  UAppli* a = null;
  if (isDef(0,UMode::MODAL) && (a = UAppli::getApp()))
    a->removeModalwin(this);
  
  if (winviews) {    
    // views have been destroyed in UBox::deleteRelatedViews
    // except in some cases
    UView* nextview = null;
    for (UView* v = winviews; v != null; v = nextview) {
      nextview = v->getNext();
      delete v;
    }
    winviews = null;
  }

  if (impl.initialized) {
    if (is_hardwin) delete impl.hard; else delete impl.soft;
    impl.initialized = null;
  }
}

/* ==================================================== ======== ======= */

USoftwinImpl::~USoftwinImpl() {
  //pos = null;   pos n'est plus un ptr mais un objet
}

UHardwinImpl::~UHardwinImpl() {
  // enlever cette window de la liste des win managees par le disp
  if (wingraph.getDisp()) {
    wingraph.getDisp()->removeHardwin(wingraph.getHardwin());
    // warn the UMS server
    if (getNatWin()) 
      wingraph.getNatDisp()->showNotify(getNatWin()->getXWindow(), false);
  }

  delete &wingraph;

  if (sub_softwins) {
    //NB: softwins est detruit implicitement (car rajoute a la childlist) ???
    sub_softwins = null;  //securite
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
//ATT: cette fct doit etre appelee a la creation de la UWin

void UWin::setSoftwinMode(bool state) {
  if (impl.initialized) {
    error("setSoftwinMode", UError::Should_not_be_realized);
    return;
  }
  if (state) {
    is_hardwin = false;
    setCmodes(UMode::HARDWIN, false);
    setCmodes(UMode::SOFTWIN, true);
  }
  else {
    is_hardwin = true;
    setCmodes(UMode::HARDWIN, true);
    setCmodes(UMode::SOFTWIN, false);
  }
}

ULink* UWin::getSubSoftwinLink() const {
  if (is_hardwin)
    return impl.hard ? impl.hard->getSoftwinLink() : null;
  else return null;
}

UGroup* UHardwinImpl::getSoftwins() const {
  return (UGroup*)(sub_softwins ? sub_softwins->getChild() : null);
} 

void UHardwinImpl::removeSoftwin(UWin* softwin) {
  if (softwin->impl.soft) softwin->impl.soft->current_winview = null;

  // recuperer  la softlist de la hardwin
  // si dans la liste => l'enlever (sans message d'erreur!)
  ULink* prevlink = null;
  bool upd = false;
  UGroup* soft_list = getSoftwins();

  if (soft_list
      && soft_list->getChildImpl(UBox::ELEM_LIST, softwin, 0, prevlink, null))
    soft_list->removeImpl(UBox::ELEM_LIST, softwin, prevlink, 
			  UGroup::NO_DEL, upd, null);
}

void UHardwinImpl::addSoftwin(UWin* softwin,
			      UWin* hardwin, UView* hardwin_view) {
  UView* softwin_view = softwin->realizeSoftwin(hardwin, hardwin_view);
  // normalement impl.soft a ete cree par realizeSoftwin
  softwin->impl.soft->current_winview = softwin_view;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NB: disp can be null (default then used)

UView* UWin::getWinView(class UDisp* disp) const {
  // hardwins have ONLY ONE shared view in the current version !!
  if (is_hardwin) return winviews;

  // default is the first view
  if (disp == null) return winviews;

  for (UView* v = winviews; v!= null; v = v->getNext()) {
    if (v->getDisp() == disp) return v;
  }
  return null; // not found
}

void UWin::addWinView(class UView* view) {
  // hardwins have ONLY ONE shared view in the current version !!
  if (is_hardwin && winviews) {
    UError::error("internal@UWin::addWinView",
		  "Hardwindows can have only one shared view");
    return;
  }

  UView* v = getWinView(view->getDisp());
  if (v) UError::error("internal@UWin::addWinView", 
		       "View already in list for this UDisp");
  view->setNext(winviews); winviews = view;

  //cerr << " addwinview: win: " << this << " / " ;
  //for (UView* v = winviews; v!= null; v = v->getNext()) cerr << v << " ";
  //cerr << endl;
}

UWinGraph* UWin::getWinGraph(UDisp* disp) const {
  UView* winview = getWinView(disp);
  return winview ? winview->getWinGraph() : null;
}

UNatWin* UWin::getNatWin(UDisp* disp) const {
  UView* winview = getWinView(disp);
  return winview ? winview->getNatWin():null;
}

UAppli* UWin::getAppli() const {
  // only one UAppli !
  UView* winview = getWinViews();
  return winview ? winview->getAppli() : null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// ATT: voir aussi + voir cas Displays separes!!
// !!POUR L'instant winview UNIQUE!!!
// NOTE: les Windows sont creees une seule fois mais ouvertes par tous 
// leurs parents

void UWin::initView(ULink *selflink, UView *parview) {
  _initView(selflink, parview, parview->getDisp());
  if (isShowable()) show(true);
}

// optimization: les vues des enfants de la UWIn seront crees ulterieurement,
// au premier show() de la UWin, via la fonction realizeChildren()
//att: parview n'est pas la view du parent mais du 1er parent de type UBox!

UView* UWin::_initView(ULink *selflink, UView *parview, UDisp *disp) {

  // par construction
  UBoxLink *wlink = static_cast<UBoxLink*>(selflink);
  if (!wlink) {
    error("internal@UWin::initView", "incorrect link type");
    return null;
  }

  UView* winview = getWinView(disp);

  // Il n'y a qu'une seule winview par Window
  // ==> ne creer et n'initialiser les descendants que la premiere fois
  if (!winview) {
    UWinGraph *wingraph = null;

    if (is_hardwin) {
      setCmodes(UMode::HARDWIN, true);
      setCmodes(UMode::SOFTWIN, false);
      // ATT: cree une seule fois => only one shared window
      // et pas possible de changer de softwin mode apres creation

      if (!impl.hard) impl.hard = new UHardwinImpl(*this); 
      wingraph = &impl.hard->wingraph;

      // ajouter a la liste des hardwins de UDisp
      disp->addHardwin(this);
    }
    else {			// softwin
      setCmodes(UMode::HARDWIN, false);
      setCmodes(UMode::SOFTWIN, true);
      // ATT: cree une seule fois => les winviews partagent le meme upos
      // et pas possible de changer de softwin mode apres creation
      if (!impl.soft) impl.soft = new USoftwinImpl(*this); 

      // dans le cas des softwin il faut un parview, et il faut de plus
      // reinitialser la vue partagee a chaque apparition !!!!
      if (!parview) {
	//NB: parview necessaire dans ce cas pour: parview->getWinGraph(); 
	//    (et aussi optionnellement pour getStyle(closest_parent)
	error("internal@UWin::initView", "No parent view!");
	return null;
      }
      wingraph = parview->getWinGraph();
    }

    if (!wingraph) {
      error("internal@UWin::initView", UError::Cant_realize_window);
      return null;
    }
    
    const UViewStyle *render;
    if (isDef(0,UMode::HAS_RENDERER)) {
      // si renderer defini dynamiquement dans la childlist
      render = (UViewStyle*) getAnyChild(&UBrick::isInstance<UViewStyle>);
    }
    else {  // default: prendre le renderer defini par le style de la Win
      //NB: closest_parent: ne sert en fait pas a grand chose (sauf dans des
      //cas de styles contextuels ou le renderer changerait suivant parent)
      /*22sept03
      UBox *closest_parent = parview ? parview->getBox() : null;
      render = getStyle(closest_parent).viewStyle;
       */
      render = getStyle(null).viewStyle;
    }

    // !note: winview est une var. d'instance!
    if (render)
      winview = (render->makeView)(wlink, null, wingraph);
    else {
      error("init", UError::Cant_retreive_style);
      winview = new UView(wlink, null, wingraph);
    }

    // initialize le WinGraph (mais ne cree pas la X Window: ce sera fait
    // ulterieurement par UWin::realize() qunad la fenetre apparaitra
    // la 1ere fois (NB: a partir de ce moment, UGraph::isWinRealized()
    // renverra true (mais pas avant)
    if (is_hardwin) wingraph->init(disp, winview);

    // add winview to the list of the window views
    addWinView(winview);
  } //endif(!winview)

  // WinView PARTAGEE (important pour UBoxLink::deleteRelatedViews)
  // winview->setVmodes(UView::WIN_SHARED, true);

  // faire pointer BoxLink->views vers la View associee 
  // !attention: toutes les vues sont chainees (=> si chainees dans
  // winviews alors forcement chainees dans wlink)
  wlink->views = winviews;

  // Propager aux children :
  // optimization: les vues des enfants de la UWIn sont crees plus tard
  // au premier show() de la UWin, via la fonction realizeChildren()

  return winview;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UWin::realizeHardwin(bool (*xrealize)(UNatDisp*, UNatWin*, UWin*)){
  UHardwinImpl *hard = impl.hard;
  
  if (hard->wingraph.isWinRealized())   // deja realize'
    return true;
  else if ((*xrealize)(hard->wingraph.getNatDisp(), 
		       hard->wingraph.getNatWin(), this)) {

    UView* winview = getWinViews();   // only one screen !!!

    realizeChildren(winview);    // create the children views of the UWin
    return true;
  }
  else return false;
}

UView* UWin::realizeSoftwin(UWin* hardwin, UView* hardwin_view) {

  UGroup* softwins = hardwin->impl.hard->getSoftwins();

  // recuperer ou creer la softlist de la hardwin
  if (!softwins) {
    softwins = new UGroup();
    // ce mode permettra de detecter les softwins au lieu adequat
    // dans uuview.cc et consorts
    softwins->setCmodes(UMode::SOFTWIN_LIST, true);

    // not taken into account when auto_deleting children
    softwins->setCmodes(UMode::FAKE_PARENT, true);
    softwins->setCmodes(UMode::CAN_SHOW, true);
    
    //initialiser sub_softwins  
    ULink *link = softwins->makeLink();
    hardwin->impl.hard->sub_softwins = link;
    //cerr << "realizesoft hardwin: " << hardwin  << " " << hardwin->cname()
    //	 <<  " softwins " << softwins << endl;
  }

  //est-ce que this est deja dans la softlist ?
  ULink* prevlink = null;
  bool already_in_softwins = 
    softwins->getChildImpl(ELEM_LIST, this, 0, prevlink, null);

  if (!already_in_softwins) {
    // NB: modifier CAN_SHOW pour eviter recursion infinie car add()
    // appelle show() qui appelle cette fonction si CAN-SHOW is true
    softwins->add(this, false);
    softwins->getChildImpl(ELEM_LIST, this, 0, prevlink, null);
  }

  ULink *l = prevlink ? prevlink->getNext() : softwins->children.first();
  UBoxLink *wlink = static_cast<UBoxLink*>(l);
  UDisp* disp = hardwin_view->getDisp();

  UView* winview = getWinView(disp);

  // creer winview (att: l'autre version de initView() appelle show
  // ce qui provoque une boucle infinie!
  if (!winview) winview = _initView(l, hardwin_view, disp);

  // pointe sur liste complete (event. mise a jour)
  wlink->views = winviews;

  // le parent de la vue doit etre mis a jour (pour les softwins only)
  winview->setParentView(hardwin_view); // !!

  // n'aura d'effet que la premiere fois (afin que les child views
  // ne soient pas crees plusieurs fois)
  realizeChildren(winview);

  // initialiser disposition spatiale avant affichage
  /*surtout pas: boucle infinie
    UUpdate upd(UUpdate::ALL/LAYOUT);
    upd.evenIfHidden();
    updateSoftwin(upd);
  */
  return winview;
}

// !ATT: l'initialisation des descendant NE DOIT SE FAIRE qu'une seule
// fois, c'est a dire pour la seule WinView effectivement creee
// (sinon on irait creer des Views inutiles dans les descendants)

void UWin::realizeChildren(UView* winview) {
  UBoxLink *winlink = null;

  if (!winview || !(winlink = winview->getBoxLink()))
    error("UWin::realizeChildren", UError::Unrealized_window);

  // propager si pas deja fait
  else if (! winview->isVmode(UView::REALIZED_CHILDREN)) {
    winview->setVmodes(UView::REALIZED_CHILDREN, true);
    UGroup::initView(winlink, winview);   // !att: winview pas parview!
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UWin::isShown() const {
  if (!isShowable()) return false;

  for (UView* winview = getWinViews(); winview!= null; 
       winview = winview->getNext()) {
    //if (!is_hardwin) return UBox::isShown(); // isShown normal
    //else return impl.wingraph->isRealized();

    //NB: dans le cas des softwin, isShown est cense etre independant
    //de la visibilite des parents
    return winview->getWinGraph()->isWinRealized();
  }
  return false;
}

void UWin::show(bool state) {
  // hardwins have only onne shared window
  if (is_hardwin) show(state, null);

  else {
    bool found = false;
    if (impl.soft && impl.soft->current_winview) {
      // verifier si current_winview toujours dans la liste
      for (UView* v = getWinViews(); v != null; v = v->getNext()) 
	if (v == impl.soft->current_winview) found = true;
    }

    if (found) show(state, impl.soft->current_winview->getDisp());
    else show(state, null);
  }
}

void UWin::show(bool state, UDisp* disp) {
  if (isShown() == state) return;

  UAppli* a = null;
  if (isDef(0,UMode::MODAL) && (a = getAppli())) {
    if (state) a->addModalwin(this);
    else a->removeModalwin(this);
  }

  // EX: UBox::show(state);
  // FAUX: car teste: if (isShowable() == state) ce qui empeche
  // le realize des windows

  setCmodes(UMode::CAN_SHOW, state);
  if (state) {
    //fire(&e, UOn::show); // pas recursif! => faire autrement
    UUpdate upd(UUpdate::SHOW);
    update(upd, disp);
  }
  else {
    //fire(&e, UOn::hide);
    UUpdate upd(UUpdate::HIDE);
    update(upd, disp);
  }
}

void UWin::close(int status) {
  if (status < 0) status = 0;
  UAppli* a = getAppli();
  if (a) a->setModalStatus(status);
  UBox::close(status);
}

int UWin::lock(UDisp *disp) {
  setModal(true);
  if (disp) show(true, disp);
  else show(true);
  
  UAppli* a = getAppli();
  return a ? a->subLoop() : 0;
}

void UWin::setModal(bool) {
  setCmodes(UMode::MODAL, true);
}

bool UWin::isModal() const {
  return isDef(0, UMode::MODAL);
}

/* ==================================================== ======== ======= */

void UWin::toBack() {
  if (is_hardwin) {
    UNatWin* nw = impl.hard->wingraph.getNatWin();
    if (nw) nw->toBack(impl.hard->wingraph.getNatDisp());
  }
  //else ... a faire...
}
void UWin::toFront() {
  if (is_hardwin) {
    UNatWin* nw = impl.hard->wingraph.getNatWin();
    if (nw) nw->toFront(impl.hard->wingraph.getNatDisp());
  }
  //else ... a faire...
}

void UWin::highlight(bool state) {
  UBox::highlight(state);
  UWin::toFront();
}

/* ==================================================== ======== ======= */

void UWin::update(UUpdate upmode, UDisp* disp) {

  if (!impl.initialized) {
    UView* parview = null;
    ULink* selflink = null;

    for (ULinkLink *ll = parents.first(); ll != null; ll = ll->getNext()) {
      std::vector<UView*> vlist;
      ll->getParent()->getViews(vlist);

      for (unsigned int k = 0; k < vlist.size(); k++) 
	if (disp == null || vlist[k]->getDisp() == disp) {
	  parview = vlist[k]; 
	  selflink = ll->link(); 
	  break;
	}
    }

    if (parview) {
      _initView(selflink, parview, parview->getDisp());
    }

    if (!impl.initialized) {
      error("UWin::update", UError::Unrealized_window);
      return;
    }
  }

  // hardwins have only one shared window
  if (is_hardwin) impl.hard->update(upmode, this, getWinViews());

  else {
    UView* view = null;

    // NB: getWinView returns a default view if disp not found
    if (disp) view = getWinView(disp);

    else if (impl.soft->current_winview) {
      // verifier si current_winview toujours dans la liste
      for (UView* v = getWinViews(); v != null; v = v->getNext()) 
	if (v == impl.soft->current_winview) view = v;
    }

    if (!view) view = getWinViews();
    impl.soft->update(upmode, this, view);
  }
}

void UWin::update(UUpdate upmode) {
  update(upmode, null);
}

void UWin::update() {
  update(UUpdate::all, null);
}

/* ==================================================== ======== ======= */
// CAS SOFTWIN

void USoftwinImpl::update(const UUpdate& upmode, UWin* win, UView* winview) {

  UWin* hardwin = null;
  if (!winview || !(hardwin = winview->getHardwin())) {
    UError::error("USoftwinImpl::update", UError::Unrealized_window);
    return;
  }
  UDisp* disp = winview->getDisp();

  switch (upmode.ix) {
  case UUpdate::TITLE:
    UError::error("internal@USoftwinImpl::update", 
		  "update title not implemnted");
    break;

  case UUpdate::HIDE:
    hardwin->impl.hard->removeSoftwin(win);
    win->setCmodes(UMode::CAN_SHOW, false);
    break;
  
  case UUpdate::SHOW:
    // mettre ou remettre 'this' en LAST position dans la softwin list
    // si deja dans la liste => l'enlever (sans message d'erreur!)
    hardwin->impl.hard->removeSoftwin(win);
    hardwin->impl.hard->addSoftwin(win, hardwin, hardwin->getWinView(disp));
    win->setCmodes(UMode::CAN_SHOW, true);
    break;

  case UUpdate::ALL:
  case UUpdate::LAYOUT:
    if (win->isShowable() || upmode.always) {
      if (!win->realizeSoftwin(hardwin, hardwin->getWinView(disp)))
	return;
    }
    break;

  default:
    break;
  }

  //dans tous les cas
  win->UBox::update(upmode);
}

/* ==================================================== ======== ======= */
// CAS HARDWIN

void UHardwinImpl::update(const UUpdate &upmode, UWin *win, UView* winview) {
  if (!winview) {
    UError::error("UHardwinImpl::update", UError::Unrealized_window);
    return;
  } 
  UDisp* disp = winview->getDisp();

  switch (upmode.ix) {
  case UUpdate::TITLE: {
    const UStr *s;
    if (!wingraph.isWinRealized()
	|| !upmode.elem || !(s = dynamic_cast<const UStr*>(upmode.elem))
	)
      return;
    else wingraph.getNatWin()->setTitle(wingraph.getNatDisp(), *s);
    }
    break;

  case UUpdate::HIDE:
    win->setCmodes(UMode::CAN_SHOW, false);
    // NB: dans le cas state==true, le champ view.show est mis a true 
    // dans doUpdate(). 
    // Par contre, dans le cas state==false il doit par contre etre mis 
    // a false ici (car on ne passera pas dans doUpdate)
    ///winview->shown = false; supprime
    //fire(&e, UOn::hide); supprime car pas recursif: faire autrement
    wingraph.getNatWin()->show(wingraph.getNatDisp(), false);	
    break;

  case UUpdate::SHOW:{
    // voir ci-dessous
    bool initialized = winview->isVmode(UView::INITIALIZED);

    win->setCmodes(UMode::CAN_SHOW, true);
    //fire(&e, UOn::show);   supprime car pas recursif: faire autrement
    if (!wingraph.isWinRealized()) {
      if (!win->realize()) return;  // creates the X-Window and child views
    }
    updateImpl(upmode, win, winview, false, 0, 0);

    // si la view n'est pas encore initialisee, c'est qu'elle n'a jamais
    // ete affichee et il faut donc initialiser la X Window correspondante
    // if (! win->winview->isDef(UView::INITIALIZED))
    if (! initialized) {
      // set title when applicable
      UTitle *title = static_cast<UTitle*>
	(win->getAnyChild( &UBrick::isInstance<UTitle>) );

      if (title) 
	wingraph.getNatWin()->setTitle(wingraph.getNatDisp(), title->get());

      // il faut changer la taille de la X Window la 1e fois qu'elle apparait
      // (et de preference avant son apparition par show pour eviter effets
      // visuels indesirables)
      wingraph.getNatWin()->resize(wingraph.getNatDisp(),
				   win->getWidth(disp),
				   win->getHeight(disp));
    }

    // dans tous les cas pop up and raise
    wingraph.getNatWin()->show(wingraph.getNatDisp(), true);
    }break;
 
  case UUpdate::PAINT: 
    if (win->isShowable() || upmode.always) 
      win->UBox::update(upmode);
    break;

  default:    //ALL,LAYOUT, etc.
    if (win->isShowable() || upmode.always) {
      //on realize si pas deja fait
      if (!wingraph.isWinRealized()) {
	if (!win->realize()) return;  // creates the X-Window and child views
      } 
      //preserver la taille courante
      //!!NOTE:  bug dans le cas des menus ou on add/remove des objs:  !!!
      //         ne changent pas de taille!                 !! A REVOIR !!
      u_dim ww, hh;
      winview->getSize(ww, hh);
      if (ww > 0 && hh >0)  updateImpl(upmode, win, winview, true, ww, hh);
      else updateImpl(upmode, win, winview, false, 0, 0);
    }
    break;
  }
}

/* ==================================================== ======== ======= */
//NB:impose une taille donnee en mode 'resize=true' (si les valeurs sont >0)

void UHardwinImpl::updateImpl(const UUpdate& upmode, 
			      UWin* win, UView* winview,
			      bool impose_size, u_dim w, u_dim h) {
  if (w <= 0 || h <= 0) impose_size = false;

  if (win->isShowable()) {
    if (upmode.ix == UUpdate::LAYOUT)
      //que layout, pas paint => showiew = null = null 
      UBox::updateImpl(upmode, winview, winview, null, impose_size, w, h);
    else
      UBox::updateImpl(upmode, winview, winview, winview, impose_size, w, h);
  }
  else if (upmode.always) {
    //toujours faire layout (mais paint inutile) si always
    bool can_show = win->isDef(0, UMode::CAN_SHOW);
    win->setCmodes(UMode::CAN_SHOW, true);
    UBox::updateImpl(upmode, winview, winview, null, impose_size, w, h);
    win->setCmodes(UMode::CAN_SHOW, can_show);
  }
  
  // securite: une taille <= 0 fait planter X !
  if (winview->getWidth()  <= 0)  winview->setWidth(1);
  if (winview->getHeight() <= 0)  winview->setHeight(1);

  //NB1: qq part il faudrait la notion de stack/level pour les softwins
  //NB2: bug classique UBox:softwin : ne reaffiche pas la zone decourverte
  // qunad on retaille pour une nouvelle taille plus petite

  // deplace dans UWin::resize : ne modifier la X Window que si on demande
  // rellement de le faire (sinon cas de boucle infinie avce on_configure
  // et autres pbms du meme genre)
  //wingraph.getNatWin()->resize(wingraph.getNatDisp(),
  //				winview->getWidth(), winview->getHeight());
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */
//ATT: contrairement a update(), resize() impose une taille donnee
// (si les valeurs sont >0)

void UWin::resize(u_dim w, u_dim h, UDisp* disp) {
  UView* winview = getWinView(disp);

  if (!winview) error("UWin::resize", UError::Unrealized_window);
  else {

    winview->resize(w, h);

    if (isShowable()) {
      UUpdate upmode(UUpdate::ALL);
      // ici il manque la possibilite du doublebuf (trait de upmode) !
      if (is_hardwin) {

	// mettre a jour l'affichage
	impl.hard->updateImpl(upmode, this, winview, true, w, h);

	// changer la taille physique de la X window
	// autrefois dans UHardwinImpl::updateImpl 5voir ci-dessus)
	impl.hard->wingraph.getNatWin()
	  ->resize(impl.hard->wingraph.getNatDisp(),
		   // pas w et h : valeurs controlees par updateImpl
		   winview->getWidth(), winview->getHeight());
      }
      else UBox::update(upmode);
    }
  }
}

bool UWin::getSize(u_dim &w, u_dim &h, UDisp* disp) const {
  UView* winview = getWinView(disp);

  if (!winview) {
    error("UWin::getSize", UError::Unrealized_window);
    w = -1; h = -1;
    return false;
  }
  else {
    winview->getSize(w, h);
    return true;
  }
}

u_dim UWin::getWidth(UDisp* disp) const {
  u_dim w, h;
  if (getSize(w, h, disp)) return w;
  else return -1;
}

u_dim UWin::getHeight(UDisp* disp) const {
  u_dim w, h;
  if (getSize(w, h, disp)) return h;
  else return -1;
}

/* ==================================================== [Elc:01] ======= */
/* ==================================================== ======== ======= */

bool UWin::whereOnScreen(u_pos& x, u_pos& y, UDisp* disp) const {
  UView* winview = getWinView(disp);
  UNatWin *natwin = null;

  if (!winview || !(natwin = winview->getWinGraph()->getNatWin())) {
    x = -1; y = -1;
    error("UWin::whereOnScreen", UError::Unrealized_window);
    return false;
  }
  else {
    //natwin->where(impl.wingraph->getNatDisp(), x, y);
    natwin->where(winview->getWinGraph()->getNatDisp(), x, y);
    // x += winview->x; y += winview->y;
    // OU BIEN:
    if (!is_hardwin) {

      if (!impl.initialized) {
	error("UWin::whereOnScreen", UError::Unrealized_window);
	return false;
      }
      x += impl.soft->pos.getX();
      y += impl.soft->pos.getY();
    }
    return true;
  }
}

bool UWin::where(UWin *win, u_pos &x, u_pos &y, UDisp* disp) const {
  u_pos x2, y2;
  if (win && whereOnScreen(x, y, disp) && win->whereOnScreen(x2, y2, disp)) {
    x -= x2; y -= y2;
    return true;
  }
  else {
    x = -1; y = -1;
    return false;
  }
}

bool UWin::where(UView *view, u_pos &x, u_pos &y) const {
  u_pos x2, y2;
  if (view && whereOnScreen(x, y, view->getDisp()) 
      && view->whereOnScreen(x2, y2)) {
    x -= x2; y -= y2;
    return true;
  }
  else {
    x = -1; y = -1;
    return false;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UWin::moveAndCenter(UDisp* disp) {
  u_dim width, height;
  UAppli *a = getAppli();

  if (a && getSize(width, height, disp)) {
    u_dim scr_width  = a->getScreenWidth();   // disp !!!
    u_dim scr_height = a->getScreenHeight();  // disp !!
    moveOnScreen((scr_width - width) / 2, (scr_height - height) / 2, disp);
  }
}

void UWin::moveOnScreen(u_pos x, u_pos y, UDisp* disp) {
  if (!impl.initialized) {
    error("UWin::moveOnScreen", UError::Unrealized_window);
    return;
  }

  if (is_hardwin) {
    UNatWin *natwin = impl.hard->wingraph.getNatWin();
    if (!natwin) {
      error("UWin::moveOnScreen", UError::Unrealized_window);
      return;
    }
    else {
      // c'est le moment de creer la win si pas deja fait :
      // sinon le move sera ineffectif et au prochain show le menu
      // apparaitra n'importe ou
      if (!natwin->isRealized()) realize();
      natwin->move(impl.hard->wingraph.getNatDisp(), x, y);
      return;
    }
  }

  else { // softwin
    error("UWin::moveOnScreen", UError::Not_yet_implemented);   //!!!COMPLETER
    return;
  }
}

void UWin::move(UEvent* e, u_pos x, u_pos y) {
  move(*e, x, y);
}

void UWin::move(UEvent& e, u_pos x, u_pos y) {
  UWin *win = e.getHardWin();
  if (win) move(win, e.getXwin() + x, e.getYwin() + y, e.getDisp());
}

void UWin::move(UWin *win, u_pos x, u_pos y, UDisp* disp) {

  if (is_hardwin) {
    // deplacer par rapport aux coords du Frame ou du Dialog
    // et non de l'Incrust
    if (win->isCmode(UMode::INCRUST)) {
      win = win->getWinView(disp)->getParentView()->getHardwin();
    }
    u_pos screen_x, screen_y;
    if (win->whereOnScreen(screen_x, screen_y, disp))
      moveOnScreen(screen_x + x, screen_y + y, disp);
  }

  else {                                // SOFTWIN

    UView* win_winview = null;
    UWin* win_hardwin = null;
    if (!win 
	|| !(win_winview = win->getWinView(disp))
	|| !(win_hardwin = win_winview->getHardwin())
	//|| !(win_wingraph = win_winview->getWinGraph())
	|| ! impl.initialized
	){
      error("UWin::move", UError::Unrealized_window);
      return;
    }

    bool found = false;
    if (impl.soft->current_winview) {
      // verifier si current_winview toujours dans la liste
      for (UView* v = getWinViews(); v != null; v = v->getNext()) 
	if (v == impl.soft->current_winview) found = true;
    }

    UWin* current_hardwin = null;
    UView* current_hardwin_view = null;
    if (found) {
      current_hardwin = impl.soft->current_winview->getHardwin();
      current_hardwin_view = impl.soft->current_winview->getHardwinView();
    }

    /* ATT: Bugs avec les UIncrust:
     * -a- winview->wingraph = new_wingraph; peut poser pbm dans ce cas
     *     car la fenetre UIncrust recouvre la fenetre Frame
     * -b- d'autant plus que les Events sont errones dans ce cas car les
     *     UIncrust loupent tous les events (et c'est Frame qui le recupere) 
     */
    // c'est un pis aller : normalement il faudrait changer le wingraph
    // de this. le probleme c'est que cette win doit rester dans
    // la sub-hardwin (UIncrust) sinon elle ne s'affichera pas
    /* obs
    if (current_hardwin && current_hardwin->isCmode(UMode::INCRUST)) {
      impl.soft->pos.set(x - current_hardwin_view->x, 
			 y - current_hardwin_view->y);
    }
    else
    */
    {
      if (current_hardwin == win_hardwin) {
	impl.soft->pos.set(x - win_winview->x, y - win_winview->y);
      }
      else  {
	if (current_hardwin) 
	  current_hardwin->impl.hard->removeSoftwin(this);

	//cerr << " win_hardwin "<< win_hardwin 
	// << " softwin " << this << " y " << y - win_winview->y <<endl;
	win_hardwin->impl.hard->addSoftwin(this, 
					   win_hardwin, 
					   win_hardwin->getWinView(disp));
	impl.soft->pos.set(x - win_winview->x, y - win_winview->y);
      }
    }
  }
}

void UWin::move(UView *view, u_pos x, u_pos y) {
  if (!view) return;
  UWin *hardwin = view->getHardwin();

  if (!hardwin) error("warning@UWin::move", UError::Unrealized_window);
  else {
    x += view->x; y += view->y;  
    move(hardwin, x, y, view->getDisp());
  }
}

/* ==================================================== ======== ======= */
// updates object layout to get correct size when necessay

static void checkUpdate(UWin *obj, UDisp* disp) {
  u_dim w, h;
  obj->getSize(w, h, disp);

  // if <= 0 must update layout to get correct sizes
  if (w <= 0 || h <= 0) {
    // 2nd arg = true ==> will update even if NOT shown
    UUpdate upd(UUpdate::ALL);
    upd.layoutIfNotShown();
    obj->update(upd);
  }
}

//positionner par rapport a view
void UWin::move(UView *view, UPlacement &pl) {
  UDisp* disp = view->getDisp();

  u_pos _x = 0, _y = 0;  //nom a changer

  if (pl.halign) {
    if (pl.halign->get() == UHalign::left.get()) {
      if (pl.hoppositeBorder) {	  // a gauche de view
	// updates layout to get correct size when necessay
	checkUpdate(this, disp);
	_x -= this->getWidth(disp) + pl.hdist;
      }
      else _x += pl.hdist;  //alignes a dist pres
    }

    else if (pl.halign->get() == UHalign::right.get()) {
      if (pl.hoppositeBorder)	  // a droite de view
	_x += view->getWidth() + pl.hdist;
      else {
	checkUpdate(this, disp);
	_x += view->getWidth() - this->getWidth(disp) - pl.hdist;
      }
    }

    //else nop: aligne' avec view (a gauche)
    // flex devrait s'adapter en largeur (qunad plus petit que view)
  }

  if (pl.valign) {
    if (pl.valign->get() == UValign::top.get()) {
      if (pl.voppositeBorder) {	      // au dessus de view
	checkUpdate(this, disp);
	_y -= this->getHeight(disp) + pl.vdist;
      }
      else _y += pl.vdist;  //alignes a dist pres
    }
    
    else if (pl.valign->get() == UValign::bottom.get()) {
      if (pl.voppositeBorder)	          // en dessous de view
	_y += view->getHeight() + pl.vdist;
      else {
	checkUpdate(this, disp);
	_y += view->getHeight() - this->getHeight(disp) - pl.vdist;
      }
    }

    //else nop: aligne' avec view (en haut)
    // flex devrait s'dapter en hauteur(qunad plus petit que view)
  }

  move(view, _x, _y);
}

UPlacement::UPlacement() {
  halign = null;
  valign = null;
  hoppositeBorder = voppositeBorder = false;
  hdist = vdist = 0;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle* UDialog::style = null;
const UStyle& UDialog::makeStyle() {
  if (!style) style = new UStyle(UWin::makeStyle());
  style->local.padding.set(6, 3);
  return *style;
}

UDialog::UDialog(const UArgs& a) : UWin(a), open_call(null) {}

bool UDialog::realize() {
  if (is_hardwin) return realizeHardwin(&UNatWin::realizeDialog);
  else {
    error("internal@UDialog::realize", UError::Cant_realize_softwin);
    return false;
  }
}

void UDialog::addingTo(ULink *selflink, UGroup *parent) {
  UWin::addingTo(selflink, parent);

  // ajouter Handler pour ouvrir auto le Dialog si le parent est un UButton
  if (isCmode(UMode::WIN_AUTO_OPEN_MODE) && parent->isCmode(UMode::CAN_ARM)) {

    if (!open_call) open_call = &ucall(this, true, &UDialog::show);
    parent->addAttr(*open_call);                             // !!CACHE!!!
  }
}

//NB: removingFrom() requires a destructor to be defined
void UDialog::removingFrom(ULink *prevlink, UGroup *parent) {

  if (isCmode(UMode::WIN_AUTO_OPEN_MODE) && parent->isCmode(UMode::CAN_ARM)) {

    // don't delete the ucall as it is shared
    if (open_call) parent->removeAttr(*open_call, false);
  }

  UWin::removingFrom(prevlink, parent);
}

/* ==================================================== ======== ======= */

void UDialog::show(bool state, UDisp* disp) {
  if (isShown() == state) return;
  //if (state) moveAndCenter();   // deconne
  UWin::show(state, disp);
}

void UDialog::show(bool state) {
  if (isShown() == state) return;
  //if (state) moveAndCenter();   // deconne
  UWin::show(state);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle* UFrame::style = null;
const UStyle& UFrame::makeStyle() {
  if (!style) style = new UStyle(UWin::makeStyle());
  return *style;
}

UFrame::UFrame(const UArgs& a) : UDialog(a) {
  is_main_frame = false;  // defaut: NOT a main_frame
  // les frame, c'est toujours des HARDWIN !!!
  is_hardwin = true;
}

void UFrame::close(int status) {
  if (status < 0) status = 0;
  UAppli* a = getAppli();

  //mod 4aug03: toujours faire close() d'abord pour appeler CB
  UWin::close(status);
  if (a && isMainFrame()) a->quit(status);
}

bool UFrame::realize() {
  if (is_hardwin) {
    if (is_main_frame) 
      return realizeHardwin(&UNatWin::realizeMainFrame);
    else
      return realizeHardwin(&UNatWin::realizeFrame);
  }
  else {
    error("internal@UFrame::realize", UError::Cant_realize_softwin);
    return false;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStyle* UIncrust::style = null;
const UStyle& UIncrust::makeStyle() {
  if (!style) style = new UStyle(UWin::makeStyle());
  return *style;
}

UIncrust::UIncrust(const UArgs& a) : UWin(a) {
  // Contrairement aux autres UWin, les UIncrust sont affichees par defaut
  // et sont en mode BOX (= tiled)
  setCmodes(UMode::CAN_SHOW|UMode::BOX|UMode::HARDWIN|UMode::INCRUST, true);
  is_hardwin      = true;  // les Incrust c'est toujours des HARDWIN !!!
  is_external_win = false;
  xwin = None;
}

UIncrust& uincrust(const UArgs& a) {
  return *new UIncrust(a);
}

UIncrust::~UIncrust() {}

// must be redefined to init parent view
void UIncrust::initView(ULink *selflink, UView *parview) {
  UWin::_initView(selflink, parview, parview->getDisp());

  // as the Incrust will be laid out as a normal UBox, it must have
  // a parent view (in constrast with other Wins that do not have one)
  // (see UBoxLink::getViewInside())
  UView* winview = getWinView(parview->getDisp());
  if (winview) winview->setParentView(parview);

  if (isShowable()) show(true);
}

void UIncrust::setExternalXWin(unsigned long external_xwin) {
  is_external_win = true;
  //if xwin ==> xdestroy(xwin) etc...
  xwin = external_xwin;
}

bool UIncrust::realize() {
  if (is_hardwin) return realizeHardwin(&UNatWin::realizeIncrust);
  else {
    error("internal@UIncrust::realize", UError::Cant_realize_softwin);
    return false;
  }  
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */


