// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: StateManagerFactory.cpp
// 
//---------------------------------------------------------------------------

#include "StateManagerFactory.h"
#include "PeriodicStateManager.h"
#include "SimulationConfiguration.h"
#include "State.h"
#include "SimulationObject.h"
#include "TimeWarpSimulationManager.h"
#include <clutils/ConfigurationScope.h>
#include <clutils/ConfigurationChoice.h>
#include <clutils/Debug.h>

StateManagerFactory::StateManagerFactory( ){ }

StateManagerFactory::~StateManagerFactory(){
  // myStateManager is deleted by the end user - the
  // TimeWarpSimulationManager
}

Configurable *
StateManagerFactory::allocate( SimulationConfiguration &configuration,
			       Configurable *parent ) const {
  StateManager *retval = 0;
  TimeWarpSimulationManager *mySimulationManager = 
    dynamic_cast<TimeWarpSimulationManager *>( parent );
  ASSERT( mySimulationManager != 0 );

  // the following cases are possible:

  // (1) StateManager is InfrequentStateManager. In this is the case,
  //     we need to find a state period (if any; defaults to 1). Then
  //     instantiate the InfrequentStateManager with a state period
  //     (if one is found).
  
  // (2) StateManager is AdaptiveStateManager. In this case, we just
  //     instantiate the AdaptiveStateManager and go on.
   
  unsigned int statePeriod = 0;
  if( configuration.stateManagerTypeIs( "PERIODIC" )){
    configuration.getStatePeriod( statePeriod );
    retval = new PeriodicStateManager( mySimulationManager, 
				       statePeriod );
    clutils::debug << "(" << mySimulationManager->getSimulationManagerID() << 
      ") configured a Periodic State Manager with period = " << 
      statePeriod << endl;
  }
  else if( configuration.stateManagerTypeIs( "ADAPTIVE" ) ){
    cerr << "Adaptive state management broken" << endl;
    abort();
  }
  else {
    mySimulationManager->shutdown( "Unknown StateManager choice \"" +
				   configuration.getStateManagerType() +
				   "\"" );
  }
  return retval;
}

const StateManagerFactory *
StateManagerFactory::instance(){
  static const StateManagerFactory *singleton = new StateManagerFactory();

  return singleton;
}



