/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="ipchains"
%option outfile="ipchains.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#define IPLEN 16

#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "ipchains.h"
#include "common.h"
#include "defs.h"


static void ipchains_parse_data(char *input);
static void ipchains_parse_rdata(char *input);
static void ipchains_parse_ips(char *input);

static wf_logentry* logentry;
%}

MONTH	"Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec"
STRING	[a-zA-Z-][a-zA-Z0-9._-]*
LOGHOST	[0-9.a-zA-Z_-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
HEXDIGIT	[0-9a-fA-F]
HEXNUM	"0x"{HEXDIGIT}+
IPCHAINS	" kernel: Packet log: "

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}	parse_date(ipchainstext, logentry);
{IPCHAINS}	/* ignore */
{STRING}[ ]{STRING}[ ]{STRING}" PROTO="{NUMBER}	ipchains_parse_data(ipchainstext);
{STRING}" REDIRECT "{NUMBER}" "{STRING}" PROTO="{NUMBER}	ipchains_parse_rdata(ipchainstext);
{IPADDR}":"{PORT}" "{IPADDR}":"{PORT}		ipchains_parse_ips(ipchainstext);
"L="{NUMBER}	logentry->datalen = atoi(ipchainstext+2);
"S="{HEXNUM}	/* ignore */
"I="{NUMBER}	/* ignore */
"F="{HEXNUM}	/* ignore */
"T="{NUMBER}	/* ignore */
"O="{HEXNUM}	/* ignore */
"SYN"		logentry->tcpflags = TCP_SYN;
"(#"{NUMBER}")"	/* ignore */
[ ]+		/* ignore whitespace */
[\n]		/* ignore */
{STRING}	if (verbose) fprintf(stderr, _("Unrecognized token: %s\n"), ipchainstext);
.		if (verbose) fprintf(stderr, _("Unrecognized character: %s\n"), ipchainstext);

%%

static void
ipchains_parse_data(char *input) {
  char chainlabel[32], branchname[32], iface[32];
  int protocol;

  if (sscanf(input, "%30s %30s %30s PROTO=%3d",
	     chainlabel, branchname, iface, &protocol) != 4)
    return;
  logentry->chainlabel = chainlabel;
  logentry->branchname = branchname;

  if (!strcmp(chainlabel, "output") || !strcmp(chainlabel, "forward"))
    logentry->output_iface = iface;
  else /* "input" and default */
    logentry->input_iface = iface;

  logentry->protocol = protocol;
}

static void
ipchains_parse_rdata(char* input) {
  char chainlabel[32], branchname[32], iface[32];
  int port, protocol;
  
  if (sscanf(input, "%30s REDIRECT %5d %30s PROTO=%3d",
	     chainlabel, &port, iface, &protocol) != 4)
    return;

  snprintf(branchname, sizeof(branchname), "RD %d", port);
  logentry->chainlabel = chainlabel;
  logentry->branchname = branchname;

  if (!strcmp(chainlabel, "output") || !strcmp(chainlabel, "forward"))
    logentry->output_iface = iface;
  else /* "input" and default */
    logentry->input_iface = iface;

  logentry->protocol = protocol;
}

static void
ipchains_parse_ips(char *input) {
  int shost1, shost2, shost3, shost4, sport;
  int dhost1, dhost2, dhost3, dhost4, dport;
  char ip[IPLEN];

  if (sscanf(input, "%3d.%3d.%3d.%3d:%5d %3d.%3d.%3d.%3d:%5d",
	     &shost1, &shost2, &shost3, &shost4, &sport,
	     &dhost1, &dhost2, &dhost3, &dhost4, &dport) != 10)
    return;

  snprintf(ip, IPLEN, "%d.%d.%d.%d", shost1, shost2, shost3, shost4);
  if (logentry->sipaddr.set(ip) == false)
    return;

  snprintf(ip, IPLEN, "%d.%d.%d.%d", dhost1, dhost2, dhost3, dhost4);
  if (logentry->dipaddr.set(ip) == false)
    return;

  logentry->sport = sport;
  logentry->dport = dport;
}


wf_inmodule_ipchains::wf_inmodule_ipchains() {
}

bool
wf_inmodule_ipchains::match(const string& line) const {
  return (line.find(" kernel: Packet log: ") != string::npos);
}

enum wf_logentry_parsing_result
wf_inmodule_ipchains::parse(wf_logentry** retlogentry,
  const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
    return WF_LOGENTRY_PARSING_ERROR;

  ipchains_scan_string(line.c_str());
  ipchainslex();

  logentry->count = 1;
  logentry->format = "ipchains";

  /*
  if (parser == (IPCHAINS_DATE|IPCHAINS_DATA|IPCHAINS_IPS))
  */
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (verbose)
    fprintf(stderr, _("ipchains parse error in line %d, ignoring.\n"),
	    linenum);

  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_ipchains_init() {
  return new wf_inmodule_ipchains();
}
