/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "bordereddialog.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "application.h"
#include "rootwindow.h"
#include "font.h"

#include <sigc++/object_slot.h>

namespace wftk {

BorderedDialog::BorderedDialog(const std::string& title,
	const std::string& border_res_name, const Font& font) :
  hRes_(Surface::registry.get(border_res_name + "_h")),
  vRes_(Surface::registry.get(border_res_name + "_v")),
  cornerRes_(Surface::registry.get(border_res_name + "_edge")),
  title_(title.empty() ? 0 : (font.valid() ? font : Font::textFont()).getString(title)),
  hPad_(vRes_ ? vRes_->res()->width() : 0),
  vPad_(hRes_ ? hRes_->res()->height() : 0)
{
  if(hRes_)
    hRes_->bind();
  if(vRes_)
    vRes_->bind();
  if(cornerRes_)
    cornerRes_->bind();

  if(title_ && vPad_ < title_->height())
    vPad_ = title_->height();

  setPadding(vPad(), vPad(), hPad(), hPad());
}

BorderedDialog::~BorderedDialog()
{
  if(hRes_)
    hRes_->free();
  if(vRes_)
    vRes_->free();
  if(cornerRes_)
    cornerRes_->free();

  delete title_;
}

void BorderedDialog::draw(Surface& target, const Point& offset, const Region& mask)
{
  Dialog::draw(target, offset, mask);

  unsigned title_offset = 0, title_width = 0;

  if(title_ && width() > 2 * hPad_) {
    Region tmp_mask(mask);

    if(title_->width() < width() - 2 * hPad_) {
      title_offset = (width() - title_->width()) / 2;
      title_width = title_->width();
    }
    else {
      title_offset = hPad_;
      title_width = width() - 2 * hPad_;
      tmp_mask &= Rect(hPad_, 0, width() - 2 * hPad_, title_->height());
    }

    Point shift(title_offset, 0);
    title_->blit(target, offset + shift, mask);
  }

  if(hRes_ && width() > 2 * hPad_) {
    Region tmp_mask(mask);
    if(title_)
      tmp_mask -= Rect(title_offset, 0, title_width, title_->height());
    tmp_mask -= Rect(0, 0, hPad_, height());
    tmp_mask -= Rect(width() - hPad_, 0, hPad_, height());
    unsigned height_shift = height() - hRes_->res()->height();
    for(unsigned x = hPad_; x < width() - hPad_; x += hRes_->res()->width()) {
      hRes_->res()->blit(target, offset + Point(x, 0), tmp_mask);
      hRes_->res()->blit(target, offset + Point(x, height_shift), tmp_mask);
    }
  }

  if(vRes_ && height() > 2 * vPad_) {
    Region tmp_mask(mask);
    tmp_mask -= Rect(0, 0, width(), vPad_);
    tmp_mask -= Rect(0, width() - vPad_, width(), vPad_);
    unsigned width_shift = width() - vRes_->res()->width();
    for(unsigned y = vPad_; y < height() - vPad_; y += vRes_->res()->height()) {
      hRes_->res()->blit(target, offset + Point(0, y), tmp_mask);
      hRes_->res()->blit(target, offset + Point(width_shift, y), tmp_mask);
    }
  }

  if(cornerRes_) {
    cornerRes_->res()->blit(target, offset, mask);
    cornerRes_->res()->blit(target, offset + Point(width() - hPad_, 0), mask);
    cornerRes_->res()->blit(target, offset + Point(width() - hPad_, height() - vPad_),
      mask);
    cornerRes_->res()->blit(target, offset + Point(0, height() - vPad_), mask);
  }
}

}

