/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#ifndef _PAINTER_H_
#define _PAINTER_H_

#include <wftk/surface.h>

namespace wftk {

/// class that allows drawing graphics primitives onto a Surface
class Painter
{
 public:
  /// constructor - bind this Painter to the \p surf Surface
  Painter(Surface* surf = 0);

  bool empty() const {return !surface_ || surface_->empty();}

  /// behaviour of painting primtives
  enum Mode {OP_REPLACE /*!< . */, OP_AND /*!< . */, OP_OR /*!< . */, OP_XOR /*!< . */};

  /// set the active (default) Color
  void setColor(const Color& col) { color_ = col; };
  /// set fill mode for boxes, circles and other closed primitives on/off 
  void setFilling(bool flag) { fill_ = flag;};
  /// set drawing Mode for overlapping primitives
  void setMode(Mode mode) { mode_ = mode;};

  /// set the target Surface.
  void setSurface(Surface* surf) {surface_ = surf;}

  /** Set a pixel on the Surface to given Color
	 *
	 * \param pt Point on the Surface to set
	 * \param col Color to set the pixel
	 */
  void setPixel(const Point& pt, const Color& col)
	{if(surface_) surface_->setPixel(pt, col);}
	
  /** fetch the Color of pixel
	 *
	 * \param pt Point on the surface to fetch
	 */
  Color getPixel (const Point& pt)
	{return surface_ ? surface_->getPixel(pt) : Color();}

  /** Draw a horizontal line on the target Surface using the default Color
	 *
	 * \param p1 Begin point, used for vertical offset
	 * \param p2 End point, used for horizontal offset while drawing
	 */
  void hLine(const Point& p1, const Point& p2);
	
  /** Draw a vertical line on the target Surface using the default Color
	 *
	 * \param p1 Begin point, used for horizontal offset
	 * \param p2 End point, used for vertical offset while drawing
	 */
  void vLine(const Point& p1, const Point& p2);
	
  /** Draw a diagonal line on the target Surface using the default Color
	 *
	 * This method calls vLine or hLine if possible, or uses a bresenham
	 * calculation to rasterize the diagonal line if necessary.
	 *
	 * \param p1 Begin point of the line
	 * \param p2 End point of the line
	 */
  void line(const Point& p1, const Point& p2);
	
  /** Draw a horizontal line on the target Surface
	 * 
	 * \param p1 Begin point, used for vertical offset
	 * \param p2 End point, used for horizontal offset while drawing
	 * \param col Color to draw the line with
	 */
  void hLine(const Point& p1, const Point& p2, const Color& col);
	
  /** Draw a vertical line on the target Surface
	 *
	 * \param p1 Begin point, used for horizontal offset
	 * \param p2 End point, used for vertical offset while drawing
	 * \param col Color to draw the line with
	 */
  void vLine(const Point& p1, const Point& p2, const Color& col);
  
  /** Draw a diagonal line on the target Surface
	 *
	 * This method calls vLine or hLine if possible, or uses a bresenham
	 * calculation to rasterize the diagonal line if necessary.
	 *
	 * \param p1 Begin point of the line
	 * \param p2 End point of the line
	 * \param col Color to draw the line with
	 */
  void line(const Point& p1, const Point& p2, const Color& col);

  /** Draw a circle on the target Surface, using the default Color.
	 *
	 * \param c Center of circle
	 * \param r radius of circle, in pixels
	 * \param t Border thickness of circle, in pixels (defaults to 1 pixel)
	 */
  void circle(const Point& c, unsigned int r, unsigned int t=1);
	
  /** Draw an ellipse on the target Surface, using the default Color.
	 *
	 * \param c Center of ellipse (midpoint between foci)
	 * \param rx maximum width of ellipse, in pixels (x-axis)
	 * \param ry maximum height of ellipse, in pixels (y-axis)
	 * \param t Border thickness of ellipse, in pixels (defaults to 1 pixel)
	 */
  void ellipse(const Point& c, unsigned int rx, 
	       unsigned int ry, unsigned int t=1);
	
  /** Draw a box (rectangle) on the target Surface, using the default Color.
	 *
	 * \param p1 Upper left co-ordinate of Box
	 * \param p2 Lower right co-ordinate of Box
	 */
  void box(const Point& p1, const Point& p2);
	
  /** Draw a triangle on the target Surface, using the default Color.
	 *
	 * \param p1 First vertex of triangle
	 * \param p2 Second vertex of triangle
	 * \param p3 Third vertex of triangle
	 */
  void trigon(const Point& p1, const Point& p2, const Point& p3);

  /** Draw a triangle on the target Surface
	 *
	 * \param p1 First vertex of triangle
	 * \param p2 Second vertex of triangle
	 * \param p3 Third vertex of triangle
	 * \param c Color used to draw the triangle
	 */
  void trigon(const Point& p1, const Point& p2, const Point& p3, const Color& c);

  /** Draw a filled triangle on the target Surface
	 *
	 * Vertex colors are blended into each other across the bitmap.
	 * 
	 * \param d1 First vertex of triangle
	 * \param d2 Second vertex of triangle
	 * \param d3 Third vertex of triangle
	 * \param c1 Color of first vertex
	 * \param c2 Color of second vertex
	 * \param c3 Color of third vertex
	 */
  void shadedTrigon(const Point& d1, const Point& d2, const Point& d3,
		    const Color& c1, const Color& c2, const Color& c3);
	
  /** Draw a textured triangle to the target Surface
	 * 
	 * \param d1 First vertex of triangle
	 * \param d2 Second vertex of triangle
	 * \param d3 Third vertex of triangle
	 * \param s1 First UV co-ordinate of texture
	 * \param s2 Second UV co-ordinate of texture
	 * \param s3 Third UV co-ordinate of texture
	 * \param texture Surface used as texture for triangle
	 */ 
  void texturedTrigon(const Point& d1, const Point& d2, const Point& d3,
		      const Point& s1, const Point& s2, const Point& s3, Surface* texture);
  
 private:
  ///
  void  ellipseFill(const Point& c, unsigned int rx, unsigned int ry);
  ///
  void writePixel(Uint32, Uint32);

  ///
  Surface* surface_;
  ///
  Color color_;
  ///
  bool fill_;
  ///
  Mode mode_;

};


}



#endif //!_PAINTER_H_
