/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#include "pixelformat.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "mutex.h"

#include <iostream>
#include <cassert>

namespace wftk {

Pixelformat Pixelformat::displayFormat;

static void
GetShiftAndLoss(Uint32 mask, Uint8& shift, Uint8& loss)
 {
  if(!mask) {
    shift = 0;
    loss = 8;
    return;
  }

  shift = 0;
  while(!(mask & 0x01)) {
    ++shift;
    mask >>= 1;
  }

  loss = 8;
  do {
    assert(loss);
    --loss;
    mask >>= 1;
  } while(mask);
}

static const SDL_PixelFormat*
GetFormat(Pixelformat::Format format)
{
  if(format < 0 || format >= Pixelformat::NUM_FORMATS)
    return 0;

  static SDL_PixelFormat formats[Pixelformat::NUM_FORMATS] = {{0, }, };
  static Mutex init_lock;

  SDL_PixelFormat* out = &formats[format];

  // Do a quick initialization check before locking
  // the mutex, to optimize for the general case.
  if(out->BitsPerPixel != 0) // it's been initialized
    return out;

  init_lock.grab();

  // This is the check we must have for the mutex locking
  // to work properly. The other one is just there
  // so we don't have to be locking the mutex
  // all the time once the formats are initialized.
  if(out->BitsPerPixel != 0) { // it's been initialized
    init_lock.release();
    return out;
  }

  Debug::channel(Debug::GENERIC)
    << "Initializing Pixelformat number " << format << Debug::endl;

  if(format == Pixelformat::IND8) {
    /* Simple 6x6x6 colour cube */
    // Use a static palette instead of calling 'new',
    // we only do this once
    static SDL_Palette p;
    const int ncolors = 6*6*6;
    static SDL_Color colors[ncolors];
    p.ncolors = ncolors;
    p.colors = colors;
    for(int i = 0; i < ncolors; ++i) {
      colors[i].r = (i % 6) * 51;
      colors[i].g = ((i / 6) % 6) * 51;
      colors[i].b = (i / 36) * 51;
    }
    out->palette = &p;
    out->BytesPerPixel = 1;
    out->BitsPerPixel = 8; // BitsPerPixel is last, because it's our init check
    init_lock.release();
    Debug::channel(Debug::GENERIC)
      << "Initialized indexed Pixelformat" << Debug::endl;
    return out;
  }

  ///RGBA bitmasks
  const Uint32 masks[][4] = 
  {{0x000000FF, 0x0000FF00, 0x00FF0000, 0xFF000000},
   {0xFF000000, 0x00FF0000, 0x0000FF00, 0x000000FF},
   {0x00FF0000, 0x0000FF00, 0x000000FF, 0xFF000000},
   {0x0000FF00, 0x00FF0000, 0xFF000000, 0x000000FF},
   {0x00FF0000, 0x0000FF00, 0x000000FF, 0x00000000},
   {0x000000FF, 0x0000FF00, 0x00FF0000, 0x00000000},
   {0x00FF0000, 0x0000FF00, 0x000000FF, 0x00000000},
   {0x000000FF, 0x0000FF00, 0x00FF0000, 0x00000000},
   {0x0000F800, 0x000007E0, 0x0000001F, 0x00000000},
   {0x00007C00, 0x000003E0, 0x0000001F, 0x00000000},
   {0x00, 0x00, 0x00, 0x00}, // IND8, placeholder
  };

  // Make sure the array matches up with the types in the enum
  assert(sizeof(masks) / (4 * sizeof(Uint32)) == Pixelformat::NUM_FORMATS);

  out->palette = 0;

  out->Rmask = masks[format][0];
  out->Gmask = masks[format][1];
  out->Bmask = masks[format][2];
  out->Amask = masks[format][3];

  GetShiftAndLoss(out->Rmask, out->Rshift, out->Rloss);
  GetShiftAndLoss(out->Gmask, out->Gshift, out->Gloss);
  GetShiftAndLoss(out->Bmask, out->Bshift, out->Bloss);
  GetShiftAndLoss(out->Amask, out->Ashift, out->Aloss);

  // find the highest bit to compute BitsPerPixel

  Uint8 Rbits = out->Rshift + 8 - out->Rloss;
  Uint8 Gbits = out->Gshift + 8 - out->Gloss;
  Uint8 Bbits = out->Bshift + 8 - out->Bloss;
  Uint8 Abits = out->Ashift + 8 - out->Aloss;

  Uint8 bits = Rbits;
  if(Gbits > bits)
    bits = Gbits;
  if(Bbits > bits)
    bits = Bbits;
  if(Abits > bits)
    bits = Abits;

  // Get the minimum number of bytes which can contain this
  // many bits. Divide by 8, rounding up.
  out->BytesPerPixel = (bits + 7) / 8;

  // two formats have a full byte of padding
  if(format == Pixelformat::RGB0888 || format == Pixelformat::BGR0888) {
    assert(out->BytesPerPixel == 3);
    out->BytesPerPixel = 4;
  }

  if(out->Amask)
    out->colorkey = (SDL_ALPHA_TRANSPARENT >> out->Aloss) << out->Ashift;
  out->alpha = SDL_ALPHA_OPAQUE;

  // BitsPerPixel is last, because it's our init check
  out->BitsPerPixel = bits;

  init_lock.release();
  Debug::channel(Debug::GENERIC)
    << "Initialized Pixelformat number " << format << Debug::endl;
  return out;
}

Pixelformat::Pixelformat(Format format) :
	surface_(0), format_(GetFormat(format))
{

}

Pixelformat::Pixelformat(const Pixelformat& other) :
	surface_(other.surface_), format_(other.format_)
{
  if(surface_)
    ++surface_->refcount;
}

Pixelformat::Pixelformat(SDL_Surface* surface) :
	surface_(surface)
{
  if(surface) {
    ++surface->refcount;
    format_ = surface->format;
  }
  else
    format_ = 0;
}

Pixelformat::~Pixelformat()
{
  if(surface_) // decrement ref count
    SDL_FreeSurface(surface_);
}

Pixelformat::operator const SDL_PixelFormat&() const
{
  static const SDL_PixelFormat invalid = {0, }; // fill with zeros

  return format_ ? *format_ : invalid;
}


Pixelformat&
Pixelformat::operator=(const Pixelformat& pf)
{
  if(format_ == pf.format_)
    return *this;

  // We now know this != &pf, since their formats would be equal

  if(surface_) // decrement ref count
    SDL_FreeSurface(surface_);

  surface_ = pf.surface_;
  format_ = pf.format_;

  if(surface_)
    ++surface_->refcount;

  return *this;
}

Pixelformat&
Pixelformat::operator=(Format format)
{
  if(surface_) // decrement ref count
    SDL_FreeSurface(surface_);

  surface_ = 0;
  format_ = GetFormat(format);

  return *this;
}

namespace { // unnamed namespace, don't export symbols

class ColorSorter {
 public:
  enum ColorIndex {RED, GREEN, BLUE, ALPHA};

  ColorSorter(const SDL_PixelFormat& f)
  {
    _colors[0].color = RED;
    _colors[0].mask = f.Rmask;
    _colors[1].color = GREEN;
    _colors[1].mask = f.Gmask;
    _colors[2].color = BLUE;
    _colors[2].mask = f.Bmask;
    _colors[3].color = ALPHA;
    _colors[3].mask = f.Amask;

    // use qsort() to order the colors, we don't need the
    // speed but the code will be easier to read

    qsort(_colors, 4, sizeof(ColorMask),
      (int(*)(const void*, const void*)) compare);
  }

  ColorIndex color(int i) const {return _colors[i].color;}

 private:
  struct ColorMask {
    ColorIndex color;
    Uint32 mask;
  } _colors[4];

  static int compare(const ColorMask* c1, const ColorMask* c2)
  {
    // can't just subtract, dealing with unsigned ints
    if(c1->mask == c2->mask)
      return 0;
    return (c1->mask > c2->mask) ? 1 : -1;
  }
};

} // unnamed namespace

std::string 
Pixelformat::asString() const
{
  if(!format_)
    return "INVALID";

  if(format_->palette)
    return "INDEXED";

  ColorSorter sorter(*format_);

  std::string color_name;
  std::string color_bits;

  // avoid encoding problems
  const char digits[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};

  for(int i = 3; i >= 0; --i) {
    switch(sorter.color(i)) {
      case ColorSorter::RED:
        color_name += 'R';
        color_bits += digits[8 - format_->Rloss];
        break;
      case ColorSorter::GREEN:
        color_name += 'G';
        color_bits += digits[8 - format_->Gloss];
        break;
      case ColorSorter::BLUE:
        color_name += 'B';
        color_bits += digits[8 - format_->Bloss];
        break;
      case ColorSorter::ALPHA:
        if(format_->Amask) {
          color_name += 'A';
          color_bits += digits[8 - format_->Aloss];
        }
        else if(format_->BitsPerPixel == 24
          && format_->BytesPerPixel == 4) {
          assert(i == 0); // last color, rest are done
          // find where the extra 8 bits are
          Uint32 missing_mask =
            ~(format_->Rmask | format_->Gmask | format_->Bmask);
          int mask_pos = 3;
          if(missing_mask > format_->Rmask)
            --mask_pos;
          if(missing_mask > format_->Gmask)
            --mask_pos;
          if(missing_mask > format_->Bmask)
            --mask_pos;
          // insert '0' at the appropriate point in the string
          color_bits.insert(mask_pos, 1, '0');
        }
        break;
      default:
        assert(false);
        break;
    }
  }

  return color_name + color_bits;
}

std::ostream& operator<<(std::ostream& os, const Pixelformat& pf)
{
  return os << pf.asString();
}

bool
Pixelformat::operator==(const SDL_PixelFormat& format) const
{
  if(format_ == &format) // same chunk of memory
    return true;

  if(!format_)
    return false;

  if(format_->BitsPerPixel != format.BitsPerPixel
    || format_->BytesPerPixel != format.BytesPerPixel)
    return false;

  // if one has a palette and the other doesn't, they don't match;
  // the test is a trick to calculate XOR
  if(!(format_->palette) != !(format.palette))
    return false;

  if(format_->palette) { // compare palette
    const SDL_Palette *p1 = format_->palette, *p2 = format.palette;
    assert(p1 && p2);
    if(p1->ncolors != p2->ncolors)
      return false;
    for(int i = 0; i < p1->ncolors; ++i)
      if(p1->colors[i].r != p2->colors[i].r
        || p1->colors[i].g != p2->colors[i].g
        || p1->colors[i].b != p2->colors[i].b)
        return false;
  }
  else { // compare the masks
    if(format_->Rmask != format.Rmask
      || format_->Gmask != format.Gmask
      || format_->Bmask != format.Bmask
      || format_->Amask != format.Amask)
      return false;
    if(format_->Amask && (format_->colorkey != format.colorkey
      || format_->alpha != format.alpha))
      return false;
  }

  return true;
}

 
Uint32
Pixelformat::mapToPixel(const Color& col) const
{
  return format_ ? SDL_MapRGBA((SDL_PixelFormat*) format_,
    col.r, col.g, col.b, col.a) : 0;
}

Color
Pixelformat::mapToColor(Uint32 pixel) const
{
  Color col;

  if(format_)
    SDL_GetRGBA(pixel, (SDL_PixelFormat*) format_,
      &col.r, &col.g, &col.b, &col.a);

  return col;
}

Color
Pixelformat::getColor(unsigned color_num) const
{
  if(!format_)
    return Color();

  const SDL_Palette *p = format_->palette;
  if(!p || color_num >= (unsigned) p->ncolors)
    return Color();

  return Color(p->colors[color_num].r, p->colors[color_num].g, p->colors[color_num].b);
}

} // namespace
