/*
                   Object Recycling, Purging, and Recovering
 */

#ifndef EDV_RECYCLE_H
#define EDV_RECYCLE_H

#include <sys/stat.h>
#include <glib.h>
#include "edv_context.h"


/*
 *      Returns a statically allocated string describing the last error
 *      that occured when calling EDVRecycle(), EDVRecover(), or
 *      EDVPurge().  Can return NULL if there was no error.
 */
extern const gchar *EDVRecycleGetError(edv_context_struct *ctx);

/*
 *	Get statistics of a recycled object by its index.
 *
 *	The path_rtn specifies the pointer to the return full path
 *	describing the original location of the object. You may pass
 *	NULL if the return value is not requested.
 *
 *	The stat_buf specifies the pointer to the return struct stat
 *	describing the original object's statistics. You may pass
 *	NULL if the return value is not requested.
 *
 *	The struct stat members st_rdev, st_blksize, and st_blocks
 *	members will be undefined.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint EDVRecycledObjectStat(
	edv_context_struct *ctx, const guint index,
	gchar **path_rtn,
	struct stat *stat_buf
);

/*
 *	Gets a listing of all the recycled objects.
 *
 *	The path_rtn specifies the pointer to the return list for full
 *	paths to the original object. You may pass NULL if the return
 *	value is not requested.
 *
 *	The index_rtn specifies the pointer to the return list of
 *	recycled object indices. You may pass NULL if the return value
 *	is not requested.
 *
 *	The stat_buf specifies the pointer to the return list of
 *	recycled object stats. You may pass NULL if the return value
 *	is not requested.
 *
 *	The returned pointer arrays and each pointer to location must
 *	be deleted by the calling function.
 *
 *	The struct stat members st_rdev, st_blksize, and st_blocks
 *	members will be undefined.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint EDVRecycledObjectStatAll(
	edv_context_struct *ctx,
	gchar ***path_rtn,
	guint **index_rtn,
	struct stat ***stat_buf,
	gint *total
);

/*
 *	Recycles an object.
 *
 *	The path specifies the object to put into the recycle bin.
 *
 *	If notify is TRUE then Endeavour will be automatically notify
 *	about this recycle.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The data specifies the user data which will be passed to the
 *	progress_cb function.
 *
 *      Returns the recycled object index number or 0 on error.
 */
extern guint EDVRecycle(
	edv_context_struct *ctx, const gchar *path,
	const gboolean notify,
        gint (*progress_cb)(
                gpointer data, const gulong i, const gulong m
        ),
        gpointer data
);

/*
 *	Recovers a recycled object from the recycle bin.
 *
 *	The index specifies the index of the recycled object to
 *	recover.
 *
 *	The alternate_recovery_path specifies the full path to the 
 *	directory that is to be used as the alternate recovery location
 *	for the recycled object. If alternate_recovery_path is NULL
 *	then the recycled object's original location will be used
 *	as the recovery location.
 *
 *	If notify is TRUE then Endeavour will be automatically notify
 *	about this recover.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The data specifies the user data which will be passed to the
 *	progress_cb function.
 *
 *      Returns non-zero on error.
 */
extern gint EDVRecover(
	edv_context_struct *ctx, const guint index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
        gint (*progress_cb)(
                gpointer data, const gulong i, const gulong m
        ),
        gpointer data
);

/*
 *	Purges a recycled object from the recycle bin.
 *
 *	The index specifies the recycled object to purge.
 *
 *	If notify is TRUE then Endeavour will be automatically notify
 *	about this purge.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The data specifies the user data which will be passed to the
 *	progress_cb function.
 *
 *      Returns non-zero on error.
 */
extern gint EDVPurge(
	edv_context_struct *ctx, const guint index,
	const gboolean notify,
        gint (*progress_cb)(
                gpointer data, const gulong i, const gulong m
        ),
        gpointer data
);

#endif	/* EDV_RECYCLE_H */
