/* $Cambridge: hermes/src/prayer/session/stream.c,v 1.2 2008/05/19 15:55:58 dpc22 Exp $ */
/************************************************
 *    Prayer - a Webmail Interface              *
 ************************************************/

/* Copyright (c) University of Cambridge 2000 - 2008 */
/* See the file NOTICE for conditions of use and distribution. */

#include "prayer_session.h"

/* Helper functions for c-client stream object */

/* Mostly working around ideas that c-client seems to use text strings
 * rather than flag bits to mark messages. May just be limits in my
 * understanding of c-client, but I can't find a better interface to work
 * with at the moment */

/* ====================================================================== */

/* stream_check_uid() ***************************************************
 *
 * Check that msgno and msguid match.
 *  session:
 *   stream:
 *    msgno: Message Number which should contain this UID (bext first guess)
 *   msguid: Message UID that we are looking for
 *
 * Returns: Msgno of message containing this UID (typically msgno unless
 *          an external expunge event has just taken place).
 ***********************************************************************/

unsigned long
stream_check_uid(struct session *session,
                 MAILSTREAM * stream,
                 unsigned long msgno, unsigned long msguid)
{
    if ((msgno == 0) || (msguid == 0)) {
        session_message(session, "Reference to illegal Message UID 0");
        session_log(session,
                    "[stream_check_uid] Reference to illegal Message UID 0");
        return (0);
    }

    /* Could just do reverse lookup (msguid -> msgno).  This way is a bit
     * more efficient as don't have to scan the entire index looking for a
     * particular UID most of the time */

    if ((msgno > stream->nmsgs)
        || (ml_uid(session, stream, msgno) != msguid)) {
        /* Messages have moved: external expunge event */
        if (((msgno = ml_msgno(session, stream, msguid)) == 0) ||
            (msgno > stream->nmsgs)) {
            /* Couldn't find this UID in folder */
            session_message(session, "Message UID %lu has been expunged",
                            msguid);
            return (0);
        }
    }
    return (msgno);
}

/* ====================================================================== */

/* stream_fill_overview() ***********************************************
 *
 * Fill message overview (ENVELOPE etc) for messages in a range.
 *  session:
 *     pool: Scratch pool (typically request->pool)
 *   stream:
 *    first: First message in sequence
 *     last: Final message in sequence
 *
 * Returns: T on sucess. NIL on stream error
 ***********************************************************************/

BOOL
stream_fill_overview(struct session * session,
                     struct pool * pool,
                     MAILSTREAM * stream,
                     unsigned long first, unsigned long last)
{
    char *uidseq, *p;
    unsigned long current;
    MESSAGECACHE *elt;

    /* Need to generate a c-client sequence string for any messages which
     * need overview fetch. Following code stolen from Pine 4 (pine/mailindx.c)
     *
     * How big a buf do we need?  How about eleven per -- 10 for
     * decimal representation of 32bit value plus a trailing
     * comma or terminating null...
     */
    uidseq = pool_alloc(pool, ((last - first + 1) * 11) * sizeof(char));
    p = uidseq;
    *p = '\0';

    for (current = first; current <= last; current++) {
        if (!(elt = ml_elt(session, stream, current)))
            return (NIL);

        if (!elt->private.msg.env) {
            /* Need to fetch overview for this message */
            if (p != uidseq) {
                p += strlen(p);
                *p++ = ',';
            }
            sprintf(p, "%lu", ml_uid(session, stream, current));
            p += strlen(p);
        }
    }

    if (uidseq && uidseq[0]) {
        if (!ml_fetch_overview(session, stream, uidseq, NIL))
            return (NIL);
    }

    return (T);
}

/* ====================================================================== */

/* stream_fill_overview_msgmap() ***************************************
 *
 * Fill message overview (ENVELOPE etc) for messages in a range with
 * msgmap applied.
 *  session:
 *     pool: Scratch pool (typically request->pool)
 *   stream:
 *       zm: Zoommap to apply
 *    first: First message in sequence
 *     last: Final message in sequence
 *
 * Returns: T on success. NIL on stream error
 ***********************************************************************/

BOOL
stream_fill_overview_msgmap(struct session * session,
                            struct pool * pool,
                            MAILSTREAM * stream,
                            struct msgmap * zm,
                            unsigned long first, unsigned long last)
{
    char *uidseq, *p;
    unsigned long current;
    MESSAGECACHE *elt;

    /* Need to generate a c-client sequence string for any messages which
     * need overview fetch. Following code stolen from Pine 4 (pine/mailindx.c)
     *
     * How big a buf do we need?  How about eleven per -- 10 for
     * decimal representation of 32bit value plus a trailing
     * comma or terminating null...
     */
    uidseq = pool_alloc(pool, ((last - first + 1) * 11) * sizeof(char));
    p = uidseq;
    *p = '\0';

    for (current = first; current <= last; current++) {
        if (!(elt = ml_elt(session, stream, msgmap_value(zm, current))))
            return (NIL);

        if (!elt->private.msg.env) {
            /* Need to fetch overview for this message */
            if (p != uidseq) {
                p += strlen(p);
                *p++ = ',';
            }
            sprintf(p, "%lu", ml_uid(session, stream, current));
            p += strlen(p);
        }
    }

    if (uidseq && uidseq[0]) {
        if (!ml_fetch_overview(session, stream, uidseq, NIL))
            return (NIL);
    }
    return (T);
}

/* stream_find_unread() **************************************************
 *
 * Find first unread message in stream.
 ************************************************************************/

unsigned long
stream_find_unread(struct session *session, MAILSTREAM * stream)
{
    unsigned long msgno;
    char seq[64];

    if (stream->nmsgs == 0)
        return (0);

    sprintf(seq, "1:%lu", stream->nmsgs);
    ml_fetch_fast(session, stream, seq, 0);

    /* Record last unread message from inbox */
    for (msgno = 1; msgno <= stream->nmsgs; msgno++) {
        if (!mail_elt(stream, msgno)->seen)
            return (msgno);
    }

    return (0);
}
