# arch-tag: ae12a27d-b9f0-47e9-bed5-a08c716dd6a4
# Copyright (C) 2005 Canonical Limited
#                    Author: David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Framework for integration tests of output parsing.
"""

from pybaz import pathname


class ExpectedChange(object):

    _all_optional = ['oldname', 'binary']

    def __init__(self, type_, name, isdir=False):
        self.type_ = type_
        self.name = name
        self._directory = isdir
        self._binary = None
        self._oldname = None
        self._string = None

    def is_binary(self, isbin):
        assert isinstance(isbin, bool)
        self._binary = isbin

    def has_oldname(self, oldname):
        assert oldname is not None
        self._oldname = oldname

    def string_value(self, strval):
        assert strval is not None
        self._string = strval

    def filter(self, obj):
        return True

    def check(self, test, changes):
        filtered = filter(self.filter, changes)
        test.assertEqual(1, len(filtered))
        change = filtered[0]
        test.assertEqual(self.type_, type(change))
        test.assertEqual(self.name, change.name)
        test.assertEqual(self._directory, change.directory)
        if self._string is not None:
            test.assertEqual(self._string, str(change))
        if self._directory:
            nametype = pathname.DirName
        else:
            nametype = pathname.FileName
        test.assertEqual(nametype, type(change.name))
        if self._binary is not None:
            test.assertEqual(self._binary, change.binary)
        else:
            test.failIf(hasattr(change, 'binary'))
        if self._oldname is not None:
            test.assertEqual(self._oldname, change.oldname)
            test.assertEqual(nametype, type(change.oldname))
        else:
            test.failIf(hasattr(change, 'oldname'))


class ExpectedCompositeChange(object):

    def __init__(self, expected_changes=()):
        self._items = {}
        for expected in expected_changes:
            self.add(expected)

    def add(self, expected):
        key = (expected.type_, expected.name)
        assert key not in self._items
        def filter_by_key(C):
            return type(C) is expected.type_ and C.name == expected.name
        expected.filter = filter_by_key
        self._items[key] = expected

    def filter(self, obj):
        return True

    def check(self, test, changes):
        filtered = filter(self.filter, changes)
        test.assertEqual(len(self._items), len(filtered))
        for change in filtered:
            key = (type(change), change.name)
            self._items[key].check(test, (change,))
