#!/usr/bin/env python
# arch-tag: 8dfdb11a-a27c-4b25-8d66-c36850b6a9a2

# Copyright (C) 2005 Canonical Limited
# Author: David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for archive location support.
"""

import os
import stat
import shutil

import pybaz as arch
from pybaz import errors

import framework
import fixtures


class TestLocation(framework.NewTestCase):
    fixture = fixtures.NullFixture()

    def testContructor(self):
        """ArchiveLocation.__init__ raises when url is invalid."""
        self.assertRaises(TypeError, arch.ArchiveLocation, None)
        self.assertRaises(TypeError, arch.ArchiveLocation, u'/unicode/string/')
        self.assertRaises(ValueError, arch.ArchiveLocation, '')
        self.assertRaises(ValueError, arch.ArchiveLocation, 'relative/path')
        self.assertRaises(ValueError, arch.ArchiveLocation,
                          'unknown://example.com/archive')
        self.assertRaises(ValueError, arch.ArchiveLocation,
                          'file://example.com/archive')

    def testUrlProperty(self):
        """ArchiveLocation.url retrieves __init__ parameter."""
        url = 'ftp://example.com/archive'
        location = arch.ArchiveLocation(url)
        self.assertEqual(location.url, url)
        url = 'http://example.com/archive'
        location = arch.ArchiveLocation(url)
        self.assertEqual(location.url, url)
        url = 'sftp://example.com/archive'
        location = arch.ArchiveLocation(url)
        self.assertEqual(location.url, url)
        url = 'file:///archive'
        location = arch.ArchiveLocation(url)
        self.assertEqual(location.url, url)
        url = '/archive'
        location = arch.ArchiveLocation(url)
        self.assertEqual(location.url, url)
        # XXX: win32 absolute paths? -- David Allouche 2005-07-08


    def testUrlReadonly(self):
        """ArchiveLocation.url is read-only."""
        location = arch.ArchiveLocation('/archive')
        self.assertRaises(AttributeError, setattr, location, 'url', '/other')

    def testEqual(self):
        """ArchiveLocation.__eq__ and __ne__ work."""
        # use self.assert_ instead of assertEqual and assertNotEqual to be sure
        # to test __eq__ and __ne__
        here = arch.ArchiveLocation('/archive')
        there = arch.ArchiveLocation('/archive')
        self.failUnless(here == there)
        self.failIf(here != there)
        there = arch.ArchiveLocation('/other/archive')
        self.failIf(here == there)
        self.failUnless(here != there)
        class NotLocation(object):
            pass
        notloc = NotLocation()
        notloc.url = here.url
        self.failIf(here == notloc)
        self.failUnless(here != notloc)

    def testRepr(self):
        """ArchiveLocation.__repr__ works."""
        url = '/archive/location'
        location = arch.ArchiveLocation(url)
        self.assertEqual(repr(location), 'pybaz.ArchiveLocation(%r)' % url)


class TestCreateMaster(framework.NewTestCase):
    fixture = fixtures.ArchiveDirFixture()

    def testCreateMaster(self):
        """create_master creates a directory and registers the archive."""
        archive = arch.Archive('alice@example.com')
        self.assertEqual(archive.is_registered(), False)
        location = arch.ArchiveLocation(self.fixture.location(archive.name))
        assert not os.path.exists(location.url)
        params = arch.ArchiveLocationParams()
        location.create_master(archive, params)
        self.assert_(os.path.exists(location.url))
        self.assertEqual(archive.is_registered(), True)

    def testCreateMasterSanity(self):
        """create_master checks the type of its arguments."""
        archive = arch.Archive('alice@example.com')
        location = arch.ArchiveLocation(self.fixture.location(archive.name))
        params = arch.ArchiveLocationParams()
        self.assertRaises(TypeError, location.create_master, object(), params)
        self.assertRaises(TypeError, location.create_master, archive, object())


class TestCreateMirror(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def testCreateMirror(self):
        """create_mirror creates a directory with =meta-info/mirror."""
        archive = self.fixture.archive
        # do not use the -MIRROR suffix at it might work for some wrong reason
        mirror_path = self.fixture.location(archive.name + '-FOO')
        assert not os.path.exists(mirror_path)
        mirror = arch.ArchiveLocation(mirror_path)
        params = arch.ArchiveLocationParams()
        mirror.create_mirror(archive, params)
        self.assert_(os.path.exists(mirror_path))
        meta_mirror = os.path.join(mirror_path, '=meta-info', 'mirror')
        self.assert_(os.path.exists(meta_mirror))

    def testCreateMirrorSanity(self):
        """create_mirror checks the type of its arguments."""
        archive = self.fixture.archive
        mirror_path = self.fixture.location(archive.name + '-MIRROR')
        location = arch.ArchiveLocation(mirror_path)
        params = arch.ArchiveLocationParams()
        self.assertRaises(TypeError, location.create_mirror, object(), params)
        self.assertRaises(TypeError, location.create_mirror, archive, object())


class TestArchiveLocations(framework.NewTestCase):
    fixture = fixtures.MirrorLocationFixture()

    def testAllLocations(self):
        """all_locations return a list of locations."""
        archive = self.fixture.archive
        all_locations = [self.fixture.master, self.fixture.mirror]
        self.assertEqual(archive.all_locations(), all_locations)


class TestLocationIsRegistered(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def testIsRegistered(self):
        """is_registered works."""
        master = self.fixture.master
        self.assertEqual(master.is_registered(), True)
        not_registered = arch.ArchiveLocation('/not/registered')
        self.assertEqual(not_registered.is_registered(), False)


class TestLocationUnregister(framework.NewTestCase):
    fixture = fixtures.MirrorLocationFixture()

    def testUnregisterMirror(self):
        """unregister a mirror location"""
        archive = self.fixture.archive
        self.fixture.mirror.unregister()
        self.assertEqual(archive.all_locations(), [self.fixture.master])

    def testUnregisterMaster(self):
        """unregister a master location."""
        archive = self.fixture.archive
        self.fixture.master.unregister()
        self.assertEqual(archive.all_locations(), [self.fixture.mirror])

    def testUnregisterNonExistentRaises(self):
        """unregister raises LocationNotRegistered for unexistent locations."""
        location = arch.ArchiveLocation('/not/registered')
        self.assertRaises(errors.LocationNotRegistered, location.unregister)

    def testUnregisterTwiceRaises(self):
        """unregister raises LocationNotRegistered the second time."""
        location = self.fixture.master
        location.unregister()
        self.assertRaises(errors.LocationNotRegistered, location.unregister)


class TestArchiveLocationsUnregistered(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def testAllLocationUnregistered(self):
        """all_locations returns empty if archive has been unregistered."""
        self.fixture.master.unregister()
        self.assertEqual(self.fixture.archive.all_locations(), [])


class TestArchiveLocationsNotRegistered(framework.NewTestCase):
    fixture = fixtures.EmptySandboxFixture()

    def testAllLocationsNotRegisteredEmpty(self):
        """all_locations returns empty if archive was not registered."""
        archive = arch.Archive('alice@example.com')
        self.assertEqual(archive.all_locations(), [])


class TestArchiveUnregister(framework.NewTestCase):
    fixture = fixtures.MirrorLocationFixture()

    def testUnregisterArchive(self):
        """unregistering an archive unregister all locations."""
        archive = self.fixture.archive
        archive.unregister()
        self.assertEqual(archive.all_locations(), [])


class TwoUnregisteredLocationsFixture(fixtures.MirrorLocationFixture):
    """A fixture that provides an archive and mirror both unregistered."""

    def setUp(self):
        super(TwoUnregisteredLocationsFixture, self).setUp()
        self.archive.unregister()


class TestLocationRegister(framework.NewTestCase):
    fixture = TwoUnregisteredLocationsFixture()

    def testRegisterOne(self):
        """register() one location for an archive."""
        master = self.fixture.master
        master.register()
        archive = self.fixture.archive
        self.assertEqual(archive.all_locations(), [master])

    def testRegisterTwo(self):
        """register() two locations for an archive."""
        master, mirror = self.fixture.master, self.fixture.mirror
        master.register()
        mirror.register()
        archive = self.fixture.archive
        self.assertEqual(archive.all_locations(), [master, mirror])


class TestLocationRegisterFailure(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def testAlreadRegistered(self):
        """register() registered location raises LocationAlreadyRegistered."""
        master = self.fixture.master
        self.assertRaises(errors.LocationAlreadyRegistered, master.register)

    def testMissingLocation(self):
        """register() with inexistent location raises ExecProblem."""
        not_here = arch.ArchiveLocation('/not/here')
        self.assertRaises(errors.ExecProblem, not_here.register)

class TestLocationMetaInfo(framework.NewTestCase):
    fixture = fixtures.MirrorLocationFixture()

    def testMetaInfo(self):
        """meta_info works."""
        archive = self.fixture.archive
        mirror = self.fixture.mirror
        self.assertEqual(mirror.meta_info('name'), archive.name)
        self.assertEqual(mirror.meta_info('mirror'), archive.name)

    def testMetaInfoSanity(self):
        """meta_info checks the sanity of its argument."""
        master = self.fixture.master
        self.assertRaises(TypeError, master.meta_info, object())
        self.assertRaises(ValueError, master.meta_info, str())

    def testMetaInfoError(self):
        """meta_info raises MetaInfoError when meta-info is inexistent."""
        master = self.fixture.master
        self.assertRaises(errors.MetaInfoError, master.meta_info, 'mirror')

# XXX: Cannot run that test because baz does not give us enough information.
# See the meta_info method implementation for details.
# -- David Allouche 2005-07-12

#     def testMetaInfoFailure(self):
#         """meta_info raises ExecProblem when location is missing."""
#         master = self.fixture.master
#         shutil.rmtree(master.url)
#         self.assertRaises(errors.ExecProblem, master.meta_info, 'name')

    def testMetaInfoUnregistered(self):
        """meta_info raises LocationNotRegistered when not registered."""
        not_registered = arch.ArchiveLocation('/not/registered')
        self.assertRaises(errors.LocationNotRegistered,
                          not_registered.meta_info, 'name')

    def testArchive(self):
        """ArchiveLocation.archive() works."""
        # NOTE: relies on the use of the meta_info method for proper handling
        # of error cases.
        archive = self.fixture.archive
        master = self.fixture.master
        meta_info_messages = []
        original_meta_info = master.meta_info
        def instrumented_meta_info(key):
            meta_info_messages.append(key)
            return original_meta_info(key)
        master.meta_info = instrumented_meta_info
        self.assertEqual(master.archive(), archive)
        self.assertEqual(meta_info_messages, ['name'])

    def testDefaultOptions(self):
        """check default archive options."""
        # that is depended upon by TestCreateMasterOptions and
        # TestCreateMirrorOptions.
        master = self.fixture.master
        self.assertEqual(master._meta_info_present('signed-archive'), False)
        self.assertEqual(master._meta_info_present('http-blows'), False)
        self.assertEqual(master._version_string(), baz_version_str)
        mirror = self.fixture.mirror
        self.assertEqual(mirror._meta_info_present('signed-archive'), False)
        self.assertEqual(mirror._meta_info_present('http-blows'), False)
        self.assertEqual(mirror._version_string(), baz_version_str)


tla_version_str = 'Hackerlab arch archive directory, format version 2.'
baz_version_str = 'Bazaar archive format 1 0'


class TestCreateMasterOptions(framework.NewTestCase):
    fixture = fixtures.ArchiveDirFixture()

    def makeMaster(self, params):
        archive = arch.Archive('alice@example.com')
        master_url = self.fixture.location(archive.name)
        master = arch.ArchiveLocation(master_url)
        master.create_master(archive, params)
        return master

    def testMasterSigned(self):
        params = arch.ArchiveLocationParams()
        params.signed = True
        master = self.makeMaster(params)
        self.assertEqual(master._meta_info_present('signed-archive'), True)
        self.assertEqual(master._meta_info_present('http-blows'), False)
        self.assertEqual(master._version_string(), baz_version_str)

    def testMasterListings(self):
        params = arch.ArchiveLocationParams()
        params.listing = True
        master = self.makeMaster(params)
        self.assertEqual(master._meta_info_present('signed-archive'), False)
        self.assertEqual(master._meta_info_present('http-blows'), True)
        self.assertEqual(master._version_string(), baz_version_str)

    def testMasterTla(self):
        params = arch.ArchiveLocationParams()
        params.tla_format()
        master = self.makeMaster(params)
        self.assertEqual(master._meta_info_present('signed-archive'), False)
        self.assertEqual(master._meta_info_present('http-blows'), False)
        self.assertEqual(master._version_string(), tla_version_str)


class TestCreateMirrorOptions(framework.NewTestCase):
    fixture = fixtures.ArchiveFixture()

    def makeMirror(self, params):
        archive = self.fixture.archive
        mirror_url = self.fixture.location(archive.name + '-MIRROR')
        mirror = arch.ArchiveLocation(mirror_url)
        mirror.create_mirror(archive, params)
        return mirror

    def testMasterSigned(self):
        params = arch.ArchiveLocationParams()
        params.signed = True
        mirror = self.makeMirror(params)
        self.assertEqual(mirror._meta_info_present('signed-archive'), True)
        self.assertEqual(mirror._meta_info_present('http-blows'), False)
        self.assertEqual(mirror._version_string(), baz_version_str)

    def testMasterListings(self):
        params = arch.ArchiveLocationParams()
        params.listing = True
        mirror = self.makeMirror(params)
        self.assertEqual(mirror._meta_info_present('signed-archive'), False)
        self.assertEqual(mirror._meta_info_present('http-blows'), True)
        self.assertEqual(mirror._version_string(), baz_version_str)

    def testMasterTla(self):
        params = arch.ArchiveLocationParams()
        params.tla_format()
        mirror = self.makeMirror(params)
        self.assertEqual(mirror._meta_info_present('signed-archive'), False)
        self.assertEqual(mirror._meta_info_present('http-blows'), False)
        self.assertEqual(mirror._version_string(), tla_version_str)


class TestMirroring(framework.NewTestCase):

    fixture = fixtures.MirroringFixture()

    def testLocationMirroring(self):
        """Test simple mirroring between two locations."""
        archive = self.fixture.archive
        all_revisions = self.fixture.all_revisions
        master, mirror = self.fixture.master, self.fixture.mirror1
        mirrorer = master.make_mirrorer(mirror)
        mirrorer.mirror()
        mirror1_revisions = list(archive.iter_location_revisions(mirror))
        self.assertEqual(mirror1_revisions, all_revisions)
        mirror2 = self.fixture.mirror2
        mirror2_revisions = list(archive.iter_location_revisions(mirror2))
        self.assertEqual(mirror2_revisions, [])

    def testLocationMirroringSanity(self):
        """ArchiveLocation.make_mirrorer sanity-checks its parameter."""
        master = self.fixture.master
        same_as_master = arch.ArchiveLocation(master.url)
        self.assertRaises(TypeError, master.make_mirrorer, object())
        self.assertRaises(ValueError, master.make_mirrorer, same_as_master)
        mirror = self.fixture.mirror1
        mirror.unregister()
        self.assertRaises(errors.LocationNotRegistered,
                          master.make_mirrorer, mirror)
        mirror.register()
        master.unregister()
        self.assertRaises(errors.LocationNotRegistered,
                          master.make_mirrorer, mirror)
        master.register()
        other_master = self.fixture.other_master
        self.assertRaises(errors.MirrorLocationMismatch,
                          master.make_mirrorer, other_master)

    def testLocationMirroringVersion(self):
        """Test mirroring a single version between two locations."""
        archive = self.fixture.archive
        master, mirror = self.fixture.master, self.fixture.mirror1
        mirrorer = master.make_mirrorer(mirror)
        mirrorer.mirror(self.fixture.version1)
        mirror1_revisions = list(archive.iter_location_revisions(mirror))
        version = self.fixture.version1
        version1_revisions = list(version.iter_location_revisions(master))
        self.assertEqual(mirror1_revisions, version1_revisions)

    def testLocationMirroringLimitSanity(self):
        """MirrorMethod.mirror() sanity-checks its limit argument."""
        master, mirror = self.fixture.master, self.fixture.mirror1
        other_archive = self.fixture.other_archive
        version = self.fixture.version1
        other_version = arch.Version(other_archive.name
                                     + '/' + version.nonarch)
        mirrorer = master.make_mirrorer(mirror)
        self.assertRaises(TypeError, mirrorer.mirror, limit=object())
        self.assertRaises(ValueError, mirrorer.mirror, limit=other_version)


framework.register(__name__)
