/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * Validate.cc
 *  Class to check domain, email, name
 *
 * $Id: Validate.cc,v 1.20 2002/03/27 05:46:28 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@step.polymtl.ca>
 * Copyright (c) 2000 Benoit Joly <benoit@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * Licensed under the GPLv2
 */

#include <stdio.h>
#include <iostream.h>
#include <strstream.h>
#include <sys/types.h>
#include <regex.h>

#include "Exception.h"
#include "Validate.h"

// FIXME: what about multiple domains ?
// by that I mean, what if we want people to be able to register either under
// .dhis.net or .ddts.net ...
Validate::Validate(const char *bannedHostFileName, const char *reservedHostFileName)
{
    this->bannedHostFileName = strdup(bannedHostFileName);
    this->reservedHostFileName = strdup(reservedHostFileName);
    rxx = new Regexx;
}

Validate::~Validate()
{
    delete rxx;
}

bool Validate::isThirdLevelDomain(const char *fqdn)
{
    char fqdnRegExp[256];
    snprintf(fqdnRegExp, 128, "%s", "^[^\\.]*\\.[^\\.]*\\.[^\\.]*\\.*$");

    if (rxx->exec(fqdn, fqdnRegExp))
    {
        return true;
    }
    return false;
}

bool Validate::isValidDomain(const char *domain)
{
    char domainReg[1024];
    char domainElementReg[1024];
    char domainBuf[1024];
    char domainElement[1024];

    int isTested = 0;


    strncpy(domainBuf, domain, 1023);

    strncpy(domainReg, "^([A-Za-z0-9]+)\\.(.+)", 1023);
    
    strncpy(domainElementReg, "^(([[:alnum:]-]{0,61}[[:alnum:]])"
            "|([[:alpha:]])){1,}$", 1023);


    while(rxx->exec(domainBuf, domainReg))
    {
        isTested = 1;

        strncpy(domainElement, rxx->match[0].atom[0].str().c_str(), 1023);
        strncpy(domainBuf, rxx->match[0].atom[1].str().c_str(), 1023);

        if (!rxx->exec(domainElement, domainElementReg))
        {
            return false;
        }
    }

    if (!rxx->exec(domainBuf, domainElementReg))
    {
        return false;
    }

    return (isTested ? true : false);
}

bool Validate::isValidDomain (const char *domain, const char *baseDomain)
{
    char domainReg[256];
    // satisfies conditions expressed in RFC 1035 for domain names
    snprintf(domainReg, 128, "%s%s$",
             "^(([[:alpha:]][[:alnum:]-]{0,61}[[:alnum:]]\\.)"
             "|([[:alpha:]]\\.)){1,}",
             baseDomain);
    if (rxx->exec(domain, domainReg))
    {
        return true; 
    }
    return false; 
}

bool Validate::isReservedDomain(const char *domain)
{
    ifstream ifs(reservedHostFileName);
    char line[128];

    while (ifs.getline(line, 128))
    {
        // check for reserved names
        if (rxx->exec(domain, line))
        {
            return true;
        }
    }
    return false; 
}

bool Validate::isBannedDomain(const char *domain)
{
    ifstream ifs(bannedHostFileName);
    char line[128];

    while (ifs.getline(line, 128))
    {
        // check for banned names
        if (rxx->exec(domain, line))
        {
            return true;
        }
    }
    return false; 
}

// check validity of a record request
bool Validate::recordAdd(DdtManager *manager, int id, const char *dname, 
                         DnsRecordType type, const char *data, 
                         const char *baseDomain)
{
    char fqdn[256];

    manager->fetchAccountInfo(id, "fqdn", fqdn, sizeof(fqdn));
    
    switch (type)
    {
    case CNAME:
        // check syntax for cnames

        // 1) check if alias is valid
        if (isValidDomain(dname) == false)
        {
            cout << "Invalid CNAME.<br>\n";
            return false;
        }

        // 2) check if it's under the users' fqdn
        if (isValidDomain(dname, fqdn) == false)
        {
            cout << "Invalid CNAME. dname is not part of your fqdn " 
                 << fqdn << "<br>" << endl;
            return false;
        }

        // 3) check if its banned
        if (isBannedDomain(dname) == true)
        {
            cout << "Hostname banned" << endl;
            return false;
        }
        
        // 4) check that it's not used in db either.
        if (manager->findUserAccountIdFromFqdn((char *)dname) != -1)
        {
            cout << "Hostname already in use.<br>\n";
            return false;
        }
            
/*
        // check the dname is really the users'
        if (manager->findUserAccountIdFromFqdn((char *)data) != id)
        {
            cout << "dname must be yours.<br>\n";
            return false;
        }
*/

        return true;
        break;
            
    case MX:
        // check syntax for MX records
            
        // 1) check if host is really bellonging to the one
        if (manager->findUserAccountIdFromFqdn((char *)dname) != id)
        {
            cout << "dname must be yours.<br>\n";
            return false;
        }

        // 2) split priority and hostname
        rxx->exec (data, "([0-9]+)[[:space:]]([^[:space:]]+)");
            
        // 3) make sure there were too parts
        if (!(rxx->match.size() > 0 && rxx->match[0].atom.size() == 2))
        {
            cout << "Bad format<br>\n";
            return false;
        }
            
        // 4) assign the parts
        char priority[4];
        char mx[1024];
        strncpy(priority, rxx->match[0].atom[0].str().c_str(), 3);
        strncpy(mx, rxx->match[0].atom[1].str().c_str(), 1023);


//        cout << "prio: " << priority << endl;
//        cout << "mx: " << mx << endl;


        // 5) check the priority syntax
        if (rxx->exec(priority, "[0-9]{1,3}") == false)
        {
            cout << "Bad priority<br>\n";
            return false;
        }

        // 6) check if mx is valid
        if (isValidDomain(mx) == false)
        {
            cout << "Bad mx name<br>\n";
            return false;
        }
        return true;            
        break;
            
    default:
        cout << "Unknown type<br>\n";
        return false;
        break;
    }
    
    return true;
}

bool Validate::name(const char *name)
{
    // FIXME: should accept 8 bit characters ?
    if (rxx->exec(name, "^[[:alpha:][:space:]\\.-]{1,64}$"))
    {
        return true;
    }

    return false;
}

bool Validate::email(const char *email, const char *baseDomain)
{
    char reg[256];

    // check email syntax -- FIXME: could be more accurate. check this up
    if (!rxx->exec(email, "^(([[:alnum:]\\.-_]{1,64}\\@[[:alnum:]\\.-_]{1,64})){1,}$"))
    {
        return false;
    }

    snprintf(reg, 128, "@.+\\.%s", baseDomain);

    if(rxx->exec(email, reg))
    {
        return false;
    }
    
    return true;
}

bool Validate::ip (const char *ip)
{
    // check IP addr syntax
    // FIXME: should check that every byte does not exceed 255
    if (rxx->exec (ip, "^[[:digit:]]{1,3}\\.[[:digit:]]{1,3}\\.[[:digit:]]{1,3}\\.[[:digit:]]{1,3}$") != 1)
    {
        return false;
    }
    return true;
}
