#!/usr/bin/perl -w 

# ddtbot - an IRC bot for the DDT Service
# Copyright (C) 2000 Luca Filipozzi and Remi Lefebvre
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use POSIX;
use Net::IRC;
use Net::DNS;
use Text::Tabs;
use DBI::DBD;

use strict;

## read in the configuration file
my %cfg;
# FIXME: should be /etc/ddtbot.conf
&read_config("/home/ddtbot/ddtbot.conf");

## become a daemon
&become_daemon;

## instantiate the objects we need
#  first, the IRC object
my $irc = new Net::IRC;
my $conn = $irc->newconn(
    Server  => $cfg{lc 'IrcServer'},
    Port    => $cfg{lc 'IrcPort'},
    Nick    => $cfg{lc 'IrcNick'},
    Ircname => $cfg{lc 'IrcUsername'});

$conn->debug(0);
$conn->add_global_handler('endofmotd',      \&on_endofmotd);
$conn->add_global_handler('notice',         \&on_notice);
$conn->add_global_handler('nicknameinuse',  \&on_nicknameinuse);
$conn->add_global_handler('disconnect',     \&on_disconnect);
$conn->add_global_handler('kick',           \&on_kick);
$conn->add_global_handler('public',         \&on_public);
$conn->add_global_handler('msg',            \&on_msg);

#  second, the DNS object
my $res = new Net::DNS::Resolver;
$res->nameservers("216.191.16.126");

#  finally, the DBI objects
my $dbh = DBI->connect(
    "dbi:Pg:dbname=$cfg{lc 'DbDatabase'}",
    $cfg{lc 'DbUsername'},
    $cfg{lc 'DbPassword'});
my $sth_name    = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where contactName like ?");
my $sth_id      = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where useraccountid = ?");
my $sth_email   = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where contactEmail like ?");
my $sth_fqdn    = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where fqdn like ?");
my $sth_online  = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where ipAddress <> '169.254.0.0'");
my $sth_offline = $dbh->prepare("select useraccountid, contactname, fqdn, ipaddress from useraccounts where ipAddress = '169.254.0.0'");

## start everything up
$irc->start;

## should never get here
$sth_name->finish;
$sth_id->finish;
$sth_email->finish;
$sth_fqdn->finish;
$sth_online->finish;
$sth_offline->finish;
$dbh->disconnect;
exit 0;
            
######
sub on_endofmotd
{
    my ($self, $event) = @_;

    $self->join($cfg{lc 'IrcChannel'});
}

sub on_notice
{
    my ($self, $event) = @_;
    my $nick = $event->nick();
    my $args = ($event->args)[0];

    if ($nick =~ /^NickServ$/i)
    {
        if ($args =~ /^This nickname is registered/i)
        {
	    logger ("Authentication requested.");
            $self->privmsg($nick, "IDENTIFY " . $cfg{lc 'IrcPassword'});
        }
        elsif ($args =~ /^Password accepted - you are now recognized/i)
        {
	    logger ("Authentication success with nickserv");
	    # now that we are authenticated, we need to part/join from our
	    # channels to get priviledges
	    $self->part($cfg{lc 'IrcChannel'});
	    $self->join($cfg{lc 'IrcChannel'});
        }
    }
}

sub on_nicknameinuse
{
    my ($self, $event) = @_;

    $self->nick($self->nick . "_");
    $self->privmsg("NickServ", "GHOST " . $cfg{lc 'IrcNick'} . " " . $cfg{lc 'IrcPassword'});
    $self->nick($cfg{lc 'IrcNick'});
}

sub on_disconnect
{
    my ($self, $event) = @_;
    #my $from = $event->from();
    #my $what = ($event->args)[0];

    $self->connect();

    #&status("disconnect from $from ($what).");
    #$ircstats{'DisconnectReason'} = $what;

    # clear any variables on reconnection.
    #$nickserv = 0;

    #&clearIRCVars();

    #if (!$self->connect()) {
        #&WARN("not connected? help me. ircCheck() should reconnect me");
    #}
}

sub on_kick
{
    my ($self, $event) = @_;
    my $nick = $event->nick;

    # err, it always matches :p -- FIXME 
    if ($nick == $cfg{lc 'IrcNick'})
    {
	logger ("Some bastard kicked me off!!!");
	$self->join($cfg{lc 'IrcChannel'});
    }
    else
    {
	logger ("$nick got kicked off.");
    }
}

sub on_public
{
    my ($self, $event) = @_;
    my $args = ($event->args)[0];
    my $chan = lc(($event->to)[0]);
    my $nick = $event->nick;
    my $mynick = $self->nick;
    
    if ($args =~ /^\s*${mynick}\s*[-\:\,\> ]+ (\w+) *(.*)/i)
    {
	logger ("public msg from $nick on $chan: $args");
	process($self, $chan, "$nick: ", $1, $2);
    }
}

sub on_msg
{
    my ($self, $event) = @_;
    my $args = ($event->args)[0];
    my $chan = lc(($event->to)[0]);
    my $nick = $event->nick;

    logger ("private msg from $nick: $args");

    if ($args =~ /^!?(\w+) *(.*)/i)
    {
        process($self, $nick, "", $1, $2);
    }
}

sub process
{
    my ($self, $nick, $pre, $cmd, $cmdline) = @_;

    my @cmdline = split(/[ \t\n]+/, $cmdline);

    if ($cmd =~ /^help$/)
    {
        $self->privmsg($nick, $pre . "help|hello|status|uptime|online|offline|nslookup <query>|dblookup [name|email|ip|fqdn] <query>");
    }
    elsif ($cmd =~ /^hello$/)
    {
        $self->privmsg($nick, $pre . "hello");
    }
    elsif ($cmd =~ /^status$/)
    {
        $self->privmsg($nick, $pre . "I am ddtbot, #ddt's bot. Contact the channel ops for more information.");
    }
    elsif ($cmd =~ /^uptime$/)
    {
        $self->privmsg($nick, $pre . `uptime`);
    }
    elsif ($cmd =~ /^nslookup$\|^ns$/)
    {
        my $packet = $res->search(@cmdline);
        if ($packet)
        {
            my $rr;
            foreach $rr ($packet->answer)
            {
                $self->privmsg($nick, $pre . expand($rr->string));
            }
        }
        else
        {
            $self->privmsg($nick, $pre . "lookup failed");
        }
    }
    elsif ($cmd =~ /^dblookup$\|^db$/)
    {
        my $action = shift(@cmdline);
        if ($action =~ /^name$/)
        {
            $sth_name->execute("\%" . join(' ',@cmdline) . "\%");
            $self->privmsg($nick, $pre . $sth_name->rows . " rows returned");
            if ($sth_name->rows <= 5 or $pre eq "")
            {
                while (my @row = $sth_name->fetchrow_array)
                {
                    $self->privmsg($nick, $pre . join(' ', @row));
                }
            }
            else
            {
                $self->privmsg($nick, $pre . "too many results; restrict your search");
            }
        }
        elsif ($action =~ /^id$/)
        {
            $sth_id->execute(join(' ',@cmdline));
            $self->privmsg($nick, $pre . $sth_id->rows . " rows returned");
            if ($sth_id->rows <= 5 or $pre eq "")
            {
                while (my @row = $sth_id->fetchrow_array)
                {
                    $self->privmsg($nick, $pre . join(' ', @row));
                }
            }
            else
            {
                $self->privmsg($nick, $pre . "too many results; restrict your search");
            }
        }
        elsif ($action =~ /^email$/)
        {
            $sth_email->execute("\%" . join(' ',@cmdline) . "\%");
            $self->privmsg($nick, $pre . $sth_email->rows . " rows returned");
            if ($sth_email->rows <= 5 or $pre eq "")
            {
                while (my @row = $sth_email->fetchrow_array)
                {
                    $self->privmsg($nick, $pre . join(' ', @row));
                }
            }
            else
            {
                $self->privmsg($nick, $pre . "too many results; restrict your search");
            }
        }
        elsif ($action =~ /^fqdn$/)
        {
            $sth_fqdn->execute("\%" . join(' ',@cmdline) . "\%");
            $self->privmsg($nick, $pre . $sth_fqdn->rows . " rows returned");
            if ($sth_fqdn->rows <= 5 or $pre eq "")
            {
                while (my @row = $sth_fqdn->fetchrow_array)
                {
                    $self->privmsg($nick, $pre . join(' ', @row));
                }
            }
            else
            {
                $self->privmsg($nick, $pre . "too many results; restrict your search");
            }
        }
        # add more dblookup actions here
        else
        {
            #$self->privmsg($chan, $pre . "$action is not a valid dblookup action");
        }
    }
    elsif ($cmd =~ /^online$/)
    {
        # we don't take any options yet
        $sth_online->execute;
        $self->privmsg($nick, $pre . $sth_online->rows . " hosts online");
    }
    elsif ($cmd =~ /^offline$/)
    {
        # we don't take any options yet
        $sth_offline->execute;
        $self->privmsg($nick, $pre . $sth_offline->rows . " hosts offline");
    }
    # add more commands here
    else
    {
        #$self->privmsg($chan, $pre . "unknown command: $cmd");
    }
}

######
sub read_config
{
    my $file = shift;
    my $num = 0;

    open CONFIG, $file or die "Can't read config:";

    while (<CONFIG>)
    {
        chomp;
        if ( ! /^\s*#/ )
        {
            if ( /\s*=\s*#/ )
            {
                s/(.*=\s*\#.*)\#.*/$1/;
            }
            else
            {
                s/(.*)\#.*/$1/;
            }
            (my $var, my $val) = split /\s*=\s*/;
            $var = (lc $var);
            $cfg{$var} = $val;
        }
    }
    close CONFIG;
}

sub become_daemon
{
    # become a daemon (i.e, fork and kill parent)
    my $pid = fork;
    die "couldn't fork: $!" unless defined($pid);
    exit if $pid; # we are the parent
    POSIX::setsid() or die "couldn't start a new session: $!";

    # open logfile, before dropping root so we can have it in /var/log or
    # where ever we like.
    open LOGFILE, ">>" . $cfg{lc 'LogFile'} or die "can't open logfile: $!";
    LOGFILE->autoflush(1);
    logger ("DDT Bot Started.");

    # drop root
    my $uid = POSIX::getpwnam($cfg{lc 'RunAsUser'});
    my $gid = POSIX::getgrnam($cfg{lc 'RunAsGroup'});
    ($<, $>) = ($uid, $uid) unless $>;
    ($(, $)) = ($gid, $gid) unless $);

    logger ("Dropped root.");

    # write the pid file
    open  PIDFILE, ">" . $cfg{lc 'PidFile'} or die "can't open pidfile: $!";
    print PIDFILE POSIX::getpid();
    close PIDFILE;

    close STDIN; close STDOUT; close STDERR;
}

sub logger
{
    # this function takes one argument, the message to be logged.
    # the current date and time is prefixed

    my $string = shift;
    my $now_string = gmtime;

    print LOGFILE $now_string, " - ", $string, "\n";
}
