/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * ddtcd.cc
 *   This is the main file for the client daemon.
 *
 * $Id: ddtcd.cc,v 1.70 2002/04/08 07:11:01 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@dhis.net>
 * Copyright (c) 2000 Luca Filipozzi <lfilipoz@dhis.net>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */

#include "ddtcd.h"
#include "DdtMD.h"

using std::cout;
using std::endl;
using std::cerr;

Options *Options::i_ = NULL;
Options *Client::opts = NULL;
Logger *Client::log = NULL;
UDP *Client::udp = NULL;
DdtpPacket *Client::pkt = NULL;

// main routine
int main(int argc, char **argv)
{
    Client *ddtcd = new Client;

    // check we are not already running
    ddtcd->alreadyRunning();

    if (ddtcd->parseArgs(&argc, &argv) == -1)
    {
        exit(1);
    }

    ddtcd->becomeDaemon();
    ddtcd->receiver();

    return 0;
}

Client::Client()
{
    opts = Options::Instance();
    forceIP = 0;
    
    // trap terminating signals
    atexit(atexit_handler);

    /* that may look hackish at first but that is not, believe me ;) */
    signal(SIGPIPE, SIG_IGN);
    signal(SIGHUP, signal_handler);
    signal(SIGINT, signal_handler);
    signal(SIGQUIT, signal_handler);
    signal(SIGABRT, signal_handler);
    signal(SIGTERM, signal_handler);

    md5_password = new char[2 * DdtMD::getAlgoDigestLength(DdtMD::MD_MD5) + 1];
}

Client::~Client()
{
    
    if (md5_password) delete [] md5_password;

    if (pkt) delete pkt;
    if (udp) delete udp;
    if (opts) delete opts;
    if (log) delete log;
}

int Client::alreadyRunning()
{
    // check if there's a pidfile .. if yes, check if it is stale
    FILE *pidfp;
    FILE *cmdlinefilefp;
    char cmdline[64];
    char cmdlinefile[64];
    int pid = 0;
    struct stat statbuf;
   
    // pid dir must exist or fopen() would segfault
    if (stat(PID_DIR, &statbuf) == 0)
    {
        if (!S_ISDIR(statbuf.st_mode))
        {
            cout << PID_DIR << " is not a directory.";
            exit (1);
        }
    }
    if (stat(PID_DIR "/ddt", &statbuf) == 0)
    {
        if (!S_ISDIR(statbuf.st_mode))
        {
            cout << PID_DIR "/ddt exists but is not a directory.";
            exit (1);
        }
    }
    else
    {
        cout << "creating " PID_DIR "/ddt ...";
        mkdir(PID_DIR "/ddt",
              S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    }
    
    // we check if theres a pidfile
    if ((pidfp = fopen(PID_FILE, "r")) != NULL)
    {
        // what's the pid
        if (fscanf(pidfp, "%d", &pid) == 1)
        {
            // if process exists
            if (kill(pid, 0) == 0)
            {
                // we check if it's a ddtcd or any other random process
                snprintf(cmdlinefile, sizeof (cmdlinefile), "/proc/%d/cmdline", pid);
                
                if (((cmdlinefilefp = fopen(cmdlinefile, "r"))) != NULL)
                {
                    if (fscanf(cmdlinefilefp, "%s", cmdline) == 1)
                    {
                        if (strstr(cmdline, "ddtcd") != NULL)
                        {
                            // it was a ddtcd, exit
                            cout << "daemon already running, exiting\n";
                            exit(0);
                        }
                        else
                        {
                            // it was really stale, remove the pid file and
                            // continue
                            cout << "removing stale pid file\n";
                            unlink(PID_FILE);
                        }
                    }
                }
                fclose(cmdlinefilefp);
            }
            else
            {
                cout << "removing stale pid file\n";
                unlink(PID_FILE);
                fclose(pidfp);
            }
        }
    }
    return 0;
}

int Client::parseArgs(int *argc, char ***argv)
{
    optTitle("ddtcd - Dynamic DNS Tools Client Daemon\n");
    optrega(&opts->verbose,   OPT_FLAG,  'v', "verbose",   "  verbose mode");
    optrega(&opts->debug,     OPT_FLAG,  'd', "debug",     "  debug mode");
    optrega(&opts->encrypt,   OPT_FLAG,  'e', "encrypt",   "  encrypt packet");
    optrega(&opts->hostaddr,  OPT_STRING,'H', "hostaddr",  " *server address");
    optrega(&opts->serverport,OPT_UINT,  'S', "serverport"," *server udp port");
    optrega(&opts->clientport,OPT_UINT,  'C', "clientport"," *client udp port");
    optrega(&opts->interface, OPT_STRING,'I', "interface", " *interface");
    optrega(&opts->accountId, OPT_INT,   'i', "accountid", "  user account Id");
    optrega(&opts->password,  OPT_STRING,'p', "password",  "  update password");
    optrega(&opts->version,   OPT_FLAG,  'V', "version",   "  version info");
    optAdditionalUsage(opts->additionalUsage);
    optDefaultFile(CONF_FILE);

    // parse command line and/or read config file
    opt(argc, argv);

    // validate other options
    bool exitRequired = false;
    if (opts->version)
    {
        cout << "ddtcd v" << VERSION << endl;
        exit (0);
    }

    if (!optinvoked(&opts->hostaddr))
    {
        cerr << "E: no server address!" << endl;
        exitRequired = true;
    }
    
    if (!optinvoked(&opts->serverport))
    {
        cerr << "E: no server port!" << endl;
        exitRequired = true;
    }

    if (!optinvoked(&opts->clientport))
    {
        cerr << "E: no client port!" << endl;
        exitRequired = true;
    }
    
    if (!optinvoked(&opts->interface))
    {
        cerr << "E: no interface!" << endl;
        exitRequired = true;
    }

    if (!optinvoked(&opts->accountId))
    {
        cerr << "E: no account ID!" << endl;
        exitRequired = true;
    }
   
    if (!optinvoked(&opts->password))
    {
        cerr << "E: no password!" << endl;
        exitRequired = true;
    }

    if (optinvoked(&opts->password) && strlen(opts->password) > 8)
    {
        opts->password[8] = '\0';
        cerr << "W: password longer than 8 chars, truncating" << endl;
    //    exitRequired = true;
    }
    
    if (exitRequired)
    {
        cerr << "missing or bad options, aborting." << endl;
        return -1;
    }
    
    try
    {
        pkt = new DdtpPacket;
        udp = new UDP(opts->hostaddr, opts->clientport, opts->serverport);
        log = Logger::Instance();
    }
    catch (DdtException &e)
    {
        cerr << e.message() << endl;
        return -1;
    }

    log->setIdent("ddtcd");
    log->openSyslog(LOG_USER, LOG_NOTICE);
    log->openFilelog(LOG_FILE, LOG_NOTICE);

    // validate verbosity/debugging options
    if (opts->verbose)
    {
        log->setFilelogPriority (LOG_INFO);
    }
    if (opts->debug)
    {
        log->setFilelogPriority (LOG_DEBUG);
    }

    // initialise variables
    serverStamp   = 0;
    retryStamp    = 0;
    retryCount    = 0;
    retryIn       = 3;
    currentStatus = OFFLINE;
    changeStatus  = OFFLINE;
    lastIP        = 0; // offline at start
    
    // generate the md5_password
    int md5Length = DdtMD::getAlgoDigestLength(DdtMD::MD_MD5);
    unsigned char digest[md5Length];
    DdtMD::digest(DdtMD::MD_MD5, (unsigned char*)opts->password, strlen(opts->password), digest, md5Length);
    for (int i = 0; i < md5Length; i++)
    {
        sprintf(&(md5_password[i*2]), "%02x", digest[i]);
    }
    return 0;
}

int Client::becomeDaemon()
{
    // become a daemon
#ifndef DDT_DEBUG
#ifdef HAVE_DAEMON
    daemon(true, false); // XXX: RH6.0's daemon(), undef HAVE_DAEMON and it should work
#else
    if (fork ()) { exit (0); }
    close (0); close (1); close (2);
#endif // HAVE_DAEMON
#endif // DDT_DEBUG

    // and let's start! :)
    log->notice ("daemon starting");
    
    struct passwd *nobody;
    struct group *nogroup;
    nobody = getpwnam ("nobody");
    nogroup = getgrnam ("nogroup");

    // FIXME: to check or not to check ?
    // create the dir. we do not check since we will create anyway
    mkdir (PID_DIR "/ddt", S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    mkdir (VAR_DIR "/ddt-client",
           S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
        
    // create our FIFO if they don't exist
    if ((mkfifo (INFIFO, S_IRUSR | S_IWUSR) == -1) && (errno != EEXIST))
    {
        log->error ("failed to create input FIFO: %s", strerror (errno)); 
        cerr << "failed to create FIFO" << endl;
        exit (1);
    }
    
    if ((mkfifo (OUTFIFO, S_IRUSR | S_IWUSR) == -1) && (errno != EEXIST))
    {
        log->error ("failed to create output FIFO: %s", strerror (errno)); 
        cerr << "failed to create FIFO" << endl;
        exit (1);
    }

    // if we are root
    if (getuid () == 0)
    {
        // if nobody and nogroup exists on system
        if (nobody != NULL && nogroup != NULL)
        {
            // give the files to nobody.nogroup since the daemon gives up permissions
            // maybe we should use a 'ddt' user and group someday. we are not at that
            // point yet though.
            if (chown (PID_DIR "/ddt", nobody->pw_uid, nogroup->gr_gid) == -1)
            {
                log->debug ("failed to give piddir to nobody.nogroup: %s",
                            strerror (errno));
            }

            if (chown (VAR_DIR "/ddt-client", nobody->pw_uid, nogroup->gr_gid) == -1)
            {
                log->debug ("failed to give var dir to nobody.nogroup: %s",
                            strerror (errno));
            }

            if (chown (INFIFO, nobody->pw_uid, nogroup->gr_gid) == -1)
            {
                log->debug ("failed to give input fifo to nobody.nogroup: %s",
                            strerror (errno));
            }
            
            if (chown (OUTFIFO, nobody->pw_uid, nogroup->gr_gid) == -1)
            {
                log->debug ("failed to give output fifo to nobody.nogroup: %s",
                            strerror (errno));
            }

            // drop group priviliedges first. :)
            if (setgid (nogroup->gr_gid) != -1)
            {
                log->debug ("setgid nogroup; dropping root");
            }
            else
            {
                log->error ("failed to setgid nogroup: %s", strerror (errno));
            }

            // drop user priviledges
            if (setuid (nobody->pw_uid) != -1)
            {
                log->debug ("setuid nobody; dropping root");
            }
            else
            {
                log->error ("failed to setuid nobody: %s", strerror (errno));
            }
        }
        else
        {
            log->error ("user nobody and/or group nogroup not found on this "
                        "system; continuing anyway");
        }
    }
    else
    {
        log->error ("not root, cannot get nobody uid");
    }

    // write pid file
    int fd;
    if ((fd = creat (PID_FILE, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) != -1)
    {
        char buf[8];
        sprintf (buf, "%d\n", getpid ());
        if (write (fd, buf, strlen (buf)) == -1)
        {
            close (fd);
        }
    }
    else
    {
        log->error ("failed to create pid file (%s): %s", PID_FILE,
                    strerror (errno));
    }

    if ((infifofd = open (INFIFO, O_RDWR | O_NONBLOCK, 0)) == -1)
    {
        log->error ("failed to open infifo: %s", strerror (errno));
        return -1;
    }

    return 0;
}

// main listen routine
int Client::receiver ()
{
    fd_set rset;
    struct timeval tv;

    FD_ZERO (&rset);

    outfifofd = -1;

    // wait for data on either the socket (from server) or the FIFO (from client)
    while (1)
    {
        tv.tv_sec = 5;
        tv.tv_usec = 0;
        FD_SET (udp->recv_sd_, &rset);
        FD_SET (infifofd, &rset);
        select (FD_SETSIZE, &rset, NULL, NULL, &tv);

        if (FD_ISSET (infifofd, &rset))
        {
            readFifo ();
        }

        if (FD_ISSET (udp->recv_sd_, &rset))
        {
            readSocket ();
        }
        processUserRequest ();
    }
}

// this is called when socket becomes readable
int Client::readSocket ()
{
    unsigned long   from;
    DdtpPacket      pkt;
    int             bufSize = 2 * sizeof(DdtpPacket);
    char            buf[bufSize];

    // receive packets and process them
    switch (udp->recv (buf, bufSize, &from))
    {
        case -1:
            log->error ("udp receive failed: %s", strerror (errno));
            break;

#ifndef DDT_DEBUG
        case -2:
            log->error ("received packet from unknown server");
            break;
#endif
    }

    if (Packet::decode (&pkt, buf, bufSize) == -1)
    {
        log->error ("failed to decode packet");
        return -1;
    }

    if (Packet::decrypt (&pkt, md5_password) == -1)
    {
        log->error ("failed to decrypt packet");
        return -1;
    }

    if (processReceivedPacket (&pkt) == -1)
    {
        // note that this is a little hack, in that
        // if we need to send a reply to the server
        // processReceivedPacket returns 0 otherwise
        // we short circuit the loop at this point
        return -1;
    }

    if (Packet::encrypt (&pkt, md5_password) == -1)
    {
        log->error ("failed to encrypt packet");
        return -1;
    }

    if (Packet::encode (&pkt, buf, bufSize) == -1)
    {
        log->error ("failed to encode packet");
        return -1;
    }
        
    if (udp->send (buf, bufSize) == -1)
    {
        log->error ("udp send failed: %s", strerror (errno));
        return -1;
    }

    return 0;
}

int Client::readFifo ()
{
    // here we check the user query from the fifo
    char input[256];
    int n;

    /*
     * we received something on the fifo so we try opening
     * the output fifo to give feedback to client. if the
     * request fails, client will simply not receive any
     * message.
     */
    
    /* make sure it's closed */
    if (close (outfifofd) == -1)
    {
//        log->debug ("failed to close outfifofd: %s", strerror (errno));
    }
    
    if ((outfifofd = open(OUTFIFO, O_WRONLY | O_NONBLOCK)) == -1)
    {
//        log->debug("failed to open outfifo: %s", strerror (errno));
        outfifofd = -1;
    }

    // valid commands are 'online' and 'offline'
    while ((n = read(infifofd, input, sizeof(input))) > 0)
    {
        if (strstr(input, "online") != NULL)
        {
            char command[256];
            
            sscanf(input, "%s %lu", command, &forceIP);
            
            changeStatus = ONLINE;
            if (changeStatus == currentStatus)
            {
                strncpy(msgbuf, "already online\n", sizeof(msgbuf));
                write(outfifofd, msgbuf, sizeof(msgbuf));
                close(outfifofd);
                continue;
            }
            strncpy(msgbuf, "sending online query\n", sizeof(msgbuf));
            write(outfifofd, msgbuf, sizeof(msgbuf));
            
            log->debug("user requested online status");
            retryIn = 0;
            retryCount = 0;
            retryStamp = time(NULL);
        }
        else if (strncmp(input, "offline", n) == 0)
        {
            changeStatus = OFFLINE;
            if (changeStatus == currentStatus)
            {
                strncpy (msgbuf, "already offline\n", sizeof(msgbuf));
                write (outfifofd, msgbuf, sizeof(msgbuf));
                close (outfifofd);
                continue;
            }
            strncpy(msgbuf, "sending offline query\n", sizeof(msgbuf));
            write(outfifofd, msgbuf, sizeof(msgbuf));

            log->debug("user requested offline status");
            retryIn = 0;
            retryCount = 0;
            retryStamp = time(NULL);
        }
        else if (strncmp(input, "status", n) == 0)
        {
            log->debug("user asked for his status");

            
            snprintf(msgbuf, sizeof(msgbuf), "current status is %s\n", 
                     ((currentStatus == OFFLINE) ? "offline" : "online"));
            write(outfifofd, msgbuf, sizeof(msgbuf));
            
            if (currentStatus != changeStatus)
            {
                snprintf(msgbuf, sizeof(msgbuf), "trying to change status to %s\n", 
                         ((changeStatus == OFFLINE) ? "offline" : "online"));
                write(outfifofd, msgbuf, sizeof(msgbuf));
            }
            close(outfifofd);
        }
        else
        {
            log->debug("invalid user request");

            strncpy(msgbuf, "invalid request\n", sizeof(msgbuf));
            write(outfifofd, msgbuf, sizeof(msgbuf));
            close(outfifofd);
        }
    }
    return 0;
}

void Client::processUserRequest(void)
{
    EncryptionType encryption = (opts->encrypt ? BLOWFISH : PLAINTEXT);
    
    int bufSize = 2 * sizeof(DdtpPacket);
    char buf[bufSize];

    // check if our IP has changed since last time.
    if (Client::lastIP != getIfaceAddr (opts->interface))
    {
        log->debug ("interface IP address changed.");
        if (currentStatus == ONLINE)
        {
            currentStatus = OFFLINE;
            changeStatus = ONLINE;
        }
    }

    if ((Client::lastIP = getIfaceAddr(opts->interface)) == 0)
    {
        currentStatus = OFFLINE;
        changeStatus = OFFLINE;
    }

    // first check if the timeout since we last heard from server has expired
    // if so, mark ourselve offline.
    if (currentStatus == ONLINE && (time(NULL) - serverStamp) > TIMEOUT)
    {
        log->error("ping timeout from server. marking offline.");
        retryIn = 0;
        currentStatus = OFFLINE;
    }

    // if a change of status is not requested, do nothing
    if (changeStatus == currentStatus)
    {
        return;
    }

    // if we reach here, it is because our wanted status is not the same as
    // our current status. we do not want to send a packet everytime we
    // loop here at that is the case, therefore, we check the retryStamp to
    // see if it is time to retry
    if ((time(NULL) - retryStamp) < retryIn)
    {
        //log->debug ("will retry in %d", 
        //            (retryIn - ((time(NULL) - retryStamp))));
        return;
    }

    // if we exceed MAX_RETRIES... the server must be down
    if (++retryCount > MAX_RETRIES)
    {
        log->error ("unable to change status. retryIn set to one hour.");
        retryCount = 0;
        retryIn = (60*60);
        return;
    }

    retryStamp = time(NULL);

    // put some #define's there .. 3 ? 2 ? should be in ddtcd.h
    if (retryIn == 0)
    {
        retryIn = 3;
    }
    else
    {
        retryIn *= 2;
    }
    log->debug ("retryIn set to %d", retryIn);
    

    // if there is still a change to make, do it
    switch (changeStatus)
    {
        case ONLINE:
            if (getIfaceAddr (opts->interface) == 0)
            {
                log->error ("interface is inexistant");
                changeStatus = OFFLINE; // stop retries

                strncpy (msgbuf, "interface is invalid\n", sizeof(msgbuf));
                write (outfifofd, msgbuf, sizeof(msgbuf));
                close (outfifofd);
                break;
            }

            log->notice ("requesting new status: online");
            Packet::createUpdateQuery(pkt, DDTPv1, encryption,
                                      opts->accountId, MARK_ONLINE,
                                      (forceIP == 0 ?
                                       getIfaceAddr(opts->interface)
                                       : forceIP), opts->password);

            log->debug ("proto->%d, id->%d, query->online, addr->%d, "
                        "password->********", DDTPv1, opts->accountId,
                        getIfaceAddr (opts->interface));

            // let's inform the client script
            strncpy (msgbuf, "sending online request\n", sizeof(msgbuf));
            write (outfifofd, msgbuf, sizeof(msgbuf));
            break;

        case OFFLINE:
            log->notice ("requesting new status: offline");
            Packet::createUpdateQuery (pkt, DDTPv1, encryption,
                                       opts->accountId, MARK_OFFLINE,
                                       0, opts->password);

            log->debug ("proto->%d, id->%d, query->offline, addr->0, "
                        "password->********", DDTPv1, opts->accountId);

            // inform the client
            strncpy (msgbuf, "sending offline request\n", sizeof(msgbuf));
            write (outfifofd, msgbuf, sizeof(msgbuf));
            break;

        default:
            log->error ("what status is that?");
            return;
            break;
    }
    
    if (Packet::encrypt (pkt, md5_password) == -1)
    {
        log->error ("failed to encrypt packet");
        return;
    }

    if (Packet::encode (pkt, buf, bufSize) == -1)
    {
        log->error ("failed to encode packet");
        return;
    }

    if (udp->send (buf, bufSize) == -1)
    {
        log->debug ("udp send failed: %s", strerror (errno));
        return;
    }
}

int Client::processReceivedPacket(DdtpPacket* pkt)
{
    switch (pkt->protocolVersion)
    {
        case DDTPv1:
        {
            DDTPv1Message *msg = &pkt->DdtpPacket_u.ddtpv1.DDTPv1Packet_u.pt.msg;
            switch (msg->messageType)
            {
                case UPDATE_REPLY:
                    switch (msg->DDTPv1Message_u.updateReply.status)
                    {
                        case UPDATE_SUCCEEDED:
                            log->debug ("update succesful, stopping retries");

                            // change our status and clear the flags
                            currentStatus = changeStatus;
                            retryCount = 0;
                            
                            if (currentStatus == ONLINE)
                            {
                                // if we are online, update the timestamp
                                serverStamp = time (NULL);
                                log->notice ("new status confirmed: online");
                            }
                            else
                            {
                                log->notice ("new status confirmed: offline");
                            }

                            strncpy (msgbuf, "new status confirmed\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            close (outfifofd);
                            break;

                        case UPDATE_FAILED:
                            log->error ("request failed, retrying");
                            retryCount++;

                            strncpy (msgbuf, "request failed, retrying\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            break;

                        case INVALID_PASSWORD:
                            log->error ("bad password!");
                            currentStatus = OFFLINE;
                            changeStatus  = OFFLINE;
                            
                            strncpy (msgbuf, "bad password!\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            close (outfifofd);
                            break;
                            
                        case INVALID_ACCOUNT:
                            log->error ("your account is invalid!");
                            currentStatus = OFFLINE;
                            changeStatus  = OFFLINE;
                            
                            strncpy (msgbuf, "account invalid!\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            close (outfifofd);
                            break;
                            
                        case INVALID_OPCODE:
                            log->error ("what opcode is that?");
                            currentStatus = OFFLINE;
                            changeStatus  = OFFLINE;
                            
                            strncpy (msgbuf, "huh?\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            close (outfifofd);
                            break;
                            
                        default:
                            log->error ("huh?");
                            currentStatus = OFFLINE;
                            changeStatus  = OFFLINE;

                            strncpy (msgbuf, "huh?\n", sizeof(msgbuf));
                            write (outfifofd, msgbuf, sizeof(msgbuf));
                            close (outfifofd);
                            break;
                    }
                    break;

                case ALIVE_QUERY:
                    log->debug ("are you alive? yes mama");
                    serverStamp = time (NULL);
                    
                    // we are online and we want to be.
                    currentStatus = ONLINE;
                    changeStatus  = ONLINE;
                    Packet::createAliveReply (pkt, DDTPv1, opts->accountId);
                    return 0;   // return 0 to indicate that we need to reply
                    break;

                case MESSAGE_ERROR:
                    log->debug ("you screwed something up");
                    currentStatus = OFFLINE;
                    changeStatus  = OFFLINE;
                    break;
                default:
                    break;
            }
            break;
        }
        case DDTPv2:
            // protocol not yet implemented
            break;
        default:
            break;
    }
    return -1;  // return -1 by default (see hack comment above)
}

void Client::atexit_handler ()
{
    FILE *fp;
    int pid;

    if (log) log->notice("daemon stopping");

    // remove pid file
    if ((fp = fopen (PID_FILE, "r")) == NULL)
    {
        if (log) log->error ("failed to open pid file: %s", strerror (errno));
    }
    else
    {
        if ((fscanf (fp, "%d", &pid) == 1) && (pid == getpid()))
        {
            if (unlink (PID_FILE) == -1)
            {
                if (log) log->debug ("failed to unlink pid file: %s", strerror (errno));
            }
        }
    }
    
    if (unlink (INFIFO) == -1)
    {
        if (log) log->debug ("failed to unlink input fifo: %s", strerror (errno));
    }
    
    if (unlink (OUTFIFO) == -1)
    {
        if (log) log->debug ("failed to unlink output fifo: %s", strerror (errno));
    }
}


void Client::signal_handler (int signum)
{
    switch (signum)
    {
        case SIGPIPE:
            log->debug ("damn, that was a SIGPIPE");
            exit (0);
            break;
        case SIGHUP:
        case SIGINT:
        case SIGQUIT:
        case SIGABRT:
        case SIGTERM:
            log->debug ("received signal (%d)", signum);
            exit (0);
            break;
    }
}

unsigned long Client::getIfaceAddr (char *iface)
{
    struct ifconf ifc;
    struct ifreq *ifr;
    struct ifreq *ifrFound;
    int sd;

    if ((sd = socket (AF_INET, SOCK_DGRAM, 0)) == -1)
    {
        log->error ("socket() failed");
        return 0;
    }
    
    ifc.ifc_len = 1024;
    ifc.ifc_buf = (char*)malloc(sizeof(char) * 1024);

    if (ioctl (sd, SIOCGIFCONF, &ifc) == -1)
    {
//        log->debug ("SIOCGIFADDR: %s", strerror (errno));
        close (sd);
        return 0;
    }

    close (sd);


    int nbReq = ifc.ifc_len / sizeof(struct ifreq);
    
    ifrFound = NULL;
    int i;
    for(i = 0; i < nbReq; i++)
    {
        ifr = (struct ifreq*)&ifc.ifc_req[i];

        //log->debug ("iface name: %s", ifr->ifr_name);
        if(strcmp(iface, ifr->ifr_name) == 0)
        {
            //log->debug("ip: %s", inet_ntoa(((struct sockaddr_in *)&ifr->ifr_addr)->sin_addr));
            ifrFound = ifr;
        }
    }

    unsigned long addr = 0;
    if(ifrFound != NULL)
    {
        addr = htonl(((struct sockaddr_in *)&ifrFound->ifr_addr)->sin_addr.s_addr);
    }
    free(ifc.ifc_buf);
    return addr;
}
