# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn status command process.
"""

import types

from qt import *

from StatusForm import StatusForm

class SvnStatusDialog(StatusForm):
    """
    Module implementing a dialog to show the output of the svn status command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs -- reference to the vcs object
        @param parent -- parent widget (QWidget)
        """
        StatusForm.__init__(self, parent)
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.statusList.setSorting(9)
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
        self.status = {
            ' ' : self.trUtf8('normal'),
            'A' : self.trUtf8('added'),
            'D' : self.trUtf8('deleted'),
            'M' : self.trUtf8('modified'),
            'C' : self.trUtf8('conflict'),
            'X' : self.trUtf8('external'),
            'I' : self.trUtf8('ignored'),
            '?' : self.trUtf8('unversioned'),
            '!' : self.trUtf8('missing'),
            '~' : self.trUtf8('type error')
        }
        self.propStatus = {
            ' ' : self.trUtf8('normal'),
            'M' : self.trUtf8('modified'),
            'C' : self.trUtf8('conflict')
        }
        self.locked = {
            ' ' : self.trUtf8('no'),
            'L' : self.trUtf8('yes')
        }
        self.history = {
            ' ' : self.trUtf8('no'),
            '+' : self.trUtf8('yes')
        }
        self.switched = {
            ' ' : self.trUtf8('no'),
            'S' : self.trUtf8('yes')
        }
        self.uptodate = {
            ' ' : self.trUtf8('yes'),
            '*' : self.trUtf8('no')
        }
        
        self.rx_status = QRegExp('(.{8})\\s+([0-9]+)\\s+([0-9?]+)\\s+([\\w?]+)\\s+(.+)')
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e -- close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the svn status command.
        
        @param fn -- filename(s) to show the status of (string or list of strings)
        """
        self.intercept = 0
        
        self.process.kill()
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('status')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.vcs.addArguments(self.process, self.vcs.options['status'])
        self.process.addArgument('--verbose')
        if '--show-updates' in self.vcs.options['status'] or \
           '-u' in self.vcs.options['status']:
            try:
                self.setActiveWindow()
                self.raiseW()
            except:
                pass
        if type(fn) is types.ListType:
            dname, fnames = self.vcs.splitPathList(fn)
            self.vcs.addArguments(self.process, fnames)
        else:
            dname, fname = self.vcs.splitPath(fn)
            self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('Subversion Status'))
        
    def finish(self):
        """
        Private slot called when the process finished or the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.cancelButton.setText(self.trUtf8('OK'))
        self.cancelButton.setDefault(1)
        self.cancelButton.setFocus()
        
        self.inputGroup.setEnabled(0)
        
        self.process = None
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.close()
        else:
            self.finish()
            
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Just call finish().
        """
        self.finish()
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            if self.rx_status.exactMatch(s):
                flags = str(self.rx_status.cap(1))
                rev = self.rx_status.cap(2)
                change = self.rx_status.cap(3)
                author = self.rx_status.cap(4)
                path = self.rx_status.cap(5)
                itm = QListViewItem(self.statusList,
                    self.status[flags[0]], self.propStatus[flags[1]],
                    self.locked[flags[2]], self.history[flags[3]],
                    self.switched[flags[4]], self.uptodate[flags[5]],
                    "%7s" % str(rev), "%7s" % str(change))
                itm.setText(8, author)
                itm.setText(9, path)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        if self.process is not None:
            s = unicode(self.process.readStderr())
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(s)
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to the subversion process.
        """
        input = self.input.text()
        input.append(os.linesep)
        
        self.errors.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.errors.insert(os.linesep)
        else:
            self.errors.insert(input)
        
        self.proc.writeToStdin(input)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        StatusForm.keyPressEvent(self, evt)
