/*
 *  PGPopUpTextFieldCell.m
 *  graphviz
 *
 *  Created by Glen Low on Fri Apr 23 2004.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#import "PGPopUpTextFieldCell.h"

@interface PGPopUpTextFormatter: NSFormatter
	{
		PGPopUpTextFieldCell* cell_;
	}
	
- (void) setCell: (PGPopUpTextFieldCell*) cell;

@end

static int FindIndexOfMenuItemWithTitle (NSMenu* menu, NSString* stringValue, int start)
	{
		int select = 0;
		int finish = [menu numberOfItems];
		
		while (start < finish)
			{
				select = (start + finish - 1) / 2;
				
				switch ([[[menu itemAtIndex: select] title] compare: stringValue])
					{
						case NSOrderedAscending:
							start = select + 1;
							break;
						case NSOrderedDescending:
							finish = select;
							break;
						case NSOrderedSame:
							return select;
							break;
					}
			}
			
		return ~start;
	}

@implementation PGPopUpTextFieldCell

- (id) initTextCell: (NSString*) string
	{
		if ((self = [super initTextCell: string]))
			{
				popup_ = [[NSPopUpButtonCell alloc] initTextCell: @"" pullsDown: NO];
				[popup_ setBordered: NO];
				
				PGPopUpTextFormatter* formatter = [[[PGPopUpTextFormatter alloc] init] autorelease];
				[formatter setCell: self];
				[self setFormatter: formatter];
			}
			
		return self;
	}
	
- (id) copyWithZone: (NSZone*) zone
	{
		PGPopUpTextFieldCell* copy = [super copyWithZone: zone];
		copy->popup_ = [popup_ copyWithZone: zone];
		
		NSFormatter* formatter = [copy formatter];
		if ([formatter respondsToSelector: @selector (setCell:)])
			[formatter performSelector: @selector (setCell:) withObject: copy];
		
		return copy;
	}
	
- (void) setFont: (NSFont*) font
	{
		[popup_ setFont: font];
		[super setFont: font];
	}
	
- (void) setControlSize: (NSControlSize) size
	{
		[popup_ setControlSize: size];
		[super setControlSize: size];
	}
	
- (NSMenu*) menu
	{
		return [popup_ menu];
	}
	
- (void) setMenu: (NSMenu*) menu
	{
		[popup_ setMenu: menu];
	}
	
- (void) drawWithFrame: (NSRect) cellFrame inView: (NSView*) controlView
	{
		NSRect popupTitleRect = [popup_ titleRectForBounds: cellFrame];
		float split = NSMaxX (popupTitleRect);
		
		NSRect textFrame = NSMakeRect (
			cellFrame.origin.x,
			cellFrame.origin.y,
			split - cellFrame.origin.x,
			cellFrame.size.height);
		[super drawWithFrame: textFrame inView: controlView];
		
		NSRect popupFrame = NSMakeRect (
			split,
			cellFrame.origin.y,
			NSMaxX (cellFrame) - split,
			cellFrame.size.height);
		[popup_ drawWithFrame: popupFrame inView: controlView];
	}
	
- (BOOL) trackMouse: (NSEvent*) event inRect: (NSRect) cellFrame ofView: (NSView*) controlView untilMouseUp: (BOOL) untilMouseUp
	{
		NSRect popupTitleRect = [popup_ titleRectForBounds: cellFrame];
		if ([controlView convertPoint: [event locationInWindow] fromView: nil].x < NSMaxX (popupTitleRect))
			return [super trackMouse: event inRect: cellFrame ofView: controlView untilMouseUp: untilMouseUp];
		else
			{
				NSMenu* menu = [popup_ menu];
				NSString* stringValue = [self stringValue];
				
				int select = 0;
				BOOL unfound = YES;
				
				[menu insertItem: [NSMenuItem separatorItem] atIndex: 0];
				[menu insertItemWithTitle: @"unchanged" action: nil keyEquivalent: @"" atIndex: 0];
				
				if ([stringValue isEqualToString: @""])
					{
						// use the default value
						select = 0;
						unfound = NO;
					}
				else
					{
						// binary search in the menu for the description
						select = FindIndexOfMenuItemWithTitle (menu, stringValue, 2);
						if (select < 0)
							{
								select = ~select;
								unfound = YES;
							}
						else
							unfound = NO;
					}
					
				// if not found, insert a custom item
				if (unfound)
					[menu insertItemWithTitle: [NSString stringWithFormat: @"%@%C", stringValue, 0x2026]
						action: nil keyEquivalent: @"" atIndex: select];
		
				// track the popup menu
				[popup_ selectItemAtIndex: select];
				cellFrame.origin.x -= 9.0;
				cellFrame.origin.y += 1.0;
				cellFrame.size.width += 9.0;
				BOOL result = [popup_ trackMouse: event inRect: cellFrame ofView: controlView untilMouseUp: untilMouseUp];
				
				// set own object to the selected item
				int selected = [popup_ indexOfSelectedItem];
				if (selected != select)
					{
						if (selected == 0)
							[self setObjectValue: nil];
						else
							{
								id <NSMenuItem> selectedItem = [menu itemAtIndex: selected];
								id selectedObject = [selectedItem representedObject];
								[self setObjectValue: selectedObject ? selectedObject : [selectedItem title]];
							}
					}
				
				// remove the custom item if any
				if (unfound)
					[menu removeItemAtIndex: select];
					
				// remove the default item and separator
				[menu removeItemAtIndex: 0];
				[menu removeItemAtIndex: 0];
						
				return result;
			}
	}

- (void) editWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object event: (NSEvent*) event
	{
		NSRect popupTitleRect = [popup_ titleRectForBounds: rect];
		rect.size.width = NSMaxX (popupTitleRect) - rect.origin.x;
	
		if ([controlView convertPoint: [event locationInWindow] fromView: nil].x < NSMaxX (popupTitleRect))
			[super editWithFrame: rect inView: controlView editor: text delegate: object event: event];
			
	}

- (void) selectWithFrame: (NSRect) rect inView: (NSView*) controlView editor: (NSText*) text delegate: (id) object start: (int) selStart length: (int) selLength
	{
		NSRect popupTitleRect = [popup_ titleRectForBounds: rect];
		rect.size.width = NSMaxX (popupTitleRect) - rect.origin.x;

		[super selectWithFrame: rect inView: controlView editor: text delegate: object start: selStart length: selLength];
	}
	
- (NSText*) setUpFieldEditorAttributes: (NSText*) text
	{
		return text;
	}
	
- (void) dealloc
	{
		[popup_ release];
		popup_ = nil;
		
		[super dealloc];
	}
@end

@implementation PGPopUpTextFormatter

- (id) init
	{
		if ((self = [super init]))
			cell_ = nil;
		return self;
	}

- (void) setCell: (PGPopUpTextFieldCell*) cell
	{
		cell_ = cell;
	}
	
- (NSString*) stringForObjectValue: (id) object
	{
		return object ? [object description] : @"";
	}

- (BOOL) getObjectValue: (id*) object forString: (NSString*) string errorDescription: (NSString**) error
	{
		NSMenu* menu = [cell_ menu];
		int select = FindIndexOfMenuItemWithTitle (menu, string, 0);
		if (select < 0)
			*object = [string copy];
		else
			{
				id <NSMenuItem> selectedItem = [menu itemAtIndex: select];
				id selectedObject = [selectedItem representedObject];
				*object = selectedObject ? selectedObject : [selectedItem title];
			}
		
		return YES;
	}
	
- (BOOL) isPartialStringValid: (NSString**) partialStringPtr proposedSelectedRange: (NSRangePointer) proposedSelRangePtr originalString: (NSString*) origString originalSelectedRange: (NSRange) origSelRange errorDescription: (NSString**) error
	{
		int partialStringLength = [*partialStringPtr length];
		if (origSelRange.location + origSelRange.length == [origString length]
			&& proposedSelRangePtr->length == 0
			&& proposedSelRangePtr->location == partialStringLength
			&& partialStringLength > origSelRange.location)
			{
				NSMenu* menu = [cell_ menu];
				int select = FindIndexOfMenuItemWithTitle (menu, *partialStringPtr, 0);
				if (select < 0 && ~select < [menu numberOfItems])
					{
						NSString* selectedItemTitle = [[menu itemAtIndex: ~select] title];
						if ([selectedItemTitle hasPrefix: *partialStringPtr])
							{
								*partialStringPtr = selectedItemTitle;
								*proposedSelRangePtr = NSMakeRange (partialStringLength, [selectedItemTitle length] - partialStringLength);
								return NO;
							}
					}
			}
		
		return YES;
	}

@end

