/*************************************************************************/
/* module:         SyncmML Decoder                                       */
/* file:           XLTDec.c                                              */
/* target system:  all                                                   */
/* target OS:      all                                                   */
/*************************************************************************/

/*
 * Copyright Notice
 * Copyright (c) Ericsson, IBM, Lotus, Matsushita Communication 
 * Industrial Co., LTD,Motorola, Nokia, Palm, Inc., Psion, 
 * Starfish Software (2001).
 * All Rights Reserved.
 * Implementation of all or part of any Specification may require 
 * licenses under third party intellectual property rights, 
 * including without limitation, patent rights (such a third party 
 * may or may not be a Supporter). The Sponsors of the Specification 
 * are not responsible and shall not be held responsible in any 
 * manner for identifying or failing to identify any or all such 
 * third party intellectual property rights.
 * 
 * THIS DOCUMENT AND THE INFORMATION CONTAINED HEREIN ARE PROVIDED 
 * ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND AND ERICSSON, IBM, 
 * LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO. LTD, MOTOROLA, 
 * NOKIA, PALM INC., PSION, STARFISH SOFTWARE AND ALL OTHER SYNCML 
 * SPONSORS DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION 
 * HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF 
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT 
 * SHALL ERICSSON, IBM, LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO., 
 * LTD, MOTOROLA, NOKIA, PALM INC., PSION, STARFISH SOFTWARE OR ANY 
 * OTHER SYNCML SPONSOR BE LIABLE TO ANY PARTY FOR ANY LOSS OF 
 * PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF 
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTAL, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH 
 * THIS DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 * 
 * The above notice and this paragraph must be included on all copies 
 * of this document that are made.
 * 
 */

/**
 * The SyncML parser.
 */

/*************************************************************************/
/* Definitions                                                           */
/*************************************************************************/
#include "xltdec.h"
#include "xltdeccom.h"
#include "xlttags.h"
#include "xltutilstack.h"

#include <smldef.h>
#include <smldtd.h>
#include <smlerr.h>

#include <libmem.h>
#include <libstr.h>


#define IS_START(tok) ((tok)->type == TOK_TAG_START)
#define IS_END(tok) ((tok)->type == TOK_TAG_END)
#define IS_EMPTY(tok) ((tok)->type == TOK_TAG_EMPTY)
#define IS_TAG(tok) (IS_START(tok) || IS_EMPTY(tok) || IS_END(tok))
#define IS_START_OR_EMPTY(tok) (IS_START(tok) || IS_EMPTY(tok))
#define IS_CONTENT(tok) ((tok)->type == TOK_CONT)

/**
 * In addition to the "public" methods and attributes of the
 * XltDecoderPtr_t type, the decoderPriv type contains the "private"
 * methods and attributes of the decoder object.
 * The public methods and attributes are documented in the XLTDec header
 * file.
 *
 * Private methods and attributes of the decoder:
 *
 *
 * ATTRIBUTE: scanner
 *
 * Pointer to the scanner status object used by this decoder. The scanner
 * will be created during the initialization of the decoder as either a XML
 * or WBXML scanner.
 *
 * 
 * ATTRIBUTE: tagstack
 *
 * The decoder uses an internal stack to check that for every start tag
 * there is a corresponding end tag.
 */
typedef struct decoderPriv_s
{
    /* public */
    Long_t charset;                /* character set used in the
                                      document - this is the MIBEnum
                                      value assigned by the IANA */
    String_t charsetStr;           /* name of the character set - valid
                                      only when charset == 0 */
    Flag_t finished;

    Boolean_t final;
    /* private */
    XltDecScannerPtr_t scanner;
    XltUtilStackPtr_t tagstack;
} decoderPriv_t, *decoderPrivPtr_t;

/**
 * nextToken and discardToken are just wrappers around the scanner's
 * nextTok and pushTok methods that do some error checking.
 */
static Ret_t nextToken(decoderPrivPtr_t pDecoder);
static Ret_t discardToken(decoderPrivPtr_t pDecoder);

/**
 * FUNCTION: concatPCData
 *
 * Tries to concatenate two Pcdata elements. Only works when the two
 * elements are of the same type (e.g. SML_PCDATA_STRING). Returns a
 * pointer to the new Pcdata element or NULL if concatenation failed.
 */
static SmlPcdataPtr_t concatPCData(SmlPcdataPtr_t pDat1, const SmlPcdataPtr_t pDat2);

/**
 * FUNCTION: buildXXX
 *
 * These functions each decode one single SyncML element starting at the
 * current position within the SyncML document. Child elements are build
 * recursively.
 * The functions check that the pointer to the memory structures are
 * NULL when called and return an error otherwise. This will only happen
 * when a SyncML element contains several child elements of the same type
 * for which this is not allowed according to the SyncML DTD; e.g. a
 * SyncHdr with two or more MsgID tags. Items and other list types
 * are handled separately by the appendXXXList functions (see below).
 *
 * PRE-Condition:
 *                 ppElem is NULL
 *                 The scanner's current token is the start tag (may be
 *                 empty) of the SyncML element to be decoded.
 *
 * POST-Condition:
 *                 The scanner's current token is the end tag (or empty
 *                 start tag) of the SyncML element to be decoded.
 */
static Ret_t buildAlert(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
#ifndef __SML_LITE__
static Ret_t buildAtomOrSeq(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
#endif
static Ret_t buildChal(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildCred(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildDelete(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildGenericCmd(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildItem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildMap(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildMapItem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildPCData(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildPutOrGet(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildResults(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
#ifndef __SML_LITE__
static Ret_t buildExec(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildSearch(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
#endif
static Ret_t buildStatus(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildSync(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildSyncHdr(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t buildTargetOrSource(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
/* build Meta structure for Meta Information extension */
#ifdef __META_EXT__
static Ret_t extBuildMeta(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t extBuildMetaAnchor(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t extBuildMetaMem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
static Ret_t appendEmiList(decoderPrivPtr_t pDecoder, SmlEmiListPtr_t *ppEmiList);
#endif
/**
 * FUNCTION: appendXXXList
 *
 * These are auxiliary functions for building SyncML elements that contain
 * lists of certain other data structures (e.g. Items). They take an
 * existing list (e.g. of type ItemListPtr_t) and append an appropriate
 * element at the end. If the ListPtr points to NULL a new list is created.
 *
 * PRE-Condition:
 *                 The scanner's current token is the start tag (may be
 *                 empty) of the SyncML element to be appended to the list.
 *
 * POST-Condition:
 *                 The scanner's current token is the end tag (or empty
 *                 start tag) of the SyncML element that was added to the
 *                 list.
 *
 * IN/OUT:         pDecoder, the decoder
 *                 ppXXXList, NULL or an initialized list, to which element
 *                 will be appended
 *
 * RETURNS:        SML_ERR_OK, if an element was successfully appended
 *                 else error code
 */
static Ret_t appendItemList(decoderPrivPtr_t pDecoder, SmlItemListPtr_t *ppItemList);
static Ret_t appendSourceList(decoderPrivPtr_t pDecoder, SmlSourceListPtr_t *ppSourceList);
static Ret_t appendMapItemList(decoderPrivPtr_t pDecoder, SmlMapItemListPtr_t *ppMapItemList);
static Ret_t appendTargetRefList(decoderPrivPtr_t pDecoder, SmlTargetRefListPtr_t *ppTargetRefList);
static Ret_t appendSourceRefList(decoderPrivPtr_t pDecoder, SmlSourceRefListPtr_t *ppSourceRefList);

typedef struct PEBuilder_s
{
    XltTagID_t     tagid;
    SmlProtoElement_t type;
    Ret_t (*build)(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem);
} PEBuilder_t, *PEBuilderPtr_t;

PEBuilderPtr_t getPETable(void);

PEBuilderPtr_t getPETable(void)
{ 
  PEBuilderPtr_t _tmpPEPtr;
  static PEBuilder_t PE[] =
  {
    { TN_ADD,       SML_PE_ADD,             buildGenericCmd },
    { TN_ALERT,     SML_PE_ALERT,           buildAlert      },
#ifndef __SML_LITE__    
    { TN_ATOMIC,    SML_PE_ATOMIC_START,    buildAtomOrSeq  },
#endif    
    { TN_COPY,      SML_PE_COPY,            buildGenericCmd },
    { TN_DELETE,    SML_PE_DELETE,          buildGenericCmd },
#ifndef __SML_LITE__    
    { TN_EXEC,      SML_PE_EXEC,            buildExec       },
#endif
    { TN_GET,       SML_PE_GET,             buildPutOrGet   },
    { TN_MAP,       SML_PE_MAP,             buildMap        },
    { TN_PUT,       SML_PE_PUT,             buildPutOrGet   },
    { TN_RESULTS,   SML_PE_RESULTS,         buildResults    },
#ifndef __SML_LITE__        
    { TN_SEARCH,    SML_PE_SEARCH,          buildSearch     },    
    { TN_SEQUENCE,  SML_PE_SEQUENCE_START,  buildAtomOrSeq  },
#endif    
    { TN_STATUS,    SML_PE_STATUS,          buildStatus     },
    { TN_SYNC,      SML_PE_SYNC_START,      buildSync       },
    { TN_REPLACE,   SML_PE_REPLACE,         buildGenericCmd },
    { TN_UNDEF,     SML_PE_UNDEF,           0               }
  };
    
  _tmpPEPtr = smlLibMalloc(sizeof(PE));
  if (_tmpPEPtr == NULL) return NULL;
  smlLibMemcpy(_tmpPEPtr, &PE, sizeof(PE));
  return _tmpPEPtr; 
}

/*************************************************************************/
/* External Functions                                                    */
/*************************************************************************/

/**
 * Description see XLTDec.h header file.
 */
Ret_t
xltDecInit(const SmlEncoding_t enc,
        const MemPtr_t pBufEnd,
        MemPtr_t *ppBufPos,
        XltDecoderPtr_t *ppDecoder, 
        SmlSyncHdrPtr_t *ppSyncHdr)
{
    decoderPrivPtr_t pDecoder;
    Ret_t rc;


    /* create new decoder object */
    if ((pDecoder = (decoderPrivPtr_t)smlLibMalloc(sizeof(decoderPriv_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    pDecoder->finished = 0;
    pDecoder->final = 0;
    pDecoder->scanner = NULL;
    if ((rc = xltUtilCreateStack(&pDecoder->tagstack, 10)) != SML_ERR_OK) {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return rc;
    }

#ifdef __SML_WBXML__
    if (enc == SML_WBXML) 
    {
        rc = xltDecWbxmlInit(pBufEnd, ppBufPos, &pDecoder->scanner);
        if (rc == SML_ERR_OK) 
        {
            pDecoder->charset = pDecoder->scanner->charset;
            pDecoder->charsetStr = NULL;
        }
    } 
#endif

#ifdef __SML_XML__
    if (enc == SML_XML) 
    {

        rc = xltDecXmlInit(pBufEnd, ppBufPos, &pDecoder->scanner);
        if (rc == SML_ERR_OK) 
        {
            pDecoder->charset = 0;
            pDecoder->charsetStr = pDecoder->scanner->charsetStr;
        }
    } 
#endif

    else if ((enc != SML_XML) && (enc != SML_WBXML))
    {
        rc = SML_ERR_XLT_ENC_UNK;
    }
    if (rc != SML_ERR_OK) 
    {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return rc;
    }

    /* try to find SyncHdr element, first comes the SyncML tag... */
    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return rc;
    }
    if (!IS_START(pDecoder->scanner->curtok) ||
            (pDecoder->scanner->curtok->tagid != TN_SYNCML_END)) {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    /* ... then the SyncHdr */
    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return rc;
    }
    if ((rc = buildSyncHdr(pDecoder, (VoidPtr_t)ppSyncHdr)) != SML_ERR_OK) {
        xltDecTerminate((XltDecoderPtr_t)pDecoder);
        return rc;
    }

    *ppBufPos = pDecoder->scanner->getPos(pDecoder->scanner);

    *ppDecoder = (XltDecoderPtr_t)pDecoder;

    return SML_ERR_OK;
}

/**
 * Description see XLTDec.h header file.
 */
Ret_t
xltDecNext(XltDecoderPtr_t pDecoder,
        const MemPtr_t pBufEnd,
        MemPtr_t *ppBufPos,
        SmlProtoElement_t *pe,
        VoidPtr_t *ppContent)
{
    decoderPrivPtr_t pDecPriv = (decoderPrivPtr_t)pDecoder;
    XltDecScannerPtr_t pScanner = pDecPriv->scanner;
    XltTagID_t tagid;
    Ret_t rc;
    int i;

    pScanner->setBuf(pScanner, *ppBufPos, pBufEnd);

    /* if we are still outside the SyncBody, look for SyncBody start tag */
    if ((rc = pDecPriv->tagstack->top(pDecPriv->tagstack, &tagid)) != SML_ERR_OK)
        return rc;
    if (tagid == TN_SYNCML_END) {
        if (((rc = nextToken(pDecPriv)) != SML_ERR_OK)) {
            return rc;
        }
        if (!((IS_START(pScanner->curtok)) &&
             (pScanner->curtok->tagid == TN_SYNCBODY))) {
            return SML_ERR_XLT_INVAL_PROTO_ELEM;
        }
    }

    if ((rc = nextToken(pDecPriv)) != SML_ERR_OK)
        return rc;

    /* if we find a SyncML protocol element build the corresponding
       data structure */
    if ((IS_START_OR_EMPTY(pScanner->curtok)) && (pScanner->curtok->tagid != TN_FINAL)) {

        PEBuilderPtr_t pPEs = getPETable();
        if (pPEs == NULL)
        {
          smlLibFree(pPEs);
          return SML_ERR_NOT_ENOUGH_SPACE;
        }
        i = 0;
        while (((pPEs+i)->tagid) != TN_UNDEF)
        { 
            if (((pPEs+i)->tagid) == pScanner->curtok->tagid) 
            {
                *pe = ((pPEs+i)->type);
                if ((rc = (pPEs+i)->build(pDecPriv, ppContent)) != SML_ERR_OK)
                {
                    smlLibFree(pPEs);
                    return rc;
                }    
                break;
            }
            i++;
        }
        if (((pPEs+i)->tagid) == TN_UNDEF) 
        {
                *pe = SML_PE_UNDEF;
                *ppContent = NULL;
                smlLibFree(pPEs);
                return SML_ERR_XLT_INVAL_PROTO_ELEM;
        }
        smlLibFree(pPEs);
    } else {

        /* found end tag */
        switch (pScanner->curtok->tagid) {
            case TN_ATOMIC:
                *pe = SML_PE_ATOMIC_END;
                *ppContent = NULL;
                break;
            case TN_SEQUENCE:
                *pe = SML_PE_SEQUENCE_END;
                *ppContent = NULL;
                break;
            case TN_SYNC:
                *pe = SML_PE_SYNC_END;
                *ppContent = NULL;
                break;
            case TN_FINAL:
                *pe = SML_PE_FINAL;
                *ppContent = NULL;
                pDecPriv->final = 1;
                break;
            case TN_SYNCBODY:
                /* next comes the SyncML end tag, then we're done */
                if ((rc = nextToken(pDecPriv)) != SML_ERR_OK)
                    return rc;
                if ((pScanner->curtok->type == TOK_TAG_END) &&
                        (pScanner->curtok->tagid == TN_SYNCML_END)) {
                    *pe = SML_PE_UNDEF;
                    *ppContent = NULL;
                    pDecPriv->finished = 1;
                } else {
                    return SML_ERR_XLT_INVAL_SYNCML_DOC;
                }
                break;
            default: 
                return SML_ERR_XLT_INVAL_PROTO_ELEM;
        }
    }

    *ppBufPos = pScanner->getPos(pScanner);

    return SML_ERR_OK;
}

/**
 * Description see XLTDec.h header file.
 */
Ret_t
xltDecTerminate(XltDecoderPtr_t pDecoder)
{
    decoderPrivPtr_t pDecPriv;

    if (pDecoder == NULL)
        return SML_ERR_OK;

    pDecPriv = (decoderPrivPtr_t)pDecoder;
    if (pDecPriv->scanner != NULL)
        pDecPriv->scanner->destroy(pDecPriv->scanner);
    if (pDecPriv->tagstack != NULL)
        pDecPriv->tagstack->destroy(pDecPriv->tagstack);
    smlLibFree(pDecPriv);

    return SML_ERR_OK;
}


Ret_t xltDecReset(XltDecoderPtr_t pDecoder)
{
  return xltDecTerminate(pDecoder);
}

/*************************************************************************/
/* Internal Functions                                                    */
/*************************************************************************/

/**
 * Gets the next token from the scanner.
 * Checks if the current tag is an end tag and if so, whether the last
 * open start tag has the same tag id as the current end tag. An open start
 * tag is one which matching end tag has not been seen yet.
 * If the current tag is a start tag its tag ID will be pushed onto the
 * tag stack.
 * If the current tag is an empty tag or not a tag at all nothing will be
 * done.
 */
static Ret_t
nextToken(decoderPrivPtr_t pDecoder)
{
    XltUtilStackPtr_t pTagStack;
    XltDecTokenPtr_t pToken;
    Ret_t rc;

    if ((rc = pDecoder->scanner->nextTok(pDecoder->scanner)) != SML_ERR_OK)
        return rc;

    pToken = pDecoder->scanner->curtok;
    pTagStack = pDecoder->tagstack;

    if (IS_START(pToken)) {
        if (pTagStack->push(pTagStack, pToken->tagid))
            return SML_ERR_UNSPECIFIC;
    } else if (IS_END(pToken)) {
        XltTagID_t lastopen;
        if (pTagStack->pop(pTagStack, &lastopen))
            return SML_ERR_UNSPECIFIC;
        if (pToken->tagid != lastopen)
            return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }
    return SML_ERR_OK;
}

static Ret_t discardToken(decoderPrivPtr_t pDecoder)
{
    Ret_t rc;
    XltTagID_t tmp;
    if ((rc = pDecoder->scanner->pushTok(pDecoder->scanner)) != SML_ERR_OK)
        return rc;
    if ((rc = pDecoder->tagstack->pop(pDecoder->tagstack, &tmp)) != SML_ERR_OK)
        return rc;
    return SML_ERR_OK;
}

static SmlPcdataPtr_t
concatPCData(SmlPcdataPtr_t pDat1, const SmlPcdataPtr_t pDat2)
{
    if (pDat1->contentType != pDat2->contentType)
        return NULL;

    switch (pDat1->contentType) {
        case SML_PCDATA_STRING:
            pDat1->content = (VoidPtr_t)smlLibStrcat(pDat1->content, pDat2->content);
            pDat1->length += pDat2->length;
            break;
        case SML_PCDATA_OPAQUE:
            if ((pDat1->content = smlLibRealloc(pDat1->content, pDat1->length + pDat2->length)) == NULL) 
                return NULL;
            smlLibMemmove(((Byte_t*)pDat1->content) + pDat1->length, pDat2->content, pDat2->length);
            pDat1->length += pDat2->length;
            break;
        default:
            return NULL;
    }
    return pDat1;
}

static Ret_t
buildSyncHdr(decoderPrivPtr_t pDecoder, VoidPtr_t *ppSyncHdr)
{
    XltDecScannerPtr_t pScanner;
    SmlSyncHdrPtr_t pSyncHdr;
    Ret_t rc;
    Long_t sessionid = 0, msgid = 0, source = 0, target = 0, version = 0, proto = 0;

    /* shortcut to the scanner object */
    pScanner = pDecoder->scanner;

    /* if ppSyncHdr is not NULL we've already 
       found a SyncHdr before! */
    if (*ppSyncHdr != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    /* initialize new SmlSyncHdr */
    if ((pSyncHdr = (SmlSyncHdrPtr_t)smlLibMalloc(sizeof(SmlSyncHdr_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pSyncHdr, 0, sizeof(SmlSyncHdr_t));

    /* initialize the element type field */
    pSyncHdr->elementType = SML_PE_HEADER;

    /* empty SmlSyncHdr is possible */
    if (IS_EMPTY(pScanner->curtok)) {
        *ppSyncHdr = pSyncHdr;
        return SML_ERR_OK;
    }

    /* get next Token */
    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pSyncHdr);
        return rc;
    }

    /* parse child elements until we find a matching end tag */
    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_VERSION:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->version);
                version++;
                break;
            case TN_PROTO:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->proto);
                proto++;
                break;
            case TN_SESSIONID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->sessionID);
                sessionid++;
                break;
            case TN_MSGID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->msgID);
                msgid++;
                break;
            case TN_RESPURI:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->respURI);
                break;

                /* child tags */
            case TN_TARGET:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pSyncHdr->target);
                target++;
                break;
            case TN_SOURCE:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pSyncHdr->source);
                source++;
                break;
            case TN_CRED:
                rc = buildCred(pDecoder, (VoidPtr_t)&pSyncHdr->cred);
                break;
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSyncHdr->meta);
                break;

                /* flags (empty tags) */
            case TN_NORESP:
                pSyncHdr->flags |= SmlNoResp_f;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }

        /* decoding of child element went ok? */
        if (rc != SML_ERR_OK) {
            smlLibFree(pSyncHdr);
            return rc;
        }

        /* get next token */
        if ((rc = nextToken(pDecoder)) != SML_ERR_OK) {
            smlLibFree(pSyncHdr);
            return rc;
        }
    }

    if ((sessionid == 0) || (msgid == 0) || (target == 0) || (source == 0) || (version == 0) || (proto == 0))
    {
      smlLibFree(pSyncHdr);
      return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppSyncHdr = pSyncHdr;

    return SML_ERR_OK;
}

static Ret_t
buildSync(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlSyncPtr_t pSync;
    Ret_t rc;
    Long_t cmdid = 0;

    /* stop decoding the Sync when we find a SyncML command */
    Byte_t break_sync = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    /* initialize a new Sync */
    if ((pSync = (SmlSyncPtr_t)smlLibMalloc(sizeof(SmlSync_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pSync, 0, sizeof(SmlSync_t));

    /* initialize the element type field */
    pSync->elementType = SML_PE_SYNC_START;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pSync);
        return SML_ERR_OK;
    }

    /* get next token */
    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pSync);
        return rc;
    }

    /* parse child elements until we find a matching end tag
       or until we find a TN_ADD, TN_ATOMIC, etc. start tag */
    while ((pScanner->curtok->type != TOK_TAG_END) && !break_sync) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSync->cmdID);
                cmdid++;
                break;
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSync->meta);
                break;

                /* child tags */
            case TN_CRED:
                rc = buildCred(pDecoder, (VoidPtr_t)&pSync->cred);
                break;
            case TN_TARGET:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pSync->target);
                break;
            case TN_SOURCE:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pSync->source);
                break;

                /* flags */
            case TN_NORESP:
                pSync->flags |= SmlNoResp_f;
                break;

                /* quit if we find an Add, Atomic, etc.
                   element */
            case TN_ADD:
            case TN_ATOMIC:
            case TN_COPY:
            case TN_DELETE:
            case TN_SEQUENCE:
            case TN_REPLACE:
                break_sync = 1;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pSync);
            return rc;
        }
        if (!break_sync) {
            /* get next token and continue as usual */
            if ((rc = nextToken(pDecoder)) != SML_ERR_OK) {
                smlLibFree(pSync);
                return rc;
            }
        } else {
            /* we've found a SyncML command - we need to go
               back one token and correct the tagstack */
            if ((rc = discardToken(pDecoder)) != SML_ERR_OK) {
                smlLibFree(pSync);
                return rc;
            }
        }
    }

    if (!break_sync)  {
      if ((pScanner->curtok->tagid) != TN_SYNC)
      {
        smlLibFree(pSync);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
      }
      else
      {
         if ((rc = pDecoder->scanner->pushTok(pDecoder->scanner)) != SML_ERR_OK)
         {
           smlLibFree(pSync);
           return rc;
         }
         if (pDecoder->tagstack->push(pDecoder->tagstack, pScanner->curtok->tagid))
         { 
            smlLibFree(pSync);
            return SML_ERR_UNSPECIFIC;
         }
      } 
    }

    *ppElem = pSync;

    return SML_ERR_OK;
}

#ifndef __SML_LITE__
static Ret_t
buildAtomOrSeq(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlAtomicPtr_t pAoS;        /* SmlAtomicPtr_t and SequencePtr_t are pointer
                                to the same structure! */
    Ret_t rc;
    Byte_t break_aos = 0;    /* stop decoding the Atomic when we find a
                                SyncML command */
    Long_t cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pAoS = (SmlAtomicPtr_t)smlLibMalloc(sizeof(SmlAtomic_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pAoS, 0, sizeof(SmlAtomic_t));

    /* initialize the element type field */
    pAoS->elementType = SML_PE_CMD_GROUP;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pAoS);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    /* get next token */
    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree (pAoS);
        return rc;
    }

    /* parse child elements until we find a matching end tag
       or until we find a TN_ADD, TN_ATOMIC, etc. start tag */
    while ((pScanner->curtok->type != TOK_TAG_END) && !break_aos) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAoS->cmdID);
                cmdid++;
                break;
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAoS->meta);
                break;

                /* flags */
            case TN_NORESP:
                pAoS->flags |= SmlNoResp_f;
                break;

                /* quit if we find an Add, Atomic, etc.
                   element */
            case TN_ADD:
            case TN_REPLACE:
            case TN_DELETE:
            case TN_COPY:
            case TN_ATOMIC:
            case TN_MAP:
            case TN_SYNC:
                break_aos = 1;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pAoS);
            return rc;
        }
        if (!break_aos) {
            if ((rc = nextToken(pDecoder)) != SML_ERR_OK) {
                smlLibFree(pAoS);
                return rc;
            }
        } else {
            /* we've found a SyncML command - we need to go
               back one token and correct the tagstack */
            if ((rc = discardToken(pDecoder)) != SML_ERR_OK) {
                smlLibFree(pAoS);
                return rc;
            }
        }
    }

    if (!break_aos) {
        /* Atomic/Sequence must contain at least one SyncML command */
        smlLibFree(pAoS);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (cmdid == 0)
    {
        smlLibFree(pAoS);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pAoS;

    return SML_ERR_OK;
}
#endif

#ifndef __SML_LITE__
static Ret_t
buildExec(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlExecPtr_t pExec;
    Ret_t rc;
    Long_t items = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pExec = (SmlExecPtr_t)smlLibMalloc(sizeof(SmlExec_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pExec, 0, sizeof(SmlExec_t));

    /* initialize the element type field */
    pExec->elementType = SML_PE_EXEC;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pExec);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pExec);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCData */
            case TN_CMDID: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pExec->cmdID);
                cmdid++;
                break;

                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pExec->cred);
                break;

            case TN_ITEM: 
                rc = buildItem(pDecoder, (VoidPtr_t)&pExec->item);
                items++;
                break;

                /* flags */
            case TN_NORESP:
                pExec->flags |= SmlNoResp_f;
                break;

            default:
                rc =  SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pExec);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pExec);
            return rc;
        }
    }

    if ((items == 0) || (cmdid == 0)) {
        smlLibFree(pExec);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pExec;

    return SML_ERR_OK;
}
#endif

static Ret_t
buildGenericCmd(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlGenericCmdPtr_t pGenCmd;
    Ret_t rc;
    Long_t items = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    /* initialize a new GenericCmd */
    if ((pGenCmd = (SmlGenericCmdPtr_t)smlLibMalloc(sizeof(SmlGenericCmd_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pGenCmd, 0, sizeof(SmlGenericCmd_t));

    /* initialize the element type field */
    pGenCmd->elementType = SML_PE_GENERIC;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pGenCmd);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pGenCmd);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pGenCmd->cmdID);
                cmdid++;
                break;
            case TN_META: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pGenCmd->meta);
                break;

                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pGenCmd->cred);
                break;

                /* flags (empty tags) */
            case TN_NORESP:
                pGenCmd->flags |= SmlNoResp_f;
                break;
            case TN_ARCHIVE:
                pGenCmd->flags |= SmlArchive_f;
                break;
            case TN_SFTDEL:
                pGenCmd->flags |= SmlSftDel_f;
                break;

                /* Lists */
            case TN_ITEM:
                rc = appendItemList(pDecoder, &pGenCmd->itemList);
                items++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pGenCmd);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pGenCmd);
            return rc;
        }
    }

    if ((items == 0) || (cmdid == 0))
    {
        smlLibFree(pGenCmd);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pGenCmd;

    return SML_ERR_OK;
}

static Ret_t
buildAlert(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlAlertPtr_t pAlert;
    Ret_t rc;
    Long_t items = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pAlert = (SmlAlertPtr_t)smlLibMalloc(sizeof(SmlAlert_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pAlert, 0, sizeof(SmlAlert_t));

    /* initialize the element type field */
    pAlert->elementType = SML_PE_ALERT;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pAlert);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pAlert);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAlert->cmdID);
                cmdid++;
                break;
            case TN_DATA: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAlert->data);
                break;

                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pAlert->cred);
                break;

                /* flags (empty tags) */
            case TN_NORESP:
                pAlert->flags |= SmlNoResp_f;
                break;

                /* Lists */
            case TN_ITEM:
                rc = appendItemList(pDecoder, &pAlert->itemList);
                items++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pAlert);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pAlert);
            return rc;
        }
    }

    if ((items == 0) || (cmdid == 0))
    {
        smlLibFree(pAlert);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pAlert;

    return SML_ERR_OK;
}

static Ret_t
buildMap(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlMapPtr_t pMap;
    Ret_t rc;
    Long_t target = 0, source = 0, mapitems = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pMap = (SmlMapPtr_t)smlLibMalloc(sizeof(SmlMap_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pMap, 0, sizeof(SmlMap_t));

    /* initialize the element type field */
    pMap->elementType = SML_PE_MAP;

    /* Source is required */
    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pMap);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pMap);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMap->cmdID);
                cmdid++;
                break;
            case TN_META: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMap->meta);
                break;

                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pMap->cred);
                break;
            case TN_SOURCE: 
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pMap->source);
                source++;
                break;
            case TN_TARGET: 
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pMap->target);
                target++;
                break;

                /* Lists */
            case TN_MAPITEM:
                rc = appendMapItemList(pDecoder, &pMap->mapItemList);
                mapitems++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pMap);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pMap);
            return rc;
        }
    }

    if ((source == 0) || (mapitems == 0) || (target == 0) || (cmdid == 0)) {
        smlLibFree(pMap);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pMap;

    return SML_ERR_OK;
}

#ifndef __SML_LITE__
static Ret_t
buildSearch(decoderPrivPtr_t pDecoder, VoidPtr_t *ppSearch)
{
    XltDecScannerPtr_t pScanner;
    SmlSearchPtr_t pSearch;
    Ret_t rc;
    Long_t source = 0, meta = 0, data = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppSearch != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pSearch = (SmlSearchPtr_t)smlLibMalloc(sizeof(SmlSearch_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pSearch, 0, sizeof(SmlSearch_t));

    /* initialize the element type field */
    pSearch->elementType = SML_PE_SEARCH;

    /* Meta is required */
    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pSearch);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pSearch);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSearch->cmdID);
                cmdid++;
                break;
            case TN_LANG:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSearch->lang);
                break;
            case TN_META: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSearch->meta);
                meta++;
                break;
            case TN_DATA: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pSearch->data);
                data++;
                break;


                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pSearch->cred);
                break;
            case TN_TARGET: 
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pSearch->target);
                break;

                /* flags */
            case TN_NORESP:
                pSearch->flags |= SmlNoResp_f;
                break;
            case TN_NORESULTS:
                pSearch->flags |= SmlNoResults_f;
                break;

                /* Lists */
            case TN_SOURCE:
                rc = appendSourceList(pDecoder, &pSearch->sourceList);
                source++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pSearch);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pSearch);
            return rc;
        }
    }

    if ((source == 0) || (meta == 0) || (data == 0) || (cmdid == 0)) {
        smlLibFree(pSearch);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppSearch = pSearch;

    return SML_ERR_OK;
}
#endif

static Ret_t
buildPutOrGet(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlGetPtr_t pGet;
    Ret_t rc;
    Long_t items = 0, cmdid = 0;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pGet = (SmlGetPtr_t)smlLibMalloc(sizeof(SmlGet_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pGet, 0, sizeof(SmlGet_t));

    /* initialize the element type field */
    pGet->elementType = SML_PE_PUT_GET;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pGet);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pGet);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pGet->cmdID);
                cmdid++;
                break;
            case TN_LANG:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pGet->lang);
                break;
            case TN_META: 
                rc = buildPCData(pDecoder, (VoidPtr_t)&pGet->meta);
                break;

                /* child tags */
            case TN_CRED: 
                rc = buildCred(pDecoder, (VoidPtr_t)&pGet->cred);
                break;

                /* flags */
            case TN_NORESP:
                pGet->flags |= SmlNoResp_f;
                break;

                /* Lists */

            case TN_ITEM:
                rc = appendItemList(pDecoder, &pGet->itemList);
                items++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pGet);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pGet);
            return rc;
        }
    }

    if ((items == 0) || (cmdid == 0))
    {
        smlLibFree(pGet);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pGet;

    return SML_ERR_OK;
}

static Ret_t
buildTargetOrSource(decoderPrivPtr_t pDecoder, VoidPtr_t *ppTarget)
{
    XltDecScannerPtr_t pScanner;
    SmlTargetPtr_t pTarget;
    Long_t locuri = 0;
    Ret_t rc;

    pScanner = pDecoder->scanner;

    if (*ppTarget != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pTarget = (SmlTargetPtr_t)smlLibMalloc(sizeof(SmlTarget_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pTarget, 0, sizeof(SmlTarget_t));

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pTarget);
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pTarget);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_LOCURI:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pTarget->locURI);
                locuri++;
                break;
            case TN_LOCNAME:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pTarget->locName);
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pTarget);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pTarget);
            return rc;
        }
    }

    if (locuri == 0) 
    {
        smlLibFree(pTarget);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppTarget = pTarget;

    return SML_ERR_OK;
}

static Ret_t
buildChal(decoderPrivPtr_t pDecoder, VoidPtr_t *ppChal)
{
    XltDecScannerPtr_t pScanner;
    SmlChalPtr_t pChal;
    Long_t meta = 0;
    Ret_t rc;

    pScanner = pDecoder->scanner;

    if (*ppChal != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pChal = (SmlChalPtr_t)smlLibMalloc(sizeof(SmlChal_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pChal, 0, sizeof(SmlChal_t));

    if (IS_EMPTY(pScanner->curtok)) {
        *ppChal = pChal;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pChal);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pChal->meta);
                meta++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pChal);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pChal);
            return rc;
        }
    }

    if (meta == 0) 
    {
        smlLibFree(pChal);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppChal = pChal;

    return SML_ERR_OK;
}

static Ret_t
buildCred(decoderPrivPtr_t pDecoder, VoidPtr_t *ppCred)
{
    XltDecScannerPtr_t pScanner;
    SmlCredPtr_t pCred;
    Ret_t rc;
    Long_t data = 0; 

    pScanner = pDecoder->scanner;

    if (*ppCred != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pCred = (SmlCredPtr_t)smlLibMalloc(sizeof(SmlCred_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pCred, 0, sizeof(SmlCred_t));

    if (IS_EMPTY(pScanner->curtok)) {
        *ppCred = pCred;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pCred);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_DATA:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pCred->data);
                data++;
                break;
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pCred->meta);
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pCred);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pCred);
            return rc;
        }
    }

    if (data == 0)
    {
      smlLibFree(pCred);
      return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppCred = pCred;

    return SML_ERR_OK;
}

static Ret_t
buildItem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlItemPtr_t pItem;
    Ret_t rc;

    pScanner = pDecoder->scanner;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pItem = (SmlItemPtr_t)smlLibMalloc(sizeof(SmlItem_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pItem, 0, sizeof(SmlItem_t));

    /* Item might be empty */
    if (IS_EMPTY(pScanner->curtok)) {
        *ppElem = pItem;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pItem);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pItem->meta);
                break;
            case TN_DATA:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pItem->data);
                break;

                /* child tags */
            case TN_TARGET:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pItem->target);
                break;
            case TN_SOURCE:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pItem->source);
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pItem);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pItem);
            return rc;
        }
    }

    *ppElem = pItem;

    return SML_ERR_OK;
}

static Ret_t
buildMapItem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlMapItemPtr_t pMapItem;
    Long_t target = 0, source = 0;
    Ret_t rc;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    pScanner = pDecoder->scanner;

    if ((pMapItem = (SmlMapItemPtr_t)smlLibMalloc(sizeof(SmlMapItem_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pMapItem, 0, sizeof(SmlMapItem_t));

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pMapItem);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pMapItem);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* child tags */
            case TN_TARGET:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pMapItem->target);
                target++;
                break;
            case TN_SOURCE:
                rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pMapItem->source);
                source++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pMapItem);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pMapItem);
            return rc;
        }
    }

    if ((target == 0) || (source == 0)) {
        smlLibFree(pMapItem);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pMapItem;

    return SML_ERR_OK;
}

static Ret_t
buildStatus(decoderPrivPtr_t pDecoder, VoidPtr_t *ppElem)
{
    XltDecScannerPtr_t pScanner;
    SmlStatusPtr_t pStatus;
    Ret_t rc;
    Long_t cmd = 0, data = 0, cmdid = 0;

    if (*ppElem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    pScanner = pDecoder->scanner;

    if ((pStatus = (SmlStatusPtr_t)smlLibMalloc(sizeof(SmlStatus_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pStatus, 0, sizeof(SmlStatus_t));

    /* initialize the element type field */
    pStatus->elementType = SML_PE_STATUS;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pStatus);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pStatus);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCData elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pStatus->cmdID);
                cmdid++;
                break;
            case TN_MSGREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pStatus->msgRef);
                break;
            case TN_CMDREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pStatus->cmdRef);
                break;
            case TN_CMD:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pStatus->cmd);
                cmd++;
                break;
            case TN_DATA:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pStatus->data);
                data++;
                break;
            case TN_CHAL:
                rc = buildChal(pDecoder, (VoidPtr_t)&pStatus->chal);
                break;
            case TN_CRED:
                rc = buildCred(pDecoder, (VoidPtr_t)&pStatus->cred);
                break;

            /* Lists */
            case TN_ITEM:
                rc = appendItemList(pDecoder, (VoidPtr_t)&pStatus->itemList);
                break;
            case TN_TARGETREF:
                rc = appendTargetRefList(pDecoder, (VoidPtr_t)&pStatus->targetRefList);
                break;
            case TN_SOURCEREF:
                rc = appendSourceRefList(pDecoder, (VoidPtr_t)&pStatus->sourceRefList);
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pStatus);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pStatus);
            return rc;
        }
    }

    if ((cmd == 0) || (data == 0) || (cmdid == 0)) 
    {
        smlLibFree(pStatus);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppElem = pStatus;

    return SML_ERR_OK;
}

static Ret_t
buildResults(decoderPrivPtr_t pDecoder, VoidPtr_t *ppResults)
{
    XltDecScannerPtr_t pScanner;
    SmlResultsPtr_t pResults;
    Ret_t rc;
    Long_t cmdref = 0, items = 0, cmdid = 0;

    if (*ppResults != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    pScanner = pDecoder->scanner;

    if ((pResults = (SmlResultsPtr_t)smlLibMalloc(sizeof(SmlResults_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pResults, 0, sizeof(SmlResults_t));

    /* initialize the element type field */
    pResults->elementType = SML_PE_RESULTS;

    if (IS_EMPTY(pScanner->curtok)) {
        smlLibFree(pResults);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pResults);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_CMDID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->cmdID);
                cmdid++;
                break;
            case TN_MSGREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->msgRef);
                break;
            case TN_CMDREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->cmdRef);
                cmdref++;
                break;
            case TN_META:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->meta);
                break;
            case TN_TARGETREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->targetRef);
                break;
            case TN_SOURCEREF:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pResults->sourceRef);
                break;

                /* Lists */
            case TN_ITEM:
                rc = appendItemList(pDecoder, &pResults->itemList);
                items++;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pResults);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pResults);
            return rc;
        }
    }

    if ((cmdref == 0) || (items == 0) || (cmdid == 0))
    {
        smlLibFree(pResults);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppResults = pResults;

    return SML_ERR_OK;
}

static Ret_t
buildPCData(decoderPrivPtr_t pDecoder, VoidPtr_t *ppPCData)
{
    XltDecScannerPtr_t pScanner;
    SmlPcdataPtr_t pPCData;
    SmlPcdataExtension_t ext;
    Ret_t rc;

    pScanner = pDecoder->scanner;

    if (*ppPCData != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if (IS_EMPTY(pScanner->curtok)) {
        if ((pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t))) == NULL)
            return SML_ERR_NOT_ENOUGH_SPACE;

        smlLibMemset(pPCData, 0, sizeof(SmlPcdata_t));

        *ppPCData = pPCData;
        return SML_ERR_OK;
    } 

    pPCData = NULL;

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        return rc;
    }

    if (IS_CONTENT(pScanner->curtok)) {
        /* PCData element has a regular string or opaque content */
        while (pScanner->curtok->type == TOK_CONT) {
            if (pPCData == NULL)
                pPCData = pScanner->curtok->pcdata;
            else {
                pPCData = concatPCData(pPCData, pScanner->curtok->pcdata);
                smlLibFree(pScanner->curtok->pcdata->content);
                smlLibFree(pScanner->curtok->pcdata);
                if (pPCData == NULL)
                    return SML_ERR_XLT_INVAL_PCDATA;
            }
                
            if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
                return rc;
            }
        }
    } else if (IS_START_OR_EMPTY(pScanner->curtok)) {
        /* PCData element contains an XML dokument that is handled by an
           extension mechanism - supported is only Meta Information */
        ext = pScanner->curtok->ext;
        if ((rc = discardToken(pDecoder)) != SML_ERR_OK)
            return rc;
        if ((pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t))) == NULL)
            return SML_ERR_NOT_ENOUGH_SPACE;
        smlLibMemset(pPCData, 0, sizeof(SmlPcdata_t));
        pPCData->contentType = SML_PCDATA_EXTENSION;
        pPCData->extension = ext;
        switch (ext) {
#ifdef __META_EXT__
            case SML_EXT_META:
                extBuildMeta(pDecoder, (VoidPtr_t)&pPCData->content);
                break;
#endif
            default:
                smlLibFree(pPCData);
                return SML_ERR_XLT_INVAL_EXT;
        }
    } else if (IS_END(pScanner->curtok)) {
        /* PCData element is empty */
    } else {
        return SML_ERR_XLT_INVAL_PCDATA;
    }
        
    if (pScanner->curtok->type != TOK_TAG_END)
        return SML_ERR_XLT_INVAL_PCDATA;

    if (pPCData == NULL) {
        if ((pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t))) == NULL)
            return SML_ERR_NOT_ENOUGH_SPACE;
        smlLibMemset(pPCData, 0, sizeof(SmlPcdata_t));
    }

    *ppPCData = pPCData;

    return SML_ERR_OK;
}

static Ret_t
appendItemList(decoderPrivPtr_t pDecoder, SmlItemListPtr_t *ppItemList)
{
    SmlItemListPtr_t pNewItemList;
    SmlItemListPtr_t pItemList;
    Ret_t rc;

    pItemList = *ppItemList;
    if (pItemList != NULL)
        while (pItemList->next != NULL)
            pItemList = pItemList->next;

    if ((pNewItemList = (SmlItemListPtr_t)smlLibMalloc(sizeof(SmlItemList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewItemList, 0, sizeof(SmlItemList_t));

    if ((rc = buildItem(pDecoder, (VoidPtr_t)&pNewItemList->item)) != SML_ERR_OK) {
        smlLibFree(pNewItemList);
        return rc;
    }

    if (pItemList == NULL)
        *ppItemList = pNewItemList;
    else
        pItemList->next = pNewItemList;

    return SML_ERR_OK;
}

static Ret_t
appendSourceList(decoderPrivPtr_t pDecoder, SmlSourceListPtr_t *ppSourceList)
{
    SmlSourceListPtr_t pNewSourceList;
    SmlSourceListPtr_t pSourceList;
    Ret_t rc;

    pSourceList = *ppSourceList;
    if (pSourceList != NULL)
        while (pSourceList->next != NULL)
            pSourceList = pSourceList->next;

    if ((pNewSourceList = (SmlSourceListPtr_t)smlLibMalloc(sizeof(SmlSourceList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewSourceList, 0, sizeof(SmlSourceList_t));

    if ((rc = buildTargetOrSource(pDecoder, (VoidPtr_t)&pNewSourceList->source)) != SML_ERR_OK) {
        smlLibFree(pNewSourceList);
        return rc;
    }

    if (pSourceList == NULL)
        *ppSourceList = pNewSourceList;
    else
        pSourceList->next = pNewSourceList;

    return SML_ERR_OK;
}

static Ret_t
appendMapItemList(decoderPrivPtr_t pDecoder, SmlMapItemListPtr_t *ppMapItemList)
{
    SmlMapItemListPtr_t pNewMapItemList;
    SmlMapItemListPtr_t pMapItemList;
    Ret_t rc;

    pMapItemList = *ppMapItemList;
    if (pMapItemList != NULL)
        while (pMapItemList->next != NULL)
            pMapItemList = pMapItemList->next;

    if ((pNewMapItemList = (SmlMapItemListPtr_t)smlLibMalloc(sizeof(SmlMapItemList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewMapItemList, 0, sizeof(SmlMapItemList_t));

    if ((rc = buildMapItem(pDecoder, (VoidPtr_t)&pNewMapItemList->mapItem)) != SML_ERR_OK) {
        smlLibFree(pNewMapItemList);
        return rc;
    }

    if (pMapItemList == NULL)
        *ppMapItemList = pNewMapItemList;
    else
        pMapItemList->next = pNewMapItemList;

    return SML_ERR_OK;
}

static Ret_t
appendTargetRefList(decoderPrivPtr_t pDecoder, SmlTargetRefListPtr_t *ppTargetRefList)
{
    SmlTargetRefListPtr_t pNewTargetRefList;
    SmlTargetRefListPtr_t pTargetRefList;
    Ret_t rc;

    pTargetRefList = *ppTargetRefList;
    if (pTargetRefList != NULL)
        while (pTargetRefList->next != NULL)
            pTargetRefList = pTargetRefList->next;

    if ((pNewTargetRefList = (SmlTargetRefListPtr_t)smlLibMalloc(sizeof(SmlTargetRefList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewTargetRefList, 0, sizeof(SmlTargetRefList_t));

    if ((rc = buildPCData(pDecoder, (VoidPtr_t)&pNewTargetRefList->targetRef)) != SML_ERR_OK) {
        smlLibFree(pNewTargetRefList);
        return rc;
    }

    if (pTargetRefList == NULL)
        *ppTargetRefList = pNewTargetRefList;
    else
        pTargetRefList->next = pNewTargetRefList;

    return SML_ERR_OK;
}

static Ret_t
appendSourceRefList(decoderPrivPtr_t pDecoder, SmlSourceRefListPtr_t *ppSourceRefList)
{
    SmlSourceRefListPtr_t pNewSourceRefList;
    SmlSourceRefListPtr_t pSourceRefList;
    Ret_t rc;

    pSourceRefList = *ppSourceRefList;
    if (pSourceRefList != NULL)
        while (pSourceRefList->next != NULL)
            pSourceRefList = pSourceRefList->next;

    if ((pNewSourceRefList = (SmlSourceRefListPtr_t)smlLibMalloc(sizeof(SmlSourceRefList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewSourceRefList, 0, sizeof(SmlSourceRefList_t));

    if ((rc = buildPCData(pDecoder, (VoidPtr_t)&pNewSourceRefList->sourceRef)) != SML_ERR_OK) {
        smlLibFree(pNewSourceRefList);
        return rc;
    }

    if (pSourceRefList == NULL)
        *ppSourceRefList = pNewSourceRefList;
    else
        pSourceRefList->next = pNewSourceRefList;

    return SML_ERR_OK;
}

/************************************************************************/
/* EXTENTION: Meta Information                                          */
/************************************************************************/

#ifdef __META_EXT__

static Ret_t
extBuildMeta(decoderPrivPtr_t pDecoder, VoidPtr_t *ppMeta)
{
    XltDecScannerPtr_t pScanner;
    SmlMetaPtr_t pMeta;
    Ret_t rc;

    pScanner = pDecoder->scanner;

    if (*ppMeta != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pMeta = (SmlMetaPtr_t)smlLibMalloc(sizeof(SmlMeta_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pMeta, 0, sizeof(SmlMeta_t));

    if (IS_EMPTY(pScanner->curtok)) {
        *ppMeta = pMeta;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pMeta);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_META_METINF:
                smlLibFree(pMeta);
                pMeta = NULL; 
                rc = extBuildMeta(pDecoder, (VoidPtr_t)&pMeta);
                break;
            case TN_META_FORMAT:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->format);
                break;
            case TN_META_TYPE:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->type);
                break;
            case TN_META_MARK:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->mark);
                break;
            case TN_META_SIZE:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->size);
                break;
            case TN_META_VERSION:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->version);
                break;
            case TN_META_NEXTNONCE:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->nextnonce);
                break;
            case TN_META_MAXMSGSIZE:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMeta->maxmsgsize);
                break;
            case TN_META_MEM:
                rc = extBuildMetaMem(pDecoder, (VoidPtr_t)&pMeta->mem);
                break;
            case TN_META_ANCHOR:
                rc = extBuildMetaAnchor(pDecoder, (VoidPtr_t)&pMeta->anchor);
                break;

            //Lists  
            case TN_META_EMI:
                rc = appendEmiList(pDecoder, &pMeta->emilist);
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pMeta);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pMeta);
            return rc;
        }
    }
    *ppMeta = pMeta;

    return SML_ERR_OK;
}

static Ret_t
extBuildMetaAnchor(decoderPrivPtr_t pDecoder, VoidPtr_t *ppAnchor)
{
    XltDecScannerPtr_t pScanner;
    SmlAnchorPtr_t pAnchor;
    Ret_t rc;
    Long_t next = 0;

    pScanner = pDecoder->scanner;

    if (*ppAnchor != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pAnchor = (SmlAnchorPtr_t)smlLibMalloc(sizeof(SmlAnchor_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pAnchor, 0, sizeof(SmlAnchor_t));

    if (IS_EMPTY(pScanner->curtok)) {
        *ppAnchor = pAnchor;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pAnchor);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_META_LAST:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAnchor->last);
                break;
            case TN_META_NEXT:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pAnchor->next);
                next++;
                break;
            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pAnchor);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pAnchor);
            return rc;
        }
    }
    if (next == 0)
    {
        smlLibFree(pAnchor);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }

    *ppAnchor = pAnchor;

    return SML_ERR_OK;
}

static Ret_t
extBuildMetaMem(decoderPrivPtr_t pDecoder, VoidPtr_t *ppMem)
{
    XltDecScannerPtr_t pScanner;
    SmlMemPtr_t pMem;
    Ret_t rc;
    Long_t freemem=0, freeid=0;

    pScanner = pDecoder->scanner;

    if (*ppMem != NULL)
        return SML_ERR_XLT_INVAL_SYNCML_DOC;

    if ((pMem = (SmlMemPtr_t)smlLibMalloc(sizeof(SmlMem_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pMem, 0, sizeof(SmlMem_t));

    if (IS_EMPTY(pScanner->curtok)) {
        *ppMem = pMem;
        return SML_ERR_OK;
    }

    if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
        smlLibFree(pMem);
        return rc;
    }

    while (pScanner->curtok->type != TOK_TAG_END) {
        switch (pScanner->curtok->tagid) {

            /* PCDATA elements */
            case TN_META_FREEMEM:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMem->freemem);
                freemem++;
                break;
            case TN_META_FREEID:
                rc = buildPCData(pDecoder, (VoidPtr_t)&pMem->freeid);
                freeid++;
                break;

                /* flags */
            case TN_META_SHAREDMEM:
                pMem->sharedmem |= SmlSharedMem_f;
                break;

            default:
                rc = SML_ERR_XLT_INVAL_SYNCML_DOC;
        }
        if (rc != SML_ERR_OK) {
            smlLibFree(pMem);
            return rc;
        }
        if (((rc = nextToken(pDecoder)) != SML_ERR_OK)) {
            smlLibFree(pMem);
            return rc;
        }
    }

    if ((freemem == 0) || (freeid == 0))
    {
        smlLibFree(pMem);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
    }
    *ppMem = pMem;

    return SML_ERR_OK;
}

static Ret_t
appendEmiList(decoderPrivPtr_t pDecoder, SmlEmiListPtr_t *ppEmiList)
{
    SmlEmiListPtr_t pNewEmiList;
    SmlEmiListPtr_t pEmiList;
    Ret_t rc;

    pEmiList = *ppEmiList;
    if (pEmiList != NULL)
        while (pEmiList->next != NULL)
            pEmiList = pEmiList->next;

    if ((pNewEmiList = (SmlEmiListPtr_t)smlLibMalloc(sizeof(SmlEmiList_t))) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(pNewEmiList, 0, sizeof(SmlEmiList_t));

    if ((rc = buildPCData(pDecoder, (VoidPtr_t)&pNewEmiList->emi)) != SML_ERR_OK) {
        smlLibFree(pNewEmiList);
        return rc;
    }

    if (pEmiList == NULL)
        *ppEmiList = pNewEmiList;
    else
        pEmiList->next = pNewEmiList;

    return SML_ERR_OK;
}

#endif