/*
 * Config file handling
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include "resmgrd.h"

static const char *	config_file_name = _PATH_RESMGR_CONF;
const char *		res_config_lock_dir = _PATH_LOCKS;


static void	res_config_parse(const char *fname);

/*
 * Set the name of the static ACL file
 */
void
res_config_set_filename(const char *name)
{
	if (name == NULL)
		name = _PATH_RESMGR_CONF;
	config_file_name = name;
}

/*
 * Reload the static device ACLs
 */
void
res_config_reload(void)
{
	res_class_t	*cl;

	/* Clear all static ACLs */
	while ((cl = res_classes) != NULL) {
		res_classes = cl->next;
		res_class_free(cl);
	}

	/* The reload the ACL file */
	res_config_parse(config_file_name);
}

/*
 * Parse the config file.
 */
static void
res_config_parse(const char *fname)
{
	char		buffer[512];
	unsigned int	line = 0;
	FILE		*fp;

	if ((fp = fopen(fname, "r")) == NULL)
		fatal("unable to open %s: %m", fname);

	while (fgets(buffer, sizeof(buffer), fp) != NULL) {
		unsigned int	n = 0, argc;
		char		*argv[16];
		char		*kwd;

		line++;
		if (!(argc = line_split(buffer, argv, 16)))
			continue;

		kwd = argv[n++];
		if (!strcmp(kwd, "class")) {
			char	*name;

			if (argc != 2) {
				fatal("%s:%u: bad number of arguments "
				      "in \"class\" statement\n",
				      fname, line);
			}
			name = argv[1];

			if (res_class_get(name)) {
				fatal("%s:%u: class \"%s\" redefined",
					fname, line, name);
			}
			res_class_create(name);
		} else
		if (!strcmp(kwd, "add")) {
			char		*devname, *clsname;
			res_device_t	*dev;
			res_class_t	*cls;

			if (argc < 3) {
				fatal("%s:%u: missing argument "
				      "in \"add\" statement\n",
				      fname, line);
			}

			devname = argv[1];
			clsname = argv[2];

			if (!(cls = res_class_get(clsname))) {
				fatal("%s:%u: unknown class \"%s\"",
					fname, line, clsname);
			}
			dev = res_class_add(cls, devname, 0);

			for (n = 3; n < argc; n++) {
				res_family_t	*family;
				const char	*opt = argv[n];

				if (!strcmp(opt, "read-only"))
					dev->flags |= DEV_FLAGS_RO;
				else
				if ((family = res_family_get(opt)) != NULL)
					dev->flags |= family->access_flags;
				else {
					fatal("%s:%u: invalid flag \"%s\"",
						fname, line, opt);
				}
			}
		} else
		if (!strcmp(kwd, "allow") || !strcmp(kwd, "deny")) {
			res_class_t	*cls;
			char		*name;
			res_acl_t	*acl;

			if (n + 1 >= argc) {
				fatal("%s:%u: missing argument "
				      "in \"allow\" statement\n",
				      fname, line);
			}

			name = argv[n++];
			if (!(cls = res_class_get(name))) {
				fatal("%s:%u: unknown class \"%s\"",
					fname, line, name);
			}

			if (!(acl = res_acl_parse(argc-n, argv+n))) {
				fatal("%s:%u: failed to parse ACL",
					fname, line);
			}
			res_acl_add(&cls->acl, acl, kwd[0] == 'a'? 1 : -1);
		} else {
			fatal("%s:%u: invalid keyword \"%s\"",
				fname, line, kwd);
		}
	}
	fclose(fp);
}
