/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

#include "mutt.h"
#include "mutt_curses.h"
#include "keymap.h"

#include <termios.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>

/* global vars used for the string-history routines */
static char **Hist = NULL;
static short HistCur = 0;
static short HistLast = 0;

/* not possible to unget more than one char under some curses libs, and it
 * is impossible to unget function keys in SLang, so roll our own input
 * buffering routines.
 */
static short UngetCount = 0;

#define UngetBufLen 128
static int UngetBuf[UngetBufLen];

void mutt_init_history (void)
{
  int i;
  static int OldSize = 0;
  
  if (Hist)
  {
    for (i = 0 ; i < OldSize ; i ++)
      safe_free ((void **) &Hist[i]);
    safe_free ((void **) &Hist);
  }
  
  if (HistSize)
    Hist = safe_calloc (HistSize, sizeof (char *));
  HistCur = 0;
  HistLast = 0;
  OldSize = HistSize;
}

static void sh_add (char *s)
{
  int prev;

  if (!HistSize)
    return; /* disabled */

  if (*s)
  {
    prev = HistLast - 1;
    if (prev < 0) prev = HistSize - 1;
    if (!Hist[prev] || strcmp (Hist[prev], s) != 0)
    {
      safe_free ((void **) &Hist[HistLast]);
      Hist[HistLast++] = safe_strdup (s);
      if (HistLast > HistSize - 1)
	HistLast = 0;
    }
  }
  HistCur = HistLast; /* reset to the last entry */
}

static char *sh_next (void)
{
  int next;

  if (!HistSize)
    return (""); /* disabled */

  next = HistCur + 1;
  if (next > HistLast - 1)
    next = 0;
  if (Hist[next])
    HistCur = next;
  return (Hist[HistCur] ? Hist[HistCur] : "");
}

static char *sh_prev (void)
{
  int prev;

  if (!HistSize)
    return (""); /* disabled */

  prev = HistCur - 1;
  if (prev < 0)
  {
    prev = HistLast - 1;
    if (prev < 0)
    {
      prev = HistSize - 1;
      while (prev > 0 && Hist[prev] == NULL)
	prev--;
    }
  }
  if (Hist[prev])
    HistCur = prev;
  return (Hist[HistCur] ? Hist[HistCur] : "");
}

/* This is the routine for user input into Mutt.  It looks at the user's
 * Charset to see whether or not it is ISO-8859 compatible (superset of
 * US-ASCII).  If not, it is assumed to be Chinese/Japanese/Korean
 * two-byte charset.
 *
 * Returns:
 *	1 need to redraw the screen and call me again
 *	0 if input was given
 * 	-1 if abort.
 *
 */
int
ci_enter_string (unsigned char *buf, size_t buflen, int y, int x, int flags)
{
  int ch, lastchar, i, j, beg = 0, ch8bit, ch8pair = 0;
  int pass = (flags == M_PASS);
  int iso8859 = 1;
  int first = 1;
  char tempbuf[_POSIX_PATH_MAX] = "";
  int len = COLS - x - 1; /* width of field */

redraw_it:

  move (y,x);
  lastchar = i = strlen ((char *) buf);
  if (!pass && i > 0)
  {
    beg = lastchar - len;
    if (beg < 0)
      beg = 0;
    else if (beg > 0)
      clrtoeol ();
    for (j = beg; j < lastchar && j < beg + len; j++)
      addch (buf[j]);
  }
  clrtoeol ();
  refresh ();

  FOREVER
  {
    /* first look to see if a keypress is an editor operation.  km_dokey()
     * returns 0 if there is no entry in the keymap, so restore the last
     * keypress and continue normally.
     */
    ch = km_dokey (MENU_EDITOR);
    if (ch == -1)
    {
      buf[i] = 0;
      return (-1);
    }

    if (ch != 0)
    {
      first = 0; /* make sure not to clear the buffer */
      switch (ch)
      {
	case OP_EDITOR_HISTORY_UP:
	  if (!pass)
	  {
	    strfcpy ((char *) buf, sh_prev (), buflen);
	    goto redraw_it;
	  }
	  break;
	case OP_EDITOR_HISTORY_DOWN:
	  if (!pass)
	  {
	    strfcpy ((char *) buf, sh_next (), buflen);
	    goto redraw_it;
	  }
	  break;
	case OP_EDITOR_BACKSPACE:
	  if (i == 0)
	  {
	    BEEP ();
	    break;
	  }
	  if (i == lastchar)
	  {
	    lastchar--;
	    i--;
	    if ( ! iso8859 )
	    {
	      if ((buf[i] & 0x80) & (buf[i-1] & 0x80))
	      {
		lastchar--;
		i--;
	      }
	    }
	    if (!pass)
	    {
	      if (i < beg)
	      {
		move (y, x);
		clrtoeol ();
		beg -= len;
		if (beg < 0)
		  beg = 0;
		for(j=beg; j<lastchar && j<beg+len; j++)
		  addch (buf[j]);
	      }
	      else
	      {
		move (y, x+i-beg);
		clrtoeol ();
		if (! iso8859)
		{
		  if ((buf[i-1] & 0x80) & (buf[i-2] & 0x80))
		    delch ();
		}
		if (i > beg)
		  delch ();
	      }
	    }
	  }
	  else
	  {
	    if (! iso8859)
	      ch8bit = ((buf[i-1] & 0x80) & (buf[i-2] & 0x80)) ? 1: 0; 
	    else
	      ch8bit = 0;
	    for (j = i ; j < lastchar ; j++)
	    {
	      if (ch8bit)
		buf[j-2] = buf[j];
	      else
		buf[j-1] = buf[j];
	    }
	    i--;
	    lastchar--;
	    if (ch8bit)
	    {
	      i--;
	      lastchar--;
	    }
	    if (!pass)
	    {
	      if (i >= beg)
	      {
		buf[lastchar] = 0;
		move (y, x + i - beg);
		for (j = i; j < lastchar && j < beg + len; j++)
		  addch (buf[j]);
	      }
	      else
	      {
		move (y, x);
		beg -= len;
		if (beg < 0)
		  beg = 0;
		for (j = beg; j <= beg + len; j++)
		  addch (buf[j]);
	      }
	      clrtoeol ();
	      move (y, x + i - beg);
	    }
	  }
	  break;
	case OP_EDITOR_BOL:
	  /* reposition the cursor at the beginning of the line */
	  i = 0;
	  if (!pass)
	  {
	    move (y,x);
	    if (beg != 0)
	    {
	      /* the first char is not displayed, so readjust */
	      clrtoeol ();
	      for (j = 0; j < len; j++)
		addch (buf[j]);
	      move (y, x);
	      beg = 0;
	    }
	  }
	  break;
	case OP_EDITOR_EOL:
	  if (!pass)
	  {
	    if (lastchar < beg + len)
	      move (y, x + lastchar - beg);
	    else
	    {
	      move (y,x);
	      clrtoeol ();
	      beg = lastchar - len / 2;
	      for (j = beg; j < lastchar; j++)
		addch (buf[j]);
	    }
	  }
	  i = lastchar;
	  break;
	case OP_EDITOR_KILL_LINE:
	  lastchar = i = 0;
	  buf[0] = 0;
	  if (!pass)
	  {
	    move (y, x);
	    clrtoeol ();
	    beg = 0;
	  }
	  break;
	case OP_EDITOR_KILL_EOL:
	  lastchar = i;
	  buf[i] = 0;
	  if (!pass)
	    clrtoeol ();
	  break;
	case OP_EDITOR_BACKWARD_CHAR:
	  if (i == 0)
	  {
	    BEEP ();
	  }
	  else
	  {
	    i--;
	    if (! iso8859)
	    {
	      if ((buf[i] & 0x80) & (buf[i-1] & 0x80)) i--;
	    }
	    if (!pass)
	    {
	      if (i < beg)
	      {
		beg -= len / 2;
		if (beg < 0) beg = 0;
		move (y, x);
		clrtoeol ();
		for (j=beg; j<beg+len; j++)
		  addch (buf[j]);
	      }
	    }
	    move (y, x+i-beg);
	  }
	  break;
	case OP_EDITOR_FORWARD_CHAR:
	  if (i == lastchar)
	  {
	    BEEP ();
	  }
	  else
	  {
	    i++;
	    if (! iso8859)
	    {
	      if ((buf[i] & 0x80) & (buf[i+1] & 0x80)) i++;
	    }
	    if (!pass)
	    {
	      if (i >= beg + len)
	      {
		beg = i;
		move (y, x);
		clrtoeol ();
		for (j=beg; j<lastchar && j<beg+len; j++)
		  addch (buf[j]);
		move (y,x);
	      }
	      else
		move (y,x+i-beg);
	    }
	  }
	  break;
	case OP_EDITOR_DELETE_CHAR:
	  if (i != lastchar)
	  {
	    for (j = i; j < lastchar; j++)
	      buf[j] = buf[j + 1];
	    lastchar--;
	    if (!pass)
	    {
	      clrtoeol ();
	      for (j = i; j < lastchar && j < beg + len; j++)
		addch (buf[j]);
	      move (y, x + i - beg);
	    }
	  }
	  else
	    BEEP ();
	  break;
	case OP_EDITOR_KILL_WORD:
	  /* delete to begin of word */
	  if (i != 0)
	  {
	    j = i;
	    while (j > 0 && ISSPACE (buf[j-1]))
	      j--;
	    if (j > 0)
	    {
	      if (isalnum (buf[j-1]))
	      {
		for (j--; j > 0 && isalnum (buf[j-1]); j--)
		  ;
	      }
	      else
		j--;
	    }
	    ch = j; /* save current position */
	    while (i < lastchar)
	      buf[j++] = buf[i++];
	    lastchar = j;
	    i = ch; /* restore current position */
	    /* update screen */
	    if (!pass)
	    {
	      if (i < beg)
	      {
		/* scrolled past the beginning of the line */
		while (i < beg)
		  beg -= len / 2;
		if (beg < 0)
		  beg = 0;
	      }
	      move (y, x + i - beg);
	      for (j=i; j<lastchar; j++)
		addch (buf[j]);
	      clrtoeol ();
	      move (y, x + i - beg);
	    }
	  }
	  break;
	default:
	  BEEP ();
      }
    }
    else
    {
      /* use the raw keypress */
      ch = LastKey;

      if ((flags & M_EFILE) && ch == ' ')
      {
	buf[i] = 0;
	mutt_buffy ((char *) buf);
	goto redraw_it;
      }

      if (first && (flags & M_CLEAR))
      {
	first = 0;
	if (IsPrint (ch))
	{
	  buf[0] = 0;
	  mutt_ungetch (ch);
	  goto redraw_it;
	}
      }

      if ((flags & (M_FILE | M_EFILE)) && (ch == '\t' || ch == ' '))
      {
	buf[i] = 0;

	/* see if the path has changed from the last time */
	if (strcmp (tempbuf, (char *) buf) == 0)
	{
	  mutt_select_file ((char *) buf, buflen, 0);
	  set_option (OPTNEEDREDRAW);
	  if (buf[0])
	  {
	    mutt_pretty_mailbox ((char *) buf);
	    sh_add ((char *) buf);
	    return (0);
	  }
	  return (-1);
	}

	if (mutt_complete ((char *) buf) == 0)
	  strfcpy (tempbuf, (char *) buf, sizeof (tempbuf));
	else
	  BEEP (); /* let the user know that nothing matched */
	goto redraw_it;
      }

      if ((flags & M_CMD) && ch == '\t')
      {
	buf[i] = 0;
	for (j=i-1; j>=0 && buf[j] != ' '; j--);
	if (strcmp (tempbuf, (char *) buf) == 0)
	{
	  mutt_select_file ((char *) buf + j + 1, buflen - j - 1, 0);
	  set_option (OPTNEEDREDRAW);
	  return (1);
	}
	if (mutt_complete ((char *) buf + j + 1) == 0)
	  strfcpy (tempbuf, (char *) buf + j + 1, sizeof (tempbuf));
	else
	  BEEP ();
	goto redraw_it;
      }

      if (ch == '\t' && (flags & M_ALIAS))
      {
	/* invoke the alias-menu to get more addresses */
	buf[i] = 0;
	if (i)
	{
	  for (j = i - 1 ; j >= 0 && buf[j] != ' ' && buf[j] != ',' ; j--);
	  if (mutt_alias_complete ((char *) buf + j + 1, buflen - j - 1))
	    goto redraw_it;
	}
	else
	  mutt_alias_menu ((char *) buf, buflen, Aliases);
	return (1);
      }
      else if (CI_is_return (ch))
      {
	buf[lastchar] = 0;
	if (!pass)
	  sh_add ((char *) buf);
	return (0);
      }
      else if (IsPrint (ch) && (lastchar + 1 < buflen))
      {
	if (! iso8859)
	{
	  ch8pair = (ch & 0x80) ? ci_getch (): 0;
	  ch8bit = (ch8pair && IsPrint (ch8pair));
	}
	else
	  ch8bit = 0;

	if (i == lastchar)
	{
	  buf[i++] = ch;
	  lastchar++;

	  if (ch8bit)
	  {
	    buf[i++] = ch8pair;
	    lastchar++;
	  }

	  if (!pass)
	  {
	    if (i > beg+len)
	    {
	      beg = i - len/2;
	      move(y,x);
	      clrtoeol();
	      for (j = beg; j < lastchar; j++)
		addch (buf[j]);
	    }
	    else
	    {
	      addch (ch);
	      if (ch8bit)
		addch (ch8pair);
	    }
	  }
	}
	else
	{
	  for (j = lastchar; j > i; j--)
	  {
	    if (ch8bit)
	      buf[j+1] = buf[j-1];
	    else
	      buf[j] = buf[j-1];
	  }
	  buf[i++] = ch;
	  lastchar++;

	  if (ch8bit)
	  {
	    buf[i++] = ch8pair;
	    lastchar++;
	  }

	  if (!pass)
	  {
	    if (i > beg + len)
	    {
	      move (y,x);
	      clrtoeol ();
	      beg = ch8bit ? i-2 : i-1;
	      for (j=beg; j<lastchar && j<beg+len; j++)
		addch (buf[j]);
	    }
	    else
	    {
	      clrtoeol();
	      j = ch8bit ? i-2 : i-1;
	      for (; j < lastchar && j < beg + len ; j++)
		addch(buf[j]);
	    }
	    move (y,x+i-beg);
	  }
	}
	if (ch8bit)
	  ch8pair = 0;
      }
      else
      {
	mutt_flushinp ();
	BEEP ();
      }
    }
    refresh ();
  }
  /* not reached */
}

int ci_get_field (char *field, char *buf, size_t buflen, int complete)
{
  int ret;
  int len = strlen (field); /* in case field==buffer */

  do
  {
    CLEARLINE (LINES-1);
    addstr (field);
    refresh ();
    ret = ci_enter_string ((unsigned char *) buf, buflen, LINES-1, len, complete);
  }
  while (ret == 1);
  CLEARLINE (LINES-1);
  return (ret);
}

void mutt_clear_error (void)
{
  Errorbuf[0] = 0;
  CLEARLINE (LINES-1);
}

void mutt_edit_file (const char *editor, const char *data)
{
  char cmd[LONG_STRING];

  endwin ();
  mutt_expand_fmt (cmd, sizeof (cmd), editor, data);
  mutt_system (cmd);
  keypad (stdscr, TRUE);
  clearok (stdscr, TRUE);
}

int mutt_yesorno (const char *msg, int def)
{
  int ch;

  CLEARLINE(LINES-1);
  printw("%s: [%c] ", msg, def ? 'y' : 'n');
  FOREVER
  {
    refresh ();
    ch = ci_getch ();
    if (ch == ERR) return(-1);
    if (CI_is_return (ch))
      break;
    else if (ch == 'y')
    {
      def = 1;
      break;
    }
    else if (ch == 'n')
    {
      def = 0;
      break;
    }
    else
    {
      BEEP();
    }
  }
  addstr (def ? "Yes" : "No");
  refresh ();
  return (def);
}

/* this function is called when the user presses the abort key */
void mutt_query_exit (void)
{
  mutt_flushinp ();
  curs_set (1);
  if (mutt_yesorno ("Exit Mutt?", 1) == 1)
  {
    endwin ();
    exit (0);
  }
  mutt_curs_set (-1);
  Signals &= ~S_INTERRUPT;
}

int ci_getch (void)
{
  int ch;

  if (UngetCount)
    return (UngetBuf[--UngetCount]);

  Signals &= ~S_INTERRUPT;
  ch = getch ();
  if (Signals & S_INTERRUPT)
    mutt_query_exit ();
  return (ch == ctrl ('G') ? ERR : ch);
}

int mutt_get_password (char *msg, char *buf, size_t buflen)
{
  int rc;

  CLEARLINE (LINES-1);
  addstr (msg);
  rc = ci_enter_string ((unsigned char *) buf, buflen, LINES - 1, strlen (msg), M_PASS);
  CLEARLINE (LINES-1);
  return (rc);
}

void mutt_error (const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vsnprintf (Errorbuf, sizeof (Errorbuf), fmt, ap);
  va_end (ap);
  
  Errorbuf[ (COLS < sizeof (Errorbuf) ? COLS : sizeof (Errorbuf)) - 2 ] = 0;

  BEEP ();
  SETCOLOR (MT_COLOR_ERROR);
  mvaddstr (LINES-1, 0, Errorbuf);
  clrtoeol ();
  SETCOLOR (MT_COLOR_NORMAL);
  refresh ();

  set_option (OPTMSGERR);
}

void mutt_message (const char *fmt, ...)
{
  va_list ap;

  va_start (ap, fmt);
  vsnprintf (Errorbuf, sizeof (Errorbuf), fmt, ap);
  va_end (ap);

  Errorbuf[ (COLS < sizeof (Errorbuf) ? COLS : sizeof (Errorbuf)) - 2 ] = 0;

  SETCOLOR (MT_COLOR_MESSAGE);
  mvaddstr (LINES - 1, 0, Errorbuf);
  clrtoeol ();
  SETCOLOR (MT_COLOR_NORMAL);
  refresh ();

  unset_option (OPTMSGERR);
}

void mutt_show_error (void)
{
  SETCOLOR (option (OPTMSGERR) ? MT_COLOR_ERROR : MT_COLOR_MESSAGE);
  CLEARLINE (LINES-1);
  addstr (Errorbuf);
  SETCOLOR (MT_COLOR_NORMAL);
}

void mutt_endwin (const char *msg)
{
  move (LINES-1, COLS-1);
  attrset (A_NORMAL);
  refresh ();
  endwin ();
  if (msg)
  {
    fputc ('\n', stdout);
    puts (msg);
  }
}

void mutt_perror (const char *s)
{
  char *p = strerror (errno);

  mutt_error ("%s: %s (errno = %d)", s, p ? p : "unknown error", errno);
}

int ci_any_key_to_continue (const char *s)
{
  struct termios t;
  struct termios old;
  int f, ch;

  f = open ("/dev/tty", O_RDONLY);
  tcgetattr (f, &t);
  memcpy ((void *)&old, (void *)&t, sizeof(struct termios)); /* save original state */
  t.c_lflag &= ~(ICANON | ECHO);
  t.c_cc[VMIN] = 1;
  t.c_cc[VTIME] = 0;
  tcsetattr (f, TCSADRAIN, &t);
  fflush (stdout);
  if (s)
    fputs (s, stdout);
  else
    fputs ("Press any key to continue...", stdout);
  fflush (stdout);
  ch = fgetc (stdin);
  fflush (stdin);
  tcsetattr (f, TCSADRAIN, &old);
  close (f);
  fputs ("\r\n", stdout);
  return (ch);
}

int mutt_do_pager (char *banner, char *tempfile, int do_color)
{
  int rc;

  if (strcmp (Pager, "builtin") == 0)
    rc = mutt_pager (banner, tempfile, NULL, do_color);
  else
  {
    char cmd[STRING];
    
    endwin ();
    snprintf (cmd, sizeof (cmd), "%s %s", Pager, tempfile);
    mutt_system (cmd);
    mutt_unlink (tempfile);
    rc = 0;
  }

  return rc;
}

int mutt_enter_fname (const char *prompt, char *buf, size_t blen, short *redraw)
{
  int i;

  mvaddstr (LINES-1, 0, (char *) prompt);
  addstr (" ('?' for list): ");
  if (buf[0])
    addstr (buf);
  clrtoeol ();
  refresh ();

  if ((i = ci_getch ()) == ERR)
  {
    CLEARLINE (LINES-1);
    return (-1);
  }
  else if (i == '?')
  {
    refresh ();
    buf[0] = 0;
    mutt_select_file (buf, blen, 0);
    *redraw = REDRAW_FULL;
  }
  else
  {
    char *pc = safe_malloc (strlen (prompt) + 3);

    sprintf (pc, "%s: ", prompt);
    mutt_ungetch (i);
    if (ci_get_field (pc, buf, blen, M_EFILE | M_CLEAR) != 0)
      buf[0] = 0;
    MAYBE_REDRAW (*redraw);
    free (pc);
  }

  return 0;
}

/* FOO - this could be made more efficient by allocating/deallocating memory
 * instead of using a fixed array
 */
void mutt_ungetch (int ch)
{
  if (UngetCount < UngetBufLen) /* make sure not to overflow */
    UngetBuf[UngetCount++] = ch;
}

void mutt_flushinp (void)
{
  UngetCount = 0;
  flushinp ();
}

#if (defined(USE_SLANG_CURSES) || defined(HAVE_CURS_SET))
/* The argument can take 3 values:
 * -1: restore the value of the last call
 *  0: make the cursor invisible
 *  1: make the cursor visible
 */
void mutt_curs_set (int cursor)
{
  static int SavedCursor = 1;
  
  if (cursor < 0)
    cursor = SavedCursor;
  else
    SavedCursor = cursor;
  
  curs_set (cursor);
}
#endif
