/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* This file defines all numerical functions */

#ifdef __GNUC__
#pragma implementation				// gcc: Class implementation
#endif

#include "mysql_priv.h"
#include <m_ctype.h>
#include <time.h>

/* return TRUE if item is a constant */


bool
eval_const_cond(COND *cond)
{
  return ((Item_func*) cond)->val_int() ? TRUE : FALSE;
}


Item_func::Item_func(List<Item> &list)
{
  arg_count=list.elements;
  args=(Item**) sql_alloc(sizeof(Item*)*arg_count);

  uint i=0;
  List_iterator<Item> li(list);
  Item *item;

  while ((item=li++))
    args[i++]= item;
  list.empty();					// Fields are used
}

Item_func::~Item_func()
{
#ifdef DELETE_ITEMS
  for (uint i=0 ; i < arg_count ; i++)
    delete args[i];
  arg_count=0;					// Safeguard
#endif
}

bool
Item_func::fix_fields(THD *thd,TABLE_LIST *tables)
{
  Item **arg,**arg_end;
  char buff[sizeof(double)];			// Max argument in function
  binary=0;
  used_tables_cache=0;

  if (thd && check_stack_overrun(thd,buff))
    return 0;					// Fatal error flag is set!
  if (arg_count)
  {						// Print purify happy
    for (arg=args, arg_end=args+arg_count; arg != arg_end ; arg++)
    {
      if ((*arg)->fix_fields(thd,tables))
	return 1;				/* purecov: inspected */
      if ((*arg)->maybe_null)
	maybe_null=1;
      if ((*arg)->binary)
	binary=1;
      if ((*arg)->type() == SUM_FUNC_ITEM)
	with_sum_func=1;
      else
	with_sum_func= with_sum_func || (*arg)->with_sum_func;
      used_tables_cache|=(*arg)->used_tables();
    }
  }
  fix_length_and_dec();
  return 0;
}


void Item_func::split_sum_func(List<Item> &fields)
{
  Item **arg,**arg_end;
  for (arg=args, arg_end=args+arg_count; arg != arg_end ; arg++)
  {
    if ((*arg)->with_sum_func)
      (*arg)->split_sum_func(fields);
    else if ((*arg)->used_tables() || (*arg)->type() == SUM_FUNC_ITEM)
    {
      fields.push_front(*arg);
      *arg=new Item_ref((Item**) fields.head_ref(),0,(*arg)->name);
    }
  }
}


void Item_func::update_used_tables()
{
  used_tables_cache=0;
  for (uint i=0 ; i < arg_count ; i++)
  {
    args[i]->update_used_tables();
    used_tables_cache|=args[i]->used_tables();
  }
}


ulong Item_func::used_tables() const
{
  return used_tables_cache;
}

void Item_func::print(String *str)
{
  str->append(func_name());
  str->append('(');
  for (uint i=0 ; i < arg_count ; i++)
  {
    if (i)
      str->append(',');
    args[i]->print(str);
  }
  str->append(')');
}


void Item_func::print_op(String *str)
{
  str->append('(');
  for (uint i=0 ; i < arg_count-1 ; i++)
  {
    args[i]->print(str);
    str->append(' ');
    str->append(func_name());
    str->append(' ');
  }
  args[arg_count-1]->print(str);
  str->append(')');
}


String *Item_num_func::str(String *str)
{
  double nr=val();
  if (null_value)
    return 0; /* purecov: inspected */
  else
    str->set(nr,decimals);
  return str;
}

void Item_num_func::fix_length_and_dec()
{
  decimals=0;
  for (uint i=0 ; i < arg_count ; i++)
    set_if_bigger(decimals,args[i]->decimals);
  max_length=float_length(decimals);
}

String *Item_int_func::str(String *str)
{
  longlong nr=val_int();
  if (null_value)
    return 0;
  else
    str->set(nr);
  return str;
}

double Item_func_plus::val()
{
  double val=args[0]->val()+args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val;
}

double Item_func_minus::val()
{
  double val=args[0]->val() - args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val;
}


double Item_func_mul::val()
{
  double val=args[0]->val()*args[1]->val();
  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0; /* purecov: inspected */
  return val;
}


double Item_func_div::val()
{
  double val=args[0]->val();
  double val2=args[1]->val();
  if ((null_value= val2 == 0.0 || args[0]->null_value || args[1]->null_value))
    return 0.0;
  return val/val2;
}

void Item_func_div::fix_length_and_dec()
{
  decimals=max(args[0]->decimals,args[1]->decimals)+2;
  max_length=args[0]->max_length - args[0]->decimals + decimals;
  uint tmp=float_length(decimals);
  set_if_smaller(max_length,tmp);
}


double Item_func_mod::val()
{
  double val= floor(args[0]->val()+0.5);
  double val2=floor(args[1]->val()+0.5);
  if ((null_value=val2 == 0.0 || args[0]->null_value || args[1]->null_value))
    return 0.0; /* purecov: inspected */
  return fmod(val,val2);
}

longlong Item_func_mod::val_int()
{
  longlong val=  args[0]->val_int();
  longlong val2= args[1]->val_int();
  if ((null_value=val2 == 0 || args[0]->null_value || args[1]->null_value))
    return 0; /* purecov: inspected */
  return val % val2;
}

void Item_func_mod::fix_length_and_dec()
{
  max_length=args[1]->max_length;
  decimals=0;
}


double Item_func_neg::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return -val;
}

void Item_func_neg::fix_length_and_dec()
{
  decimals=args[0]->decimals;
  max_length=args[0]->max_length;
}


double Item_func_abs::val()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return fabs(val);
}

double Item_func_log::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val <= 0.0)))
    return 0.0; /* purecov: inspected */
  return log(val);
}

double Item_func_log10::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val <= 0.0)))
    return 0.0; /* purecov: inspected */
  return log10(val);
}

double Item_func_exp::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0; /* purecov: inspected */
  return exp(val);
}

double Item_func_sqrt::val()
{
  double val=args[0]->val();
  if ((null_value=(args[0]->null_value || val < 0)))
    return 0.0; /* purecov: inspected */
  return sqrt(val);
}

double Item_func_pow::val()
{
  double val=args[0]->val();
  double val2=args[1]->val();
  if ((null_value=(args[0]->null_value || args[1]->null_value)))
    return 0.0; /* purecov: inspected */
  return pow(val,val2);
}

// Trigonometric functions

double Item_func_acos::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return fix_result(acos(val));
}

double Item_func_asin::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return fix_result(asin(val));
}

double Item_func_atan::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  if (arg_count == 2)
  {
    double val2= args[1]->val();
    if ((null_value=args[1]->null_value))
      return 0.0;
    return fix_result(atan2(val,val2));
  }
  return fix_result(atan(val));
}

double Item_func_cos::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return fix_result(cos(val));
}

double Item_func_sin::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return fix_result(sin(val));
}

double Item_func_tan::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0.0;
  return fix_result(tan(val));
}


// Conversion functions

void Item_func_integer::fix_length_and_dec()
{
  max_length=args[0]->max_length - args[0]->decimals+1;
  uint tmp=float_length(decimals);
  set_if_smaller(max_length,tmp);
  decimals=0;
}

longlong Item_func_ceiling::val_int()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return (longlong) ceil(val);
}

longlong Item_func_floor::val_int()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return (longlong) floor(val);
}

void Item_func_round::fix_length_and_dec()
{
  max_length=args[0]->max_length;
  decimals=args[0]->decimals;
  if (args[1]->const_item())
  {
    int tmp=args[1]->val_int();
    if (tmp < 0)
      decimals=0;
    else
    {
      set_if_smaller(decimals,(uint) tmp);
    }
  }
}

double Item_func_round::val()
{
  double val=args[0]->val();
  int decimals=args[1]->val_int();
  uint abs_decimals=abs(decimals);

  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0.0;
  double tmp=(abs_decimals < array_elements(log_10) ? 
	      log_10[abs_decimals] : pow(10,abs_decimals));

  if (truncate)
    return decimals < 0 ? floor(val/tmp)*tmp : floor(val*tmp)/tmp;
  return decimals < 0 ? rint(val/tmp)*tmp : rint(val*tmp)/tmp;
}


double Item_func_rand::val()
{
  if (arg_count)
  {					// Only use argument once in query
    ulong tmp=((ulong) args[0]->val_int())+55555555L;
    randominit(&current_thd->rand,tmp,tmp/2);
#ifdef DELETE_ITEMS
    delete args[0];
#endif
    arg_count=0;
  }
  return rnd(&current_thd->rand);
}

longlong Item_func_sign::val_int()
{
  double val=args[0]->val();
  null_value=args[0]->null_value;
  return val < 0.0 ? -1 : (val  > 0 ? 1 : 0);
}


double Item_func_units::val()
{
  double val=args[0]->val();
  if ((null_value=args[0]->null_value))
    return 0;
  return val*mul+add;
}


void Item_func_min_max::fix_length_and_dec()
{
  decimals=0;
  max_length=0;
  maybe_null=1;
  for (uint i=0 ; i < arg_count ; i++)
  {
    if (max_length < args[i]->max_length)
      max_length=args[i]->max_length;
    if (decimals < args[i]->decimals)
      decimals=args[i]->decimals;
    if (!args[i]->maybe_null)
      maybe_null=0;
  }
}


double Item_func_min::val()
{
  double value=0.0;
  null_value=1;
  for (uint i=0; i < arg_count ; i++)
  {
    if (null_value)
    {
      value=args[i]->val();
      null_value=args[i]->null_value;
    }
    else
    {
      double tmp=args[i]->val();
      if (tmp < value)
	value=tmp;
    }
  }
  return value;
}


double Item_func_max::val()
{
  double value=0.0;
  null_value=1;
  for (uint i=0; i < arg_count ; i++)
  {
    if (null_value)
    {
      value=args[i]->val();
      null_value=args[i]->null_value;
    }
    else
    {
      double tmp=args[i]->val();
      if (tmp > value)
	value=tmp;
    }
  }
  return value;
}


longlong Item_func_length::val_int()
{
  String *res=args[0]->str(&value);
  if (!res)
  {
    null_value=1;
    return 0; /* purecov: inspected */
  }
  return (longlong) res->length();
}


longlong Item_func_locate::val_int()
{
  String *a=args[0]->str(&value1);
  String *b=args[1]->str(&value2);
  uint start=0;
  if (!a || !b)
    return 0; /* purecov: inspected */
  if (arg_count == 3)
  {
    start=(uint) args[2]->val_int()-1;
    if (start > a->length() || start+b->length() > a->length())
      return 0;
  }
  if (!b->length())				// Found empty string at start
    return (longlong) (start+1);
  return (longlong) (a->strstr(*b,start)+1) ;
}


longlong Item_func_field::val_int()
{
  String *field;
  if (!(field=item->str(&value)))
    return 0;					// -1 if null
  for (uint i=0 ; i < arg_count ; i++)
  {
    String *tmp_value=args[i]->str(&tmp);
    if (tmp_value && field->length() == tmp_value->length() &&
	!memcmp(field->ptr(),tmp_value->ptr(),tmp_value->length()))
      return (longlong) (i+1);
  }
  return 0;
}


longlong Item_func_ascii::val_int()
{
  String *res=args[0]->str(&value);
  if (!res)
  {
    null_value=1;
    return 0;
  }
  null_value=0;
  return (longlong) (res->length() ? (uchar) (*res)[0] : (uchar) 0);
}


longlong Item_func_period_add::val_int()
{
  ulong period=(ulong) args[0]->val_int();
  int months=(int) args[1]->val_int();

  if ((null_value=args[0]->null_value || args[1]->null_value) ||
      period == 0L)
    return 0; /* purecov: inspected */
  return (longlong)
    convert_month_to_period((uint) ((int) convert_period_to_month(period)+
				    months));
}


longlong Item_func_period_diff::val_int()
{
  ulong period1=(ulong) args[0]->val_int();
  ulong period2=(ulong) args[1]->val_int();

  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0; /* purecov: inspected */
  return (longlong) ((long) convert_period_to_month(period1)-
		     (long) convert_period_to_month(period2));
}


longlong Item_func_to_days::val_int()
{
  null_value=0;
  if (args[0]->result_type() == STRING_RESULT)
  {
    String *str;
    char *pos,*start;
    uint length,date[3];

    if (!(str=args[0]->str(&value)))
    {
      null_value=1; /* purecov: inspected */
      return 0; /* purecov: inspected */
    }
    // Fix at least the following formats
    // YYMMDD, YYYYMMDD,  YY-MM-DD, YYYY-MM-DD

    for (pos=str->c_ptr(); *pos && !isdigit(*pos) ; pos++) ;
    length=(uint) ((str->ptr()+str->length())-pos);
    for (uint i=0 ; i < 3; i++)
    {
      uint tmp_value=0;
      while (!isdigit(*pos) && *pos)
	pos++;
      start=pos;
      while (isdigit(pos[0]) &&
	     ((pos-start) < 2 || ((pos-start) < 4 && i == 0 && length >= 8)))
      {
	tmp_value=tmp_value*10 + (uint) (uchar) (*pos - '0');
	pos++;
      }
      date[i]=tmp_value;
    }
    return (longlong) calc_daynr(date[0],date[1],date[2]);
  }
  else
  {
    double tmp=args[0]->val();
    if (tmp > 99999999.0)
    {						// Probably timestamp
      tmp=floor(tmp/1000000.0);			// Fix to YYYYMMDD
    }
    ulong date=(ulong) tmp;
    if (args[0]->null_value)
    {
      null_value=1; /* purecov: inspected */
      return 0; /* purecov: inspected */
    }
    return (longlong) calc_daynr(date/10000,(date/100)%100,date%100);
  }
}

/* weekday() has a automatic to_days() on item */

longlong Item_func_weekday::val_int()
{
  ulong tmp_value=(ulong) args[0]->val_int();
  uint weekday;
  if ((null_value=args[0]->null_value) || !tmp_value)
    return 0; /* purecov: inspected */

  weekday=calc_weekday(tmp_value);
  return (longlong) (odbc_type  ? ((weekday+1) % 7) +1 : weekday);
}


longlong Item_func_unix_timestamp::val_int()
{
  if (arg_count == 0)
    return (longlong) current_thd->query_start();
  if (args[0]->type() == FIELD_ITEM)
  {						// Optimize timestamp field
    Field *field=((Item_field*) args[0])->field;
    if (field->type() == FIELD_TYPE_TIMESTAMP)
      return ((Field_timestamp*) field)->get_time();
  }
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
  {
    return 0; /* purecov: inspected */
  }
  return (longlong) str_to_timestamp(str->ptr(),str->length());
}


longlong Item_func_time_to_sec::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  ulong tmp=str_to_time(str->ptr(),str->length());
  return (longlong) ((tmp/10000)*3600+((tmp/100)%100)*60+(tmp%100));
}


static char nbits[256] = {
  0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7,
  4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8,
};

longlong Item_func_bit_count::val_int()
{
  ulonglong value= (ulonglong) args[0]->val_int();
  if (args[0]->null_value)
  {
    null_value=1; /* purecov: inspected */
    return 0; /* purecov: inspected */
  }
  ulong v1=(ulong) value;
  ulong v2=(ulong) (value >> 32);
  uint bits= (uint) (uchar) (nbits[(uchar) v1] +
			     nbits[(uchar) (v1 >> 8)] +
			     nbits[(uchar) (v1 >> 16)] +
			     nbits[(uchar) (v1 >> 24)] +
			     nbits[(uchar) v2] +
			     nbits[(uchar) (v2 >> 8)] +
			     nbits[(uchar) (v2 >> 16)] +
			     nbits[(uchar) (v2 >> 24)]);
  return (longlong) bits;
}


/*
** Handle the ANSI interval definition
*/

void Item_interval::fix_length_and_dec()
{
  switch (int_type) {
  case YEAR:		max_length=4; break;	// 0-9999
  case MONTH:		max_length=6; break;	// 0-9999*12
  case DAY:		max_length=7; break;	// 0-9999*366
  case HOUR:		max_length=8; break;	// 0-9999*366*24
  case MINUTE:
  case SECOND:		max_length=11;break;	// Should be enough.
  case YEAR_MONTH:	max_length=7; break;	// YYYY-MM
  case DAY_HOUR:	max_length=9; break;	// DDDDDD HH
  case DAY_MINUTE:	max_length=12; break;	// DDDDDD HH:MM
  case DAY_SECOND:	max_length=15; break;	// DDDDDD HH:MM:SS
  case HOUR_MINUTE:	max_length=11; break;	// HHHHHHHH:MM
  case HOUR_SECOND:	max_length=14; break;	// HHHHHHHH:MM:SS
  case MINUTE_SECOND:	max_length=13; break;	// MMMMMMMMMM:SS
  }
  max_length++;					// place for sign
  str_value.set(buff,sizeof(buff));
  maybe_null=1;					// NULL on errors
}


/*
** Get a array of numbers from a string object.
** The string object may have separators or not. Anything is allowed as a
** separator.  To allow no separators, start from the end from the string.
** All numbers, except the first one, must be 2 characters (or 1 + separator).
** This functions only works for count > 1
*/

static bool get_interval_info(const char *str,uint length,uint count,
			      long *values)
{
  ulong value;
  if (length < (count-1)*2+1 || !isdigit(str[length-1]))
    return 1;
  bool sign=0;
  if (str[0] == '-')
  {
    sign=1;
    str++;
    length--;
  }
  length--;					// For a better offset
  while (--count)
  {
    if (length < 2)
      return 1;
    value= (uint) (str[length--] - '0');
    if (isdigit(str[length]))
    {
      value+=(uint) (str[length--] - '0')*10;
      if (length > 1 && !isdigit(str[length]))
	length--;
    }
    else
      length--;
    if (!isdigit(str[length]))
      return 1;					// Too short or wrong format
    values[count]=value;
  }

  /* Get first value, that may be big */
  const char *end=str+length;
  value=0;
  do
  {
    if (!isdigit(*str))
      return 1;
    value= value*10+ (uint) (*str - '0');
  } while (str++ != end);
  values[0]= sign ? -(long) value : (long) value;
  return 0;
}


/*
** Get an easily manupulatable INTERVAL object from an interval.
** To make code easy, allow interval objects without separators.
*/


bool Item_interval::val_time(INTERVAL *t)
{
  String *res;
  long array[4],value;
  LINT_INIT(res);
  LINT_INIT(value);

  if ((int) int_type <= SECOND)
  {
    value=(long) args[0]->val_int();
    if ((null_value=args[0]->null_value))
      return 1;
  }
  else if (!(res=args[0]->str(&str_value)))
    return ((null_value=1));

  null_value=0;
  bzero(t,sizeof(*t));

  switch (int_type) {
  case YEAR:
    t->year=value;
    break;
  case MONTH:
    t->month=value;
    break;
  case DAY:
    t->day=value;
    break;
  case HOUR:
    t->hour=value;
    break;
  case MINUTE:
    t->minute=value;
    break;
  case SECOND:
    t->second=value;
    break;
  case YEAR_MONTH:				// Allow YEAR-MONTH YYYYYMM
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->year=array[0];
    t->month=array[1];
    break;
  case DAY_HOUR:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    break;
  case DAY_MINUTE:
    if (get_interval_info(res->ptr(),res->length(),3,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    t->minute=array[2];
    break;
  case DAY_SECOND:
    if (get_interval_info(res->ptr(),res->length(),4,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    t->minute=array[2];
    t->second=array[3];
    break;
  case HOUR_MINUTE:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->hour=array[0];
    t->minute=array[1];
    break;
  case HOUR_SECOND:
    if (get_interval_info(res->ptr(),res->length(),3,array))
      return ((null_value=1));
    t->hour=array[1];
    t->minute=array[2];
    t->second=array[3];
    break;
  case MINUTE_SECOND:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->minute=array[2];
    t->second=array[3];
    break;
  }
  return 0;
}
