/* 
 * Based on README.aztcd from the linux-sources (Tiny Audio CD Player)
 *       and CDPlay 0.31 from Sariel Har-Peled
 * There is not much left from the original code.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/cdrom.h>
#include <errno.h>
#include <ncurses.h>

#include "cthugha.h"
#include "cd_player.h"
#include "keys.h"
#include "display.h"
#include "action.h"
#include "sound.h"


/*
 * This file is used for the "normal" cthugha-L and for the cthugha-server.
 * cthugha-server displays no graphic and accepts only a subset of the
 * options cthugha-L supports.
 *
 * If CTH_SERVER is defined, then the cthugha-server version should be 
 * generated.
 */


/* use slitly changed error print macros here */
#ifdef CTH_SERVER

#define sprintfee(fmt, args...)	{					  \
   mvprintw(22,0,"ERROR:");						  \
   refresh();								  \
   printf("\n");							  \
   fprintf(stderr, fmt " (%d - %s)\n" , ## args, errno, strerror(errno)); \
}
#define sprintfe(fmt, args...)	{					  \
   mvprintw(22,0,"ERROR:");						  \
   refresh();								  \
   printf("\n");							  \
   fprintf(stderr, fmt, ## args);					  \
}

#else

#define sprintfee(fmt, args...)	{					  \
   printf("\n");							  \
   fprintf(stderr, fmt " (%d - %s)\n" , ## args, errno, strerror(errno)); \
}
#define sprintfe(fmt, args...)	{					  \
   printf("\n");							  \
   fprintf(stderr, fmt, ## args);					  \
}

#endif

static struct cdrom_tochdr   tocHdr;
static struct cdrom_subchnl  subchnl; 
static struct cdrom_tocentry entry;
static struct cdrom_msf      msf; 
union  { 
    struct cdrom_msf msf;
    unsigned char buf[2336];
} azt;
struct cdrom_volctrl  volctrl;
static int cd_playing = 0;			/* already playing ? */
static int first, last;
  
int cd_handle = -1;				/* handle of /dev/cdrom */
int cd_in_pause = 0;				/* now in pause-mode */
int cd_first_track = -1;			/* start with track nr */
int cd_loop = 0;
int cd_randomplay = 0;
int cd_eject_on_end = 0;
int *random_played;
int cd_stop_on_exit = 0;				
static int ejected=0;
static int no_disk=0;
static int rand_ini=0;
static int stop_track=-1;

/*
 * Initialization. If cd_first_track is >= 0 then start playing at the
 *  specified track.
 */
int init_cd() {

    printfv("Initializing CD...\n");

    cd_handle=open("/dev/cdrom", O_RDONLY);
  
    if (cd_handle <= 0) {
	sprintfee("CD: already playing, no audio disk, door open");
	no_disk = 1;
	return 1;
    }
	
    if ( cd_first_track >= 0) {
	if ( cd_randomplay) {
	    printfv("  Starting random play of CD\n");
	    cd_next(0);
	} else {
	    printfv("  Starting CD at track: %d.\n", cd_first_track);
	    cd_track(cd_first_track);
	}
    }

    if ( cd_loop)
	cd_eject_on_end = 0;

    return 0;
}

/* 
 * Close the CD-Player. 
 * Stop a possibly running CD if that is desired.
 */
int exit_cd() {
    if ( cd_handle == -1)		/* not initialized */
	return 0;

    if ( cd_stop_on_exit && cd_playing)
	cd_stop();
		
    if (close(cd_handle)) {
	sprintfe("Can not close CD.\n");
	return 1;
    }
    return 0;
}

/*
 *  CD-player check for complete routine
 */
int cd_check() {

    if ( ! cd_playing )			/* player is not played -> OK */
	return 0;

    cd_subchnl();			/* get information from CD */
	
    /* Test wether CD-ROM has completed operation, or if the desired
       track has been reached. Not all CD-ROMs give the CDROM_AUDIO_COMPL.
       status (like mine)
    */
    if ( (subchnl.cdsc_audiostatus == CDROM_AUDIO_COMPLETED) /* 'nice' CD */
	|| (subchnl.cdsc_trk >= stop_track)	/* reached next track */
	|| ( (subchnl.cdsc_trk == 0)		/* again from start, */
	    && (stop_track > last) ) ) {	/*  while plaing last track*/
       	
	if(cd_randomplay)
	    cd_next(0);
	else if(cd_loop)
	    cd_track(0);
	else if(cd_eject_on_end)
	    cd_eject();
	else
	    cd_stop();
    }
    return 0;
}

/* 
 *  CD-Player-user-interface (and mixer)
 *  This is much more than a simple CD-Player. It's more a control-panel
 *  for sound-related stuff.
 */
int cd_player() {

#ifndef CTH_SS

    char str[512];
    char str2[1024];
    char vstr1[16], vstr2[16], vstr3[16], vstr4[16];
    static int key, sel = 0;
    int update_snd = 0;
    int skip=0;
    int fast=0;

    char * voltostr(int volume, char * str) {
	if ( volume < 0) {
	    strcpy( str, "default");
	    return str;
	}
	sprintf(str, "%d", volume & 255);
	return str;
    }

#ifndef CTH_SERVER
    do {
#endif
	cd_check();

	if ( subchnl.cdsc_audiostatus == CDROM_AUDIO_PLAY )
	    sprintf(str, 
		    "CD-Player Track: %d %d:%02d (%d - %d) %d:%02d",
		    subchnl.cdsc_trk,
		    subchnl.cdsc_reladdr.msf.minute,
		    subchnl.cdsc_reladdr.msf.second,
   		    first, last,
		    subchnl.cdsc_absaddr.msf.minute,
		    subchnl.cdsc_absaddr.msf.second);
	else if ( subchnl.cdsc_audiostatus == CDROM_AUDIO_PAUSED )
	    sprintf(str, 
		    "CD-Player Track: %d %d:%02d (%d - %d) %d:%02d pause",
		    subchnl.cdsc_trk,
		    subchnl.cdsc_reladdr.msf.minute,
		    subchnl.cdsc_reladdr.msf.second,
   		    first, last,
		    subchnl.cdsc_absaddr.msf.minute,
		    subchnl.cdsc_absaddr.msf.second);
	else
	    if(ejected)
		sprintf(str,"CD-Player: ejected");
	    else if (no_disk)
		sprintf(str,"CD-Player: no disk in drive");
	    else
		sprintf(str,"CD-Player Track: stopped");

	sprintf(str2,
		"%s\n"
		"<- prev(7)  pause(8)  -> next(9) \n"
		"<< rev(4)   stop(5)   >> fwd(6)  \n"
		"            %s             \n\n"
		"%cAutoplay:       %10s%c\n"
		"%cPlayloop:       %10s%c\n"
		"%cRandomplay:     %10s%c\n"
		"%cEject On End:   %10s%c\n"
		"\n"
		"%cVolume CD:      %10s%c\n"
		"%cVolume Line-In: %10s%c\n"
		"%cVolume MIC:     %10s%c\n"
		"%cVolume rec:     %10s%c\n"
		"%cSample rate:    %10d%c\n"
		"%cChannels:       %10s%c\n",
		str, ejected ? "close(2)":"eject(2)",
		
		(sel == 0) ? '>' : ' ',
		(cd_first_track == -1) ? "off" : "on",
		(sel == 0) ? '<' : ' ',

		(sel == 1) ? '>' : ' ',
		(cd_loop) ? "on" : "off",
		(sel == 1) ? '<' : ' ',

		(sel == 2) ? '>' : ' ',
		(cd_randomplay) ? "on" : "off",
		(sel == 2) ? '<' : ' ',

		(sel == 3) ? '>' : ' ',
		(cd_eject_on_end) ? "on" : "off",
		(sel == 3) ? '<' : ' ',

		(sel == 4) ? '>' : ' ',	
		voltostr( sound_volume_cd, vstr1),
		(sel == 4) ? '<' : ' ',
		
		(sel == 5) ? '>' : ' ', 
		voltostr( sound_volume_line, vstr2),
		(sel == 5) ? '<' : ' ',

		(sel == 6) ? '>' : ' ', 
		voltostr( sound_volume_mic, vstr3),
		(sel == 6) ? '<' : ' ',

		(sel == 7) ? '>' : ' ', 
		voltostr( sound_volume_rec, vstr4),
		(sel == 7) ? '<' : ' ',
		
		(sel == 8) ? '>' : ' ',
		sound_sample_rate,
		(sel == 8) ? '<' : ' ',

		(sel == 9) ? '>' : ' ',
		(sound_stereo == 1) ? "mono" : "stereo",
		(sel == 9) ? '<' : ' '
	    );


#ifndef CTH_SERVER
	display_print(str2, 0,1);

	display_sound();
	smooth_setpalette();

#else
	mvprintw(0,0,"Cthugha-server Control Panel\n"
		     "----------------------------\n");
	printw(str2);
	printw("\n\n\n");
#endif
	update_snd = 0;
	skip = 0;
	fast = 0;
	while( (key = getkey()) != Z_NOKEY) {

	    switch( key ) {
	    case Z_CD_NEXT:
		skip ++;
		break;
	    case Z_CD_PREV:
		skip --;
		break;
	    case Z_CD_FWD:
		fast += 15;
		break;
	    case Z_CD_REV:
		fast -= 15;
		break;
	    case Z_CD_PAUSE:
		cd_pause();
		break;
	    case Z_CD_STOP:
		cd_stop();
		break;
	    case Z_CD_EJECT:
		cd_eject();
		break;

	    case Z_UP:
		if( -- sel < 0)
		    sel = 9;
		break;
	    case Z_DOWN:
		if ( ++ sel > 9)
		    sel = 0;
		break;
	    case Z_PLUS: case Z_RIGHT:
		/* change some settings for CD player */
		switch( sel) {
		case 0:
		    cd_first_track = 1;
		    break;
		case 1:
		    cd_loop = 1;
		    cd_eject_on_end = 0;
		    break;
		case 2:
		    cd_randomplay = 1;
		    break;
		case 3:
		    cd_eject_on_end = 1;
		    cd_loop = 0;
		    break;
		/* increment the volume for the given input-source
		   the volume is incremented for the left and the right 
		   channel */
		case 4:
		    sound_volume_cd < 0 ? 
			(sound_volume_cd = 25700):
			(sound_volume_cd += 257);	
		    sound_source |= SNDSRC_CD;
		    break;
		case 5:
		    sound_volume_line < 0 ? 
			(sound_volume_line = 25700):
			(sound_volume_line += 257);	
		    sound_source |= SNDSRC_LINE;
		    break;
		case 6:
		    sound_volume_mic < 0 ? 
			(sound_volume_mic = 25700):
			(sound_volume_mic += 257);
		    sound_source |= SNDSRC_MIC;
		    break;
		case 7:
		    sound_volume_rec += 257;	
		    break;
		case 8:
		    sound_sample_rate += 1000;
		    break;
		case 9:
		    sound_stereo = (sound_stereo == 1) ? 2 : 1;
		    break;
		}
		update_snd = (sel > 3)?1:0;
		break;
	    case Z_MINUS: case Z_LEFT:
		switch( sel) {
		case 0:
		    cd_first_track = -1;
		    break;
		case 1:
		    cd_loop = 0;
		    break;
		case 2:
		    cd_randomplay = 0;
		    if ( random_played) {
			free(random_played);
			random_played = NULL;
		    }
		    cd_track(subchnl.cdsc_trk);
		    break;
		case 3:
		    cd_eject_on_end = 0;
		    break;
		case 4:
		    if ( (sound_volume_cd -= 257) < 0)  {
			sound_volume_cd = 0;
			sound_source &= ~SNDSRC_CD;
		    }
		    break;
		case 5:
		    if ( (sound_volume_line -= 257) < 0) {
			sound_volume_line = 0;
			sound_source &= ~SNDSRC_LINE;
		    }
		    break;
		case 6:
		    if ( (sound_volume_mic -= 257) < 0) {
			sound_volume_mic = 0;
			sound_source &= ~SNDSRC_CD;
		    }
		    break;
		case 7:
		    if ( (sound_volume_rec -= 257) < 0) {
			sound_volume_rec = 0;
			sound_source &= ~SNDSRC_CD;
		    }
		    break;
		case 8:
		    if ( (sound_sample_rate -= 1000) <= 0)
			sound_sample_rate = 0;
		    break;
		case 9:
		    sound_stereo = (sound_stereo == 1) ? 2 : 1;
		    break;
		}
		update_snd = (sel > 3)?1:0;
		break;

#ifndef CTH_SERVER
	    case Z_ESC:				/* Exit on 'q' and ESC */
		if ( update_snd) {
		    init_sound_mixer();
		    update_dsp();
		}
		display_print(NULL, 0,1);
		return -1;
#else
	    default:
		return key;
#endif
	    }
	}
	if ( update_snd) {
	    init_sound_mixer();
	    update_dsp();
	}
	if ( skip != 0) {
#ifdef CTH_SERVER
	    printw("please wait...");
	    refresh();
#endif
	    cd_next(skip);
	}
	if ( fast != 0) {
#ifdef CTH_SERVER
	    printw("please wait...");
	    refresh();
#endif
	    cd_fast(fast);
	}

#ifndef CTH_SERVER
    } while(1);
#endif

#endif
    return 0;
}


/*****************************************************************************
 *  Primitives to access the CD:
 */

int rand_track(void)
{
    int num,loop=0;
    
    if(!rand_ini){
	srand((unsigned int)time((long *)NULL));
	rand_ini++;
    }

    num=rand()%last;
    while(random_played[num]){
	if(++num >= last){
	    num=first-1;
	    loop++;
	}
	if(loop>2)
	    break;
    }
    return loop>2?last+1:num+1;
}

#define CD_INIT    if ( cd_handle == -1) if( init_cd() ) return 1;


/* eject the CD */
int cd_eject() 
{
    int ret;
    CD_INIT

    if(cd_randomplay && random_played) {
	free(random_played);
	random_played = NULL;
    }
    cd_playing=0;

    if(ejected) {
	no_disk=ejected=0;
	ret = ioctl(cd_handle,CDROMSTART);	/* close the door ;) */
	if ( !ret && cd_first_track >= 0) {
	    if(!cd_randomplay)
		cd_track(cd_first_track);
	    else
		cd_next(0);
	}
    }else{
	cd_stop();
	ret = ioctl(cd_handle,CDROMEJECT);
	ejected = ! ret;
    }
    return ret;
}

/* pause CD */
int cd_pause() {
	
    if(ejected || no_disk)
	return 0;

    if (!cd_playing || (cd_handle == -1))
	return 0;			/* ignore if not started playing */

    cd_subchnl();

    if (subchnl.cdsc_audiostatus == CDROM_AUDIO_PAUSED ) {
	cd_in_pause = 0;
	if (ioctl(cd_handle,CDROMRESUME)) 
	    return 1;
    } else {
       	cd_in_pause = 1;
	if (ioctl(cd_handle,CDROMPAUSE)) 
	    return 1;
    }
    return 0;
}

/* stop CD */
int cd_stop() {
    CD_INIT;

    if(ejected || no_disk)
	return 0;

    cd_playing = 0;

    ioctl(cd_handle,CDROMSTOP);

    return 0;
}

int cd_readtoc() {
    int cmd;

    if(ejected || no_disk)
	return 0;

    cmd=CDROMREADTOCHDR;
    if (ioctl(cd_handle,cmd,&tocHdr)) {
	sprintfe("Can't read TOC\n");
	return 1;
    }
    first=tocHdr.cdth_trk0;
    last= tocHdr.cdth_trk1;
    if ((first==0)||(first>last)) {
	sprintfe("Illegal TOC\n");
	return 1;
    } 
    /* set end-time in msf for cd_fast() */
    entry.cdte_track = CDROM_LEADOUT;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	sprintfe("Can't read TOC\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    return 0;
}

/* 
 * start play starting at track 
 */
int cd_track(int track) {

    if(ejected || no_disk)
	return 0;

    cd_stop();
		
    cd_readtoc();

    if (track < first)
	track=first;
    if (track > last) 
	track=last;

    entry.cdte_track = track;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	sprintfee("Can't read TOC\n");
	return 1;
    }

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec0   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame0 = entry.cdte_addr.msf.frame;

    /* set stop_track */
    stop_track = last + 1;
    
    /* set end-time to end of CD */
    entry.cdte_track = CDROM_LEADOUT;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	sprintfe("Can't read TOC\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    if( ioctl(cd_handle, CDROMSTART)) {
	sprintfee("Can't start CD");
	return 1;
    }
    if (ioctl(cd_handle,CDROMPLAYMSF,&msf)) {
	sprintfe("Can't play CD.\n");
	return 1;
    }
    cd_playing=1;

    return 0;
}

/*
 *  Read information from CD-ROM (audio-status, trk, addr)
 */
int cd_subchnl() {

    if ( cd_handle == -1)
	if(init_cd())
		return 0;
    if ( ! cd_playing )
	cd_readtoc();

    subchnl.cdsc_format=CDROM_MSF;
    if (ioctl(cd_handle,CDROMSUBCHNL,&subchnl)) {
	subchnl.cdsc_audiostatus=CDROM_AUDIO_NO_STATUS;
	if(ejected)
	    return 0;
	else if(no_disk)
	    return 0;
	sprintfee("Can't SUBCHNL CD-Rom.");
	no_disk=1;
	cd_playing=0;
	return 1;
    }
    if(ejected || no_disk){
	no_disk=ejected=0;
	if(cd_first_track >= 0)
	    if(!cd_randomplay)
		cd_track(cd_first_track);
	    else
		cd_next(0);
    }
#ifdef CTH_SERVER
    printfv("\nAudioStatus:%s (%d)   Track:%d  Mode:%d   MSF=%d:%d:%d       \n",
	    subchnl.cdsc_audiostatus==CDROM_AUDIO_PLAY ? 
	    "PLAYING":"NOT PLAYING",
	    subchnl.cdsc_audiostatus,
	    subchnl.cdsc_trk,subchnl.cdsc_adr, 
	    subchnl.cdsc_absaddr.msf.minute, subchnl.cdsc_absaddr.msf.second, 
	    subchnl.cdsc_absaddr.msf.frame);
#endif

    if( (subchnl.cdsc_audiostatus != CDROM_AUDIO_PLAY) &&
       (subchnl.cdsc_audiostatus != CDROM_AUDIO_PAUSED) )
	subchnl.cdsc_trk = 0;
    else if( !cd_playing)
	cd_playing++;
	
    return 0;
}

/* go to next track */
int cd_next(int skip) {
    int track;
    CD_INIT;

    if(ejected || no_disk)
	return 0;

    if (!cd_playing++) {		/* CD not playing */
	cd_readtoc();				/* get table of contents */

	track=cd_randomplay ? rand_track()	/* random track */
	    : first;				/* start at first track */

	if(cd_randomplay) {
	    if ( random_played)
		free( random_played);
	    random_played=calloc(last-first+1,sizeof(int));
	}
    } else {				/* CD is already playing */
	cd_subchnl();				/* get information from CD */

	cd_stop();
	track = cd_randomplay ? rand_track() 	/* next random track  */
	    : (subchnl.cdsc_trk + skip);	/* increment or drecrement */
    }
    if (cd_randomplay) {
	if(track>last)
	    if(cd_loop){
		register z;
		for(z=0;z<last;z++)
		    random_played[z]=0;
		track=rand_track();				
	    }
	    else if(cd_eject_on_end){
		cd_eject();
		return 0;
	    }
	    else{
		cd_stop();
		return 0;
	    }
	
	random_played[track-1]=1;
    }
    /* Check for first and last track */
    if ( track > last) 
	track=first;
    if ( track < first)
	track=last;

    entry.cdte_track = track;
    entry.cdte_format = CDROM_MSF;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	sprintfe("Can't read TOC entry\n");
	return 1;
    }

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec0   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame0 = entry.cdte_addr.msf.frame;

    entry.cdte_format = CDROM_MSF;
    stop_track = entry.cdte_track = 
    	( !cd_randomplay || track==last ) ?CDROM_LEADOUT:track+1;
    if (ioctl (cd_handle, CDROMREADTOCENTRY, &entry)) {
	sprintfe("Can't read TOC entry\n");
	return 1;
    }
    msf.cdmsf_min1   = entry.cdte_addr.msf.minute;
    msf.cdmsf_sec1   = entry.cdte_addr.msf.second;
    msf.cdmsf_frame1 = entry.cdte_addr.msf.frame;

    if( ioctl(cd_handle, CDROMSTART)) {
	sprintfee("Can't start CD");
	return 1;
    }

    printf("Playing from %d:%d to %d:%d\n", 
	   msf.cdmsf_min0,msf.cdmsf_sec0,
	   msf.cdmsf_min1,msf.cdmsf_sec1);

    if (ioctl(cd_handle,CDROMPLAYMSF,&msf))
	printf("Drive Error\n");
    cd_playing=1;
    
    return 0;
}

int cd_fast(int skip) {
    int pos;

    CD_INIT;

    cd_subchnl();			/* get information from CD */

    if(ejected || no_disk)
	return 0;

    /* calculate the current position in frames */
    pos = subchnl.cdsc_absaddr.msf.minute * CD_SECS * CD_FRAMES +
	  subchnl.cdsc_absaddr.msf.second * CD_FRAMES +
	  subchnl.cdsc_absaddr.msf.frame;

    /* Skip by seconds */
    pos += skip * CD_FRAMES;
    if ( pos < 0)			/* play first if a begin */
	return cd_randomplay?cd_next(0):cd_track(0);

    /* subtract the skip-value from the current position */
    msf.cdmsf_min0   = pos / (CD_SECS*CD_FRAMES);
    msf.cdmsf_sec0   = (pos / CD_FRAMES) % CD_SECS;
    msf.cdmsf_frame0 = pos % CD_FRAMES;

    /* we don't set end values - it is already set before, but check */
    if(((msf.cdmsf_min1 * 60 + msf.cdmsf_sec1)*CD_FRAMES+msf.cdmsf_frame1) 
       < pos ){
	sprintfe("Can't skip, CD untouched");
	return 0;
    }
    /* Play it */
    if( ioctl(cd_handle, CDROMSTART)) {
	sprintfee("Can't start CD");
	return 1;
    }
    if (ioctl(cd_handle,CDROMPLAYMSF,&msf)) { 
	sprintfee("Drive error or invalid address\n");
    }

    cd_subchnl();			/* get information from CD */
    return 0;
}

/*
int cd_volume(int volume) {
    cmd=CDROMVOLCTRL;
    printf("--Channel 0 Left  (0-255): ");
    scanf("%d",&arg1);
    printf("--Channel 1 Right (0-255): ");
    scanf("%d",&arg2);
    volctrl.channel0=arg1;
    volctrl.channel1=arg2;
    volctrl.channel2=0;
    volctrl.channel3=0;
    if (ioctl(handle,cmd,&volctrl)) 
	{ printf("Drive error or unsupported command\n");
      }
    break;  
}
*/
