/****************************************************************************/
/*                                                                          */
/* IMG* Image Processing Tools Library                                      */
/* Program:   imgFoveate.c                                                  */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Thu Oct 20 20:45:42 1994                                      */
/* Copyright (C) 1994 Simon A.J. Winder                                     */
/*                                                                          */
/****************************************************************************/

#include "tools.h"

#define PRGNAME "Foveate"
#define ERROR(e) imgError(e,PRGNAME)

int main(int argc,char **argv)
{
  it_image *f1,*f2;
  it_float *flt_ptr,*in_ptr;
  int width,height,x,y,xx,yy,centx,centy,centset;
  int xdist,ydist,krad,kx,kxs,ky,kysqd,xstart,ystart,xend,yend;
  double sum,sumk,incsig,centsig,sigma,const1;
  double kern,val,max,min;

  IFHELP
    {
      fprintf(stderr,"img%s - Foveally smooth image with Gaussian kernel\n",
	      PRGNAME);
      fprintf(stderr,"img%s [incr [sigma [x y]]]\n",
	      PRGNAME);
      fprintf(stderr,"  stdin: Float\n");
      fprintf(stderr,"  stdout: Float\n");
      fprintf(stderr,"  incr is the delta_sigma/pixel increment rate\n");
      fprintf(stderr,"  sigma is the e^(-0.5) radius at the foveal centre\n");
      fprintf(stderr,"  xy is the foveal centre position\n");
      exit(0);
    }

  imgStart(PRGNAME);
  centx=centy=0; /* make gcc happy */
  centset=0;
  centsig=0.0;
  incsig=0.025;
  if(argc>5)
    ERROR("too many arguments");
  if(argc>1)
    incsig=atof(argv[1]);
  if(argc>2)
    centsig=atof(argv[2]);
  if(argc>3)
    {
      if(argc!=5)
	ERROR("x and y must both be specified");
      centx=atoi(argv[3]);
      centy=atoi(argv[4]);
      centset=1;
    }

  /* Loop for all images */
  do {
    f1=i_read_image_file(stdin,IT_FLOAT,IM_FRAGMENT);
    if(f1==NULL)
      ERROR("can't import image file");
    width=f1->width;
    height=f1->height;
    if(!centset)
      {
	centx=width/2;
	centy=height/2;
      }
    f2=i_create_image(width,height,IT_FLOAT,IM_FRAGMENT);
    if(f2==NULL)
      ERROR("out of memory");

    imgInitMinMax(min,max);
    for(y=0;y<height;y++)
      {
	fprintf(stderr,"[%d]",y);
	flt_ptr=im_float_row(f2,y);
	for(x=0;x<width;x++)
	  {
	    /* Calculate amount of smoothing necessary for this x,y */
	    xdist=x-centx;
	    ydist=y-centy;
	    sigma=centsig+incsig*sqrt((double)(xdist*xdist+ydist*ydist));
	    if(sigma==0.0)
	      const1= -100.0;
	    else
	      const1= -1.0/(2.0*sigma*sigma);
	    krad=(int)(3.0*sigma+0.5);

	    /* Inner loop limits */
	    kxs=ky= -krad;
	    xstart=x-krad;
	    if(xstart<0)
	      {
		kxs= -x;
		xstart=0;
	      }
	    xend=x+krad;
	    if(xend>=width)
	      xend=width-1;
	    ystart=y-krad;
	    if(ystart<0)
	      {
		ky= -y;
		ystart=0;
	      }
	    yend=y+krad;
	    if(yend>=height)
	      yend=height-1;

	    /* Inner loop */
	    sum=0.0;
	    sumk=0.0;
	    for(yy=ystart;yy<=yend;yy++,ky++)
	      {
		kysqd=ky*ky;
		in_ptr=im_float_row(f1,yy)+xstart;
		for(kx=kxs,xx=xstart;xx<=xend;xx++,kx++)
		  {
		    kern=exp(const1*(double)(kx*kx+kysqd));
		    sumk+=kern;
		    sum+=(kern*(*in_ptr++));
		  }
	      }
	    
	    /* Normalise sum so that kernel integrates to unity */
	    val=sum/sumk;
	    if(val>max)
	      max=val;
	    if(val<min)
	      min=val;
	    *flt_ptr++ =val;
	  }
      }
    fprintf(stderr,"\n");

    f2->min_value=min;
    f2->max_value=max;
    f2->valid_x=f1->valid_x;
    f2->valid_y=f1->valid_y;
    f2->valid_width=f1->valid_width;
    f2->valid_height=f1->valid_height;

    i_destroy_image(f1);
    i_write_image_file(stdout,f2,IF_BINARY);
    i_destroy_image(f2);
  } while(!feof(stdin));

  imgFinish(PRGNAME);
  return(0);
}
/* Version 1.0 (Oct 1994) */
/* Version 1.1 (Nov 1994) */
