/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
/*
**  output.c
*/

#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "sources.h"
#include "message.h"
#include "output.h"
#include "buffer.h"
#include "edit.h"
#include <ctype.h>
#include <memory.h>
#include "intstack.h"

#define MAXSIZE    5000

#define NR_CURSOR     3
#define NR_MARK       4
#define MAX_SIGN_SIZE 5
#define CWH CURSORWIDTH/2
#define CHH CURSORHEIGHT/2

static XPoint cursor_h[] = {{0, 0},    {CWH,   CHH},   {-CWH,   CHH}};
static XPoint cursor[]   = {{0, 0},    {CWH+1, CHH+1}, {-CWH-1, CHH+1}};
/*static XPoint mark_h[]   = {{0, -CHH}, {CWH, 0}, {0, CHH}, {-CWH, 0}}; */
static XPoint mark[]     = {{0, -CHH}, {CWH, 0}, {0, CHH}, {-CWH, 0}};


static XPoint *sign[] = { cursor, mark, cursor_h, mark };
static int sign_size[] = { NR_CURSOR, NR_MARK, NR_CURSOR, NR_MARK };
static Bool sign_filled[] = { True, True, False, False };

#define LATEX_TAB(c)       (latex_tab[Num2Tab(c)])
#define ASCII_TAB(c)       (ascii_tab[Num2Tab(c)])
#define SYMBOL_TAB(c)      (symbol_tab[Num2Tab(c)])
#define LATEX_PH(c)        (latex_ph[Ph2Num(c)])
#define ASCII_PH(c)        (ascii_ph[Ph2Num(c)])
#define SYMBOL_PH(c)       (symbol_ph[Ph2Num(c)]|0x100)
#define LATEX_PHNUM(c)     (latex_phnum[Num(c)])
#define TEXTOPEN           (Font2Char(1, 183))
#define TEXTCLOSE          (Font2Char(1, 183))
#define LASTFONT(A,B)      (((A)<<8)+128+(B))


static
char *latex_tab[41] = { "\\\\\n\t", "\\=",  "\t\\>",    "\\<",
			      "\\+",      "\\-",  "\\push",   "\\pop",
			      "\n\t",     "",     "",         "",
			      "",         "}",    "",         "}",
			      "{",        "}",    "{",        "}",
			      "{",        "}",    "\\hfill",  "\\hrulefill",
			      "\\dotfill", "\\stackrel{", "\\stackrel{",
			      "}{\\stackrel{", "}{", "}}", "",
			      "\\begin{mptabbing}", "\\end{mptabbing}",
			      "\\begin{mpdisplay}", "\\end{mpdisplay}", "|",
			      "\\ms{%i}", "", "", "", "" };

static
char *ascii_tab[41] = { "\n", "", "\t", "",   "", "", "", "",
			      "\n", "", "",   "",   "", "", "", "",
			      "",   "", "",   "",   "", "", "", "",
			      "",   "", "",   "",   "", "", "", "",
			      "",   "", "",   "",   "", "", "", "", ""  };

static
char *symbol_tab[41] = { 0,   "[set]", "[tab]", "[back]",
			       "[plus]", "[minus]", "[push]", "[pop]",
			       "[NwLn]", "[TEXT]", "[text]", "[math]",
			       "[disp]", "]", "[Hide:", "]",
			       "[Top]", "[top]", "[Bot]", "[bot]",
			       "[Gap]", "[gap]", "[Fill]", "[Line]", "[Dots]",
			       "[StackB:", "[StackC:", ":", ":", "]", "]",
			       "[Tabbing:", "]", "[Display:", "]", "[Bar]",
			       "[Space]", "[MATH]", "[BOTH]", "[Name:", "]" };
static
char *latex_ph[7] = { "\\mpE ", "\\mpO ", "\\mpI ", "\\mpV ", 
			    "\\mpT ", "\\mpL ", "\\mpD " };

static
char *ascii_ph[7] = { "Expr", "Op", "Id", "Var",
			    "Text", "List", "Disp" };

static
Char symbol_ph[7] = {  'E', 'O', 'I', 'V', 'T', 'L', 'D'         };

static
char *latex_phnum[16] = { "", "_1", "_2", "_3", "_4", "_5", "_6", "_7",
				"_8", "_9", "_{10}", "_{11}",
				"_{12}", "_{13}", "_{14}", "_{15}"};

#define FONTSYMBOL  "["
#define SIZESYMBOL  "[SizeA"
#define RELSIZESYMBOL "[SizeR"
#define NAMEDHEAD    "\\mpN{"
#define NAMEDTAIL    "}"

static
char *size_latex[10] = {
    "{\\tiny ",   "{\\scriptsize ", "{\\footnotesize ",
    "{\\small ",  "{\\normalsize ", "{\\large ",
    "{\\Large ",  "{\\LARGE ",      "{\\huge ",
    "{\\Huge " };

#define FONTLATEX(A) ((A)>3? font_latex[0]:font_latex[A])
#define SIZELATEX(A) ((A)>128? ((A)>197? size_latex[9]: \
				((A)<188? size_latex[0]: \
				 size_latex[(A)-188]))\
		      :  ((A)>69? size_latex[9]: \
			 ((A)<60?size_latex[0]: size_latex[(A)-60])))

static Window output_win = 0;
static Window message_win = 0;
static Window test_win = 0;
static Bool drindex = False;
static char messtxt[400];
static char messlen = 0;
static Bool messcurs = False;
static int display_delta = 0;
static int def_thinspace=0;

#define PLACENAMESIZE 40
static char placename[PLACENAMESIZE+1];
static int placepos=0;
static int nameset=0;
static int inplacename=0;

static int base_x;
static int base_y;
static int move_x;
#define move_y  line_height()

static int maxlen;
static char *string;
static int maxitem;
static XTextItem *items;
static int initem;
static int limitchar;
static int lastfont=LASTFONT(255,0);
static int instring;
static int left_margin;
static int smart_mode = INVISIBLE;
static int smart_height = 0;
static int search_x, search_y;
static void (*search_func)(void*) = NULL;
static Bool can_backtab;

static int base_color = 0;
static int base_style = 0;
static int identifier_font = 3;
static int hiding = 0;

#define TextBox  1
#define BackBox   2
#define DefaultBox 3
#define NodeBox    4
#define NewlineBox 5
#define StackCenterBox 6
#define StackBaseBox 7
#define BarBox 8

#define IsStackBox(A) ((A)==StackCenterBox || (A)==StackBaseBox)

#define GapBox 16
#define TopBox 17
#define BottomBox 18
#define IsFieldBox(A) (((A)&0x1F0)==0x10)

#define StippleBox 32
#define LineBox    33
#define SpaceBox   34
#define IsLineBox(A)  (((A)&0x1F0)==0x20)

#define CursorBox 64
#define IsCursorBox(A)  (((A)&0x1F0)==0x40)
#define CursorKind(A)   ((A)&0x03)

#define EmptyBox(A) (IsFieldBox(A)||IsLineBox(A)||IsCursorBox(A)|| \
		     (A)==NodeBox||(A)==NewlineBox)

typedef struct BOX BOX;

struct BOX {
    void *funcarg;
    short width, ascent, descent, x, y, itemnr,
          itemtot, lasc, ldesc, lright, maxwidth, special,size,fontnr;
    char color, style;
    BOX *fbox, *lbox, *nbox;
};

static BOX *freebox = NULL;
static BOX *boxstack[1000];
static void *funcargs[1000];
static BOX *cursorstack = NULL;
static INTSTACK *savestack=NULL;
static int bsd = 0;

INTSTACK *fontstack = NULL;
INTSTACK *sizestack = NULL;

#define pushfont(A) push_int(&fontstack,A)
#define popfont()   pop_int(&fontstack)
#define headfont()  head_int(fontstack)
#define pushsize(A) push_int(&sizestack,A)
#define popsize()   pop_int(&sizestack)
#define headsize()  head_int(sizestack)

static void init_item(int i)
{
    items[i].chars  = &string[instring];
    items[i].nchars = 0;
    items[i].delta  = 0;
    items[i].font   = None;
    limitchar = maxlen - i*2;
}

static char spaces[1000];

static void print_box(BOX *b, int d)
{
    BOX *h;

    spaces[d]='\0';
    fprintf(stdout,"%s(X%i,Y%i)(W%i,H%i):(A%i,D%i)[A%i,D%i,R%i]:S%i:[F%i,S%i]\n",
	    spaces, b->x, b->y, b->width, b->ascent+b->descent,b->ascent,
	    b->descent, b->lasc, b->ldesc, b->lright, b->special, b->fontnr,
	    b->size);
    spaces[d]=' ';
    h = b->fbox;
    while (h) {
	print_box(h, d+1);
	h = h->nbox;
    }
}

static BOX *new_box(void)
{
    BOX *t=freebox;
    if (t)
	freebox = freebox->nbox;
    else
	t = (BOX*) malloc(sizeof(BOX));
    memset(t,0,sizeof(BOX));
    return t;
}

static void free_box(BOX *b)
{
    BOX *h;

    if (!b) return;
    h=b;
    b->lbox = NULL;
    while (h) {
	if (h->fbox) {
	    h->fbox->lbox = h;
	    h = h->fbox;
	} else if (h->nbox) {
	    h->nbox->lbox = h->lbox;
	    h = h->nbox;
	} else if (h->lbox) {
	    h->nbox = freebox;
	    freebox = h->lbox->fbox;
	    h = h->lbox;
	    h->fbox = NULL;
	} else {
	    h->nbox = freebox;
	    freebox = b;
	    h = NULL;
	}
    }
}

static void close_box(void);
static int split_text=False;
static int after_node=False;

static void close_textbox(void)
{
    if (bsd && boxstack[bsd-1]->special==TextBox)
	close_box();
}

static BOX *open_box(int special)
{
    BOX *b;

    if (bsd>=1000) return NULL;
    if (bsd && special==TextBox && boxstack[bsd-1]->special==TextBox)
	return boxstack[bsd-1];
    close_textbox();
    b = new_box();

    boxstack[bsd] = b;
    b->color = base_color;
    b->style = base_style;
    if (bsd) {
	b->fontnr = boxstack[bsd-1]->fontnr;
	b->size = boxstack[bsd-1]->size;
	b->x = boxstack[bsd-1]->width;
	b->funcarg = boxstack[bsd-1]->funcarg;
	b->lasc = boxstack[bsd-1]->lasc;
	b->ldesc = boxstack[bsd-1]->ldesc;
	if (special==TextBox) {
	    b->lright = boxstack[bsd-1]->lright;
	    boxstack[bsd-1]->lright = 0;
	}
    }
    b->special = special;
    switch (special) {
    case DefaultBox:
	b->ascent = font_ascent(0,0);
	b->descent = font_descent(0,0);
	b->lasc = font_ascent(0,0);
	b->ldesc = font_descent(0,0);
	break;
    default:
	b->ascent = -200;
	b->descent = -200;
	break;
    }
    bsd++;
    return b;
}

static int count_lines(BOX *b)
{
    int i=0;
    if (!b) return 0;
    b=b->fbox;
    while (b) {
	i += (IsLineBox(b->special));
	b=b->nbox;
    }
    return i;
}

static void expand_gap(BOX *gap, int width)
{
    if (!gap || gap->width>=width)
	return;
    else {
	BOX *h;
	int nr=0;
	int i,j,w;

	nr = count_lines(gap);
	w = width-gap->width;
	if (!nr)
	    gap->x = w/2;
	else {
	    j=0;
	    h=gap->fbox;
	    i=0;
	    while (h) {
		h->x=j;
		if (IsLineBox(h->special)) {
		    j+= h->width = (w+i)/nr;
		    i++;
		    h->y = (gap->ascent-gap->descent)/2;
		    if (h->special!=SpaceBox) {
			switch (gap->ascent-gap->descent) {
			case 0: gap->ascent++;
			case 1: gap->descent++;
			case 2: gap->ascent++;
			default: break;
			}
		    }
		    h->ascent = h->descent=0;
		} else
		    j+= h->width;
		h = h->nbox;
	    }
	    gap->width = width;
	}
    }
}

static void add_space_boxes(BOX *b, int width)
{
    int i = count_lines(b);
    BOX *h;

    if (!b || b->width>=width) return;
    if (!i) {
	/* add space right */
	h = new_box();
	h->special=SpaceBox;
	h->color = b->color;
	h->style = b->style;
	if (b->lbox) b->lbox->nbox=h; else b->fbox = h;
	b->lbox = h;
	/* add space left */
	h=new_box();
	h->special=SpaceBox;
	h->color = b->color;
	h->style = b->style;
	h->nbox = b->fbox;
	b->fbox = h;
    }
    expand_gap(b,width);
}

static void close_box(void)
{
    BOX *b = (bsd ? boxstack[--bsd] : NULL);
    BOX *c = (bsd ? boxstack[bsd-1]: NULL);
    int la,ld, i;

    if (b) {
	boxstack[bsd]=NULL;
	if (IsStackBox(b->special)) {
	    if (!b->fbox)
		b->width = 0;
	    else {
		BOX *h = b->fbox;
		BOX *gap = NULL;
		BOX *bot = NULL;
		BOX *top = NULL;
		BOX *g=NULL;
		BOX *l=b;
		int tw = 0;
		int th = 0;
		int pos[4];

#define SET_BOX(A) if (!A && h->fbox) { A = h;if (tw<h->width) tw=h->width;} else g=h

		while (h) {
		    switch (h->special) {
		    case TopBox:
			SET_BOX(top);
			break;
		    case BottomBox:
			SET_BOX(bot);
			break;
		    case GapBox:
			SET_BOX(gap);
			break;
		    default:
			if (!IsCursorBox(h->special) &&
			    h->special!=NewlineBox)  g=h;
			break;
		    }
		    h=h->nbox;
		    if (g) {
			g->nbox = NULL;
			if (l==b) b->fbox = h; else l->nbox=h;
			free_box(g);
			g = NULL;
		    } else if (l==b) l = l->fbox; else l=l->nbox;
		}
		for (i=0; i<4; pos[i++]=0);
		pos[0]=line_space/2;
		if (top || gap || bot) {
		    b->width = tw;
		    add_space_boxes(top, tw);
		    add_space_boxes(gap, tw);
		    add_space_boxes(bot, tw);
		    if (top) {
			pos[0]=top->ascent;
			if (!gap && bot) top->descent+=line_space;
			pos[1]=top->descent;
		    }
		    pos[3]=pos[0]+pos[1];
		    if (gap) {
			if (top) gap->ascent+=line_space;
			if (bot) gap->descent+=line_space;
			pos[3]+=(gap->ascent-gap->descent)/2;
			pos[1]+=gap->ascent;
			pos[2]=gap->descent;
		    }
		    if (bot) {
			pos[2]+=bot->ascent;
			th=bot->descent;
		    }
		    pos[1]+=pos[0];
		    pos[2]+=pos[1];
		    th+=pos[2];
		    b->y = 0;
		    if (c && b->special==StackCenterBox) {
			b->y =
			    (char_ascent(Font2Char(c->fontnr,'x'), c->size)-
			     char_descent(Font2Char(c->fontnr,'x'),c->size))/2;
		    }
		    b->ascent = pos[1];
		    b->descent = th - b->ascent;
		    if (top) top->y = b->ascent-pos[0];
		    if (gap) gap->y = b->ascent-pos[1];
		    if (bot) bot->y = b->ascent-pos[2];
		} else
		    b->width = 0;
	    }
	    move_x = 0;
	    i=0;
	    while (i<bsd) move_x+=boxstack[i++]->x;
	    move_x+=b->x+b->width;
	} else if (b->ascent < -b->descent && !IsCursorBox(b->special)) {
	    b->ascent = 0;
	    b->descent = 0;
	}
	if (b->special==BarBox) {
	    move_x+=3;
	    b->width=3;
	}
	if (IsFieldBox(b->special)) {
	    move_x = 0;
	    i=0;
	    while (i<bsd) move_x+=boxstack[i++]->x;
	    move_x+=b->x; /* +b->width; */
	}
	if (c) {
	    if (b->width || EmptyBox(b->special) || b->fbox) {
		la = b->y+b->ascent;
		if (la>c->ascent) c->ascent = la;
		ld = -b->y+b->descent;
		if (ld>c->descent) c->descent = ld;
		if (b->special==TextBox) {
		    c->ldesc = b->ldesc;
		    c->lasc = b->lasc;
		    if (!split_text) {
			b->width+=b->lright;
			move_x+=b->lright;
		    } else
			c->lright = b->lright;
		    c->fontnr = b->fontnr;
		    c->size = b->size;
		} else if (!IsCursorBox(b->special)) {
		    c->lasc = la;
		    c->ldesc = ld;
		}
		if (b->maxwidth > b->width) {
		    move_x = move_x + b->maxwidth-b->width;
		    b->width = b->maxwidth;
		}
		if (!c->lbox)
		    c->fbox = c->lbox = b;
		else {
		    c->lbox->nbox = b;
		    c->lbox = b;
		}
		if (!IsStackBox(c->special))
		    c->width+= b->width;
	    } else {
		if (b->special==TextBox) {
		    if (!split_text) {
			c->width+=b->lright;
			move_x+=b->lright;
		    } else
			c->lright=b->lright;
		    c->fontnr = b->fontnr;
		    c->size = b->size;
		}
		free_box(b);
	    }
	}
    }
}

static BOX *close_boxes(void)
{
    int i;
    BOX *c;

    free_int(savestack);
    savestack = NULL;
    for (i=bsd-1; i>=0; i--) {
	push_int(&savestack, boxstack[i]->fontnr);
	push_int(&savestack, boxstack[i]->size);
	push_int(&savestack, boxstack[i]->special);
	funcargs[i] = boxstack[i]->funcarg;
    }
    c = boxstack[0];
    while (bsd>0) close_box();
    return c;
}

static void open_boxes(void)
{
    int i,j=0;
    BOX *b;

    while ((i=pop_int(&savestack))) {
	b = open_box(i);
	b->size = pop_int(&savestack);
	b->fontnr = pop_int(&savestack);
	b->funcarg = funcargs[j++];
    }
}

static void put_sign(Window win, int x, int y, int kind);

static void draw_cursors(void)
{
    BOX *h = cursorstack;
    while (h) {
	put_sign(output_win, h->x, h->y, CursorKind(h->special));
	cursorstack = h->fbox;
	h->fbox = NULL;
	h = cursorstack;
    }
}

static int draw_box(BOX *b, int x, int y, int ascent, int descent,int code)
{
    BOX *h=b->fbox;
    int ncode=0,i;
    int nl=0;

    x = x+b->x;
    y = y-b->y;
    ascent -=b->y;
    descent += b->y;
    switch (b->special) {
    case TopBox:
	if (code&0x3) descent = b->descent;
	break;
    case GapBox:
	if (code&0x1) descent = b->descent;
	if (code&0x4) ascent = b->ascent;
	break;
    case BottomBox:
	if (code&0x6) ascent = b->ascent;
	break;
    default:
	if (IsCursorBox(b->special)) {
	    b->x = x;
	    b->y = y;
	    b->fbox = cursorstack;
	    cursorstack = b;
	} else if (IsStackBox(b->special)) {
	    while (h) {
		switch (h->special) {
		case GapBox: ncode |= 0x2; break;
		case TopBox: ncode |= 0x4; break;
		case BottomBox: ncode |= 0x1; break;
		default: break;
		}
		h = h->nbox;
	    }
	    h = b->fbox;
	}
	break;
    }
    switch (b->style) {
    case SMART:
	if (smart_mode != VISIBLE) break;
    case VISIBLE:
	if (!b->fbox && b->width)
	    XFillRectangle(display, output_win, get_GC(Reverse, b->color),
			   x, y-ascent, b->width, ascent+descent);
	switch (b->special) {
	case TextBox:
	    if (b->itemtot) {
		XDrawText(display, output_win, get_GC(Normal,b->color), x,y,
			  items+b->itemnr, b->itemtot);
		undefined_font(Normal);
	    }
	    break;
	case LineBox:
	    if (b->width)
		XDrawLine(display, output_win, get_GC(Normal, b->color),
			  x, y-1, x+b->width, y-1);
	    break;
	case StippleBox:
	    if (b->width) {
		int j,yp = y-1;
		i=x;
		while (i<x+b->width) {
		    j = i+2;
		    if (j>x+b->width) j = x+b->width;
		    XDrawLine(display, output_win, get_GC(Normal, b->color),
			      i, yp, j, yp);
		    i+=5;
		}
	    }
	    break;
	case BarBox:
	    XDrawLine(display, output_win, get_GC(Normal, b->color),
		      x+1, y-ascent, x+1, y+descent);
	    break;
	default:
	    break;
	}
	if (!b->fbox && (i=must_underline(b->color)) && b->special!=GapBox
	    && b->width && !IsStackBox(b->special)) {
	    if (i!=2)
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-2, x+b->width-1, y+descent-2);
	    if (i>1) {
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-1, x+b->width-1, y+descent-1);
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-3, x+b->width-1, y+descent-3);
	    }
	}
	break;
    case SHADES:
	if (b->color && !b->fbox && b->width)
	    XFillRectangle(display, output_win, get_GCXor(b->color),
			   x, y-ascent, b->width, ascent+descent);
	if (!b->fbox && (i=must_underline(b->color)) &&
	    b->special!=GapBox && b->width && !IsStackBox(b->special)) {
	    if (i!=2)
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-2, x+b->width-1, y+descent-2);
	    if (i>1) {
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-1, x+b->width-1, y+descent-1);
		XDrawLine(display, output_win, get_GCXor(PrimSel),
			  x, y+descent-3, x+b->width-1, y+descent-3);
	    }
	}
	break;
    default:
	break;
    }
    while (h) {
	nl+=draw_box(h, x, y, ascent, descent,ncode);
	h = h->nbox;
    }
    if (b->special==NewlineBox) nl++;
    if (search_func && b->funcarg && search_x>=x && search_x<x+b->width &&
	search_y>=y-ascent && search_y < y+descent) {
	(*search_func)(b->funcarg);
	search_func = NULL;
    }
    return nl;
}

static void box_add_char(Char c)
{
    BOX *b;

    b = open_box(TextBox);
    if (!b->width) {
	b->itemnr = initem;
	init_item(initem++);
	b->itemtot = 1;
    }
    if (Char2Font(c)==TEXTFONT)
	c = Font2Char(b->fontnr, Char2ASCII(c));
    if (char_width(c, b->size)) {
	int tf = LASTFONT(Char2Font(c),b->size);
	int i = b->itemnr+b->itemtot-1;
	int j=0;
	if (tf!=lastfont && items[i].font!=None) {
	    b->itemtot++;
	    init_item(initem);
	    initem++;
	    i++;
	}
	if (items[i].font==None) {
	    j = (!split_text||tf!=lastfont ? b->lright+char_left(c,b->size):0);
	    items[i].font = font_ID(Char2Font(c), b->size);
	    items[i].delta += j;
	}
	split_text = False;
	lastfont = tf;
	string[instring] = Char2ASCII(c);
	instring++;
	items[i].nchars++;
	b->lasc = char_ascent(c, b->size);
	b->ldesc = char_descent(c,b->size);
	b->lright = char_right(c,b->size);
	j+= char_width(c,b->size);
	b->width += j;
	move_x += j;
	if (b->ascent<b->lasc) b->ascent= b->lasc;
	if (b->descent<b->ldesc) b->descent = b->ldesc; 
    }
}

static void box_neg_space(int space)
{
    BOX *b = open_box(BackBox);
    BOX *c = (bsd>1? boxstack[bsd-2]:NULL);

    b->width = space;
    b->x-= space;
    if (c) {
	if (c->maxwidth<c->width)
	    c->maxwidth = c->width;
	c->width-=space;
    }
    close_box();
    if (c) {
      c->width-=space;
      move_x-=space;
    }
}

static void box_space(int space)
{
    BOX *b = open_box(TextBox);

    if (!b->itemtot) {
	b->itemnr = initem;
	init_item(initem++);
	b->itemtot=1;
    }
    if (items[b->itemnr+b->itemtot-1].nchars) {
	init_item(initem++);
	b->itemtot++;
    }
    if (b->lright) space+= b->lright;
    b->lright = 0;
    items[b->itemnr+b->itemtot-1].delta+=space;
    b->width+=space;
    move_x+=space;
}

static void switch_size_fontnr(int size, int fontnr)
{
    BOX *b = open_box(TextBox);

    if (size!=b->size || fontnr!=b->fontnr) {
	if (!b->itemtot) b->itemnr = initem;
	init_item(initem++);
	b->itemtot +=1;
        items[b->itemnr+b->itemtot-1].delta+=b->lright;
	b->width+=b->lright;
	move_x+=b->lright;
	b->lright = 0;
	b->size = size;
	b->fontnr = fontnr;
    }
}

static void switch_sizenr(int sizenr)
{
    BOX *b = open_box(TextBox);
    if (sizenr!=b->size)
	switch_size_fontnr(sizenr, b->fontnr);
}

static void switch_fontnr(int fontnr)
{
    BOX *b = open_box(TextBox);
    if (fontnr!=b->fontnr)
	switch_size_fontnr(b->size, fontnr);
}

static void pushtabs(void);
static void poptabs(void);
static void endline(void);
static void settab(void);
static void rtab(void);
static void ltab(void);
static void tabplus(void);
static void tabminus(void);

static int bufcount = 0;
int saved_chars = 0;

int more_keys(void)
{
    XEvent report;

    saved_chars = bufcount;
    if (bufcount>10 || !XEventsQueued(display, QueuedAfterFlush))
	return (bufcount = 0);
    XPeekEvent(display, &report);
    while (report.type == KeyRelease) {
	XNextEvent(display, &report);
	if (XEventsQueued(display,QueuedAfterFlush))
	    XPeekEvent(display, &report);
	else
	    return (bufcount = 0);
    }
    if (report.type == KeyPress)
	return ++bufcount;
    else
	return (bufcount=0) ;
}

void set_message_window(void *w)
{
   message_win = *((Window *) w);
}

void draw_message(void)
{
    if (message_win) {
	XClearWindow(display, message_win);
	push_fontgroup(POPUPFONT);
	if (messlen) {
	    XDrawString(display, message_win, get_GC_font(Normal,0,TEXTFONT,0),
			0, font_ascent(0,0) + line_space/2, messtxt, messlen);
	}
	if (messcurs) {
	    put_sign(message_win, string_width(TEXTFONT, messtxt, messlen),
		     font_ascent(0,0)+line_space/2, 0);
	}
	XFlush(display);
	pop_fontgroup();
    }
}

void out_message(char *txt)
{
    if (message_win) {
	strncpy(messtxt, txt, 400);
	messlen = strlen(messtxt);
	messcurs = False;
	draw_message();
    }
}

void out_message_curs(char *txt)
{
    if (message_win) {
	strncpy(messtxt, txt, 400);
	messlen = strlen(messtxt);
	messcurs = True;
	draw_message();
    }
}

void clear_message(Bool allways)
{
    if (allways || !messcurs) {
	messtxt[0]='\0';
	messlen = 0;
	messcurs = False;
	if (message_win) draw_message();
    }
}

void *test_window(void)
{
    return &test_win;
}

void set_output_window(void *w)
{
    if (output_win || bsd) {
	BOX *c;
	fprintf(stdout, "Unset Window vergeten...\n");
	c = close_boxes();
	free_box(c);
	free_int(savestack);
	savestack = NULL;
	search_func = NULL;
    }
    output_win = *((Window *) w);
    lastfont = LASTFONT(255,0);
    def_thinspace=0;
    base_x = base_y = move_x = 0;
    left_margin = 0;
    can_backtab = True;
    drindex=False;
    bsd = 0;
    initem = 0;
    instring = 0;
    placepos=nameset=inplacename=0;
    free_int(fontstack);
    fontstack = NULL;
    free_int(sizestack);
    sizestack = NULL;
    base_style = VISIBLE;
    base_color = 0;
    split_text = 0;
    hiding = 0;
    open_box(DefaultBox);
}

static void resettabs(void);

void unset_output_window(void)
{
    BOX *c;
    flush();
    c = close_boxes();
    free_box(c);
    free_int(savestack);
    savestack = NULL;
    search_func = NULL;
    output_win = 0;
    resettabs();
}

void set_drawstyle(int style)
{
    split_text = !after_node;
    close_textbox();
    base_style = style;
}

void set_underline(Bool b)
{
    split_text = !after_node;
    close_textbox();
    if (b) {
	out_char(StackB);
	out_char(TopGap);
    } else {
	out_char(GapBottom);
	out_char(GlueLine);
	out_char(StackClose);
    }
}

void set_italic(Bool b)
{
    if (b) {
	if (b>True) {
	    switch_fontnr(b-1);
	    pushfont(b-1);
	} else {
	    switch_fontnr(identifier_font);
	    pushfont(identifier_font);
	}
    } else {
	popfont();
	switch_fontnr(headfont());
    }
}

int next_id_font(int n)
{
    int i;
    i=n+1;
    while (i!=n) {
	if (!i) i++;
	if (font_openmathtex(i,0) || font_opentexttex(i,0)) return i;
	if (i==255) i=0; else i++;
    }
    return -1;
}

void set_default_thinspace(int n)
{
    def_thinspace=n&0xff;
}

void set_index(Bool b)
{
    drindex = b;
}

void set_text_mode(int mode)
{
    split_text = !after_node;
    close_textbox();
    if ((mode && !(base_color&PrimSel)) || (!mode && base_color&PrimSel))
	base_color ^= PrimSel;
}

void switch_thick(void)
{
    split_text = !after_node;
    close_textbox();
    base_color ^= SecondSel;
}

void switch_thin(void)
{
    split_text = !after_node;
    close_textbox();
    base_color ^= ThirdSel;
}

void switch_reverse(void)
{
    split_text = !after_node;
    close_textbox();
    base_color ^= PrimSel;
}

void out_clear(void)
{
    if (output_win)
	XClearWindow(display, output_win);
    base_x = 0;
    base_y = 0;
}

void set_margin(int newmargin)
{
    left_margin = newmargin;
    can_backtab = True;
}

void set_smart_height(int height)
{
    smart_height = height;
}

void set_search_func(void (*func)(void*), int x, int y)
{
    search_func = func;
    search_x = x+left_margin;
    search_y = y;
}

void detect_margin(void)
{
    WINDOWTYPE wtype,ptype;
    Window wp,wpp;
    void *data;
    int i;

    wtype = get_window_type(output_win, &wp, &data);
    if (!data)
	ptype = get_window_type(wp, &wpp, &data);
    if (!data) data = (void*) &output_win;
    if (eventfunc[wtype]->margin)
	i = (*(eventfunc[wtype]->margin))(data);
    else
	i = 0;
    left_margin = i;
    can_backtab = True;
}

int where_x(void)
{
  return base_x+move_x;
/*    int i=0,x=0;
    while (i<bsd) {
	x+= boxstack[i]->x;
	i++;
    }
    if (i)
	x+=boxstack[i-1]->width;
    return base_x+x; */
}

int where_y(void)
{
    return base_y;
}

void set_x_y(int new_x, int new_y)
{
    flush();
    base_x = new_x;
    base_y = new_y;
}

void set_x(int new_x)
{
    base_x = new_x;
}

void set_y(int new_y)
{
    base_y = new_y;
}

int line_height(void)
{
    return font_height(0,0)+line_space;
}

static void clear_to_end_of_line(int x, int height)
{
    if (base_style==VISIBLE || base_style==SMART)
	XClearArea(display, output_win, x, base_y, 0, height, False);
}

void clear_to_end_of_page(void)
{
    if (base_style==VISIBLE) {
	endline();
	XClearArea(display, output_win, 0, base_y, 0, 0, False);
    }
}

static void clear_to_begin_of_line(int height)
{
    if (base_style==VISIBLE || base_style==SMART) {
	if (left_margin+base_x>0)
	    XClearArea(display, output_win, 0, base_y, base_x+left_margin,
		       height, False);
    }
}

void flush(void)
{
    BOX *b = close_boxes();

    if (b && output_win) {
	b->ascent+=line_space/2;
	b->descent+=line_space/2+line_space%2;
	clear_to_begin_of_line(b->ascent+b->descent);
	smart_mode = (b->ascent+b->descent!=smart_height ? VISIBLE: INVISIBLE);
	if (draw_box(b,base_x+left_margin, base_y+b->ascent,
		     b->ascent, b->descent,0)) {
	    clear_to_end_of_line(base_x+left_margin+b->width,
				 b->ascent+b->descent);
	    draw_cursors();
	    base_y+=b->ascent+b->descent;
	} else
	    draw_cursors();
    }
    initem = 0;
    instring = 0;
    init_item(0);
    smart_mode = INVISIBLE;
    smart_height = 0;
    boxstack[0] = NULL;
    if (b) free_box(b);
    open_boxes();
}

static void out_symbol(Char data)
{
    box_add_char(data);
}

void out_text_delim(int on)
{
    if (textdots && !hiding) {
	box_add_char(on ? TEXTOPEN : TEXTCLOSE);
	after_node=False;
    }
}

void out_index(int c)
{
    char str[10];
    int i = 0;

    if (!c) return;
    if (c<-500) c = -500;
    if (c>500) c = 500;
    sprintf(str, "%i",c);
    i = (bsd?boxstack[bsd-1]->size:0)-2;
    switch_size_fontnr(i,0);
    pushsize(i);
    pushfont(0);
    i=0;
    while (str[i]) {
	out_symbol(Font2Char(0,str[i]));
	i++;
    }
    popsize();
    popfont();
    switch_size_fontnr(headsize(),headfont());
    after_node=False;
}

void out_cursor(int kind)
{
    split_text = !after_node;
    open_box(CursorBox+kind);
    close_box();
}

void open_node(void* data)
{
    BOX *b;

    close_textbox();
    b = open_box(NodeBox);
    b->funcarg = data;
    after_node=True;
    if (placepos) {
	nameset=1;
	placename[placepos]='\0';
	placepos=0;
    } else nameset=0;
}

void close_node(void)
{
    after_node=False;
    while (bsd && boxstack[bsd-1]->special!=NodeBox) close_box();
    close_box();
    /* fonts and sizes should be changed */
    placepos=nameset=0;
}

static void out_bold(char *str)
{
    switch_size_fontnr(-1,1);
    pushsize(-1);
    pushfont(1);
    out_string(str);
    popsize();
    popfont();
    switch_size_fontnr(headsize(),headfont());
}

void out_char(Char data)
{
    if (hiding && !IsNewline(data) && data!=StartHide && data!=EndHide) return;
    if (inplacename) {
	if (placepos<PLACENAMESIZE && !(data&0xff00))
	    placename[placepos++]=data&0xff;
	else if (data==PlNameEnd) inplacename=0;
	return;
    }
    switch (data) {
    case Newline:
	endline();
	can_backtab = True;
	break;
    case SoftNewline:
	endline();
	break;
    case Ltab:
	ltab();
	break;
    case Rtab:
	rtab();
	can_backtab = False;
	break;
    case Settab:
	settab();
	can_backtab = False;
	break;
    case Tabplus:
	tabplus();
	can_backtab = False;
	break;
    case Tabminus:
	tabminus();
	can_backtab = False;
	break;
    case Pushtabs:
	pushtabs();
	can_backtab = False;
	break;
    case Poptabs:
	poptabs();
	can_backtab = False;
	break;
    case AskText:
    case AskMath:
    case AskBoth:
    case InText:
    case InMath:
    case InDisp:
	break;
    case StartHide:
	hiding++;
	break;
    case EndHide:
	if (hiding) hiding--;
	break;
    case CloseStack:
    case CloseTop:
    case CloseGap:
    case CloseBottom:
	close_textbox();
	if (bsd>1) close_box();
	break;
    case StackClose:
	close_textbox();
	if (bsd>1) close_box();
	if (bsd>1) close_box();
	break;
    case TopGap:
	close_textbox();
	if (bsd>1) close_box();
	open_box(GapBox);
	break;
    case GapBottom:
	close_textbox();
	if (bsd>1) close_box();
	open_box(BottomBox);
	break;
    case StackC:
	open_box(StackCenterBox);
	open_box(TopBox);
	break;
    case StackB:
	open_box(StackBaseBox);
	open_box(TopBox);
	break;
    case VerLine:
	close_textbox();
	open_box(BarBox);
	close_box();
	break;
    case ThinSpace:
	thinspace(def_thinspace*screen_space);
	break;
    case PopSize:
	popsize();
	switch_sizenr(headsize());
	break;
    case OpenTop:
	open_box(TopBox);
	break;
    case OpenBottom:
	open_box(BottomBox);
	break;
    case OpenGap:
	open_box(GapBox);
	break;
    case GlueLine:
	open_box(LineBox);
	close_box();
	break;
    case GlueSpace:
	open_box(SpaceBox);
	close_box();
	break;
    case GlueStipple:
	open_box(StippleBox);
	close_box();
	break;
    case TabOpen:
	open_tabbing();
	break;
    case TabClose:
	close_tabbing();
	break;
    case DisplayOpen:
	open_display();
	break;
    case DisplayClose:
	close_display();
	break;
    case PlName:
	inplacename=1;
	placepos=0;
	break;
    case PlNameEnd:
	inplacename=0;
	break;
    default:
	if (IsPh(data)) {
	    if (placepos || nameset) {
		if (!nameset) placename[placepos]='\0';
		out_bold(placename);
		placepos=nameset=0;
	    } else
		box_add_char(SYMBOL_PH(data));
	    if (drindex) out_index(Num(data));
	} else if (Char2Font(data)==SpaceFont) {
	    thinspace(Char2ASCII(data)*screen_space);
	} else if (Char2Font(data)==StackFont) {
	    open_box(Char2ASCII(data)==63?StackBaseBox:StackCenterBox);
	} else if (Char2Font(data)==FontFont) {
	    switch_fontnr(Char2ASCII(data));
	    pushfont(Char2ASCII(data));
	} else if (Char2Font(data)==PopFont) {
	    popfont();
	    switch_fontnr(headfont());
	} else if (Char2Font(data)==SizeFont) {
	    int c = Char2ASCII(data);
	    if (c<128) {
		int i = (bsd?boxstack[bsd-1]->size:0) +c-64;
		switch_sizenr(i);
		pushsize(i);
	    } else {
		switch_sizenr(c-192);
		pushsize(c-192);
	    }
	} else {
	    out_symbol(data);
	    can_backtab = False;
	}
	break;
    }
    after_node=False;
}

Bool display_tab(Char data)
{
    if (IsTab(data)) {
	out_bold(SYMBOL_TAB(data));
	return True;
    } else {
	char *c;
	int i, colon=True,ind,fr=1;
	switch (Char2Font(data)) {
	case StackFont:
	    c = (Char2ASCII(data)==63?SYMBOL_TAB(StackB):
		       SYMBOL_TAB(StackC));
	    fr=0;
	    colon = False;
	    break;
	case FontFont:
	    if (font_name(Char2ASCII(data)))
		c = concat("[", font_name(Char2ASCII(data)));
	    else {
		c = "[Font??";
		fr=0;
	    }
	    break;
	case SizeFont:
	    if ((int)Char2ASCII(data)>128) {
		c = concat(SIZESYMBOL,"xxxx");
		i = strlen(SIZESYMBOL);
		ind = Char2ASCII(data)-192;
	    } else {
		c = concat(RELSIZESYMBOL,"xxxx");
		i = strlen(RELSIZESYMBOL);
		ind = Char2ASCII(data)-64;
	    }
	    sprintf(c+i, "%i", ind);
	    break;
	case PopFont:
	    c = "]";
	    fr = 0;
	    colon = False;
	    break;
	default:
	    return False;
	}
	out_bold(c);
	if (colon) out_bold(":");
	if (fr) myfree(c);
	return True;
    }
}

void out_string(char *str)
{
    int i, l = mystrlen(str);

    for (i=0; i<l; i++)
	box_add_char(str[i]);
    can_backtab=False;
}

void thinspace(int spacing)
{
    if (spacing)
	if (spacing >=0)
	    box_space(spacing);
	else
	    box_neg_space(-spacing);
    else
	close_textbox();
}

static void put_sign(Window win, int x, int y, int kind)
{
    GC   tempgc;
    XPoint temp1[MAX_SIGN_SIZE];
    XPoint *temp2 = sign[kind];
    int nr = sign_size[kind];
    int i;

    for (i=0; i<nr; i++) {
	temp1[i].x = temp2[i].x+x;
	temp1[i].y = temp2[i].y+y;
    }
    temp1[i] = temp1[0];
    tempgc = get_GCXor(PrimSel);
    if (sign_filled[kind])
	XFillPolygon(display, win, tempgc, temp1,
		     nr, Convex, CoordModeOrigin);
    else
	XDrawLines(display, win, tempgc, temp1, nr, CoordModeOrigin);
}

void put_mark(int x, int y)
{
    put_sign(output_win, x+left_margin, y, 1);
}

/*
**  tabbing functies
*/

#define MAXTAB 20

typedef struct TABBING TABBING;
struct TABBING {
                short stops[MAXTAB];
                unsigned char curtabmar;
                unsigned char hightab;
                unsigned char curtab;
                TABBING *prev;
               };

static TABBING *stacktop = 0, *freelist = 0;
static int freel=0;

static int tabsize = 24;

static int simpletabsize; /* = 8 * "width of an n" */

static int in_tab_env = 0;

/*
static void stopfunc(void)
{
    int i=5;
    while (i) i--;
}

static void check(void)
{
    int i=0;
    TABBING *h=freelist;

    if (!freelist && !freel) return;
    if (!freel || !freelist) { stopfunc(); return; }
    while (h) {
	i++;
	h=h->prev;
    }
    if (i!=freel) stopfunc();
}
*/
static void check(void) { }

#define GET_FREE(A) { if (freel) { A=freelist;freelist=A->prev;freel--; }\
                      else  A = (TABBING*) malloc(sizeof(TABBING)); check(); }
#define SET_FREE(A) { TABBING *h=A->prev; A->prev = freelist; freelist=A;\
		      freel++; A=h; check(); }

static void pushtabs(void)
{
    TABBING *h;

    if (!in_tab_env) return;
    GET_FREE(h);
    if (h) {
	if (stacktop)
	    *h = *stacktop;
	else {
	    /* open tabbing. */
	    h->stops[0] = 0;
	    h->hightab = h->curtab = h->curtabmar = 0;
	}
	h->prev = stacktop;
	stacktop = h;
    }
}

static void poptabs(void)
{
    if (!in_tab_env) return;
    if (stacktop && stacktop->prev) SET_FREE(stacktop);
}

void tab_unlock(void *ts)
{
    int i=1;
    TABBING *h = (TABBING*) ts;
    if (!ts) return;
    while (h->prev) { h=h->prev; i++; }
    h->prev=freelist;
    freelist=(TABBING*) ts;
    freel+=i;
    check();
    while (freel>100) {
	h=freelist; freelist=h->prev;
	free(h);
	freel--;
    }
    check();
}

void *tab_lock(void)
{
    TABBING *a,*b,*c;
    if (!stacktop) return NULL;
    GET_FREE(a);
    *a=*stacktop;
    b=a->prev;
    c=a;
    while (b) {
	GET_FREE(c->prev);
	c=c->prev;
	*c=*b;
	b=b->prev;
    }
    return (void*) a;
}

static void resettabs(void)
{
    while (stacktop) SET_FREE(stacktop);
    in_tab_env = 0;
}


void set_tab_stack(void *ts, int nropen)
{
    resettabs();
    stacktop = (TABBING*) ts;
    stacktop = (TABBING*) tab_lock();
    in_tab_env = nropen;
    if (in_tab_env)
	base_x = stacktop->stops[stacktop->curtab=stacktop->curtabmar];
    else
	base_x = 0;
    move_x = 0;
}

Bool tab_equal(void *ts)
{
    TABBING *a,*b;

    if (!ts && !stacktop) return True;
    if (!ts || !stacktop) return False;
    a= (TABBING*) ts;
    b= stacktop;
    while (a && b) {
	int i;
	if (a->curtabmar!=b->curtabmar || a->hightab!=b->hightab ||
	    a->curtab!=b->curtab || (!a->prev)^(!b->prev)) return False;
	i=a->hightab+1;
	while (i) {
	    i--;
	    if (a->stops[i]!=b->stops[i]) return False;
	}
	a = a->prev;
	b = b->prev;
    }
    return True;
}

void open_tabbing(void)
{
    in_tab_env++;
    pushtabs();
}

void open_display(void)
{
    int i, j;

    open_tabbing();
    if (in_tab_env==1) {
	j = (display_delta+latex_side<0 ? 0 : display_delta+latex_side);
	for (i=0; i<j; i++) {
	    rtab();
	    tabplus();
	}
    }
    display_delta = 0;
}

void close_tabbing(void)
{
    if (in_tab_env) {
	in_tab_env--;
	if (in_tab_env)
	    poptabs();
	else
	    resettabs();
    }
}

void close_display(void)
{
    close_tabbing();
}

void settabsize(int newsize)
{
    tabsize = newsize;
}

int getsimpletabsize(void)
{
    return simpletabsize;
}

void setsimpletabsize(int newsize)
{
    simpletabsize = newsize;
}

void opspace(int size)
{
    thinspace(size * screen_space);
}

static void endline(void)
{
    close_textbox();
    open_box(NewlineBox);
    close_box();
    lastfont=LASTFONT(255,0);
    flush();
    if (in_tab_env)
	base_x = stacktop->stops[stacktop->curtab=stacktop->curtabmar];
    else
	base_x = 0;
    move_x = 0;
}

static void settab(void)
{
    if (in_tab_env && (int)stacktop->curtab < MAXTAB-1) {
	stacktop->curtab++;
	stacktop->stops[stacktop->curtab] = where_x();
	if (stacktop->hightab < stacktop->curtab) 
	    stacktop->hightab = stacktop->curtab;
    }
}

static void rtab(void)
{
    if (in_tab_env) {
	if ((int)stacktop->curtab < MAXTAB-1) {
	    stacktop->curtab++;
	    if (stacktop->curtab > stacktop->hightab) {
		thinspace(tabsize);
		stacktop->stops[stacktop->curtab] = where_x();
		stacktop->hightab++;
	    } else
		thinspace(stacktop->stops[stacktop->curtab]-where_x());
	}
    } else
	thinspace(simpletabsize - where_x()%simpletabsize);
}

static void ltab(void)
{
    if (in_tab_env)
	if (stacktop->curtab)
	    if (can_backtab)
		thinspace(stacktop->stops[--stacktop->curtab]-where_x());
}

static void tabplus(void)
{
    if (in_tab_env) {
	if (stacktop->curtabmar < stacktop->hightab)
	    stacktop->curtabmar++;
    }
}

static void tabminus(void)
{
    if (in_tab_env) {
	if (stacktop->curtabmar)
	    stacktop->curtabmar--;
    }
}

void output_init(void)
{
    long i;

    i = (XMaxRequestSize(display) -4) *4;
    if (i>MAXSIZE) i=MAXSIZE;
    maxlen = i -2;
    if (!(string = (char *) malloc(maxlen)))
	message(EXIT -1, "Out of memory in output.");
    maxitem = i /3;
    if (!(items = (XTextItem *) malloc(maxitem * sizeof (XTextItem))))
	message(EXIT -1, "Out of memory in output.");
    initem = 0;
    limitchar = maxlen;
    instring = 0;
    init_item(initem);
    lastfont = LASTFONT(255,0);
    boxstack[0] = NULL;
    simpletabsize = 8 * char_width(Font2Char(TEXTFONT,'n'),0);
    tabsize = screen_tab;
    test_win = XCreateSimpleWindow(display, root_window, 0,0,1,1,0,0,0);
}

/*
**  Een aantal functies om het tex-en van strings te vereenvoudigen
**  en te controleren
*/

static Bool in_math, in_tabbing, tex_plhl, after_slash, after_macro,
            secure, setms;
static FILE *texf = NULL;
static char **texstring = NULL;       /* LaTeX -> string */
static int stringlen=0, stringmax=0;  /* length of string, size allocated */
static int pref_math[50] = {0};
static int pref_count[50] = {1};
static int total_count = 1;
static int soft_push[50] = {0};
static int soft_count = 0;  
static int nrpref = 0;
static int last_close = 0;
static int nrdisp = 0;
static int texmode = 0;
static int texoutput=0; /* 0=FILE 1=SCHERM 2=STRING */


#define TrueTrue  2
#define prefmath     pref_math[nrpref]

static void add_string(char *str)
{
    int i;
    char *c;

    if (!str || !(i = strlen(str))) return;
    if (stringlen+i>=stringmax) {
	stringmax = (((stringlen+i)>>11)+1)<<11;
	c = (char*) malloc(stringmax);
	if (stringlen) {
	    strcpy(c, *texstring);
	    free(*texstring);
	} else
	    *c=0;
	*texstring = c;
    }
    strcpy(*texstring+stringlen, str);
    stringlen+=i;
}

static void printlatex(Bool bld, char *str)
{
    switch (texoutput) {
    case 0:
	if (str) fputs(str,texf);
	break;
    case 1:
	switch (bld) {
	case False:
	    out_string(str);
	    break;
	default:
	    out_bold(str);
	    break;
	}
	break;
    case 2:
	add_string(str);
	break;
    default:
	break;
    }
}

static void close_slash(void)
{
    if (after_slash) {
	printlatex(0," ");
	after_slash = after_macro = False;
    }
}

static void close_macro(void)
{
    if (after_macro) {
	if (after_slash)
	    printlatex(False," ");
	else
	    printlatex(False,"{}");
	after_slash=after_macro=False;
    }
}

static void close_math(void)
{
    if (after_slash) {
	printlatex(False," ");
	after_macro = after_slash = False;
    }
    if (in_math == True) {
	printlatex(True,"$");
	last_close = False;
    } else if (in_math == TrueTrue) {
	printlatex(True,"\\mbox{");
	last_close = True;
    }
    after_macro = after_macro && !in_math;
    in_math = False;
}

static void open_math(void)
{
    if (after_slash) {
	printlatex(False," ");
	after_slash = after_macro = False;
    }
    after_macro = after_macro && in_math;
    if (!in_math)
	if (prefmath == TrueTrue || last_close) {
	    printlatex(True,"}");
	    in_math = TrueTrue;
	    last_close = False;
	} else {
	    printlatex(True,"$");
	    in_math = True;
	}
}

void push_math_pref(Bool premath)
{
    if (pref_math[nrpref]==premath)
	pref_count[nrpref]++;
    else {
	pref_math[++nrpref]=premath;
	pref_count[nrpref]=1;
    }
    total_count++;
}

static void soft_math_pref(Bool premath)
{
    push_math_pref(premath);
    soft_push[++soft_count] = total_count;
} 

void pop_math_pref(void)
{
    while (soft_push[soft_count]==total_count) {
	pref_count[nrpref]--;
	if (pref_count[nrpref]<1 && nrpref>0) nrpref--;
	in_math = pref_math[nrpref];
	total_count--;
	soft_count--;
    }
    pref_count[nrpref]--;
    if (pref_count[nrpref]<1 && nrpref>0) nrpref--;
    total_count--;
}

static void clear_state(void)
{
    pref_math[0] =
	in_tabbing = 
	in_math =
	tex_plhl =
	after_slash =
	after_macro =
	secure =
	last_close = False;
    pref_count[0]= 1;
    soft_count = 0;
    def_thinspace=0;
    total_count = 1;
    texmode = output_mode;
    setms = (output_mode!=MPTEX);
    nrdisp = 0;
    nrpref = 0;
}

void tex_set_file(FILE *f)
{
    texf=f;
    clear_state();
    texoutput = (!f);
}

void tex_set_string(char **str)
{
    texstring = str;
    stringlen = stringmax = 0;
    if (*str) free(*str);
    *str = NULL;
    clear_state();
    texoutput = 2;
    setms=True;
}

int tex_current_pos(void)
{
    return stringlen;
}

void tex_mode(int mode)
{
    texmode = mode;
    setms = setms || mode!=MPTEX;
}

void tex_open_proof(void)
{
    int d = (display_delta+latex_side<0 ? 0 : display_delta+latex_side); 
    if (!in_tabbing) {
	char buf[500];
	close_math();
	if (texoutput!=1) {
	    sprintf(buf, "\\begin{mpdisplay}{%s}{%s}{%s}{%i}\n\t",
		    latex_space_unit, latex_tab_unit,
		    latex_line_unit, d);
	    printlatex(True, buf);
	} else
	    open_tabbing();
	in_tabbing = True;
	after_slash = False;
	after_macro = False;
	secure = False;
	in_math = False;
	nrdisp = 1;
    } else nrdisp++;
    display_delta = 0;
}

void tex_close_proof(void)
{
    if (in_tabbing && nrdisp==1) {
	if (last_close) {
	    close_slash();
	    printlatex(True,"}");
	    last_close = False;
	}
	close_math();
	if (texoutput!=1)
	    printlatex(True,"\n\\end{mpdisplay}");
	else
	    close_tabbing();
	in_tabbing = 
	    after_slash =
	    after_macro =
	    secure =
	    in_math = False;
	nrdisp = 0;
    } else if (nrdisp) nrdisp--;
}

void tex_unset(void)
{
    close_math();
    if (in_tabbing) 
	if (texoutput!=1) 
	    printlatex(True,"\\end{mpdisplay}");
	else
	    close_tabbing();
    if (!texstring) printlatex(False,"\n");
    clear_state();
    texf = NULL;
    texstring=0;
    texoutput=0;
    stringlen=0;
    stringmax=0;    
}

void tex_placeholders(Bool texthem)
{
    tex_plhl = texthem;
}

static void update_after(char *str)
{
    while (*str) {
	if (*str=='\\')
	    after_slash= after_macro=!after_slash;
	else {
	    after_macro=after_macro && isalpha(*str);
	    after_slash = False;
	}
	str++;
    }
}

void out_latex_char(Char c)
{
    int newmode;
    char *lc=NULL;

    if (!c) return;
    if (inplacename) {
	if (placepos<PLACENAMESIZE && !(c&0xff00))
	    placename[placepos++]=c&0xff;
	else if (c==PlNameEnd) inplacename=0;
	return;
    }
    if (IsTab(c)) {
	if (c==PlName) { placepos=0; inplacename=1; } else
	if (texmode==ASCII) {
	    printlatex(True,ASCII_TAB(c));
	} else 	if (in_tabbing && (c>=SoftNewline)) {
	    if (c!=SoftNewline)	close_math();
	    printlatex(True,LATEX_TAB(c));
	} else {
	    close_slash();
	    if (!prefmath) close_math();
	    switch (c) {
	    case Rtab:
		printlatex(False,"\t");
		break;
	    case Newline:
	    case SoftNewline:
		printlatex(False,"\n");
		break;
	    case AskText:
	    case AskMath:
	    case AskBoth:
		break;
	    case InText:
		if (!in_math && prefmath==TrueTrue) open_math();
		in_math = False;
		soft_math_pref(False);
		if (texoutput==1) out_bold(SYMBOL_TAB(c));
		break;
	    case InMath:
		in_math = True;
		soft_math_pref(True);
		if (texoutput==1) out_bold(SYMBOL_TAB(c));
		break;
	    case InDisp:
		in_math = TrueTrue;
		soft_math_pref(TrueTrue);
		if (texoutput==1) out_bold(SYMBOL_TAB(c));
		break;
	    case ThinSpace:
		if (def_thinspace || texoutput==1) {
		    char oss[10];
		    sprintf(oss, LATEX_TAB(ThinSpace), def_thinspace);
		    close_slash();
		    printlatex(True,oss);
		}
		break;
	    case PopSize:
	    case CloseStack:
	    case OpenTop:
	    case CloseTop:
	    case OpenBottom:
	    case CloseBottom:
	    case OpenGap:
	    case CloseGap:
	    case StackClose:
	    case StackB:
	    case StackC:
	    case TopGap:
	    case GapBottom:
		if (prefmath) open_math();
		printlatex(True,LATEX_TAB(c));
		break;
	    case TabOpen:
		if (!in_tabbing) {
		    in_tabbing = True;
		    nrdisp=1;
		    close_math();
		    printlatex(True, LATEX_TAB(TabOpen));
		} else if (texoutput==1) {
		    nrdisp++;
		    close_math();
		    printlatex(True, LATEX_TAB(TabOpen));
		} else nrdisp++;
		break;
	    case TabClose:
		nrdisp--;
		if (!nrdisp) {
		    in_tabbing = False;
		    close_math();
		    printlatex(True, LATEX_TAB(TabClose));
		} else if (texoutput==1) {
		    close_math();
		    printlatex(True, LATEX_TAB(TabClose));
		}
		break;
	    case DisplayOpen:
		tex_open_proof();
		if (texoutput==1)
		    printlatex(True, LATEX_TAB(DisplayOpen));
		break;
	    case DisplayClose:
		tex_close_proof();
		if (texoutput==1)
		    printlatex(True, LATEX_TAB(DisplayClose));
		break;
	    default:
		printlatex(True,LATEX_TAB(c));
		break;
	    }
	}
    } else if (IsPh(c)) {
	if (texmode==ASCII) {
	    if (texoutput==1)
		out_char(c);
	    else if (placepos || nameset) {
		if (!nameset) placename[placepos]='\0';
		printlatex(True,placename);
		placepos=nameset=0;
	    } else
		printlatex(True,ASCII_PH(c));
	} else {
	    if (tex_plhl) {
		if (Ph(c)!=Text) open_math();
		if (placepos || nameset) {
		    if (!nameset) placename[placepos]='\0';
		    printlatex(True,NAMEDHEAD);
		    printlatex(True,placename);
		    printlatex(True,NAMEDTAIL);
		    placepos=nameset=0;
		} else printlatex(True,LATEX_PH(c));
		if (Ph(c)!=Text) printlatex(True,LATEX_PHNUM(c));
	    }
	    if (texoutput==1) {
		if (Ph(c) == Text)
		    close_math();
		else
		    open_math();
		out_char(c);
	    }
	}
    } else if (IsOpspace(c)) {
	if (texmode!=ASCII) {
	    char oss[10];
	    int i = Char2ASCII(c);

	    if (i>0) {
		switch (texmode) {
		case PROOFTEX:
		    close_slash();
		    while (i) {
			printlatex(False, "\\,");
			i--;
		    }
		    break;
		case PLAINTEX:
		    break;
		case MPTEX:
		default:
		    sprintf(oss, "\\ms{%i}", i);
		    close_slash();
		    printlatex(True,oss);
		    break;
		}
	    }
	}
    } else if (Char2Font(c)==StackFont) {
	if (texmode!=ASCII) {
	    open_math();
	    close_slash();
	    printlatex(False,LATEX_TAB(StackC));
	}
    } else if (Char2Font(c)==FontFont) {
	if (texmode!=ASCII) {
	    char *h=NULL;
	    int npm=prefmath;
	    if (npm) {
		h = font_openmathtex(Char2ASCII(c),0);
		if (!h && (h= font_opentexttex(Char2ASCII(c),0)))
		    npm=0;
	    } else {
		h = font_opentexttex(Char2ASCII(c),0);
		if (!h && (h= font_openmathtex(Char2ASCII(c),0)))
		    npm=1;
	    }
	    push_math_pref(npm);
	    if (h) {
		if (!prefmath)
		    close_math();
		else
		    open_math();
		secure=True;
		printlatex(True,h);
		update_after(h);
	    }
	}
    } else if (Char2Font(c)==PopFont) {
	if (texmode!=ASCII) {
	    char *h=font_closetex(Char2ASCII(c),0);
	    if (h) {
		if (!prefmath)
		    close_math();
		else
		    open_math();
		secure=True;
		printlatex(True,h);
		update_after(h);
	    }
	    pop_math_pref();
	}
    } else if (Char2Font(c)==SizeFont) {
	if (texmode!=ASCII) {
	    if (!prefmath)
		close_math();
	    else
		open_math();
	    printlatex(True,SIZELATEX((int)Char2ASCII(c)));
	}
    } else if (c<(Char)(BUTTONFONT*256)) {
	if (!(lc = char_latex(c, pref_math[nrpref])))
	    if (!(lc = char_latex(c, !pref_math[nrpref])))
		newmode = pref_math[nrpref];
	    else
		newmode = !pref_math[nrpref];
	else
	    newmode = pref_math[nrpref];
	if (texmode!=ASCII && newmode && !in_math) open_math();
	if (texmode!=ASCII && !newmode && in_math) close_math();
	if (!lc) {
	    close_slash();
	    printlatex(False," ");
	    after_macro = False;
	} else {
	    if (texmode!=ASCII) {
		if (strlen(lc)>1) {
		    if (isalpha(*lc))
			close_macro();
		    else
			close_slash();
		    secure=True;
		    printlatex(True,lc);
		} else {
		    if (secure) {
			if (isalpha(*lc))
			    close_macro();
			else
			    close_slash();
			secure=False;
		    }
		    printlatex(False,lc);
		}
		update_after(lc);
	    } else
		if (strlen(lc)>1)
		    printlatex(True,lc);
		else
		    printlatex(False,lc);
	}
    }
}

void set_display_delta(int d)
{
    display_delta = d;
}

void tex_to_mode(int tm)
{
    switch (tm) {
    case LTEXTMODE:
	if (prefmath!=False) {
	    pop_math_pref();
	    push_math_pref(False);
	}
	break;
    case LMATHMODE:
	if (prefmath!=True) {
	    pop_math_pref();
	    push_math_pref(True);
	}
	break;
    case LBOTHMODE:
	pop_math_pref();
	push_math_pref(prefmath);
	break;
    default: break;
    }
}

void tex_code(TexCode c)
{
    placepos=nameset=0;
    if (texmode!=ASCII) {
	secure = True;
	switch (c) {
	case ExprOpen:
	    if (!setms && texf) {
		close_slash();
		fprintf(texf, "\\setms{%s}",latex_space_unit);
		setms=True;
	    }
	    push_math_pref(True);
	    break;
	case ExprClose:
	    pop_math_pref();
	    break;
	case SOpOpen:
	    switch (texmode) {
	    case PLAINTEX:
		break;
	    case PROOFTEX:
	    case MPTEX:
	    default:
		open_math();
		printlatex(False,"{");
		after_macro = False;
		break;
	    }
	    push_math_pref(True);
	    break;
	case SOpClose:
	    switch (texmode) {
	    case PLAINTEX:
		break;
	    case PROOFTEX:
	    case MPTEX:
	    default:
		open_math();
		printlatex(False,"}");
		after_macro = False;
		break;
	    }
	    pop_math_pref();
	    break;
	case LOpOpen:
	    push_math_pref(True);
	    break;
	case LOpClose:
	    pop_math_pref();
	    break;
	case SIdOpen:
	    open_math();
	    push_math_pref(True);
	    break;
	case SIdClose:
	    pop_math_pref();
	    break;
	case LIdOpen:
	    open_math();
	    printlatex(False,"\\mathit{");
	    after_macro = False;
	    push_math_pref(in_math);
	    break;
	case LIdClose:
	    if (prefmath != in_math) {
		if (in_math)
		    close_math();
		else
		    open_math();
	    } else close_slash();
	    in_math = prefmath;
	    pop_math_pref();
	    printlatex(False,"\\/}");
	    after_macro = False;
	    break;
	case VarOpen:
	    push_math_pref(True);
	    break;
	case VarClose:
	    pop_math_pref();
	    break;
	case TextOpen:
	    secure = False;
	    if (prefmath == TrueTrue && in_math) {
		close_slash();
		printlatex(False,"\\mbox{");
		after_macro = False;
		in_math = False;
	    }
	    push_math_pref(False);
	    break;
	case TextClose:
	    secure = False;
	    pop_math_pref();
	    if (prefmath == TrueTrue) {
		close_math();
		printlatex(False,"}");
		after_macro = False;
		in_math = TrueTrue;
	    }
	    break;
	case DispOpen:
	    if (!setms && texf) {
		close_slash();
		fprintf(texf, "\\setms{%s}",latex_space_unit);
		setms=True;
	    }
	    tex_open_proof();
	    push_math_pref(True);
	    display_delta = 0;
	    break;
	case DispClose:
	    tex_close_proof();
	    pop_math_pref();
	    break;
	default:
	    break;
	}
    }
}
