open Printf
open Mstring
open Uri
open Url

(* An hypertext(media) link on the Web *)

(* This is currently for HTTP and derived, but ... *)
(* Contains only the one we support *)
type link_method =
   GET 
 | HEAD
 | POST of string

let parse_method = function
   "GET" -> GET
 | "HEAD" -> HEAD
 | "POST" -> POST ""
 | _ -> raise Not_found (* other cases should be caught by caller ! *)


type link = {
  h_uri : string;
  h_context: string option;
  h_method : link_method		(* default is GET *)
  }

type link_error =
    LinkResolve of string
  | UrlLexing of string * int

exception Invalid_link of link_error

(* parsed Absolute URL + URL -> Absolute URL *)
(* NO FRAGMENT HANDLING *)

let urlconcat contextp newuri =
  let l = String.length newuri in 
    if l = 0 then string_of contextp 
    else if l > 2 & newuri.[0] = '/' & newuri.[1] = '/' then
      (* this is probably a gopher relative uri *)
      sprintf "%s:%s" (string_of_protocol contextp.protocol) newuri
    else if newuri.[0] = '/' then (* start from root *)
      string_of {
      	 protocol = contextp.protocol;
	 user = contextp.user;
	 password = contextp.password;
      	 host = contextp.host;
      	 port = contextp.port;
	 path = Some (String.sub newuri 1 (String.length newuri - 1));
	 search = None }
    else if newuri.[0] = '?' then (* change only search part *)
      string_of {
      	 protocol = contextp.protocol;
	 user = contextp.user;
	 password = contextp.password;
      	 host = contextp.host;
      	 port = contextp.port;
	 path = contextp.path;
	 search = Some (String.sub newuri 1 (String.length newuri - 1))}
    else match contextp.path with
      None | Some "" -> 
      	 string_of {
	    protocol = contextp.protocol;
	    user = contextp.user;
	    password = contextp.password;
	    host = contextp.host;
	    port = contextp.port;
	    path = Some (Lexurl.remove_dots newuri);
	    search = None}
    | Some old ->
        (* only the "dirname" part of the context path is important *)
        (* e.g  .../d/e/f becomes /d/e/ *)
      	let path = sprintf "%s/%s" (Filename.dirname old) newuri in
        (* we then have to remove dots *)
	let reduced = Lexurl.remove_dots path in
      	 string_of {
	    protocol = contextp.protocol;
	    user = contextp.user;
	    password = contextp.password;
	    host = contextp.host;
	    port = contextp.port;
	    path = Some reduced;
	    search = None}
	      
(* Produces an URL *)
let resolve link =
  (* First remove the possible fragment of the uri *)
  let newuri, frag =
    try
      let pos = first_char_pos '#' link.h_uri in
      	String.sub link.h_uri 0 pos, 
        Some (String.sub link.h_uri (succ pos) 
                    (String.length link.h_uri - pos - 1))
    with
       	Not_found -> link.h_uri, None 
  in
  if Uri.is_absolute newuri then
    try
     {uri_url = Lexurl.normalize newuri;
      uri_frag = frag}
    with
      Url_Lexing _ ->
	raise (Invalid_link
	          (LinkResolve (I18n.sprintf "not a legal absolute uri")))

  else begin (* It is a relative uri *)
    let context =
      match link.h_context with 
	 None -> 
	  raise (Invalid_link (LinkResolve (I18n.sprintf 
				  "no context and not an absolute url")))
       | Some c -> c in

    let contextp = 
       try Lexurl.make context
       with
	Url_Lexing (err,pos) ->
	 raise (Invalid_link (UrlLexing (err,pos)))
       in
    {uri_url = urlconcat contextp newuri;
     uri_frag = frag}
     end

let string_of link =
  let uri = resolve link in
   match uri.uri_frag with 
      None -> uri.uri_url
    | Some f -> Printf.sprintf "%s#%s" uri.uri_url f


type func = {
  hyper_visible : bool;
  hyper_title : string;
  hyper_func : link -> unit
  }
