/*  Light-weight streams.
    These are meant to be used with the same syntax as the C++ streams
    classes, but are much more economical.  (Use of iostreams makes the
    executable files much bigger, e.g., over 300K on UNIX.)

    Author: John Collins, collins@phys.psu.edu.
    21 Jan 96

    (C) John Collins & Penn State University.
*/

#include "sos.h"
// That includes an implicit <stdio.h>


/* ==================== class sos ========================
   "sos" means "Simple_ostream".
   It is used for output files that have already been opened,
   e.g., stdout.
   Note that assignment works.  E.g.,
      sos oldf(stderr);
      sos newf = oldf;
      sos newf1;
      newf1 = oldf;
   At present, this is done by C++'s implicitly generated copy constructor,
   which just copies the data members of the object, i.e., the pointer myfile.
*/


sos::sos(FILE *f) {
   // For debugging, to show how many constructors are called.
   //   fprintf (stderr, "=== Opening an sos(FILE*).\n");
   myfile = f;
}

sos::~sos() {flush();}

sos & sos::put(char c) {
   if (myfile) fputc (c, myfile);
   return *this;
}

sos & sos::write(const char *string, int length) {
   if (myfile) fwrite (string, sizeof(char), length, myfile);
   return *this;
}

sos & sos::operator<<(char *s) {
   if (myfile) fputs (s, myfile);
   return *this;
}

sos & sos::operator<<(char c) {
   if (myfile) fputc (c, myfile);
   return *this;
}

sos & sos::operator << (int i) {
   // Write a string representation of the integer to the stream
   fprintf (myfile, "%i", i);
   return *this;
}

sos & sos::operator << (void manipulator(sos &)) {
// For manipulators
   manipulator(*this);
   return *this;
}


void sos::endl () { put('\n'); flush(); }
void endl (sos & f) { f.endl(); }

void sos::flush() { if (myfile) fflush(myfile); }
void flush (sos & f) { f.flush(); }





/* ==================== class fsos ========================
   This is sos with means to open named files and close them.
   Note that assignment works.  E.g.,
      fsos oldf("FILENAME");
      sos  newf = oldf;
      fsos newfsos = newf
      sos  newf1;
      newf1 = oldf;
   At present, this is done by C++'s implicitly generated copy constructor,
   which just copies the data members of the object, i.e., the pointer myfile.
*/

fsos::fsos (char *filename) {
   open(filename);
}

fsos::~fsos() { close(); }

int fsos::open (char *filename) {
   myfile = fopen (filename, "w");
   return IsOpen();
}

int fsos::close() {
   // Close the associated file.  Return the error code given by fclose().
  if (IsOpen()) {
     flush();
     int retcode = fclose(myfile);
     myfile = NULL;
     return retcode;
  }
  else {
     myfile = NULL;
     return EOF;
  }
}


// ============== Standard streams:
sos lite_cout(stdout), lite_cerr(stderr);

