/*
 * config.c  -  Read configuration files
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define NO_BINARY 1	/* No need to include binary support here */
#include "common.h"
#include "makerom.h"



/*
 * Variables private to this module
 */
static int curline;			/* current line in input file */
static int errors;			/* number of errors found */
static struct bootdef bootd;		/* boot definition record */



/*
 * List of recognized section names in configuration file
 */
typedef enum {
	sect_none,
	sect_makerom,
	sect_kernel,
	sect_pktdrv
} sectypes;

struct sectlist {
	char     *name;
	sectypes  type;
};

static struct sectlist sections[] = {
	{ "[makerom]",	sect_makerom },
	{ "[kernel]",	sect_kernel },
	{ "[pktdrv]",	sect_pktdrv },
	{ NULL,		sect_none }
};



/*
 * List of recognized options for binary files
 */
struct optlist {
	char *optname;
	int   optval;
};

static struct optlist pktoptlist[] = {
	{ "HW_IRQ",	HW_IRQ },
	{ "IO_ADDR",	IO_ADDR },
	{ "BASE_MEM",	BASE_MEM },
	{ "AUI_TYPE",	AUI_TYPE },
	{ "DMA_NUM",	DMA_NUM },
	{ NULL,		0 }
};

static struct optlist kernoptlist[] = {
	{ "MINIMAL",	K_MINIMAL },
	{ "X86",	K_X86 },
	{ NULL,		0 }
};



/*
 * List of recognized parameters
 */
typedef enum {
	par_null,
	par_string,
	par_int,
	par_bool
} paramtypes;

struct paramlist {
	char       *name;
	paramtypes  type;
	union {
		char **charptr;
		int   *intptr;
		int   *boolptr;
	} var;
};

/* Paramters in makrom section of configuration file */
static struct paramlist configparams[] = {
	{ "netbootdir",	par_string,	{&netbootdir}},
	{ "database",	par_string,	{&dbname}},
	{ "kerneldir",	par_string,	{&kerneldir}},
	{ "loaderdir",	par_string,	{&loaderdir}},
	{ "pktdrvdir",	par_string,	{&pktdrvdir}},
	{ "utilsdir",	par_string,	{&utilsdir}},
	{ NULL,		par_null,	{NULL}}
};

/* Parameters in each bootrom section of database file */
static struct paramlist dbparams[] = {
	{ "kernel",	par_string,	{&bootd.kernelname}},
	{ "useint18",	par_bool,	{(char **)&bootd.use_int18}},
#if MAXLOADERS < 3
#error Invalid number of loaders
#endif
	{ "loader1",	par_string,	{&bootd.loadernames[0]}},
	{ "loader2",	par_string,	{&bootd.loadernames[1]}},
	{ "loader3",	par_string,	{&bootd.loadernames[2]}},
	{ "outname1",	par_string,	{&bootd.outnames[0]}},
	{ "outname2",	par_string,	{&bootd.outnames[1]}},
	{ "outname3",	par_string,	{&bootd.outnames[2]}},
	{ "flash1",	par_bool,	{(char **)&bootd.flashflags[0]}},
	{ "flash2",	par_bool,	{(char **)&bootd.flashflags[1]}},
	{ "flash3",	par_bool,	{(char **)&bootd.flashflags[2]}},
#if MAXDRIVERS < 8
#error Invalid number of drivers
#endif
	{ "driver1",	par_string,	{&bootd.drivernames[0]}},
	{ "driver2",	par_string,	{&bootd.drivernames[1]}},
	{ "driver3",	par_string,	{&bootd.drivernames[2]}},
	{ "driver4",	par_string,	{&bootd.drivernames[3]}},
	{ "driver5",	par_string,	{&bootd.drivernames[4]}},
	{ "driver6",	par_string,	{&bootd.drivernames[5]}},
	{ "driver7",	par_string,	{&bootd.drivernames[6]}},
	{ "driver8",	par_string,	{&bootd.drivernames[7]}},
	{ "argument1",	par_string,	{&bootd.driverargs[0]}},
	{ "argument2",	par_string,	{&bootd.driverargs[1]}},
	{ "argument3",	par_string,	{&bootd.driverargs[2]}},
	{ "argument4",	par_string,	{&bootd.driverargs[3]}},
	{ "argument5",	par_string,	{&bootd.driverargs[4]}},
	{ "argument6",	par_string,	{&bootd.driverargs[5]}},
	{ "argument7",	par_string,	{&bootd.driverargs[6]}},
	{ "argument8",	par_string,	{&bootd.driverargs[7]}},
	{ NULL,		par_null,	{NULL}}
};



/*
 * Read a line from the setup file
 */
static char *readline(fd)
FILE *fd;
{
  int len;
  char c, *cp = NULL;
  static char buf[1024];

  while (cp == NULL) {
	/* Read one line */
	if (fgets(buf, sizeof(buf)-1, fd) == NULL)
		return(NULL);
	curline++;
	len = strlen(buf);
	/* Skip everything in the line which is longer than the buffer */
	if (buf[len - 1] != '\n') {
		while ((c = fgetc(fd)) != '\n' && c != EOF)
			;
	} else {
		len--;
		buf[len] = '\0';
	}
	/* Remove trailing comments */
	if ((cp = strchr(buf, '#')) != NULL)
		*cp = '\0';
	else
		cp = &buf[len];
	cp--;
	/* Remove trailing whitespace */
	while (cp >= buf && (*cp == ' ' || *cp == '\t'))
		cp--;
	*(++cp) = '\0';
	/* Skip empty lines */
	if (cp == buf)
		cp = NULL;
  }

  /* Skip leading whitespace */
  for (cp = buf; *cp && (*cp == ' ' || *cp == '\t'); cp++)
	;
  return(cp);
}



/*
 * Decode one line of the configuration file
 */
static void decode(params, buf)
struct paramlist *params;
char *buf;
{
  char *cp;
  char *arg;
  int i;

  /* Find equal sign and isolate parameter name */
  if ((cp = strchr(buf, '=')) == NULL) {
	fprintf(stderr, "%s: [%d] invalid line\n", progname, curline);
	errors++;
	return;
  }
  arg = cp + 1;
  for (--cp; cp > buf && (*cp == ' ' || *cp == '\t'); cp--)
	;
  *(++cp) = '\0';

  /* Find parameter name in list */
  for (i = 0; params[i].name != NULL; i++)
	if (!strcmp(buf, params[i].name))
		break;
  if (params[i].name == NULL) {
	fprintf(stderr, "%s: [%d] invalid parameter\n", progname, curline);
	errors++;
	return;
  }

  /* Decode argument */
  while (*arg && (*arg == ' ' || *arg == '\t'))
	arg++;
  switch (params[i].type) {
	case par_string:
		if ((cp = strdup(arg)) == NULL) {
			perror(progname);
			exit(EXIT_MEMORY);
		}
		if (*params[i].var.charptr != NULL)
			free(*params[i].var.charptr);
		*params[i].var.charptr = cp;
		break;
	case par_bool:
		if (!strcmp(arg, "true"))
			*params[i].var.boolptr = 1;
		else if (!strcmp(arg, "false"))
			*params[i].var.boolptr = 0;
		else {
			fprintf(stderr, "%s: [%d] invalid boolean argument\n",
							progname, curline);
			errors++;
		}
		break;
	case par_int:
		if (sscanf(arg, "%d", params[i].var.intptr) != 1) {
			fprintf(stderr, "%s: [%d] invalid integer argument\n",
							progname, curline);
			errors++;
		}
		break;
	case par_null:
	default:
		break;
  }
}



/*
 * Add new item to description list
 */
static void additem(dlist, buf)
struct desclist **dlist;
char *buf;
{
  struct desclist *ditem;
  struct optlist *opts;
  char *cp, *end;
  int i;

  /* Allocate memory for current description item */
  if ((ditem = malloc(sizeof(struct desclist))) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }

  /* Get name of item */
  for (cp = buf; *cp && *cp != ' ' && *cp != '\t'; cp++)
	;
  *cp = '\0';
  if (strlen(buf) == 0 || (ditem->progname = strdup(buf)) == NULL) {
	fprintf(stderr, "%s: [%d] invalid entry\n", progname, curline);
	free(ditem);
	return;
  }
  for (buf = cp + 1; *buf && (*buf == ' ' || *buf == '\t'); buf++)
	;

  /* Get description string */
  ditem->descript = NULL;
  if (*buf == '"') {
	buf++;
	for (cp = buf; *cp && *cp != '"'; cp++)
		;
	*cp = '\0';
  } else {
	for (cp = buf; *cp && *cp != ' ' && *cp != '\t'; cp++)
		;
	*cp = '\0';
  }
  if (strlen(buf) > 0)
	ditem->descript = strdup(buf);
  for (buf = cp + 1; *buf && (*buf == ' ' || *buf == '\t'); buf++)
	;

  /* Get program options - we only have options for packet drivers */
  ditem->options = 0;
  if (dlist == &pktdrvdesc)
	opts = pktoptlist;
  else if (dlist == &kerneldesc)
	opts = kernoptlist;
  else
	opts = NULL;

  if (opts != NULL) {
	/* Options are only seperated by ',' so skip to next blank */
	for (end = buf; *end && *end != ' ' && *end != '\t'; end++)
		*end = toupper(*end);
	*end = '\0';
	while (buf < end) {
		for (cp = buf; *cp && *cp != ','; cp++)
			;
		*cp = '\0';
		if (strlen(buf) > 0) {
			for (i = 0; opts[i].optname != NULL; i++)
				if (!strcmp(opts[i].optname, buf))
					break;
			if (opts[i].optname == NULL) {
				fprintf(stderr, "%s: [%d] invalid option %s\n",
							progname, curline, buf);
				errors++;
			} else
				ditem->options |= opts[i].optval;
		}
		buf = cp + 1;
	}
  }

  /* Insert new item into list */
  ditem->next = *dlist;
  *dlist = ditem;
}



/*
 * Open configuration file
 */
static FILE *openconfig(fname)
char *fname;
{
  FILE *fd = NULL;

  if (fname == NULL)
	return(NULL);
  fd = fopen(fname, "r");
  if (verbose > 0 && fd != NULL)
	printf("Configuration file      = %s\n", fname);
  return(fd);
}



/*
 * Read configuration file
 */
void readconfig(fname)
char *fname;
{
  sectypes cursect = sect_none;
  FILE *fd;
  char *buf;
  int i;

  /* Nothing to do if we cannot open the setup file */
  if ((fd = openconfig(fname)) == NULL)
	return;
  curline = 0;
  errors = 0;

  /* Read all lines in input file */
  while ((buf = readline(fd)) != NULL) {
	if (*buf == '[') {
		for (i = 0; sections[i].name != NULL; i++)
			if (!strcmp(buf, sections[i].name))
				break;
		if (sections[i].name == NULL) {
			fprintf(stderr, "%s: [%d] invalid section name\n",
							progname, curline);
			errors++;
		}
		cursect = sections[i].type;
	} else switch (cursect) {
		case sect_none:
			fprintf(stderr, "%s: [%d] definition outside of section\n",
							progname, curline);
			errors++;
			break;
		case sect_makerom:
			decode(configparams, buf);
			break;
		case sect_kernel:
			additem(&kerneldesc, buf);
			break;
		case sect_pktdrv:
			additem(&pktdrvdesc, buf);
			break;
		default:
			break;
	}
  }

  if (errors > 0) {
	fprintf(stderr, "%s: found %d errors in file %s, terminating\n",
						progname, errors, fname);
	exit(EXIT_MAKEROM_CONFIG);
  }
  fclose(fd);
}



/*
 * Read one entry from the database file
 */
struct bootdef *getdb(name)
char *name;
{
  FILE *fd;
  char *buf, *cp, *nambuf;
  int i, j;
  int foundit = 0;
  size_t len;
  enum {
	no_section,
	notin_section,
	in_section
  } cursect = no_section;

  /* Open database file */
  if ((fd = fopen(dbname, "r")) == NULL) {
	fprintf(stderr, "%s: unable to open database file %s\n", progname, dbname);
	exit(EXIT_MAKEROM_DBFILE);
  }
  curline = 0;
  errors = 0;

  /* Generate section name */
  len = strlen(name) + strlen(progname) + 2;
  if ((nambuf = malloc(len)) == NULL) {
	perror(progname);
	exit(EXIT_MEMORY);
  }
  sprintf(nambuf, "%s:%s", name, progname);

  /* Read all lines in the database file */
  memset(&bootd, 0, sizeof(bootd));
  while ((buf = readline(fd)) != NULL) {
	if (*buf == '[') {
		if ((cp = strchr(buf, ']')) == NULL || *(cp+1) != '\0') {
			fprintf(stderr, "%s: [%d] invalid section name\n",
							progname, curline);
			errors++;
		}
		*cp = '\0';
		if (!strcmp(buf+1, nambuf)) {
			if (foundit) {
				fprintf(stderr, "%s: section %s defined twice, skipping\n",
							progname, name);
				cursect = notin_section;
			} else {
				cursect = in_section;
				foundit++;
			}
		} else
			cursect = notin_section;
	} else switch (cursect) {
		case no_section:
			fprintf(stderr, "%s: [%d] definition outside of section\n",
							progname, curline);
			errors++;
			break;
		case in_section:
			decode(dbparams, buf);
		case notin_section:
		default:
			break;
	}
  }

  if (errors > 0) {
	fprintf(stderr, "%s: found %d errors in database file, terminating\n",
							progname, errors);
	exit(EXIT_MAKEROM_DBERR);
  }
  fclose(fd);
  free(nambuf);

  if (!foundit)
	return(NULL);

  /* Reorder the loader and driver lists, so that there are no holes */
  for (i = 0, j = -1; i < MAXLOADERS; i++) {
	if (bootd.loadernames[i] == NULL && j < 0)
		j = i;
	else if (bootd.loadernames[i] != NULL && j >= 0) {
		bootd.loadernames[j] = bootd.loadernames[i];
		bootd.outnames[j] = bootd.outnames[i];
		bootd.flashflags[j] = bootd.flashflags[i];
		bootd.loadernames[i] = NULL;
		i = j;
		j = -1;
	}
  }

  for (i = 0, j = -1; i < MAXDRIVERS; i++) {
	if (bootd.drivernames[i] == NULL && j < 0)
		j = i;
	else if (bootd.drivernames[i] != NULL && j >= 0) {
		bootd.drivernames[j] = bootd.drivernames[i];
		bootd.driverargs[j] = bootd.driverargs[i];
		bootd.drivernames[i] = NULL;
		i = j;
		j = -1;
	}
  }

  /* Check all entries */
  if (bootd.kernelname == NULL) {
	fprintf(stderr, "%s: no kernel file name given\n", progname);
	exit(EXIT_MAKEROM_NOKERN);
  }

  bootd.loadernum = 0;
  for (i = 0; i < MAXLOADERS; i++)
	if (bootd.loadernames[i] != NULL) {
		if (bootd.outnames[i] == NULL) {
			fprintf(stderr, "%s: no output file given for loader %s\n",
						progname, bootd.loadernames[i]);
			exit(EXIT_MAKEROM_NOOUT);
		}
		bootd.loadernum++;
	}
  if (bootd.loadernum == 0) {
	fprintf(stderr, "%s: no bootrom loader specified\n", progname);
	exit(EXIT_MAKEROM_NOLOADER);
  }

  bootd.drivernum = 0;
  for (i = 0; i < MAXDRIVERS; i++)
	if (bootd.drivernames[i] != NULL) {
		if (bootd.driverargs[i] == NULL)
			bootd.driverargs[i] = strdup("");
		bootd.drivernum++;
	}
  if (bootd.drivernum == 0) {
	fprintf(stderr, "%s: no driver program specified\n", progname);
	exit(EXIT_MAKEROM_NODRIVER);
  }

  return(&bootd);
}

