! first.S  -  primary boot loader for DOS
!
! Copyright (C) 1996-1998 Gero Kuhlmann   <gero@gkminix.han.de>
!
!  This program is free software; you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation; either version 2 of the License, or
!  any later version.
!
!  This program is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with this program; if not, write to the Free Software
!  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#ifndef ASM_DEBUG
#undef ASM_DEBUG
#endif
#include "first.inc"

	.text

	.org	0

	mov	dx,ds
	mov	ax,cs			! set DS
	mov	ds,ax
	mov	oldES,es
	mov	oldDS,dx		! save old register values in case
	mov	oldBP,bp		! we have to return to the boot rom
	mov	oldSI,si
	mov	oldDI,di
	mov	bp,sp
	mov	ax,[bp+4]
	mov	[header+0],ax		! load the address of the boot header
	mov	ax,[bp+6]
	mov	[header+2],ax
	mov	ax,[bp+8]
	mov	[bootp+0],ax		! load the address of the bootp block
	mov	ax,[bp+10]
	mov	[bootp+2],ax

! Tell the user who we are and that we started running

	mov	si,#sigmsg
	call	prnstr

! Check if the boot image header is correct.

	les	bx,header
	mov	si,#bmerr		! prepare for correct error message
	seg	es
	mov	ax,BOOT_HD_MAGIC+0[bx]
	cmp	ax,[bmagic+0]		! compare boot rom magic numbers
	jne	doerr1
	seg	es
	mov	ax,BOOT_HD_MAGIC+2[bx]
	cmp	ax,[bmagic+2]
	jne	doerr1

	mov	si,#vmerr		! prepare for correct error message
	seg	es
	mov	al,BOOT_HD_LENGTH[bx]
	mov	cl,#4
	shr	al,cl
	and	al,#0x0F
	cmp	al,#VENDOR_SIZE		! check vendor ID size
	jne	doerr1
	xor	di,di
dovmag:	mov	al,vmagic[di]		! check vendor ID
	or	al,al
	jz	getrd			! vendor ID ok, continue
	seg	es
	cmp	al,BOOT_HD_VENDOR[bx+di]
	jne	doerr1
	inc	di
	jmp	dovmag

doerr1:	call	prnstr			! in case of error return to the
	mov	si,oldSI		! boot rom with registers set
	mov	di,oldDI		! correctly
	mov	bp,oldBP
	mov	es,oldES
	mov	ds,oldDS
	retf

! Next get the address of the ramdisk and its size.

getrd:	mov	si,#recerr
	mov	al,#VENDOR_RAMDISK
	call	fndldr				! find load record for ramdisk
	mov	ax,es
	or	ax,di
	jz	doerr1
	seg	es
	mov	al,BOOT_LD_FLAGS[di]		! get load record flags
	test	al,#BOOT_FLAG_B0 + BOOT_FLAG_B1	! check that it has a
	jnz	doerr1				! correct flag

	seg	es
	mov	ax,BOOT_LD_ADDR+0[di]		! get base adress of ramdisk
	seg	es
	mov	bx,BOOT_LD_ADDR+2[di]
	mov	[rdaddr+0],ax
	mov	[rdaddr+2],bx

	seg	es
	mov	ax,BOOT_LD_MLENGTH+0[di]	! get ramdisk size
	seg	es
	mov	bx,BOOT_LD_MLENGTH+2[di]
	add	ax,#0x03ff			! round to nearest kb
	adc	bx,#0
	mov	cl,#10
	shr	ax,cl				! convert into kb
	mov	cl,#6
	shl	bx,cl
	or	ax,bx
	mov	rdsize,ax

! Get the disk geometry out of the vendor information block in the
! load record

	seg	es
	mov	bl,BOOT_LD_LENGTH[di]
	and	bl,#0x0f
	xor	bh,bh				! compute pointer to
	shl	bx,#1				! vendor block
	shl	bx,#1
	seg	es
	mov	ax,BOOT_LD_SECNUM[di+bx]	! get number of sectors
	mov	secnum,ax
	seg	es
	mov	ax,BOOT_LD_SPT[di+bx]		! get sectors per track
	mov	secptk,al
	seg	es
	mov	ax,BOOT_LD_CYL[di+bx]		! get number of cylinders
	mov	cylnum,ax
	seg	es
	mov	al,BOOT_LD_DRIVE[di+bx]		! get ram disk drive id
	mov	drvid,al
	seg	es
	mov	al,BOOT_LD_NOHD[di+bx]		! get no-hard-disk flag
	mov	nohd,al

! Set the address of the BIOS disk status byte

	mov	bx,#BIOS_FDSTAT
	cmp	al,#0x80
	jb	setsts
	mov	bx,#BIOS_HDSTAT
setsts:	mov	statof,bx

! Get the number of floppy or hard disk drives in the system and set
! the DOS disk parameter block

	cmp	al,#0x80
	jae	getnm2

	mov	ah,#0x08
	xor	dl,dl
	int	0x13			! get the number of floppy disk
	jc	getnm1			! drives from the BIOS
	or	dl,dl
	jnz	gotnum
	inc	dl			! indicate at least one drive
	jmp	gotnum
getnm1:	int	0x11			! if int 13h didnt work try it with
	mov	cl,#6			! the mainboard dip switch config
	shr	ah,cl
	and	ah,#0x03		! determine number of floppy disk
	inc	ah			! drives
	mov	dl,ah
	jmp	gotnum

getnm2:	mov	ah,#0x08
	mov	dl,#0x80
	int	0x13			! get the number of hard disk
	jc	getnm3			! drives from the BIOS
	inc	dl
	jmp	gotnum
getnm3:	mov	dl,1			! we have at least one drive

gotnum:	mov	drvnum,dl		! save number of disk drives
	call	setdpb			! set disk parameter block

! Now get the boot sector of the ramdisk and check that its correct. If
! we are simulating a hard disk the boot sector contains the partition
! table, which we have to analyze. Then load the partitions boot sector.

	mov	ax,#TEMP_SEGMENT
	mov	es,ax			! pointer to temporary buffer
	xor	bx,bx
	xor	al,al			! indicate read
	xor	dx,dx			! first sector
	call	rwsect			! read boot sector
	mov	si,#rderr
	jc	doerr2
	cmp	byte ptr (drvid),#0x80	! if the ram disk is simulates a
	jb	chkbot			! floppy, there is no partition table

	mov	si,#dskerr		! prepare for correct error message
	seg	es
	cmp	byte ptr PART_STATUS,#PART_ACTIVE
	jne	doerr2
	seg	es
	cmp	byte ptr PART_TYPE,#PART_FAT16
	je	partok
	seg	es
	cmp	byte ptr PART_TYPE,#PART_FAT12
	jne	doerr2
partok:	seg	es
	mov	dx,[PART_ABS_SECT+0]	! get number of first sector
	seg	es
	mov	ax,[PART_ABS_SECT+2]
	or	ax,ax
	jnz	doerr2
	xor	al,al			! indicate read
	call	rwsect			! read boot sector
	mov	si,#rderr
	jc	doerr2

chkbot:	mov	si,#dskerr		! prepare for correct error message
	mov	al,drvid
	seg	es
	cmp	byte ptr DISK_BOOT,al		! check boot disk number
	jne	doerr2
	seg	es
	cmp	word ptr DISK_BPS,#SECT_SIZE	! check sector size
	jne	doerr2
	seg	es
	cmp	word ptr DISK_HEADS,#2	! check number of heads
	jne	doerr2
	seg	es
	mov	ax,DISK_SPT		! check number of sectors per track
	cmp	al,secptk
	je	dobotp

doerr2:	call	prnstr			! in case of error return to the
	mov	si,oldSI		! boot rom with registers set
	mov	di,oldDI		! correctly
	mov	bp,oldBP
	mov	es,oldES
	mov	ds,oldDS
	retf

! Save the BOOTP record for later retrieval by a DOS program.

dobotp:	cld
	xor	dx,dx
	les	di,bootp
	mov	ax,es
	or	ax,di
	jz	dobot9
	seg	es
	mov	al,BOOTP_OP[di]		! op code must indicate reply
	cmp	al,#BOOTP_REPLY
	jne	dobot9			! it isnt
	add	di,#BOOTP_VEND
	mov	bx,di
	mov	si,#pmagic		! compare vendor ID
dobot1:	mov	di,bx
	mov	cx,#BOOTP_MAGIC_LEN
	repe
	cmpsb
	jz	dobot2			! vendor ID is valid
	add	si,cx
	cmp	byte ptr [si],#0	! check next vendor ID
	jne	dobot1
dobot9:	jmp	nobot2			! vendor ID not found

doerr6:	jmp	doerr2

dobot2:	sub	si,#BOOTP_MAGIC_LEN
	sub	si,#pmagic
	mov	ax,si
	push	ds
	mov	bx,ds
	mov	es,bx
	mov	di,#btpnew
	lds	si,bootp
	mov	bx,si
	mov	dx,#BOOTP_SIZE
	or	ax,ax			! if not RFC vendor ID the bootp
	jnz	dobot7			! record has fixed length

	xor	cx,cx
	add	si,#BOOTP_VEND + BOOTP_MAGIC_LEN
dobot3:	lodsb
	cmp	al,#BOOTP_RFC_NOP	! handle NOP tag
	jnz	dobot4
	inc	cx
	cmp	cx,#16			! more than 16 NOP tags is VERY unusual
	jae	dobot7			! so the bootp record maybe broken
	jmp	dobot3			! loop to next tag

nobot2:	jmp	nobotp

dobot4:	cmp	al,#BOOTP_RFC_END	! handle END tag
	jnz	dobot6
	mov	dx,si
	sub	dx,bx			! compute length of bootp record
	cmp	dx,#BOOTP_SIZE
	jae	dobot7
	mov	dx,#BOOTP_SIZE		! use minimum size
	jmp	dobot7
dobot6:	lodsb				! handle all other tags
	mov	cl,al
	xor	ch,ch
	add	si,cx			! jump to next tag
	xor	cx,cx			! reset NOP counter
	jmp	dobot3			! proceed with next tag

dobot7:	mov	si,#btperr
	mov	ax,#btpnew		! bootp record cannot be larger
	add	ax,dx			! than the current segment
	jc	doerr6
	mov	cx,dx
	mov	si,bx			! restore source pointer
	rep
	movsb				! save the bootp record
	pop	ds
nobotp:	mov	btplen,dx		! set length of bootp record

! Everything is right, so we can now move the resident section to the
! end of the conventional memory, thus overwriting the bootrom data
! area. Therefore there is no chance of returning to the bootrom from
! now on.
! Note that the resident section doesnt start at offset 0, so we have
! to set the segment address to somewhere lower.

#ifdef ASM_DEBUG
	mov	si,#debug1
	call	prnstr
#endif
	cli
	mov	ax,#TEMP_SEGMENT		! set new stack
	mov	ss,ax
	mov	sp,#0xFFF0

	cld	
	int	0x12			! get memory size in kB
	mov	cl,#6
	shl	ax,cl			! compute last usable segment
	mov	bx,#btpnew
	add	bx,btplen
	mov	dx,bx
	mov	cl,#4
	shr	bx,cl			! compute size of code segment in
	inc	bx			! paragraphs
	sub	ax,bx			! compute new code segment
	mov	resseg,ax
	mov	es,ax			! set source and destination ptrs
	mov	si,#start_resident
	mov	di,si
	mov	cx,dx
	sub	cx,si			! compute size of resident area
	rep
	movsb				! move it

! Setup all interrupt vectors

	mov	bx,resseg
	push	ds
	mov	ds,bx
	xor	ax,ax
	mov	es,ax
	seg	es
	mov	ax,[I13_INT+0]
	mov	[old13h+0],ax
	seg	es
	mov	ax,[I13_INT+2]
	mov	[old13h+2],ax
	seg	es
	mov	ax,[I2F_INT+0]
	mov	[old2Fh+0],ax
	seg	es
	mov	ax,[I2F_INT+2]
	mov	[old2Fh+2],ax
	seg	es
	mov	ax,[IF1_INT+0]
	mov	[oldF1h+0],ax
	seg	es
	mov	ax,[IF1_INT+2]
	mov	[oldF1h+2],ax
	seg	es
	mov	ax,[IF8_INT+0]
	mov	[oldF8h+0],ax
	seg	es
	mov	ax,[IF8_INT+2]
	mov	[oldF8h+2],ax
	pop	ds

	seg	es
	mov	word ptr [I13_INT+2],bx	! interrupt vector 13h
	seg	es
	mov	word ptr [I13_INT+0],#int13
	seg	es
	mov	word ptr [I2F_INT+2],bx	! interrupt vector 2Fh
	seg	es
	mov	word ptr [I2F_INT+0],#int2F
	seg	es
	mov	word ptr [IF8_INT+2],bx	! interrupt vector F8h
	seg	es
	mov	word ptr [IF8_INT+0],#intF8
	mov	di,#IF1_INT
	mov	si,#if1sig		! interrupt vector F1h
	mov	cx,#4			! contains the string "NetB"
	rep				! to provide as an installation
	movsb				! check
	sti

! Output some debugging messages by simply calling the interrupt vectors
! which we just created.

#ifdef DRV_DEBUG
	mov	ax,#0x4A06
	int	0x2F
	mov	si,#consz
	call	prnstr			! print out amount of conventional
	mov	ax,dx			! memory
	mov	cl,#12
	shr	ax,cl
	call	prnwrd
	mov	ax,dx
	mov	cl,#4
	shl	ax,cl
	call	prnwrd
	mov	si,#bytes
	call	prnstr

	mov	ax,#0x8800
	int	0x15
	push	ax
	mov	si,#extsz
	call	prnstr			! print out amount of extended memory
	mov	cl,#6
	shr	ax,cl
	call	prnwrd
	pop	ax
	mov	cl,#10
	shl	ax,cl
	call	prnwrd
	mov	si,#bytes
	call	prnstr
#endif

! The boot sector had to be placed into a temporary memory area to
! avoid overwriting any bootrom structures. However, a call back
! to the bootrom is no longer possible, so we can move the bootblock
! where it belongs.

	push	ds
	mov	ax,#TEMP_SEGMENT
	mov	ds,ax
	xor	ax,ax
	mov	es,ax
	xor	si,si
	mov	di,#BOOT_OFFSET
	mov	cx,#SECT_SIZE
	rep
	movsb				! move it
	pop	ds

! Finally call the boot sector

#ifdef ASM_DEBUG
	mov	si,#debug2
	call	prnstr
#endif
#ifdef ASM_FREEZE_AFTER_INIT
lop:	jmp	lop
#else
	jmpi	BOOT_OFFSET,0
#endif



!====================================================================
!
! Setup DOS drive parameter block (DPB) for floppy disk drive. The
! DPB is required to activate the floppy disk drive after the ramdisk
! has been turned off.
! Input:  none
! Output: none
! Registers changed: AX, BX, CX, DX

setdpb:	push	es
	push	si
	push	di
	mov	dl,drvid
	cmp	dl,#0x80		! can only restore floppy drives
	jae	setdp8

! First get the drive parameters from the BIOS

	mov	dpb_phys,dl		! set physical drive ID
	mov	ah,#0x08		! get drive parameters from BIOS
	int	0x13
	jc	setdp8
	xor	ah,ah
	mov	al,ch			! set max number of cylinders
	inc	ax
	mov	dpb_cyls,ax
	mov	si,#dpb_bpb_cur
	mov	al,cl			! set max number of sectors per track
	mov	BPB_SPT[si],ax
	mov	al,dh
	inc	ax			! set max number of heads
	mov	BPB_HEADS[si],ax

! Determine DOS disk parameters by drive type

	cmp	bl,#5
	jb	setdp1			! check for invalid drive type
	mov	bl,#4
setdp1:	xor	bh,bh
	dec	bx
	push	bx
	shl	bx,#1
	shl	bx,#1			! compute address into drive para-
	add	bx,#drvtab		! meter table
	xor	ah,ah
	mov	al,[bx+0]		! get # of entries in root dir
	mov	BPB_DIR[si],al
	mov	al,[bx+1]		! get # of sectors per FAT
	mov	BPB_SPF[si],ax
	mov	al,[bx+2]		! get # of sectors per cluster
	mov	BPB_SPC[si],al
	mov	al,[bx+3]		! get media ID
	mov	BPB_MEDIA_ID[si],al
	pop	bx
	mov	al,typtab[bx]		! get drive type
	mov	dpb_type,al

! Determine number of bytes per sector

	seg	es
	mov	cl,[di+3]		! get shift value from BIOS media
	mov	ax,#128			! parameter table
	shl	ax,cl			! shift base value
	mov	BPB_BPS[si],ax
	jmp	setdp4
setdp8:	jmp	setdp9			! needed for short jumps

! Determine total number of sectors

setdp4:	mov	ax,BPB_SPT[si]
	mul	word ptr (dpb_cyls)
	or	dx,dx			! should not overflow
	jnz	setdp8
	cmp	word ptr BPB_HEADS[si],#2
	jb	setdp3
	shl	ax,#1
	jc	setdp8
setdp3:	mov	BPB_TOT_SECTS[si],ax

! Determine if the drive can detect disk changes

	mov	ah,#0x15
	mov	dl,drvid
	int	0x13			! get DASD type from BIOS
	mov	bl,ah
	mov	ax,#DPB_F_DEFAULT
	jc	setdp2
	cmp	bl,#0x02		! check if drive detects disk changes
	jne	setdp2
	or	ax,#DPB_F_DOOR
setdp2:	mov	dpb_flags,ax		! set flags

! Thats it

	inc	byte ptr (dpb_valid)	! increment valid flag
setdp9:	pop	di
	pop	si
	pop	es
	ret



!====================================================================
!
! Find a load record in the boot header. The ID number of the load
! record is in AL, and ES:DI points to requested load record, or is
! the NULL pointer if load record not found.
!
! Changed registers: AX, DI, ES

fndldr:	push	cx
	mov	ch,al
	les	di,header		! examine boot image header
	seg	es
	mov	al,BOOT_HD_LENGTH[di]	! get length of image header
	call	getlen
	add	di,ax			! get the pointer to first load record
fndl1:	seg	es
	cmp	ch,BOOT_LD_TAG1[di]	! is it the desired one ?
	je	fndl3
	seg	es
	mov	al,BOOT_LD_FLAGS[di]	! no, so check if its the last record
	test	al,#BOOT_FLAG_EOF
	jnz	fndl2
	seg	es
	mov 	al,BOOT_LD_LENGTH[di]	! no, get the address of the next one
	call	getlen
	add	di,ax
	jmp	fndl1

fndl2:	xor	ax,ax			! couldnt find the desired record
	mov	es,ax
	mov	di,ax
fndl3:	pop	cx
	ret



!====================================================================
!
! Compute the length of a load record address from a length byte
! in AL. Return the offset in AX.
!
! Changed registers: AX

getlen:	push	cx
	mov 	ah,al
	mov 	cl,#4
	shr	ah,cl
	and	ax,#0x0f0f		! compute the total length in
	add	al,ah			! bytes from the length of the
	xor	ah,ah			! record and that of the vendor
	shl	ax,1			! information.
	shl	ax,1
	pop	cx
	ret



!====================================================================
! Print a string in DS:SI onto the console
!
! Changed registers: AL

prnstr:	push	si
	cld
prns1:	lodsb				! loop over all characters of
	or	al,al			! string
	jz	prns2
	push	bx
	mov	ah,#0x0E		! print it
	mov	bl,#0x07
	xor	bh,bh
	int	0x10
	pop	bx
	jmp	prns1
prns2:	pop	si
	ret



#ifdef DRV_DEBUG
!====================================================================
!
! Print hexadecimal values (in AX or AL) or characters onto the console
!
! Changed registers: AX

prnwrd:	push	ax
	mov	al,ah
	call	prnbyt			! print the upper byte
	pop	ax
prnbyt:	push	ax
	shr	al,1			! prepare upper nibble
	shr	al,1
	shr	al,1
	shr	al,1
	call	prnnib			! print it
	pop	ax
prnnib:	and	al,#0x0F		! prepare lower nibble
	add	al,#0x30
	cmp	al,#0x39		! convert it into hex
	jle	prnchr
	add	al,#7
prnchr:	push	bx
	mov	ah,#0x0E		! print it
	mov	bl,#0x07
	xor	bh,bh
	int	0x10
	pop	bx
	ret
#endif



!====================================================================
!
! String and constants definitions


! Startup signature

sigmsg:	.byte	0x0D, 0x0A
	.ascii	"DOS Net Boot Image Loader"
	.byte	0x0D, 0x0A
crlf:	.byte	0x0D, 0x0A
	.byte	0


! Magic numbers for boot record and bootp entry

bmagic:	.long	BOOT_MAGIC		! boot image magic number
vmagic:	.ascii	VENDOR_MAGIC		! vendor magic ID
	.byte	0			! end of vendor magic ID
pmagic:	.byte	BOOTP_MAGIC_RFC		! bootp magic ID for RFC 1048
	.byte	BOOTP_MAGIC_CMU		! bootp magic ID for CMU
	.byte	BOOTP_MAGIC_STA		! bootp magic ID for Stanford
	.byte	0


! Specifications for different types of disk drives. The order is:
! # dir entries, # sects per FAT, # sects per cluster, media ID

drvtab:	.byte	112, 2, 2, 0xfd		! 360kB disk
	.byte	224, 7, 1, 0xf9		! 1.2MB disk
	.byte	112, 3, 2, 0xf9		! 720kB disk
	.byte	224, 9, 1, 0xf0		! 1.44 MB disk

typtab:	.byte	DPB_T_360		! type values for drive parameter block
	.byte	DPB_T_1200
	.byte	DPB_T_720
	.byte	DPB_T_1440


! Error messages

recerr:	.ascii	"Error in load record data"
	.byte	0x0D, 0x0A
	.byte	0

bmerr:	.ascii	"Invalid boot header magic number"
	.byte	0x0D, 0x0A
	.byte	0

vmerr:	.ascii	"Invalid vendor magic ID"
	.byte	0x0D, 0x0A
	.byte	0

rderr:	.ascii	"Error while accessing ramdisk"
	.byte	0x0D, 0x0A
	.byte	0

dskerr:	.ascii	"Wrong ramdisk image"
	.byte	0x0D, 0x0A
	.byte	0

btperr:	.ascii	"BOOTP record invalid"
	.byte	0x0D, 0x0A
	.byte	0


! Debug messages

#ifdef ASM_DEBUG
debug1:	.ascii	"Making driver resident"
	.byte	0x0D, 0x0A
	.byte	0

debug2:
#ifdef ASM_FREEZE_AFTER_INIT
	.ascii  "Freezing!"
#else
	.ascii	"Calling boot block"
#endif
	.byte	0x0D, 0x0A
	.byte	0
#endif

#ifdef DRV_DEBUG
consz:	.ascii	"RAMDISK: reporting conventional memory size: "
	.byte	0
extsz:	.ascii	"RAMDISK: reporting extended memory size: "
	.byte	0
bytes:	.ascii	" bytes"
	.byte	0x0D,0x0A
	.byte	0
#endif



!====================================================================
!
! Variable definitions

header:	.long	0			! pointer to boot header from boot rom
bootp:	.long	0			! pointer to bootp block from boot rom

resseg:	.word	0			! segment of resident section

oldDS:	.word	0			! old DS from boot rom
oldES:	.word	0			! old ES from boot rom
oldBP:	.word	0			! old BP from boot rom
oldSI:	.word	0			! old SI from boot rom
oldDI:	.word	0			! old DI from boot rom



!====================================================================
!
! Start of resident section. This will be placed at the end of the
! low 640kB RAM area.
!
!====================================================================
!

	.align	16			! has to be paragraph aligned

start_resident:				! indicate start of resident section


!====================================================================
!
! New interrupt 2Fh routine. This routine gets called by IO.SYS
! in order to determine the maximum amount of memory usable to
! DOS. This only works with DOS versions 5.0 and higher. The DOS
! function which gets installed into the interrupt 2Fh vector
! does not call the old vector (i.e. it does not daisy-chain),
! and therefore we can redirect 2Fh without a problem even when
! considering to remove the ram disk lateron.
!
! NOTE THAT THIS INTERRUPT HAS TO BE THE FIRST ROUTINE IN THE
! RESIDENT SECTION!
!
! Input:  AX     -  Magic ID
!         DX     -  segment following last usable byte
! Output: DX     -  new segment  following last usable byte
! Registers changed: DX

int2F:	jmp	int2F1			! this has to be a relative jump
	nop

	.ascii	"RPL"			! magic ID string for DOS

int2F1:	cmp	ax,#0x4A06		! check for magic ID
	jne	int2F9
	push	cx
	mov	dx,#start_resident	! determine last usable segment
	mov	cl,#4			! from segment and offset of
	shr	dx,cl			! the resident section
	pop	cx
	push	ax
	push	cs
	pop	ax
	add	dx,ax			! add offset to segment
	dec	dx
	pop	ax
	iret

int2F9:	seg	cs
	jmp	far [old2Fh]		! jump to old interrupt routine



!====================================================================
!
! New interrupt F8h routine. It can be used to retrieve several
! values from the resident driver.
! Input:  AX  -  function code
! Output: depends on function:
!
! Installation check (AX = 0x9C00)
!         AX     -  contains 0x009C
!
! Return ramdisk size and address (AX = 0x9C01)
!         BX:DX  -  address of ramdisk
!         CX     -  size of ramdisk in kb
!
! Return size and address of BOOTP block (AX = 0x9C02)
!         BX:DX  -  address of BOOTP block
!         CX     -  size of BOOTP block
!
! Return miscellaneous values for handling the ramdisk (AX = 0x9C03)
!         AX     -  XMS handle for ram disk
!         BX:DX  -  address of old interrupt vector table
!         CL     -  ramdisk id
!
! Remove ramdisk (AX = 0x9C04)
!         AL     -  non-zero if error
!
! Registers changed: depends on function

intF8:	cmp	ah,#0x9C		! check for magic ID
	jne	intF89
	cmp	al,#01			! check for function number
	jne	intF81
	seg	cs
	mov	bx,[rdaddr+2]		! return ramdisk address
	seg	cs
	mov	dx,[rdaddr+0]
	seg	cs
	mov	cx,rdsize		! return ramdisk size
	iret

intF81:	cmp	al,#0x02
	jne	intF82
	mov	bx,cs			! return address of BOOTP record
	mov	dx,#btpnew
	seg	cs
	mov	cx,btplen		! return BOOTP length
	iret

intF82:	cmp	al,#0x03
	jne	intF83
	mov	bx,cs			! return address of old interrupt
	mov	dx,#oldints		! vector table
	seg	cs
	mov	cl,drvid		! return drive id
	seg	cs
	mov	ax,xmshdl		! return XMS handle
	iret

intF83:	cmp	al,#0x04
	jne	intF88
	call	rmrd			! remove ramdisk
	iret

intF88:	or	al,al
	jnz	intF89
	xchg	al,ah			! return installation check code
intF89:	iret



!====================================================================
!
! New interrupt 13h routine to handle disk accesses. It is different
! for simulating either a floppy drive or a hard disk. DOS provides
! a way for restoring this interrupt to its original value when we
! want to remove the ramdisk lateron.
! Input:  AH  -  function code
!         DL  -  driver number
! Output: carry flag set if error
! Registers changed: depends on function

int13:	sti				! we dont need interrupts disabled
	push	ax
	seg	cs
	mov	ax,word ptr (xmsadr+0)
	seg	cs
	or	ax,word ptr (xmsadr+2)	! check if XMS already initialized
	jnz	int13s
	mov	ax,#0x4300		! check if XMS available
	int	0x2f
	cmp	al,#0x80		! XMS not available
	jne	int13s
	push	bx
	push	es
	mov	ax,#0x4310		! get XMS driver address
	int	0x2f
	seg	cs
	mov	word ptr (xmsadr+0),bx	! save driver address
	seg	cs
	mov	word ptr (xmsadr+2),es
	pop	es
	call	inixms			! initialize XMS
	pop	bx
int13s:	pop	ax

	or	ah,ah			! function 0 doesnt use drive numbers
	jnz	int130
	pushf
	seg	cs
	call	far [old13h]		! call BIOS
	jmp	int13f			! always return without error

int130:	seg	cs
	cmp	dl,byte ptr (drvid)	! check if its for us
	je	int132
	seg	cs
	cmp	byte ptr (drvid),#0x80	! check if we are to simulate a hard
	jae	int13h			! disk drive


! First comes the floppy drive redirector

	cmp	dl,#0x80		! check if its for a hard disk
	jb	int133
	seg	cs
	test	byte ptr (nohd),#0xff	! check if hard disk accesses allowed
	jz	int131
	cmp	ah,#0x08		! function 0x08 should not return error
	mov	ah,#0x80		! return with error
	jne	int135
	xor	dl,dl			! indicate no hard disk present
	jmp	int13f			! return without error

! Handle function 0x08 for disk drives other than the ramdisk.

int133:	cmp	ah,#0x08
	jne	int131
	pushf
	seg	cs			! function 0x08 has to return the
	call	far [old13h]		! correct number of disk drives
	seg	cs
	mov	dl,drvnum
int13f:	xor	ah,ah			! never return an error
	jmp	int136

! Jump directly to the BIOS

int131:	seg	cs
	jmp	far [old13h]		! call the old interrupt routine

! Now handle all ramdisk functions. First check if the function number
! is correct.

int132:	cmp	ah,#0x18
	jbe	int134
	mov	ah,#0x01		! unknown command
int135:	stc
int136:	push	ds
	jmp	int13e

! Determine the handlers address according to the function number in AH
! and jump to it.

int134:	push	ds
	push	cs
	pop	ds			! set data segment
	push	bx
	mov	bl,ah
	xor	bh,bh
	dec	bx
	shl	bx,#1			! compute pointer into routine table
	jmp	near fntab[bx]


! Now comes the hard disk drive redirector

int13h:	cmp	dl,#0x80		! check if its for a floppy drive
	jb	int131

! Handle function 0x08 for hard disk drives other than the ramdisk.

	cmp	ah,#0x08
	jne	int137
	dec	dl
	pushf
	seg	cs			! function 0x08 has to return the
	call	far [old13h]		! correct number of disk drives
	seg	cs
	mov	dl,drvnum
	jmp	int13f			! always return without error

! Handle function 0x15 for disk drives other than the ramdisk. This is
! the only function besides 0x08 which returns a value in DX and therefore
! has to have special handling.

int137:	push	dx
	dec	dl
	cmp	ah,#0x15
	jne	int138
	pushf
	seg	cs
	call	far [old13h]		! call the BIOS for handling
	jc	int139
	cmp	ah,#0x03		! DX is only used if AH = 0x03
	jne	int139
	add	sp,#0x0002		! remove DX from stack if the BIOS
	jmp	int136			! returned a value in it

! Handle all other functions for drives other than the ramdisk. This will
! just call the original BIOS handler.

int138:	pushf
	seg	cs
	call	far [old13h]		! simply call the old int 13h routine
int139:	pop	dx
	jmp	int136


! Save the return status into the BIOS data area and return to the caller
! while preserving the carry flag.

int13e:	push	es			! return from function handler
	push	bx			! this code is not allowed to change
	call	getsts			! any register or flag
	seg	es
	mov	[bx],ah			! set disk operation status
	pop	bx
	pop	es
	pop	ds
intend:	push	ax			! general exit point for interrupts
	pushf
	pop	ax
	push	bp
	mov	bp,sp
	mov	[bp+8],al		! put the flags onto the stack
	pop	bp
	pop	ax
	iret


! Function table

fntab:	.word	f1301			! function 01: return last error
	.word	f1302			! function 02: read disk
	.word	f1303			! function 03: write disk
	.word	f1304			! function 04: verify disk
	.word	f1305			! function 05: format disk
	.word	f1306			! function 06: format track
	.word	f1307			! function 07: format disk
	.word	f1308			! function 08: get drive parameters
	.word	f1309			! function 09: intialize controller
	.word	f130A			! function 0A: read long sectors
	.word	f130B			! function 0B: write long sectors
	.word	f130C			! function 0C: seek for cylinder
	.word	f130D			! function 0D: disk reset
	.word	f130E			! function 0E: read sector buffer
	.word	f130F			! function 0F: write sector buffer
	.word	f1310			! function 10: check if drive ready
	.word	f1311			! function 11: recalibrate drive
	.word	f1312			! function 12: controller ram diagnostic
	.word	f1313			! function 13: drive diagnostic
	.word	f1314			! function 14: controller int diagnostic
	.word	f1315			! function 15: get disk type
	.word	f1316			! function 16: detect disk change
	.word	f1317			! function 17: set media type for format
	.word	f1318			! function 18: set media type for format

f13end:	jmp	int13e



!====================================================================
!
! Function 01 - return last error status
!

f1301:	push	es
	call	getsts			! get offset to status byte
	seg	es
	mov	ah,[bx]			! get disk operation status
	pop	es
	pop	bx
	clc
	jmp	intend



!====================================================================
!
! Function 02/03 - read/write from disk

f1302:
f1303:	pop	bx			! get old BX from stack
	push	dx
	push	ax
	call	cvtsec			! get linear sector number
	pop	ax
	jnc	f13021
	mov	ah,#0x04		! error: sector not found
f13028:	pop	dx
	stc
	jmp	f13end			! terminate

f13021:	push	cx
	push	bx
	push	ax
	mov	cl,al			! move number of sectors into CX
	xor	ch,ch
f13022:	jcxz	f13026
	cmp	dx,secnum		! check if sector is still correct
	jb	f13023
	pop	ax
	mov	ah,#0x04		! error: sector not found
f13027:	sub	al,cl			! compute number of sectors processed
	pop	bx
	pop	cx
	jmp	f13028

f13023:	pop	ax
	push	ax
	xor	al,al
	cmp	ah,#0x02		! check if read or write sector
	je	f13024
	inc	al
f13024:	call	rwsect			! actually handle request
	jnc	f13025
	pop	ax
	mov	ah,#0x20		! error: disk controller error
	jmp	f13027

f13025:	inc	dx
	dec	cx			! proceed with next sector
	add	bx,#SECT_SIZE
	jmp	f13022

f13026:	pop	ax
	pop	bx
	pop	cx
	pop	dx
	xor	ah,ah			! no error
	jmp	f13end



!====================================================================
!
! Function 08  -  get disk drive parameters

f1308:	pop	bx			! get old BX from stack
	mov	dl,byte ptr (drvnum)	! get number of disk drives
	mov	cl,byte ptr (secptk)	! get sectors per track
	mov	ch,byte ptr (cylnum)	! number of cylinders
	mov	dh,#1			! number of heads - 1
	xor	bx,bx			! ramdisk drive type
	xor	ax,ax
	dec	ch
f13e1:	jmp	f13end



!====================================================================
!
! Function 04, 05, 06, 07, 09, 0D, 10, 11, 12, 13, 14, 16  -  no operation

f1304:
f1305:
f1306:
f1307:
f1309:
f130D:
f1310:
f1311:
f1312:
f1313:
f1314:
f1316:	pop	bx			! get old BX from stack
	xor	ah,ah			! no error
	jmp	f13e1



!====================================================================
!
! Function 0A, 0B, 0E, 0F, 17, 18  -  not implemented

f130A:
f130B:
f130E:
f130F:
f1317:
f1318:	pop	bx			! get old BX from stack
	mov	ah,#0x01		! invalid opcode
	stc
	jmp	f13e1



!====================================================================
!
! Function 0C  -  seek for cylinder

f130C:	pop	bx			! get old BX from stack
	push	dx
	push	ax
	call	cvtsec			! get linear sector number
	pop	ax
	mov	ah,#0x00		! no error
	jnc	f130C1
	mov	ah,#0x04		! error: sector not found
f130C1:	pop	dx
	jmp	f13e1			! terminate



!====================================================================
!
! Function 15  -  get disk type

f1315:	pop	bx			! get old BX from stack
	mov	ah,#0x02		! indicate floppy disk
	cmp	dl,#0x80		! check if floppy disk type requested
	jb	f13159
	inc	ah			! indicate hard disk
	mov	dx,secnum		! get number of sectors on disk
	xor	cx,cx
f13159:	clc
	jmp	f13e1



!====================================================================
!
! Convert Cyl/Sec/Head notation into a linear sector number
! Input:  CH  -  cylinder number
!         CL  -  sector number
!         DH  -  head number
! Output: DX  -  linear sector number
!         carry flag set if invalid sector number
! Registers changed: AX,DX

cvtsec:	push	cx
	cmp	dh,#2			! check if head number is correct
	jae	cvts8			! maximum number of heads is always 2
	mov	dl,dh
	xor	dh,dh			! move current head number into DX
	mov	al,ch
	mov	ch,cl
	mov	ah,cl			! compute track number into AX, the
	mov	cl,#6			! upper two bits of CL are the high
	shr	ah,cl			! bits of the 10 bit cylinder number
	shl	ax,#1			! compute track number from cylinders
	add	ax,dx
	mov	dl,byte ptr (secptk)
	xor	dh,dh
	mul	dx			! compute number of track starting
	or	dx,dx			! sector
	jnz	cvts8			! should not be more than 65535 sectors

	mov	dl,ch
	and	dl,#0x3F		! move sector number into AX
	cmp	dl,byte ptr (secptk)	! check if sector number is correct
	ja	cvts8
	xor	dh,dh
	dec	dx			! sector numbers start with 1
	js	cvts8			! therefore sector 0 does not exist
	add	dx,ax			! compute final sector number
	jc	cvts8			! should never overflow
	cmp	dx,secnum		! check if the sector is valid
	jae	cvts8
	clc				! no error
	jmp	cvts9
cvts8:	stc				! return with error
cvts9:	pop	cx
	ret



!====================================================================
!
! Read/write a sector from the ram disk. This routine requires a
! sector to be 512 bytes long.
! Input:  AL     -  non-zero if write to ram disk
!         DX     -  logical sector number
!         ES:BX  -  pointer to destination buffer
! Output: carry flag set if error
! Registers changed: AX

rwsect:	push	cx
	push	dx
	mov	ch,al			! save direction indicator
	mov	dx,es
	mov	ax,dx
	mov	cl,#12
	shr	ax,cl
	mov	cl,#4
	shl	dx,cl			! compute linear buffer address
	add	dx,bx
	adc	ax,#0
	or	ch,ch			! check direction of transfer
	jz	rwsec1
	mov	word ptr (rd_srcb+0),dx	! set source address for write
	mov	byte ptr (rd_srcb+2),al
	jmp	rwsec2
rwsec1:	mov	word ptr (rd_dstb+0),dx	! set destination address for read
	mov	byte ptr (rd_dstb+2),al
rwsec2:	pop	dx

	push	dx
	mov	ax,dx
	mov	cl,#9
	shl	dx,cl			! compute linear ramdisk address
	mov	cl,#7			! from sector number
	shr	ax,cl
	add	dx,(rdaddr+0)
	adc	ax,(rdaddr+2)
	or	ch,ch			! check direction of transfer
	jz	rwsec3
	mov	word ptr (rd_dstb+0),dx	! set destination address for write
	mov	byte ptr (rd_dstb+2),al
	jmp	rwsec4
rwsec3:	mov	word ptr (rd_srcb+0),dx	! set source address for read
	mov	byte ptr (rd_srcb+2),al
rwsec4:	push	es
	push	si
	mov	ax,cs
	mov	es,ax
	mov	si,#rd_gdt
	mov	cx,#SECT_SIZE/2		! copy 512 bytes, e.g. 256 words
	mov	ax,#0x8700		! let the BIOS move the sector
	int	0x15
	pop	si
	pop	es
	pop	dx
	pop	cx
	ret



!====================================================================
!
! Return a pointer to the disk drive status byte. This routine should
! not change any flags!
! Input:  none
! Output: ES:BX  -  pointer to disk drive status byte
! Registers changed: BX, ES

getsts:	mov	bx,#BIOS_SEG		! status byte is in BIOS data
	mov	es,bx			! segment
	seg	cs
	mov	bx,statof
	ret



!====================================================================
!
! Initialize the XMS interface. This is necessary to prevent the
! ram disk from getting overwritten. The way this works is to
! first allocate all of the available XMS, then resize the memory
! block to end just above the ramdisk and lock it. Unfortunately
! we have to do it this complicated because there is no way of
! knowing how the XMS is going to allocate the available memory.
! Another problem is that at least HIMEM.SYS requires up to 256
! bytes of stack, and we cannot assume the caller of INT 13 to
! provide that much so we have to change stacks.
! Input:  none
! Output: none
! Registers changed: AX, BX

inixms:	call	setstk			! set new stack

! First check that the XMS version number is correct. To support all
! necessary functions it has to be version 2.0+.

	xor	ah,ah
	call	callxm			! get version number
	cmp	ah,#0x02
	jb	inixm8			! wrong XMS version

! Determine how much memory we can allocate.

	mov	ah,#0x08
	xor	bl,bl			! get amount of extended memory
	call	callxm
	or	bl,bl			! check for error
	jnz	inixm8
	mov	bx,rdsize		! get size of ramdisk
	add	bx,#65			! care for a missing HMA
	cmp	bx,ax			! check if enough memory for ram disk
	ja	inixm8

! Grab all of the extended memory.

	push	bx
	mov	dx,ax			! grab largest block - which is whole
	mov	ah,#0x09		! memory because there should be no
	call	callxm			! other process using XMS
	pop	bx
	or	ax,ax			! check for error
	jz	inixm8
	mov	xmshdl,dx		! save handle

! Now resize the memory block so that it will contain the ramdisk image.

	mov	ah,#0x0f		! reallocate memory block
	call	callxm
	or	ax,ax			! check for error
	jnz	inixm1

inixm8:	mov	xmshdl,#0		! in case of error dont return handle
	jmp	inixm9

! Now lock the memory block and check that the physical address of the
! memory block is correct.

inixm1:	mov	dx,xmshdl
	mov	ah,#0x0c		! lock memory block
	call	callxm
	add	bx,#0x03ff
	adc	dx,#0x0001		! add 65kb - maximum difference
	sub	bx,word ptr (rdaddr+0)	! check that ramdisk address is below
	sbb	dx,word ptr (rdaddr+2)
	jc	inixm8

! Thats it. Restore all registers and swap the stack back to its
! original state.

inixm9:	call	rststk			! restore old stack
	ret



!====================================================================
!
! Call XMS driver.
! Input:  AH  -  function code
!         other registers depending on function code
! Output: depends on called function
! Registers changed: depends on called function

callxm:	push	ax
	push	bp
	push	ax
	mov	bp,sp
	mov	ax,[bp+6]
	mov	[bp+4],ax		! make far return address from
	mov	[bp+6],cs		! near call
	pop	ax
	pop	bp
	seg	cs
	push	word ptr (xmsadr+2)	! push address of XMS driver
	seg	cs
	push	word ptr (xmsadr+0)
	retf				! call XMS driver



!====================================================================
!
! Set new stack
! Input:  none
! Output: none
! Registers changed: AX, BX, DS, SS, SP

setstk:	cli
	pop	bx			! get return address
	mov	ax,sp
	seg	cs
	mov	word ptr (oldstk+0),ax
	seg	cs
	mov	word ptr (oldstk+2),ss	! save old stack pointer
	mov	ax,cs
	mov	ss,ax
	mov	sp,#newtos - 2		! change to new stack
	sti
	push	cx
	push	dx
	push	si			! save all registers
	push	di
	push	es
	push	ds
	mov	ax,cs			! set DS to current segment
	mov	ds,ax
	jmp	bx			! return to caller



!====================================================================
!
! Reset stack to old stack
! Input:  none
! Output: none
! Registers changed: all (reset to old state)


rststk:	pop	bx			! get return address
	pop	ds
	pop	es
	pop	di
	pop	si			! restore all registers
	pop	dx
	pop	cx
	cli
	seg	cs
	mov	ax,word ptr (oldstk+0)	! restore old stack
	seg	cs
	mov	ss,word ptr (oldstk+2)
	mov	sp,ax
	sti
	jmp	bx			! return to caller



!====================================================================
!
! Remove ramdisk from memory. This involves restoring all interrupt
! vectors, freeing all used memory and restoring the DOS drive para-
! meter table. Since we need to call the XMS drive, we have to switch
! stacks like with inixms.
! Input:  none
! Output: AL  -  non-zero if error
! Registers changed: AX

rmrd:	push	bx
	call	setstk			! set new stack
	mov	al,drvid
	cmp	al,#0x80		! can only restore floppy drives
	jb	rmrd1
rmrd8:	call	rststk			! return with error
	pop	bx
	mov	al,#0x01
	ret

! First determine the address of the DOS disk parameter block for the
! ramdisk and check that the open count is zero, i.e. no open file on
! the device.

rmrd1:	push	ds
	mov	ax,#0x0803
	int	0x2f			! get address of drive parameter
	mov	ax,ds			! table from DOS
	mov	es,ax
	pop	ds

rmrd2:	seg	es
	mov	al,4[di]		! get physical unit number
	cmp	al,drvid		! is it our drive?
	je	rmrd3
	cmp	di,#0xffff		! error if we couldnt find the DPB
	je	rmrd8			! for the ramdisk
	seg	es
	les	di,[di]			! get pointer to next entry
	jmp	rmrd2

rmrd3:	mov	word ptr (dpb_addr+0),di
	mov	word ptr (dpb_addr+2),es
	seg	es
	mov	ax,0x20[di]		! get device open count
	or	ax,ax
	jnz	rmrd8

! Next restore the interrupt vectors. Int 13h is special as it is
! redirected by DOS. However, DOS provides a function to restore
! that interrupt. Interrupt 2Fh doesnt have to get restored because
! DOS does never call the old interrupt again.

	xor	ax,ax
	mov	es,ax
	mov	ax,cs
	seg	es			! first check that nobody redirected
	cmp	ax,word ptr [IF8_INT+2]	! our own interrupts. In that case
	jne	rmrd8			! there is no chance of removing the
	seg	es			! ramdisk.
	mov	ax,word ptr [IF8_INT+0]
	cmp	ax,#intF8
	jne	rmrd8
	mov	si,#if1sig
	mov	di,#IF1_INT
	mov	cx,#4			! interrupt F1h contains a signature
	repz				! and no vector
	cmpsb
	jnz	rmrd8

	push	ds
	les	bx,old13h		! get old interrupt vector 13h
	mov	dx,bx
	mov	ax,es			! save it into DS:DX and ES:BX
	mov	ds,ax
	mov	ah,#0x13
	int	0x2f			! call DOS to restore vector
	mov	ax,ds
	mov	cx,cs
	cmp	ax,cx			! check that its indeed our interrupt
	jne	rmrd4			! which we are replacing
	mov	ax,es
	cmp	ax,cx
	jne	rmrd4
	cmp	bx,#int13
	jne	rmrd4
	cmp	dx,#int13
	je	rmrd5

rmrd4:	mov	ah,#0x13
	int	0x2f			! restore old interrupt
	pop	ds			! someone redirected the interrupt
	jmp	near rmrd8		! already, cant restore

rmrd5:	pop	ds			! restore the other interrupts
	cli
	xor	ax,ax
	mov	es,ax
	mov	ax,(oldF8h+0)
	seg	es
	mov	word ptr [IF8_INT+0],ax
	mov	ax,(oldF8h+2)
	seg	es
	mov	word ptr [IF8_INT+2],ax
	sti

! OK, we can now setup the DOS drive parameter table to contain the
! correct values for the physical floppy drive. If we couldnt create
! a valid parameter table entry for this drive, simply mark the DOS
! entry as invalid. This will cause "Not Ready" errors in DOS. This
! doesnt work with DR-DOS 5.0!

	les	di,dpb_addr			! get address of DPB
	seg	es
	or	word ptr 0x1f[di],#80		! mark drive as invalid
	test	byte ptr (dpb_valid),#0xff	! check if DPB valid
	jz	rmrd6
	cld					! got correct table entry
	mov	cx,#dpb_end - dpb
	mov	si,#dpb
	add	di,#4
	rep
	movsb				! simply copy the DPB

! Next remove the ramdisk image from extended memory using the XMS driver.

rmrd6:	mov	dx,xmshdl
	or	dx,dx			! only free memory if we really
	jz	rmrd7			! assigned it with XMS
	push	dx
	mov	ah,#0x0d
	call	callxm			! unlock memory block
	pop	dx
	or	ax,ax			! dont free block if error
	jz	rmrd7
	mov	ah,#0x0a
	call	callxm			! free memory block

! Finally we can remove the memory for the ramdisk driver. We only
! reset the owner field of the memory control block to 0 to indicate
! it as free.

rmrd7:	mov	dx,#start_resident	! determine last usable segment
	mov	cl,#4			! from segment and offset of
	shr	dx,cl			! the resident section
	mov	ax,cs
	add	dx,ax			! add offset to segment
	sub	dx,#2
	mov	es,dx
	mov	di,#1
	xor	ax,ax			! set owner field to 0
	stosw
	add	di,#5
	mov	cx,#4			! clear owner name
	rep
	stosw

! Thats it. Return to caller.

rmrd9:	call	rststk			! restore old stack
	pop	bx
	xor	al,al			! return without error
	ret



!====================================================================
!
! Variables for the resident section

		.align	2

oldints:
old13h:		.long	0		! old interrupt 13h vector
old2Fh:		.long	0		! old interrupt 2Fh vector
oldF1h:		.long	0		! old interrupt F1h vector
oldF8h:		.long	0		! old interrupt F8h vector


! Disk parameters for ram disk

statof:		.word	BIOS_FDSTAT	! offset to BIOS disk status byte
rdaddr:		.long	0		! base address of ram disk
rdsize:		.word	0		! size of ram disk in kb
cylnum:		.word	80		! number of cylinders
secnum:		.word	2400		! number of sectors on disk
secptk:		.word	15		! number of sectors per track
drvnum:		.byte	1		! number of disk drives
drvid:		.byte	0		! ram disk drive id
nohd:		.byte	0		! no-hard-disk flag

		.align	2


! Variables used to access the XMS interface

xmshdl:		.word	0		! XMS handle for ram disk
xmsadr:		.long	0		! address of XMS driver interface


! Variables used to redirect the stack

oldstk:		.long	0		! old stack pointer
newstk:		.space	512		! new stack for calling XMS driver
newtos:					! new top of stack


! Signature to put into interrupt vector F1h

if1sig:	.ascii	"NetB"


! Descriptor table to access ram disk using the BIOS

rd_gdt:		.word	0,0,0,0
		.word	0,0,0,0
rd_src:		.word	0xffff		! length
rd_srcb:	.byte	0,0,0		! base
		.byte	0x93		! typebyte
		.word	0		! limit16,base24 =0
rd_dst:		.word	0xffff		! length
rd_dstb:	.byte	0,0,0		! base
		.byte	0x93		! typebyte
		.word	0		! limit16,base24 =0
		.word	0,0,0,0		! BIOS CS
		.word	0,0,0,0		! BIOS DS


! DOS disk parameter block. It contains the definitions for the
! floppy disk drive which is redirected by the ramdisk, and used
! for removing the ramdisk drive. Note that this DPB is only
! valid for DOS 4.0 and higher.

dpb_addr:	.long	0		! address of DPB in DOS data area
dpb_valid:	.byte	0		! non-zero if DPB is valid

dpb:
dpb_phys:	.byte	0		! BIOS ID of physical drive
dpb_log:	.byte	0		! logical DOS drive ID

dpb_bpb_low:	.word	512		! BIOS param block for lowest capacity
		.byte	0xff
		.word	1
		.byte	2
		.word	64
		.word	360
		.byte	0x00
		.word	2
		.word	9
		.word	1
		.long	0
		.long	0

dpb_fat:	.byte	0		! flag indicating 16-bit FAT
dpb_open:	.word	0		! device open count
dpb_type:	.byte	0x01		! device type
dpb_flags:	.word	DPB_F_DEFAULT	! flags describing drive
dpb_cyls:	.word	80		! number of cylinders

dpb_bpb_cur:	.word	512		! BIOS parameter block for current
		.byte	1
		.word	1
		.byte	2
		.word	224
		.word	2400
		.byte	0xf9
		.word	7
		.word	15
		.word	2
		.long	0
		.long	0

dpb_rsvd:	.byte	0, 0, 0, 0, 0, 0
dpb_ltrack:	.byte	0xff		! last accessed track
dpb_lacc:	.long	0xffffffff	! time of last disk access
dpb_volname:	.ascii	"NO NAME    "	! volume name
		.byte	0
dpb_sernum:	.long	0		! volume serial number
dpb_fsname:	.ascii	"FAT12   "	! file system name
		.byte	0
dpb_end:


! Copy of bootp block from bootrom. This has to be last in the data area!

btplen:		.word	0		! length of bootp block
btpnew:					! bootp block has to be at the very end

