//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  Maintains a collection of property names and values.  Upon
  construction the property object contains some reasonable default
  values for most properties as well as "Cartridge.Image" set to
  the name of the ROM image and "Cartridge.Id" set to the
  cartridge's computed identification.

  @author  Bradford W. Mott
  @version $Id: Props.cxx,v 1.2 1997/05/17 19:00:06 bwmott Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fstream.h>

#include "machine.hxx"
#include "Error.hxx"
#include "Props.hxx"

//============================================================================
// Constructor
//============================================================================
Properties::Properties(const char* filename)
{
  mySize = 64;
  myProperties = new Property[mySize];
  myNumberOfProperties = 0;

  // Set the default property values
  set("Cartridge.Directory", "");
  set("Cartridge.Name", "Unknown");
  set("Cartridge.Image", "Unknown");
  set("Cartridge.Id", "Unknown");
  set("Cartridge.Type", "4K");

  set("Display.FrameRate", "60");
  set("Display.XStart", "0");
  set("Display.Width", "160");
  set("Display.YStart", "38");
  set("Display.Height", "210");

  set("Console.LeftDifficulty", "B");
  set("Console.RightDifficulty", "B");
  set("Console.TelevisionType", "Color");

  set("Controller.Paddle0", "None");
  set("Controller.Paddle1", "None");
  set("Controller.Paddle2", "None");
  set("Controller.Paddle3", "None");

  set("Controller.Joystick0", "0");
  set("Controller.Joystick1", "1");

  set("Timer.Adjustment", "0");
  set("TIA.PlayerDelay", "1");
  set("M6507.Compatibility", "Low");

  try
  {
    // Autodetect some of the properties based on the ROM image
    autodetect(filename);

    // Make sure all the values are reasonable
    validate();
  }
  catch(...)
  {
    // Free each of the properties
    for(uLong i = 0; i < myNumberOfProperties; ++i)
    {
      delete[] myProperties[i].keyword;
      delete[] myProperties[i].value;
    }

    // Free the properties array
    delete[] myProperties;
    
    // Propagate the exception
    throw;
  }
}

//============================================================================
// Destructor
//============================================================================
Properties::~Properties()
{
  // Free each of the properties
  for(uLong i = 0; i < myNumberOfProperties; ++i)
  {
    delete[] myProperties[i].keyword;
    delete[] myProperties[i].value;
  }

  // Free the properties array
  delete[] myProperties;
}

//============================================================================
// Answer the value of the property or the empty string if it doesn't exist
//============================================================================
const char* Properties::find(const char* property) const
{
  // Find the named property and answer its value
  for(uLong i = 0; i < myNumberOfProperties; ++i)
  {
    if(strcmp(property, myProperties[i].keyword) == 0)
      return myProperties[i].value;
  }

  return 0;
}

//============================================================================
// Answer the integer value of the property
//============================================================================
int Properties::integer(const char* property) const
{
  return atoi(find(property));
}

//============================================================================
// Answer the value assigned to property
//============================================================================
const char* Properties::get(const char* property) const
{
  // Find the named property and answer its value
  for(uLong i = 0; i < myNumberOfProperties; ++i)
  {
    if(strcmp(property, myProperties[i].keyword) == 0)
      return myProperties[i].value;
  }

  return "";
}

//============================================================================
// Set the value of property to the specified value
//============================================================================
void Properties::set(const char* keyword, const char* value)
{
  // See if the property already exists
  for(uLong i = 0; i < myNumberOfProperties; ++i)
  {
    if(strcmp(keyword, myProperties[i].keyword) == 0)
    {
      delete[] myProperties[i].value;
      char* v = new char[strlen(value) + 1];
      strcpy(v, value);
      myProperties[i].value = v;
      return;
    }
  }

  // See if my array needs to be resized
  if(myNumberOfProperties == mySize)
  {
    Property* newProperties = new Property[mySize * 2];

    for(uLong i = 0; i < mySize; ++i)
      newProperties[i] = myProperties[i];

    delete[] myProperties;
    myProperties = newProperties;
    mySize *= 2;
  } 

  char* k = new char[strlen(keyword) + 1];
  char* v = new char[strlen(value) + 1];
  strcpy(k, keyword);
  strcpy(v, value);
  myProperties[myNumberOfProperties].keyword = k;
  myProperties[myNumberOfProperties].value = v;

  ++myNumberOfProperties;
}

//============================================================================
// Make sure the values assigned to properties are reasonable
//============================================================================
void Properties::validate()
{
  // TODO: Check other values to make sure they're ok

  // Make sure the starting x and width values are ok
  if(integer("Display.XStart") % 4 != 0)
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description() << "Display.XStart must be a multiple of four!";
    Throw(err);
  }

  if(integer("Display.Width") % 4 != 0)
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description() << "Display.Width must be a multiple of four!";
    Throw(err);
  }

  // Let's check the controller properties
  const char* joystick0 = find("Controller.Joystick0");
  const char* joystick1 = find("Controller.Joystick1");
  const char* paddle0 = find("Controller.Paddle0");
  const char* paddle1 = find("Controller.Paddle1");
  const char* paddle2 = find("Controller.Paddle2");
  const char* paddle3 = find("Controller.Paddle3");

  // Check the Controller.Joystick0 property
  if((strcmp(joystick0, "None") != 0) &&
     (strcmp(joystick0, "0") != 0) &&
     (strcmp(joystick0, "1") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Joystick0 value!  Please use the value " 
        << "0, 1 or None for the Controller.Joystick0 property.";
    Throw(err);
  }

  // Check the Controller.Joystick1 property
  if((strcmp(joystick1, "None") != 0) &&
     (strcmp(joystick1, "0") != 0) &&
     (strcmp(joystick1, "1") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Joystick1 value!  Please use the value " 
        << "0, 1 or None for the Controller.Joystick1 property.";
    Throw(err);
  }

  // Check the Controller.Paddle0 property
  if((strcmp(paddle0, "None") != 0) &&
     (strcmp(paddle0, "0") != 0) &&
     (strcmp(paddle0, "1") != 0) &&
     (strcmp(paddle0, "2") != 0) &&
     (strcmp(paddle0, "3") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Paddle0 value!  Please use the value " 
        << "0, 1, 2, 3 or None for the Controller.Paddle0 property.";
    Throw(err);
  }

  // Check the Controller.Paddle1 property
  if((strcmp(paddle1, "None") != 0) &&
     (strcmp(paddle1, "0") != 0) &&
     (strcmp(paddle1, "1") != 0) &&
     (strcmp(paddle1, "2") != 0) &&
     (strcmp(paddle1, "3") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Paddle1 value!  Please use the value " 
        << "0, 1, 2, 3 or None for the Controller.Paddle1 property.";
    Throw(err);
  }

  // Check the Controller.Paddle2 property
  if((strcmp(paddle2, "None") != 0) &&
     (strcmp(paddle2, "0") != 0) &&
     (strcmp(paddle2, "1") != 0) &&
     (strcmp(paddle2, "2") != 0) &&
     (strcmp(paddle2, "3") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Paddle2 value!  Please use the value " 
        << "0, 1, 2, 3 or None for the Controller.Paddle2 property.";
    Throw(err);
  }

  // Check the Controller.Paddle3 property
  if((strcmp(paddle3, "None") != 0) &&
     (strcmp(paddle3, "0") != 0) &&
     (strcmp(paddle3, "1") != 0) &&
     (strcmp(paddle3, "2") != 0) &&
     (strcmp(paddle3, "3") != 0))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "Invalid Controller.Paddle3 value!  Please use the value " 
        << "0, 1, 2, 3 or None for the Controller.Paddle3 property.";
    Throw(err);
  }

  // Make sure the setup makes a little sense
  if((strcmp(joystick0, "None") != 0) &&
      ((strcmp(paddle0, "None") != 0) || (strcmp(paddle1, "None") != 0)))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "The 'stella.vcs' has joystick 0 and paddle 0 or 1 active" << endl
        << "which is not possible.  Please modify the file and turn" << endl
        << "off either the joystick or the paddles.";
    Throw(err);
  }

  if((strcmp(joystick1, "None") != 0) &&
      ((strcmp(paddle2, "None") != 0) || (strcmp(paddle3, "None") != 0)))
  {
    Error err;
    err.message() << "There is a problem with the 'stella.vcs' file!";
    err.description()
        << "The 'stella.vcs' has joystick10 and paddle 2 or 3 active" << endl
        << "which is not possible.  Please modify the file and turn" << endl
        << "off either the joystick or the paddles.";
    Throw(err);
  }
}

//============================================================================
// Autodetect properties from the given ROM file
//============================================================================
void Properties::autodetect(const char* file)
{
  char* dirname = new char[strlen(file) + 1];
  char* filename = new char[strlen(file) + 1];

  // Break file up into dirname and filename
  const char* separator = strrchr(file, PATH_SEPARATOR);
  if(separator == 0)
  {
    strcpy(filename, file);
    strcpy(dirname, "");
  }
  else
  {
    strcpy(filename, separator + 1);
    strncpy(dirname, file, separator - file);
    dirname[separator - file] = 0;
  }

  // Calculate the cart's id
  char id[128];
  calculateId(file, id);

  // Attempt to get the cart's type based on ROM image and Cartridge Id
  const char* type = cartridgeType(file, id);

  // Supercharger games require the CPU to be in high compability mode
  if(strcmp(type, "AR") == 0)
  {
    set("M6507.Compatibility", "High");
  }
 
  set("Cartridge.Name", filename);
  set("Cartridge.Directory", dirname);
  set("Cartridge.Image", filename);
  set("Cartridge.Type", type);
  set("Cartridge.Id", id);

  delete[] dirname;
  delete[] filename;
}

//============================================================================
// Calculate the Cartridge Id associated with the ROM Image
//============================================================================
void Properties::calculateId(const char* file, char* id)
{
  // Get the filename
  const char* separator = strrchr(file, PATH_SEPARATOR);
  const char* filename = (separator == 0) ? file : separator + 1;

  ifstream in;

  #if defined UNIX_OS
    in.open(file, ios::in | ios::nocreate);
  #elif defined MAC_OS
    in.open(file, ios::in | ios::binary);
  #else
    in.open(file, ios::in | ios::nocreate | ios::binary);
  #endif

  // Make sure we were able to open the file
  if(in.fail())
  {
    // Can't open file so throw exception
    Error err;
    err.message() << "The file \"" << filename << "\" couldn't be opened...";
    Throw(err);
  }
  else
  {
    // Read ROM from the input stream
    uByte rom[32768];
    in.read(rom, 32768);

    // Calculate Cartridge ID
    uLong cartId = 0;
    for(uWord i = 1024; i < 2048; i++)
    {
      cartId += (((uLong)rom[i]) << ((i & 0x0003) * 8));
    }

    sprintf(id, "%08x", cartId); 
  }
}

//============================================================================
// Determine the cartridge's type
//============================================================================
const char* Properties::cartridgeType(const char* file, const char* id)
{
  struct IdToType {
    const char* id;
    const char* type;
  };

  // Maps a cartridge id to its type (for those hard to guess carts :-)
  static IdToType table[] = {
    {"e9422fb1", "E0"},    // Death Star
    {"b82650c9", "E0"},    // Gyruss
    {"98553b40", "E0"},    // Super Cobra
    {"45dfd9a3", "E0"},    // Tutankamn
    {"48a54df0", "E0"},    // Popeye
    {"2ca40acd", "E0"},    // Star Wars, Arcade
    {"84b60c35", "E0"},    // Frogger2
    {"80a2714d", "E0"},    // Montezuma's Revenge
    {"15dd4774", "F6SC"},  // Dig Dug
    {"9f99ada9", "F8SC"},  // Defender ][
    {"37f268a9", "FASC"},  // Tunnel runner
    {"ce3d8002", "FASC"},  // Mountain King
    {"b0cb2c2e", "FASC"},  // Omega Race
    {"d439a04a", "E7"},    // Burger Timer
    {"ffffff00", "E7"},    // Bump-N-Jump
    {"c3af1519", "E7"},    // He-man
    {"",         ""}
  };

  // Get the filename
  const char* separator = strrchr(file, PATH_SEPARATOR);
  const char* filename = (separator == 0) ? file : separator + 1;
 
  // Open ROM image so we can look at it
  ifstream in;

  #if defined UNIX_OS
    in.open(file, ios::in | ios::nocreate);
  #elif defined MAC_OS
    in.open(file, ios::in | ios::binary);
  #else
    in.open(file, ios::in | ios::nocreate | ios::binary);
  #endif

  // Make sure we were able to open the file
  if(in.fail())
  {
    // Can't open file so throw exception
    Error err;
    err.message() << "The file \"" << filename << "\" couldn't be opened...";
    Throw(err);
  }
  else
  {
    // Read ROM from the input stream and see how big it is
    uByte rom[32768];
    in.read(rom, 32768); 
    uLong size = in.gcount();

    // Make sure the size is reasonable
    if(((size % 2048) != 0) && (size != 0) && ((size % 8448) != 0))
    {
      Error err;
      err.message() << "The file \"" << filename << "\" doesn't seem to be "
          << "a ROM image file...";
      Throw(err);
    }

    // See if this cartridge is listed in the table
    for(IdToType* entry = table; *entry->id != 0; ++entry)
    {
      if(strcmp(entry->id, id) == 0)
        return entry->type;
    }

    // Calculate the type of the cartridge
    if((size % 8448) == 0)
    {
      // Supercharger ROMs are always a multiple of 8448
      return "AR";
    }
    if(size == 2048)
    {
      return "2K";
    }
    else
    {
      if(memcmp(rom, rom + 2048, 2048) == 0)
      {
        return "2K";
      }
      else if(size == 4096)
      {
        return "4K";
      }
      else
      {
        if(memcmp(rom, rom + 4096, 4096) == 0)
        {
          return "4K";
        }
        else if(size == 8192)
        {
          return "F8";
        }
        else
        {
          if(memcmp(rom, rom + 8192, 8192) == 0)
          {
            return "F8";
          }
          else if(size == 12288)
          {
            return "FASC";
          }
          else if(size == 32768)
          {
            return "F4SC";
          }
          else
          {
            // See if this cart has 128 bytes of RAM
            uByte first = rom[0];
            for(uWord i = 0; i < 256; ++i)
            {
              if(rom[i] != first)
                return "F6";
            }
            return "F6SC";
          }
        }
      }
    } 
  }

  return "4K";
}
 
