/*

Name:
MIKMODUX.C

Description:
Modplaying example of mikmod - bare unix? version.

Original by MikMak <mikmak@via.nl>, 
then Chris Conn <cconn@tohs.abacom.com>,
then Steve McIntyre <stevem@chiark.greenend.org.uk>

HISTORY
=======

v1.00 (06/12/96) - first "versioned" version

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "mikmod.h"

#include "ui.h"

#ifdef CPUTIME_SNAGGER
#include <sched.h>
#endif /* CPUTIME_SNAGGER */

curmod cur_mod;

char *d_text[]={
	"            ",	/*standard, not deleted*/
	" <del>      ",	/*marked for deletion*/
	" <deleted>  ",	/*deleted, gone!*/
	" <del error>"	/*can't delete...*/
	};

char helptext[]=

"Available switches (CaSe SeNsItIvE!):\n"
"\n"
"  -d x    use device-driver #x for output (0 is autodetect). Default=0\n"
"  -ld     List all available device-drivers\n"
"  -ll     List all available loaders\n"
"  -x      disables protracker extended speed\n"
"  -p      disables panning effects (9fingers.mod)\n"
"  -v xx   Sets volume from 0 (silence) to 100. Default=100\n"
"  -f xxxx Sets mixing frequency. Default=44100\n"
"  -m      Force mono output (so sb-pro can mix at 44100)\n"
"  -8      Force 8 bit output\n"
"  -i      Use interpolated mixing\n"
"  -r      Restart a module when it's done playing\n"
"  -q      Quiet mode (interactive commands disabled, displays only errors)\n"

#ifdef CPUTIME_SNAGGER
"  -s      Renice to -20 (more scheduling priority)\n"
"  -S      Get realtime priority (snag all the cpu needed, beware :))\n"
#endif /* CPUTIME_SNAGGER */

"  -t      Tolerant mode - do not stop on non-fatal file access errors.";

int quiet=0;           /* set if quiet mode is enabled */
UBYTE md_type=0;       /* default is a non-wavetable sound device */

void tickhandler(void)
{
	MP_HandleTick();    /* play 1 tick of the module */
	MD_SetBPM(mp_bpm);
}

#if (defined(NEEDS_USLEEP))

/*
 *  NAME:
 *      usleep     -- This is the precision timer for Test Set
 *                    Automation. It uses the select(2) system
 *                    call to delay for the desired number of
 *                    micro-seconds. This call returns ZERO
 *                    (which is usually ignored) on successful
 *                    completion, -1 otherwise.
 *
 *  ALGORITHM:
 *      1) We range check the passed in microseconds and log a
 *         warning message if appropriate. We then return without
 *         delay, flagging an error.
 *      2) Load the Seconds and micro-seconds portion of the
 *         interval timer structure.
 *      3) Call select(2) with no file descriptors set, just the
 *         timer, this results in either delaying the proper
 *         ammount of time or being interupted early by a signal.
 *
 *  HISTORY:
 *      Added when the need for a subsecond timer was evident.
 *	Modified for Solaris-specific bits by SAM 24/10/96
 *  AUTHOR:
 *      Michael J. Dyer                   Telephone:   AT&T 414.647.4044
 *      General Electric Medical Systems        GE DialComm  8 *767.4044
 *      P.O. Box 414  Mail Stop 12-27         Sect'y   AT&T 414.647.4584
 *      Milwaukee, Wisconsin  USA 53201                      8 *767.4584
 *      internet:  mike@sherlock.med.ge.com     GEMS WIZARD e-mail: DYER
 */

#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/types.h>

int     usleep( unsigned long int microSeconds )
{
        unsigned int            Seconds, uSec;
        int                     nfds;
        struct  timeval         Timer;

#ifdef SOLARIS
	fd_set                 readfds, writefds, exceptfds;

	nfds = 0;
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);
#else
	int			readfds, writefds, exceptfds;
 	nfds = readfds = writefds = exceptfds = 0;
#endif

        if( (microSeconds == (unsigned long) 0)
                || microSeconds > (unsigned long) 4000000 )
        {
                errno = ERANGE;         /* value out of range */
                perror( "usleep time out of range ( 0 -> 4000000 ) " );
                return -1;
        }

        Seconds = microSeconds / (unsigned long) 1000000;
        uSec    = microSeconds % (unsigned long) 1000000;

        Timer.tv_sec            = Seconds;
        Timer.tv_usec           = uSec;

        if( select( nfds, &readfds, &writefds, &exceptfds, &Timer ) < 0 )
        {
                perror( "usleep (select) failed" );
                return -1;
        }

        return 0;
}
#endif /*NEEDS_USLEEP*/

int main(int nargc,char *nargv[])
{
	extern int quit;
	extern int play_current;
	UNIMOD *mf;
	int cmderr=0;                   /* error in commandline flag */
	int morehelp=0;                 /* set if user wants more help */
	int t;
	extern float speed_constant;   /* tempo multiplier, initialised to 1*/
	UWORD reppos;
	/*
		Initialize soundcard parameters.. you _have_ to do this
		before calling MD_Init(), and it's illegal to change them
		after you've called MD_Init()
	*/

	int tolerant    =0;
	play_current    =1;	/* we are playing _this_ mod; jump to next 
				   when this goes to 0 */
	md_mixfreq      =44100;                     /* standard mixing freq */
	md_dmabufsize   =32768;                     /* standard dma buf size */
	md_mode         =DMODE_16BITS|DMODE_STEREO; /* standard mixing mode */
	md_device       =0;                  /* standard device: autodetect */

	/*
		Register the loaders we want to use..
	*/

	ML_RegisterLoader(&load_m15);    /* if you use m15load, register it as first! */
	ML_RegisterLoader(&load_mod);
	ML_RegisterLoader(&load_mtm);
	ML_RegisterLoader(&load_s3m);
	ML_RegisterLoader(&load_stm);
	ML_RegisterLoader(&load_ult);
	ML_RegisterLoader(&load_uni);
	ML_RegisterLoader(&load_669);
	ML_RegisterLoader(&load_xm);

	/*
		Register the drivers we want to use:
	*/

        MD_RegisterDriver(&drv_nos);
	MD_RegisterDriver(&drv_raw);

#ifdef SUN
	MD_RegisterDriver(&drv_sun);
#elif defined(SOLARIS)
	MD_RegisterDriver(&drv_sun);
#elif defined(__alpha)
        MD_RegisterDriver(&drv_AF);
#elif defined(OSS)
        MD_RegisterDriver(&drv_vox);
	#ifdef ULTRA
	       MD_RegisterDriver(&drv_ultra);
	#endif /* ULTRA */
#elif defined(__hpux)
        MD_RegisterDriver(&drv_hp);
#elif defined(AIX)
        MD_RegisterDriver(&drv_aix);
#elif defined(SGI)
        MD_RegisterDriver(&drv_sgi);
#endif

	MD_RegisterPlayer(tickhandler);

	/* Parse option switches using standard getopt function: */

	opterr=0;

	while( !cmderr &&
#ifdef CPUTIME_SNAGGER
		(t=getopt(nargc,nargv,"ohxpm8irv:f:l:d:tsS")) != EOF ){
#else
		(t=getopt(nargc,nargv,"ohxpm8irv:f:l:d:t")) != EOF ){
#endif

		switch(t){

			case 'd':
				md_device=atoi(optarg);
				break;

			case 'l':
				if(optarg[0]=='d') MD_InfoDriver();
				else if(optarg[0]=='l') ML_InfoLoader();
				else{
					cmderr=1;
					break;
				}
				exit(0);

			case 'r':
				mp_loop=1;
				break;

			case 'm':
				md_mode&=~DMODE_STEREO;
				break;

			case '8':
				md_mode&=~DMODE_16BITS;
				break;

			case 'i':
				md_mode|=DMODE_INTERP;
				break;

			case 'x':
				mp_extspd=0;
				break;

			case 'p':
				mp_panning=0;
				break;

			case 'v':
				if((mp_volume=atoi(optarg))>100) mp_volume=100;
				break;

			case 'f':
				md_mixfreq=atol(optarg);
				break;

			case 'h':
				morehelp=1;
				cmderr=1;
				break;

			case 't':
				tolerant=1;
				break;

			case 'q':
				quiet=1;
				break;

#ifdef CPUTIME_SNAGGER
			case 's':
				if (nice(-20) == -1)
					perror("renice to -20");
				break;
			case 'S':
				{
					struct sched_param sp;
					memset(&sp, 0, sizeof(struct sched_param));
					sp.sched_priority = 1;
					if (sched_setscheduler(0, SCHED_RR, &sp) == -1)
						perror("realtime priority");
				}
				break;
#endif /* CPUTIME_SNAGGER */

			case '?':
				puts("\07Invalid switch or option needs an argument\n");
				cmderr=1;
				break;
		}
	}

	if(cmderr || optind>=nargc){

		/*
			there was an error in the commandline, or there were no true
			arguments, so display a usage message
		*/

                printf("Usage: %s [switches] <fletch.mod> ... \n",nargv[0]);

                if(morehelp)
                        puts(helptext);
                else
                        printf("Type %s -h for more help.\n",nargv[0]);

                exit(-1);
	}

	if (!quiet)
		puts(mikbanner);

	/*  initialize soundcard */

	if(!MD_Init()){
		printf("Driver error: %s.\n",myerr);
		return 0;
	}

	/*  initialize volume and tempo multipliers */

	speed_constant=1.0;
	mp_volume=100;


#ifdef __hpux
#include <sys/signal.h>
		/* without the following line, quiting mikmod by CTRL-C
		 * would close the terminal window, which is probably
		 * not intended by the user
		 */
		signal(SIGINT, SIG_IGN);
#endif

	/*  initialize curses interface */
	init_display();

	for(quit=0; !quit && optind<nargc; optind++){

	/* kill audio output, as some mods sound awful left hanging... */
		MD_PatternChange();

		cur_mod.deleted=0;
		strcpy(cur_mod.filename,nargv[optind]);

		/* load the module */

		mf=ML_LoadFN(nargv[optind]);

		/* didn't work -> exit with errormsg. */

		if(mf==NULL){
			if(tolerant)
			{
				display_version();
				display_error(myerr,myerr_file);
				continue;
			}
			else
			{
				printf("MikMod Error: %s\n",myerr);
				break;
			}
		}

		/*      initialize modplayer to play this module */

		MP_Init(mf);

		update_driver_display();
		update_file_display();
		display_version();
		display_driver();
		display_file();

		cur_mod.numpat=mf->numpos;
		cur_mod.songname=mf->songname;
		cur_mod.modtype=mf->modtype;
		cur_mod.flags=mf->flags;
		update_name_display();
		display_name();

		
		/*	set the number of voices to use.. you
			could add extra channels here (e.g. md_numchn=mf->numchn+4; )
			to use for your own soundeffects:
		*/

		md_numchn=mf->numchn;

		/*  start playing the module: */

		MD_PlayStart();

		while(!MP_Ready() && !quit){ /* if we have a quit signal, exit loop */

			MD_Update();

			/* no need to wait with the unix drivers */

			if (!md_type) /* handled elsewhere for GUS cards */
			{
				/* update the status display... */
				cur_mod.sngpos=mp_sngpos+1;
				cur_mod.patpos=mp_patpos;
				update_status_display();
				display_status();
			}

			if(!play_current){ 	/*play_current=0 when the next or previous*/
				play_current=1;	/*mod is selected*/
				break;
			}
		}
		MD_PlayStop();          /* stop playing */
		ML_Free(mf);            /* and free the module */

	}

	MD_Exit();
	exit_display();
	if (quit)
		{
		return 0;
		}
	if(!quiet)
		if(mf==NULL){
			printf("MikMod Error: %s\n",myerr);
			if(myerr_file)
				printf("%s\n",myerr_file);
		}
		else
			printf("Finished playlist...\n");
	return 1;
}
