/*
 *   gettypes.c
 *
 *   Oliver Fromme  <oliver.fromme@heim3.tu-clausthal.de>
 *   Sun Jul 20 08:04:31 MET DST 1997
 *
 *   This is a very simple program that helps configuring a
 *   software package for the local system.  It computes the
 *   appropriate types for 8, 16 and 32 bit wide integers,
 *   and outputs the required typedefs with the following
 *   names:  int8, int16, int32, uint8, uint16, uint32.
 *
 *   Notes:
 *    - The "intXX" types are signed, the "uintXX" types are
 *      unsigned.
 *
 *    - The types are guaranteed to have AT LEAST the indicated
 *      number of bits.  DO NOT depend on them having exactly
 *      that many bits, because they might have more.
 *
 *    - Use the system type "char" for ASCII characters, not
 *      int8 or uint8.  DO NOT depend on "char" having exactly
 *      8 bits, because it might have more.
 *
 *    - For normal system integers (counters etc.), use "int",
 *      not int16 or uint16.  The type "int" is guaranteed to
 *      have at least 16 bits, so it should be used for things
 *      which don't exceed the range -32767 ... 32767, or 0 ...
 *      65535 for unsigned int.  Similarly, the type "long int"
 *      is guaranteed to have at least 32 bits.
 *
 *    - This program does not provide a 64 bit integer type.
 *      Many systems don't have one, and depending on such a
 *      type would seriously hut a program's portability.  If
 *      you need such large numbers, try to use floating point
 *      values, or simulate them with two 32 bit values.
 *      If you don't care about portability: Some systems have
 *      a "long long" type (for example gcc/intel, SparcWorks
 *      cc/Sun) which is 64 bits wide, and on DEC Alpha the
 *      "long" type is already 64 bits wide.  There are two
 *      defines supported which could help you in that case:
 *         NEED64BITS   - define this if you absolutely,
 *                        definitely need a 64 bit wide type.
 *         HAVELONGLONG - define this if your compiler knows
 *                        about a "long long" type.
 */

#include <stdio.h>
#include <stdlib.h>

#ifdef HAVELONGLONG
	static enum {t_char, t_short, t_int, t_long, t_longlong} type_enum;
	static const char *typename[5] =
		{"char", "short", "int", "long", "long long"};
#else
	static enum {t_char, t_short, t_int, t_long} type_enum;
	static const char *typename[4] =
		{"char", "short", "int", "long"};
#endif

static int findmin (int min, int sz[])
{
#ifdef HAVELONGLONG
	if (min > sz[t_longlong]) {
#else
	if (min > sz[t_long]) {
#endif
		fprintf (stderr, "*** ERROR:  No datatype that is at least"
			" %d bits wide!\n", min);
		exit (1);
	}
#ifdef HAVELONGLONG
	if (min > sz[t_long])
		return (t_longlong);
#endif
	if (min > sz[t_int])
		return (t_long);
	if (min > sz[t_short])
		return (t_int);
	if (min > sz[t_char])
		return (t_short);
	return (t_char);
}

/*ARGSUSED*/
int main (int argc, char *argv[])
{
	int ssz[5];
	int usz[5];
	int sig8,   sig16,   sig32;
	int unsig8, unsig16, unsig32;
#ifdef NEED64BITS
	int sig64, unsig64;
#endif

	ssz[t_char]  = 8 * sizeof(signed   char);
	usz[t_char]  = 8 * sizeof(unsigned char);
	ssz[t_short] = 8 * sizeof(signed   short);
	usz[t_short] = 8 * sizeof(unsigned short);
	ssz[t_int]   = 8 * sizeof(signed   int);
	usz[t_int]   = 8 * sizeof(unsigned int);
	ssz[t_long]  = 8 * sizeof(signed   long);
	usz[t_long]  = 8 * sizeof(unsigned long);
#ifdef HAVELONGLONG
	ssz[t_longlong] = 8 * sizeof(signed   long long);
	usz[t_longlong] = 8 * sizeof(unsigned long long);
#endif

	sig8    = findmin(8,  ssz);
	unsig8  = findmin(8,  usz);
	sig16   = findmin(16, ssz);
	unsig16 = findmin(16, usz);
	sig32   = findmin(32, ssz);
	unsig32 = findmin(32, usz);
#ifdef NEED64BITS
	sig64   = findmin(64, ssz);
	unsig64 = findmin(64, usz);
#endif

	printf ("/*\n *   This file is automatically generated.\n");
	printf (" *   Do not edit, changes will be lost.\n */\n");
	printf ("#ifndef SIZE_INT8\n");
	printf ("typedef signed   %s int8;\n",   typename[sig8]);
	printf ("typedef unsigned %s uint8;\n",  typename[unsig8]);
	printf ("typedef signed   %s int16;\n",  typename[sig16]);
	printf ("typedef unsigned %s uint16;\n", typename[unsig16]);
	printf ("typedef signed   %s int32;\n",  typename[sig32]);
	printf ("typedef unsigned %s uint32;\n", typename[unsig32]);
#ifdef NEED64BITS
	printf ("typedef signed   %s int64;\n",  typename[sig64]);
	printf ("typedef unsigned %s uint64;\n", typename[unsig64]);
#endif

	printf ("#define SIZE_INT8   %d\n", ssz[sig8]    / 8);
	printf ("#define SIZE_UINT8  %d\n", ssz[unsig8]  / 8);
	printf ("#define SIZE_INT16  %d\n", ssz[sig16]   / 8);
	printf ("#define SIZE_UINT16 %d\n", ssz[unsig16] / 8);
	printf ("#define SIZE_INT32  %d\n", ssz[sig32]   / 8);
	printf ("#define SIZE_UINT32 %d\n", ssz[unsig32] / 8);
#ifdef NEED64BITS
	printf ("#define SIZE_INT64  %d\n", ssz[sig64]   / 8);
	printf ("#define SIZE_UINT64 %d\n", ssz[unsig64] / 8);
#endif
	printf ("#endif\n");
	exit (0);
	/*NOTREACHED*/ /*OH_LINT_IS_SO_SILLY*/
	return (0);
}

/* EOF */
