#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <cstdio>
#include <strstream>
#include <gdk--/font.h>
#include <gtk--/main.h>
#include <gtk--/box.h>
#include <gtk--/table.h>
#include <gtk--/text.h>
#include <gtk--/scrollbar.h>
#include "CWText.h"

CWText::CWText(Gtk_Box* parent) : Gtk_Text()
{
  _isAtEnd = false;
  _wasAtEndLastTick = false;
  _isFileRequested = false;

  table = new Gtk_Table(2, 2);
  table->set_row_spacings(5);
  table->set_col_spacings(5);
  parent->pack_start(Gtk_ObjectHandle<Gtk_Table>(table));

  // Use a fixed font and set the size of this widget so that it
  // displays 80x24 characters.
  _fixedFont = new Gdk_Font("fixed");
  set_usize(_fixedFont->char_width('Z') * 80,
            _fixedFont->height() * 24 + 4);   // 4 is a fudge factor.

  table->attach(Gtk_ObjectHandle<Gtk_Text>(this), 0, 1, 0, 1);
  Gtk_Text::show();

  // I don't know why, but Gtk_Text::get_vadjustment() and the Gtk_VScrollbar
  // constructors are not "const"-wise compatible.  If you look at sample
  // code from gtk+, i.e. straight "C", they should be.
  vadj = const_cast<Gtk_Adjustment*>(get_vadjustment());
  vscrollbar = new Gtk_VScrollbar(*vadj);
  table->attach(Gtk_ObjectHandle<Gtk_VScrollbar>(vscrollbar),
                   1, 2, 0, 1, GTK_FILL, GTK_FILL);
  vscrollbar->show();  

  connect_to_method(vadj->value_changed, this, &CWText::ValueChangedCB);
  connect_to_method(vadj->changed, this, &CWText::ChangedCB);

  // Set up a timeout to make periodic updates.
  _tick = connect_to_method(Gtk_Main::timeout(500), this, &CWText::TickCB); 
}

CWText::~CWText()
{
  if( _file )
    {
      _file->close();
      delete _file;
      _file = 0;
    }

  if( _fixedFont )
    {
      delete _fixedFont;
      _fixedFont = 0;
    }
}


void CWText::show()
{
  table->show();
}


void CWText::Clear()
{
  freeze();
  set_point(0);
  forward_delete(get_length());
  thaw();
}


void CWText::insert(const _gtk_string& buffer, gint length)
{
  Gtk_Text::insert(*_fixedFont, 0, 0, buffer, length);
}


void CWText::LoadFile(string fileName, bool waitUntilAtEnd)
{
  if( !_isAtEnd && waitUntilAtEnd )
    {
      // Check again next tick.
      _isFileRequested = true;
      _requestedFile = fileName;
      return;
    }

  // Load the file immediately.

  // Close the previous file.
  if( _file )
    {
      _file->close();
      delete _file;
      _file = 0;
    }

  Clear();
  _currPos = 0;

  // Open the new file.
  _file = new ifstream(fileName.c_str());
  if( *_file )
    {
      // Load the new file.
      freeze();
      FileInsert();
      thaw();
    }
  else
    {
      ostrstream ostrm;
      ostrm << "\nWarning: unable to open \"" << fileName << "\"." << ends;
      insert(ostrm.str(), -1);
      ostrm.rdbuf()->freeze(0);
      fileName = "";
    }

  // Leave these here because the above FileInsert() causes _isAtEnd
  // to be set to true which in turn causes TickCB() to warp the
  // cursor to the bottom.  You don't want to warp until after
  // GotoEnd() is called.
  _isAtEnd = false;
  _wasAtEndLastTick = false;

  // It is important to invoke the Gtk-- Signal as the last thing done
  // in the function.  Otherwise, you could get weird interactions
  // with what would remain of this method.
  FileLoaded(fileName);
}



////////////////////////////////////////////////////////////////////////
// protected
////////////////////////////////////////////////////////////////////////


void CWText::ValueChangedCB()
{
  // Tells you when the adjustments "value" (and hence when the
  // slider's value) has changed.
  //
  // Use this to notify connected methods of whether the user is at
  // the end of the displayed text.

  if( _isAtEnd && vadj->gtkobj()->value < _endValue )
    {
      _isAtEnd = false;
      return;
    }

  // _fixedFont->height() is mostly a fudge factor, but it makes some sense.
  if( !_isAtEnd && vadj->gtkobj()->value + _fixedFont->height() >= _endValue )
    {
      _isAtEnd = true;
    }
}


void CWText::ChangedCB()
{
  // Tells you when the text widget changes the adjustments "upper"
  // and "lower".
  _endValue = vadj->gtkobj()->upper
              - vadj->gtkobj()->page_increment
              - _fixedFont->height();
}


void CWText::GotoEnd()
{
  vadj->set_value(_endValue);
  _isAtEnd = true;
}


////////////////////////////////////////////////////////////////////////
// private
////////////////////////////////////////////////////////////////////////

void CWText::CheckFileInput()
{
  if( !_file )
    return;

  off_t endPos;
  if( GetFileSize(endPos) )
    {
      if( _currPos < endPos )
        {
          // You have to clear the status flags, or _file will continue to
          // think it is at eof (meaning you won't be able to read the new
          // input).
          _file->clear();

          FileInsert();

          if( _isAtEnd )
            GotoEnd();
        }
    }
}


bool CWText::GetFileSize(off_t& size)
{
  struct stat buf;
  if( fstat(_file->rdbuf()->fd(), &buf) )
    return false;
  size = buf.st_size;
  return true;
}


void CWText::FileInsert()
{
  while( _file->read(_readBuffer, MAX_READ_BUFFER) )
    insert(_readBuffer, _file->gcount());
  insert(_readBuffer, _file->gcount());
  _currPos = _file->tellg();
}


gint CWText::TickCB()
{
  if( _isAtEnd && _isFileRequested )
    {
      LoadFile(_requestedFile);
      _isFileRequested = false;
    }
  else if( _isAtEnd )
    {
      // When the user moves the scrollbar off of the end position, I
      // do not update the text being displayed.  When the user returns
      // after Crafty has add much text to the end of the file, if you
      // don't first freeze, all the text scrolls on at once.
      //
      // However, you don't want to freeze every time TickCB() is called
      // because it is relatively expensive for scrolling small amounts
      // of text.
      if( !_wasAtEndLastTick )
        freeze();

      CheckFileInput();

      if( !_wasAtEndLastTick )
        {
          thaw();
          GotoEnd();
        }
      _wasAtEndLastTick = true;
    }
  else
    _wasAtEndLastTick = false;
  return 1;
}
