/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Bank_DRAM.h"

#include "Bank_ROM.h"			// IsPCInRAM (implicitly, through META_CHECK)
#include "Bank_SRAM.h"			// gRAMBank_Size, gRAM_Memory, gMemoryAccess
#include "CPU_REG.h"			// Software::BusError
#include "DebugMgr.h"			// Debug::CheckStepSpy
#include "MetaMemory.h"			// MetaMemory
#include "RAM_ROM.h"			// Memory::InitializeBanks


// ===========================================================================
//		 DRAM Bank Accessors
// ===========================================================================
// These functions provide fetch and store access to the emulator's random
// access memory.

static AddressBank	gAddressBank =
{
	DRAMBank::GetLong,
	DRAMBank::GetWord,
	DRAMBank::GetByte,
	DRAMBank::SetLong,
	DRAMBank::SetWord,
	DRAMBank::SetByte,
	DRAMBank::GetRealAddress,
	DRAMBank::ValidAddress,
	DRAMBank::GetMetaAddress,
	DRAMBank::AddOpcodeCycles
};

const uae_u32	kMemoryStart = 0x00000000;
const uaecptr	kGlobalStart = offsetof (LowMemHdrType, globals);

static uae_u32	gDynamicHeapSize;


/*
// !!! TBD Use these to determine heap size(s):

	UInt		MemNumCards(void)
	UInt		MemNumHeaps(UInt cardNo)
	UInt		MemHeapID(UInt cardNo, UInt heapNo)
	Boolean		MemHeapDynamic(UInt heapID)
	VoidPtr		MemHeapPtr(UInt heapID)
	ULong		MemHeapSize(UInt heapID)
*/

static inline int InlineValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	int	result = (iAddress + iSize) <= gRAMBank_Size;

	return result;
}


static inline uae_u8* InlineGetRealAddress (uaecptr iAddress)
{
	return (uae_u8*) &(gRAM_Memory[iAddress]);
}


static inline uae_u8* InlineGetMetaAddress (uaecptr address)
{
	return (uae_u8*) &(gRAM_MetaMemory[address]);
}


static inline void ScreenCheck (uae_u8* metaAddress, uaecptr address, uae_u32 size)
{
#if defined (macintosh)

	if (size == 1 && !MetaMemory::IsScreenBuffer8 (metaAddress))
		return;

	if (size == 2 && !MetaMemory::IsScreenBuffer16 (metaAddress))
		return;

	if (size == 4 && !MetaMemory::IsScreenBuffer32 (metaAddress))
		return;

#else

	if (MetaMemory::IsScreenBuffer (metaAddress, size))

#endif

	{
		Screen::MarkDirty (address, size);
	}
}


/***********************************************************************
 *
 * FUNCTION:	DRAMBank::Initialize
 *
 * DESCRIPTION:	Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DRAMBank::Initialize (void)
{
	DRAMBank::SetBankHandlers ();
}


/***********************************************************************
 *
 * FUNCTION:	DRAMBank::Reset
 *
 * DESCRIPTION:	Standard reset function.  Sets the sub-system to a
 *				default state.  This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DRAMBank::Reset (void)
{
}


/***********************************************************************
 *
 * FUNCTION:	DRAMBank::Save
 *
 * DESCRIPTION:	Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DRAMBank::Save (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	DRAMBank::Load
 *
 * DESCRIPTION:	Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DRAMBank::Load (SessionFile&)
{
}


/***********************************************************************
 *
 * FUNCTION:	DRAMBank::Dispose
 *
 * DESCRIPTION:	Standard dispose function.  Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void DRAMBank::Dispose (void)
{
}


/***********************************************************************
 *
 * FUNCTION:    DRAMBank::SetBankHandlers
 *
 * DESCRIPTION: Set the bank handlers UAE uses to dispatch memory
 *				access operations.
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void DRAMBank::SetBankHandlers (void)
{
	// Memory banks 0 and 1 are managed by the functions in DRAMBank.

	gDynamicHeapSize = HWRegisters::GetDynamicHeapSize ();

	uae_u32	sixtyFourK	= 64 * 1024L;
	uae_u32	numBanks	= (gDynamicHeapSize + sixtyFourK - 1) / sixtyFourK;

	Memory::InitializeBanks (	gAddressBank,
								bankindex (kMemoryStart),
								numBanks);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::GetLong
// ---------------------------------------------------------------------------

uae_u32 DRAMBank::GetLong (uaecptr iAddress)
{
	if (iAddress > gDynamicHeapSize)
		return SRAMBank::GetLong (iAddress);

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMLongRead]++;
	if (iAddress & 2)
		gMemoryAccess[kDRAMLongRead2]++;
#endif

	if (CHECK_FOR_ADDRESS_ERROR && (iAddress & 1) != 0)
	{
		Software::AddressError ();
	}

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, GetLong, long, true)

	if (VALIDATE_DRAM_GET &&
		gMemAccessFlags.fValidate_DRAMGet &&
		!InlineValidAddress (iAddress, sizeof (long)))
	{
		InvalidAccess (iAddress, sizeof (long), true);
	}

#if (HAS_PROFILING)
	CYCLE_GETLONG (WAITSTATES_DRAM);
#endif

	return do_get_mem_long (gRAM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::GetWord
// ---------------------------------------------------------------------------

uae_u32 DRAMBank::GetWord (uaecptr iAddress)
{
	if (iAddress > gDynamicHeapSize)
		return SRAMBank::GetWord (iAddress);

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMWordRead]++;
#endif

	if (CHECK_FOR_ADDRESS_ERROR && (iAddress & 1) != 0)
	{
		Software::AddressError ();
	}

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, GetWord, short, true)

	if (VALIDATE_DRAM_GET &&
		gMemAccessFlags.fValidate_DRAMGet &&
		!InlineValidAddress (iAddress, sizeof (short)))
	{
		InvalidAccess (iAddress, sizeof (short), true);
	}

#if (HAS_PROFILING)
	CYCLE_GETWORD (WAITSTATES_DRAM);
#endif

	return do_get_mem_word (gRAM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::GetByte
// ---------------------------------------------------------------------------

uae_u32 DRAMBank::GetByte (uaecptr iAddress)
{
	if (iAddress > gDynamicHeapSize)
		return SRAMBank::GetByte (iAddress);

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMByteRead]++;
#endif

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, GetByte, char, true)

	if (VALIDATE_DRAM_GET &&
		gMemAccessFlags.fValidate_DRAMGet &&
		!InlineValidAddress (iAddress, sizeof (char)))
	{
		InvalidAccess (iAddress, sizeof (char), true);
	}

#if (HAS_PROFILING)
	CYCLE_GETBYTE (WAITSTATES_DRAM);
#endif

	return do_get_mem_byte (gRAM_Memory + iAddress);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::SetLong
// ---------------------------------------------------------------------------

void DRAMBank::SetLong (uaecptr iAddress, uae_u32 iLongValue)
{
	if (iAddress > gDynamicHeapSize)
	{
		SRAMBank::SetLong (iAddress, iLongValue);
		return;
	}

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMLongWrite]++;
	if (iAddress & 2)
		gMemoryAccess[kDRAMLongWrite2]++;
#endif

	if (CHECK_FOR_ADDRESS_ERROR && (iAddress & 1) != 0)
	{
		Software::AddressError ();
	}

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, SetLong, long, false)

	if (VALIDATE_DRAM_SET &&
		gMemAccessFlags.fValidate_DRAMSet &&
		!InlineValidAddress (iAddress, sizeof (long)))
	{
		InvalidAccess (iAddress, sizeof (long), false);
	}

	ScreenCheck (metaAddress, iAddress, sizeof (long));

#if (HAS_PROFILING)
	CYCLE_PUTLONG (WAITSTATES_DRAM);
#endif

	do_put_mem_long (gRAM_Memory + iAddress, iLongValue);

#if FOR_LATER
	// Mark that this memory location can now be read from.

	MetaMemory::MarkLongInitialized (iAddress);
#endif

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 4);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::SetWord
// ---------------------------------------------------------------------------

void DRAMBank::SetWord (uaecptr iAddress, uae_u32 iWordValue)
{
	if (iAddress > gDynamicHeapSize)
	{
		SRAMBank::SetWord (iAddress, iWordValue);
		return;
	}

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMWordWrite]++;
#endif

	if (CHECK_FOR_ADDRESS_ERROR && (iAddress & 1) != 0)
	{
		Software::AddressError ();
	}

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, SetWord, short, false)

	if (VALIDATE_DRAM_SET &&
		gMemAccessFlags.fValidate_DRAMSet &&
		!InlineValidAddress (iAddress, sizeof (short)))
	{
		InvalidAccess (iAddress, sizeof (short), false);
	}

	ScreenCheck (metaAddress, iAddress, sizeof (short));

#if (HAS_PROFILING)
	CYCLE_PUTWORD (WAITSTATES_DRAM);
#endif

	do_put_mem_word (gRAM_Memory + iAddress, iWordValue);

#if FOR_LATER
	// Mark that this memory location can now be read from.

	MetaMemory::MarkWordInitialized (iAddress);
#endif

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 2);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::SetByte
// ---------------------------------------------------------------------------

void DRAMBank::SetByte (uaecptr iAddress, uae_u32 iByteValue)
{
	if (iAddress > gDynamicHeapSize)
	{
		SRAMBank::SetByte (iAddress, iByteValue);
		return;
	}

#if (PROFILE_MEMORY)
	gMemoryAccess[kDRAMByteWrite]++;
#endif

	register uae_u8*	metaAddress = InlineGetMetaAddress (iAddress);
	META_CHECK (metaAddress, iAddress, SetByte, char, false)

	if (VALIDATE_DRAM_SET &&
		gMemAccessFlags.fValidate_DRAMSet &&
		!InlineValidAddress (iAddress, sizeof (char)))
	{
		InvalidAccess (iAddress, sizeof (char), false);
	}

	ScreenCheck (metaAddress, iAddress, sizeof (char));

#if (HAS_PROFILING)
	CYCLE_PUTBYTE (WAITSTATES_DRAM);
#endif

	do_put_mem_byte (gRAM_Memory + iAddress, iByteValue);

#if FOR_LATER
	// Mark that this memory location can now be read from.

	MetaMemory::MarkByteInitialized (iAddress);
#endif

	// See if any interesting memory locations have changed.  If so,
	// CheckStepSpy will change gBreakReason, which we'll check later.

	Debug::CheckStepSpy (iAddress, 1);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::ValidAddress
// ---------------------------------------------------------------------------

int DRAMBank::ValidAddress (uaecptr iAddress, uae_u32 iSize)
{
	int	result = InlineValidAddress (iAddress, iSize);

//	assert (result);

	return result;
}


// ---------------------------------------------------------------------------
//		 DRAMBank::GetRealAddress
// ---------------------------------------------------------------------------

uae_u8* DRAMBank::GetRealAddress (uaecptr iAddress)
{
	return InlineGetRealAddress (iAddress);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::GetMetaAddress
// ---------------------------------------------------------------------------

uae_u8* DRAMBank::GetMetaAddress (uaecptr iAddress)
{
	return InlineGetMetaAddress (iAddress);
}


// ---------------------------------------------------------------------------
//		 DRAMBank::AddOpcodeCycles
// ---------------------------------------------------------------------------

void DRAMBank::AddOpcodeCycles (void)
{
#if (HAS_PROFILING)
	CYCLE_GETWORD (WAITSTATES_DRAM);
#endif
}


// ---------------------------------------------------------------------------
//		 DRAMBank::InvalidAccess
// ---------------------------------------------------------------------------

void DRAMBank::InvalidAccess (uaecptr iAddress, long size, Bool forRead)
{
	UNUSED_PARAM(iAddress)
	UNUSED_PARAM(size)
	UNUSED_PARAM(forRead)

	Software::BusError ();
}


// ---------------------------------------------------------------------------
//		 DRAMBank::ProbableCause
// ---------------------------------------------------------------------------

void DRAMBank::ProbableCause (uaecptr iAddress, long size, Bool forRead)
{
	Errors::EAccessType	whatHappened = MetaMemory::GetWhatHappened (iAddress, size, forRead);

	if (whatHappened != Errors::kOKAccess)
	{
		assert (whatHappened != Errors::kUnknownAccess);

		Memory::PreventedAccess (iAddress, size, forRead, whatHappened);
	}
}
