/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _LOGGING_H_
#define _LOGGING_H_

#include "PreferenceMgr.h"		// FOR_EACH_PREF
#include "Miscellaneous.h"		// StMemory
#include "omnithread.h"			// omni_mutex

class LogStreamInner
{
	// Non-multithread-safe version of LogStream.  LogStream
	// acquires a "logging mutex" and calls these functions.

	public:
								LogStreamInner	(const char* baseName);
								~LogStreamInner	(void);

		int						DumpHex			(const void*, long dataLen);
		int						VPrintf			(const char* fmt, va_list args);
		int						Write			(const void* buffer, long size);

		void					Clear			(void);

		long					GetLogSize		(void);
		void					SetLogSize		(long);

		void					EnsureNewFile	(void);
		void					DumpToFile		(void);

	private:
		FileReference			CreateFileReference	(void);
		void					Timestamp			(void);
		void					NewLine				(void);
		void					Append				(const char* buffer, long size);
		void					DumpToFile			(FileHandle&, const char*, long size);

		const char*				fBaseName;
		long					fFileIndex;

		StMemory				fBuffer;
		long					fBufferSize;
		long					fBufferOffset;

		uae_u32					fLastGremlinEventCounter;
		uae_u32					fLastTimestampTime;
		uae_u32					fBaseTimestampTime ;
		char					fLastTimestampString[30];
};

class LogStream
{
	public:
								LogStream		(const char* baseName);
								~LogStream		(void);

	// Multithread-safe interface for logging information.

		int						Printf			(const char* msg, ...);
		int						DataPrintf		(const void*, long dataLen, const char* msg, ...);
		int						VPrintf			(const char* fmt, va_list args);
		int						Write			(const void* buffer, long size);

		void					Clear			(void);

		long					GetLogSize		(void);
		void					SetLogSize		(long);

		void					EnsureNewFile	(void);
		void					DumpToFile		(void);

	private:
		omni_mutex				fMutex;
		LogStreamInner			fInner;
};

void		LogEvtAddEventToQueue		(uaecptr);
void		LogEvtAddUniqueEventToQueue	(uaecptr, DWord, Boolean);
void		LogEvtEnqueueKey			(UInt ascii, UInt keycode, UInt modifiers);
void		LogEvtEnqueuePenPoint		(uaecptr);

void		LogEvtGetEvent				(uaecptr, SDWord timeout);
void		LogEvtGetPen				(uaecptr, uaecptr, uaecptr);
void		LogEvtGetSysEvent			(uaecptr, Long timeout);

LogStream*	LogGetStdLog				(void);

void		LogUpdateCache				(void);

#define LogAppendMsg	LogGetStdLog ()->Printf
#define LogAppendData	LogGetStdLog ()->DataPrintf
#define LogClear		LogGetStdLog ()->Clear
#define LogDump			LogGetStdLog ()->DumpToFile
#define LogStartNew		LogGetStdLog ()->DumpToFile (); LogGetStdLog ()->Clear (); LogGetStdLog ()->EnsureNewFile


// The LogFoo macros below are called all the time.  When implemented in
// terms of a function that instantiates a Preference<> object all the
// time, they can slow down the overall execution of Poser significantly.
// Therefore, the logging sub-system caches the relevent values and
// works from those, instead.

#define DECLARE_CACHED_PREF_KEYS(name, type, init) kCachedPrefKey##name,
enum
{
	FOR_EACH_PREF(DECLARE_CACHED_PREF_KEYS)
	kCachedPrefKeyDummy
};

extern Bool gLogCache[];
inline Bool LogCommon(int key)
{
	assert (key >= kCachedPrefKeyLogErrorMessages);
	assert (key <= kCachedPrefKeyLogRPCData);
	return gLogCache[key];
}

#define LogErrorMessages()		LogCommon(kCachedPrefKeyLogErrorMessages)
#define LogWarningMessages()	LogCommon(kCachedPrefKeyLogWarningMessages)
#define LogGremlins()			LogCommon(kCachedPrefKeyLogGremlins)
#define LogCPUOpcodes()			LogCommon(kCachedPrefKeyLogCPUOpcodes)
#define LogEnqueuedEvents()		LogCommon(kCachedPrefKeyLogEnqueuedEvents)
#define LogDequeuedEvents()		LogCommon(kCachedPrefKeyLogDequeuedEvents)
#define LogSystemCalls()		LogCommon(kCachedPrefKeyLogSystemCalls)
#define LogApplicationCalls()	LogCommon(kCachedPrefKeyLogApplicationCalls)
#define LogSerial()				LogCommon(kCachedPrefKeyLogSerial)
#define LogSerialData()			LogCommon(kCachedPrefKeyLogSerialData)
#define LogNetLib()				LogCommon(kCachedPrefKeyLogNetLib)
#define LogNetLibData()			LogCommon(kCachedPrefKeyLogNetLibData)
#define LogExgMgr()				LogCommon(kCachedPrefKeyLogExgMgr)
#define LogExgMgrData()			LogCommon(kCachedPrefKeyLogExgMgrData)
#define LogHLDebugger()			LogCommon(kCachedPrefKeyLogHLDebugger)
#define LogHLDebuggerData()		LogCommon(kCachedPrefKeyLogHLDebuggerData)
#define LogLLDebugger()			LogCommon(kCachedPrefKeyLogLLDebugger)
#define LogLLDebuggerData()		LogCommon(kCachedPrefKeyLogLLDebuggerData)
#define LogRPC()			    LogCommon(kCachedPrefKeyLogRPC)
#define LogRPCData()		    LogCommon(kCachedPrefKeyLogRPCData)


#endif /* _LOGGING_H_ */
