/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _PREFERENCEMGR_H_
#define _PREFERENCEMGR_H_

#include "EmulatorTypes.h"	 	// FileRefList, DatabaseInfoList, DeviceType
#include "omnithread.h" 		// omni_mutex, omni_mutex_lock
#include "Platform_Files.h" 	// FileReference
#include "Skins.h"				// SkinNameList

#include <map>
#include <vector>


/*
	This file contains the routines for loading, saving, and accessing a collection
	of preferences/settings.

	All settings are accessed via the Preference class.  This is a templatized class
	that is specialized on the type of the setting and parameterized on the name of
	the setting.  This sounds nasty, but in practice is simple. For example, say you
	needed to access whether or not NetLib redirection was turned on.  You would use
	the following:

		Preference<bool>	pref(kPrefKeyRedirectNetLib);
		
		if (*pref)
		{
			// It's on.
		}

	Changing a value works similarly:

		Preference<DeviceType>	pref(kPrefKeyDeviceType);
		*pref = kDevicePalmIII;

	When the Preference object is destructed, it's new value is written back out to
	the Preference Manager.

	The Preference Manager is the "container" for all the preferences used by the
	application.  You can think of it as a map/dictionary/associative array.  You
	directly access the Preference Manager in order to create it, have it load the
	collection of preferences from disk, have it save them to disk, and to destroy
	it.  While there are GetPref and SetPref methods on this class, those are for
	the use of the Preference objects.

	All preferences are saved as text in the form:

		<key>=<value>

	<key> is any unique string.  It can consist of any text characters except for
	'='.  Keys can have subparts separated by '.', and can indicate array indices
	with "[#]".  For example:

		Scale=2
		GremlinInfo.fNumber=100
		PRC_MRU[0]=MyApplication.prc
		GremlinInfo.fAppList[0].name=AddressBook

	With the Preference class, it's possible to pull in all or part of the
	hierarchical information.  For instance:

		Preference<GremlinInfo>			pref("GremlinInfo");
		Preference<DatabaseInfoList>	pref("GremlinInfo.fAppList");
		Preference<DatabaseInfo>		pref("GremlinInfo.fAppList[0]");
		Preference<string>				pref("GremlinInfo.fAppList[0].name");
*/

typedef const char* PrefKeyType;

class BasePreference
{
	public:
								BasePreference		(PrefKeyType name, bool = true);
								BasePreference		(long index, bool = true);
		virtual 				~BasePreference 	(void);

		bool					Loaded				(void)	{ return fLoaded; }

		void					Flush				(void)	{ this->Save (); }

	protected:
		void					Load				(void);
		void					Save				(void);

	protected:
		virtual bool			DoLoad				(void) = 0;
		virtual void			DoSave				(void) = 0;

	protected:
		string					fName;
		bool					fLoaded;
		bool					fChanged;
		bool					fAcquireLock;
};

template <class T>
class Preference : public BasePreference
{
	public:
								Preference			(PrefKeyType name, bool = true);
								Preference			(long index, bool = true);
		virtual 				~Preference 		(void);

	// I would *like* to have these operators.	That way, I could pass in a
	// "Preference<Foo>" any place that accepts a "const Foo&" as a parameter.
	// However, CodeWarrior seems to use the conversion operators here in place
	// that accept merely a "Foo&", which has lead to some problems.
//								operator const T&() const	{ return *GetValue (); }
//								operator const T*() const	{ return GetValue (); }
		const T&				operator*() const			{ return *GetValue (); }
		const T*				operator->() const			{ return GetValue (); }

		const T*				GetValue () const			{ assert (fLoaded); return &fValue; }

		const T&				operator=(const T& rhs) 	{ fValue = rhs;
															  fLoaded = true;
															  fChanged = true;
															  this->Flush ();
															  return fValue; }

	protected:
		virtual bool			DoLoad				(void);
		virtual void			DoSave				(void);

	private:
		T						fValue;
};

typedef void (*PrefNotifyFunc)(PrefKeyType);
typedef StringList	PrefKeyList;

class Preferences
{
	public:
								Preferences 		(void);
		virtual 				~Preferences		(void);

		virtual void			Load				(void);
		virtual void			Save				(void);

// This should work, but CW doesn't appear to like it.
//	private:
//		template <class T>	friend class Preference;
	public:
		bool					GetPref 			(const string& key, string& value);
		void					SetPref 			(const string& key, const string& value);
		void					DeletePref			(const string& key);

	public:
		void					PushPrefix			(const string& prefix);
		void					PushPrefix			(long index);
		void					PopPrefix			(void);

		string					ExpandKey			(const string& name);
		string					AppendName			(string key, const string& name);

	public:
		void					AddNotification		(PrefNotifyFunc, PrefKeyType);
		void					AddNotification		(PrefNotifyFunc, const PrefKeyList&);
		void					RemoveNotification	(PrefNotifyFunc);
		void					RemoveNotification	(PrefNotifyFunc, PrefKeyType);
		void					RemoveNotification	(PrefNotifyFunc, const PrefKeyList&);
		void					DoNotify			(const string& key);

	protected:
		virtual void			WriteBanner 		(FILE*);
		virtual bool			ReadBanner			(FILE*);
		virtual void			StripUnused 		(void);

	protected:
		typedef StringStringMap			PrefList;
		typedef PrefList::value_type	PrefPairType;
		typedef PrefList::iterator		iterator;

		PrefList				fPreferences;

		typedef StringList		PrefixType;
		PrefixType				fPrefixes;

		typedef map<PrefNotifyFunc, PrefKeyList> PrefNotifyList;

		PrefNotifyList			fNotifications;

	public:
		static omni_mutex		fgPrefsMutex;
};

extern Preferences* gPrefs;


class EmulatorPreferences : public Preferences
{
	public:
								EmulatorPreferences (void);
		virtual 				~EmulatorPreferences(void);

		virtual void			Load				(void);

		FileReference			GetIndPRCMRU		(int);
		FileReference			GetIndRAMMRU		(int);
		FileReference			GetIndMRU			(const FileRefList&, int);

		void					UpdatePRCMRU		(const FileReference&);
		void					UpdateRAMMRU		(const FileReference&);
		void					UpdateMRU			(FileRefList&, const FileReference&);

		void					RemovePRCMRU		(const FileReference&);
		void					RemoveRAMMRU		(const FileReference&);
		void					RemoveMRU			(FileRefList&, const FileReference&);

		int 					UseLoggingOptions	(int newType);

	protected:
		virtual void			WriteBanner 		(FILE*);
		virtual bool			ReadBanner			(FILE*);
		virtual void			StripUnused 		(void);

		int 					fCurLoggingType;
};

extern EmulatorPreferences* gEmuPrefs;


/*
	The FOR_EACH_PREF macro is used to manage the preferences keys.  Preferences
	are accessed via keys passed to the Preference constructor.  These keys have
	the symbolic form:

		kPrefKey<Label>

	The macro below is used to declare the keys used to access the settings, define
	the keys, initialize the settings to default values.
*/

#define FOR_EACH_PREF(DO_TO_PREF)								\
	FOR_EACH_UNINIT_PREF(DO_TO_PREF)							\
	FOR_EACH_INIT_PREF(DO_TO_PREF)

#define FOR_EACH_UNINIT_PREF(DO_TO_PREF)						\
	DO_TO_PREF(WindowLocation, unused, unused)					\
	DO_TO_PREF(GCWLocation, unused, unused)						\
	DO_TO_PREF(BackgroundColor, unused, unused) 				\
	DO_TO_PREF(HighlightColor, unused, unused)

#define FOR_EACH_INIT_PREF(DO_TO_PREF)							\
	DO_TO_PREF(CloseAction, CloseActionType, (kSaveAsk))		\
	DO_TO_PREF(CommPort, string, (""))							\
	DO_TO_PREF(RedirectNetLib, bool, (false))					\
																\
	DO_TO_PREF(ReportCorruptedHeap, bool, (true))				\
	DO_TO_PREF(ReportFreeChunkAccess, bool, (true)) 			\
	DO_TO_PREF(ReportHardwareRegisterAccess, bool, (true))		\
	DO_TO_PREF(ReportInvalidPC, bool, (true))					\
	DO_TO_PREF(ReportLowMemoryAccess, bool, (true)) 			\
	DO_TO_PREF(ReportLowStackAccess, bool, (true))				\
	DO_TO_PREF(ReportMemMgrDataAccess, bool, (true))			\
	DO_TO_PREF(ReportMemMgrSemaphore, bool, (true)) 			\
	DO_TO_PREF(ReportScreenAccess, bool, (true))				\
	DO_TO_PREF(ReportStackAlmostOverflow, bool, (true)) 		\
	DO_TO_PREF(ReportStackOverflow, bool, (true))				\
	DO_TO_PREF(ReportStorageHeapAccess, bool, (true))			\
	DO_TO_PREF(ReportSysFatalAlert, bool, (true))				\
	DO_TO_PREF(ReportSystemGlobalAccess, bool, (true))			\
	DO_TO_PREF(ReportUnhandledException, bool, (true))			\
	DO_TO_PREF(ReportUnimplementedTrap, bool, (true))			\
	DO_TO_PREF(ReportUninitializedChunkAccess, bool, (true))	\
	DO_TO_PREF(ReportUninitializedStackAccess, bool, (true))	\
	DO_TO_PREF(ReportUnlockedChunkAccess, bool, (true)) 		\
																\
	DO_TO_PREF(LogErrorMessages, uae_u8, (0))					\
	DO_TO_PREF(LogWarningMessages, uae_u8, (0)) 				\
	DO_TO_PREF(LogGremlins, uae_u8, (0))						\
	DO_TO_PREF(LogCPUOpcodes, uae_u8, (0))						\
	DO_TO_PREF(LogEnqueuedEvents, uae_u8, (0))					\
	DO_TO_PREF(LogDequeuedEvents, uae_u8, (0))					\
	DO_TO_PREF(LogSystemCalls, uae_u8, (0)) 					\
	DO_TO_PREF(LogApplicationCalls, uae_u8, (0))				\
	DO_TO_PREF(LogSerial, uae_u8, (0))							\
	DO_TO_PREF(LogSerialData, uae_u8, (0))						\
	DO_TO_PREF(LogNetLib, uae_u8, (0))							\
	DO_TO_PREF(LogNetLibData, uae_u8, (0))						\
	DO_TO_PREF(LogExgMgr, uae_u8, (0))							\
	DO_TO_PREF(LogExgMgrData, uae_u8, (0))						\
	DO_TO_PREF(LogHLDebugger, uae_u8, (0))						\
	DO_TO_PREF(LogHLDebuggerData, uae_u8, (0))					\
	DO_TO_PREF(LogLLDebugger, uae_u8, (0))						\
	DO_TO_PREF(LogLLDebuggerData, uae_u8, (0))					\
	DO_TO_PREF(LogRPC, uae_u8, (0)) 							\
	DO_TO_PREF(LogRPCData, uae_u8, (0)) 						\
																\
	DO_TO_PREF(ReportMemoryLeaks, bool, (false))				\
	DO_TO_PREF(ReportLockedRecords, bool, (false))				\
																\
	DO_TO_PREF(DebuggerSocketPort, long, (6414))				\
	DO_TO_PREF(RPCSocketPort, long, (6415)) 					\
																\
	DO_TO_PREF(InterceptSysFatalAlert, bool, (true))			\
																\
	DO_TO_PREF(AskAboutStartMenu, bool, (true)) 				\
	DO_TO_PREF(StartMenuItem, FileReference, ())				\
																\
	DO_TO_PREF(FillNewBlocks, bool, (false))					\
	DO_TO_PREF(FillResizedBlocks, bool, (false))				\
	DO_TO_PREF(FillDisposedBlocks, bool, (false))				\
	DO_TO_PREF(FillStack, bool, (false))						\
																\
	DO_TO_PREF(LastConfiguration, Configuration, (kDevicePalmIII, 1024, FileReference())) 		\
																\
	DO_TO_PREF(GremlinInfo, GremlinInfo, ())					\
	DO_TO_PREF(HordeInfo, HordeInfo, ())						\
																\
	DO_TO_PREF(LastPSF, FileReference, ())						\
																\
	DO_TO_PREF(PRC_MRU, FileRefList, ())						\
	DO_TO_PREF(PSF_MRU, FileRefList, ())						\
																\
	DO_TO_PREF(Skins, SkinNameList, ())							\
	DO_TO_PREF(Scale, ScaleType, (2))							\
																\
	DO_TO_PREF(UserName, string, ("Palm OS Emulator"))

// Declare all the keys
#define DECLARE_PREF_KEYS(name, type, init) extern PrefKeyType kPrefKey##name;
FOR_EACH_PREF(DECLARE_PREF_KEYS)

#endif	// _PREFERENCEMGR_H_
