/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "fltk_Dialog.h"

#include <FL/Fl.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Browser.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Check_Button.H>
#include <FL/Fl_Choice.H>
#include <FL/Fl_Input.H>
#include <FL/Fl_Light_Button.H>
#include <FL/Fl_Output.H>
#include <FL/Fl_Pixmap.H>
#include <FL/Fl_Return_Button.H>
#include <FL/Fl_Round_Button.H>
#include <FL/Fl_Tabs.H>
#include <FL/Fl_Widget.H>
#include <FL/Fl_Window.H>
#include <FL/fl_draw.H>
#include <FL/fl_file_chooser.H>

#include <string.h>

#include "CPU_REG.h"
#include "EmulatorTypes.h"		// CloseActionType
#include "ErrorHandling.h"
#include "MapFile.h"
#include "Platform.h"
#include "PreferenceMgr.h"
#include "Skins.h"
#include "Strings.r.h"

#include "omnithread.h"
#include "fltk_main.h"

#include "poser.xpm"

static omni_mutex sDialogMutex;
static omni_semaphore sDialogDone(0);
static char *sDialogTitle = NULL;
static char *sDialogMsg = NULL;
static int sDialogFlags;
static int sDialogResult;

#define wnd_NewConfiguration			1006
#define item_cfgDeviceMenu				100
#define item_cfgSkinMenu				101
#define item_cfgRAMSizeMenu 			102
#define item_cfgBrowseButton			103

#define wnd_NewGremlin					1007
#define item_ngGremlinStart				3
#define item_ngGremlinEnd				4
#define item_ngSwitchDepth				5
#define item_ngMaxDepth					6
#define item_ngSnapshotFrequency		7
#define item_ngApplicationList			8
#define item_ngLoggingOptions			9

#define wnd_Preferences 				1008
#define item_prefPort					100
#define item_prefNetLib 				101
#define item_prefSaveAlways 			102
#define item_prefSaveAsk				103
#define item_prefSaveNever				104
#define item_prefUserName				105

#define wnd_DebugOptions				1009
#define item_dbgLowMem					100
#define item_dbgGlobals 				101
#define item_dbgScreen					102
#define item_dbgRegisters				103
#define item_dbgMemMgrData				104
#define item_dbgStorageHeap 			105
#define item_dbgMemMgrSemaphore 		106
#define item_dbgLowStack				107
#define item_dbgFreeChunk				108
#define item_dbgUnlockedChunk			109
#define item_dbgUninitStack 			110
#define item_dbgUninitChunk 			111
#define item_dbgStackAlmostOverflow 	112

#define wnd_LoggingOptions				1010
#define item_logPageController			80
#define item_logErrors					100
#define item_logWarnings				101
#define item_logGremlins				102
#define item_logOpcodes 				103
#define item_logEnqueuedEvents			104
#define item_logDequeuedEvents			105
#define item_logSystemCalls 			106
#define item_logApplicationCalls		107
#define item_logSerial					108
#define item_logSerialData				109
#define item_logNetLib					110
#define item_logNetLibData				111
#define item_logExgMgr					112
#define item_logExgMgrData				113
#define item_logHighLevelDebugger		114
#define item_logHighLevelDebuggerData	115
#define item_logLowLevelDebugger		116
#define item_logLowLevelDebuggerData	117

#define wnd_Skins						1011
#define item_skDevices					3
#define item_skSkins					4
#define item_skDoubleScale				5
#define item_skWhiteBackground			6

#define wnd_ExportDialog				1018
#define item_exDefaultButton			1
#define item_exCancelButton				2
#define item_exDatabaseList				3


// -----------------------------------------------------------------------------
// helper routines for establishing z-order independent ids for widgets.

typedef unsigned long WidgetID;
map <WidgetID, Fl_Widget*> gWidgetIDs;

void PrvClearWidgetIDs (void)
{
	gWidgetIDs.clear ();
}

void PrvSetWidgetID (Fl_Widget* o, unsigned long id)
{
	gWidgetIDs[id] = o;
}

Fl_Widget* PrvFindWidgetByID (unsigned long id)
{
	return gWidgetIDs [id];
}

// -----------------------------------------------------------------------------
// helper functions for getting data out of windows

static void SetBoolFromButton (const char* key, int buttonID)
{
	Preference<bool> pref(key);
	Fl_Button* b = (Fl_Button*) ::PrvFindWidgetByID (buttonID);
	pref = (b->value() == 1);
}

static bool GetBoolPref (const char* key)
{
	Preference<bool> pref(key);
	return *pref;
}

#define SET_BOOL_FROM_BUTTON(bn, n) \
	SetBoolFromButton(kPrefKey##bn, n)

#define GET_BOOL_PREF(bn) \
	GetBoolPref (kPrefKey##bn)

// -----------------------------------------------------------------------------
// callback used to close modal dialogs. modalResult is < 0 if not yet completed.
// set modalResult to be >=0 to complete dialog. Typically 0=cancel, 1=ok

static long modalResult;
static void modalCallback( Fl_Widget*, long int result )
{
	modalResult = result;
}


// -----------------------------------------------------------------------------
// modal dialog control loop

static int postModalDialog( Fl_Window* dlg )
{
	modalResult = -9999;
	dlg->show();

	// run our own event loop so that we have total control
	// of the UI thread
	for (;;)
	{
		Fl::wait();
		if ( modalResult != -9999 )
			break;

		if ( !dlg->visible() )
		{
			// user hit escape (cancel)
			modalResult = 0;
			break;
		}
	}

	dlg->hide();

	return modalResult;
}


// -----------------------------------------------------------------------------
// simple text output subclass which does word wrapping...
class WrapOutput : public Fl_Output
{
	bool _wrapped;
	string _unwrapped;
public:
	WrapOutput( int x, int y, int w, int h, char *l=NULL)
		: Fl_Output( x,y,w,h,l ), _wrapped(false)
		{}
	virtual void draw();
	virtual void resize( int x,int y,int w, int h ) { _wrapped = false; Fl_Output::resize(x,y,w,h); }

	void unwrappedValue( const string& s ) { _unwrapped = s; }
};


// -----------------------------------------------------------------------------
// wrap the text before drawing, give the current window size

void WrapOutput::draw()
{
	if ( !_wrapped )
	{
		int len = _unwrapped.length();
		char *inbuf = new char[ len + 1 ];
		char *outbuf = new char[ len + 1 ];

		strcpy( inbuf, _unwrapped.c_str() );

		int tw = w()+40;
		char *t = outbuf, *f = inbuf;
		char *curline = outbuf;
		char *pspace = NULL;

		// init out buffer to empty string
		*t = '\0';
		
		while (*f)
		{
			// copy a char, and make sure the dest string is always null terminated
			*t = *f;
			*(t+1) = '\0';
			
			if ( *t == ' ' )
			{
				// if we see a space, remember it
				pspace = t;
			}
			else if ( *t == '\n' )
			{
				// if we see a newline, reset the space ptr
				// and resel the current line ptr.
				pspace = NULL;
				curline = t+1;
			}
			
			if ( (fl_width( curline ) > tw) && (pspace!=NULL) )
			{
				// if the width of the current line exceeds the window size,
				// and we have a previous space, then wrap.
				*pspace = '\n';
				curline = pspace + 1;
				pspace = NULL;
			}

			t++;
			f++;
		}

		value( outbuf );
		delete inbuf;
		delete outbuf;
		_wrapped = true;
	}

	Fl_Output::draw();
}


// -----------------------------------------------------------------------------
// create the message common dialog

Fl_Window* makeCommonDialog( const string& msg, const string& title,
							 const string buttons[], const int results[] )
{
	Fl_Window* w;
	{
		Fl_Window* o = new Fl_Window(343, 156, title.c_str());
		w = o;
		o->box(FL_EMBOSSED_BOX);
		{
			Fl_Return_Button* o = new Fl_Return_Button(250, 125, 85, 25, buttons[0].c_str());
			o->labelsize(12);
			o->callback( modalCallback, results[0] );
		}
		if ( !buttons[1].empty() )
		{
			Fl_Button* o = new Fl_Button(160, 125, 85, 25, buttons[1].c_str());
			o->labelsize(12);
			o->callback( modalCallback, results[1] );
		}
		if ( !buttons[2].empty() )
		{
			Fl_Button* o = new Fl_Button(70, 125, 85, 25, buttons[2].c_str());
			o->labelsize(12);
			o->callback( modalCallback, results[2] );
		}
		{
			WrapOutput* o = new WrapOutput(5, 5, 330,115);
			o->type(4);
			o->color(FL_LIGHT1);
			o->labelsize(10);
			o->textsize(10);
			o->align(FL_ALIGN_LEFT + FL_ALIGN_WRAP);
			Fl_Group::current()->resizable(o);
			o->unwrappedValue( msg );
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the dialog (call in the context of the UI thread)

void fltkPostDialog( void )
{
	string buttons[3];
	int results[3];
	string label(" Note" );

	// get the mesage icon, and set it at appropriate:
	switch (sDialogFlags & Errors::kAlertMask)
	{
	case Errors::kNoteAlert:
		label = "Note";
		break;
	case Errors::kCautionAlert:
		label = "Caution";
		break;
	case Errors::kErrorAlert:
		label = "Error";
		break;
	}

	if (sDialogFlags & Errors::kOKCancelMask)
	{
		if ((sDialogFlags & Errors::kOKCancelMask) == Errors::kOK)
		{
			buttons[0] = Platform::GetString( kStr_OK );
			results[0] = Errors::kOK;
		}
		else if ((sDialogFlags & Errors::kOKCancelMask) == Errors::kCancel)
		{
			buttons[0] = Platform::GetString( kStr_Cancel );
			results[0] = Errors::kCancel;
		}
		else
		{
			buttons[0] = Platform::GetString( kStr_OK );
			results[0] = Errors::kOK;
			buttons[1] = Platform::GetString( kStr_Cancel );
			results[1] = Errors::kCancel;
		}
	}
	else if (sDialogFlags & Errors::kYesNoMask)
	{
		if ((sDialogFlags & Errors::kYesNoMask) == Errors::kYes)
		{
			buttons[0] = Platform::GetString( kStr_Yes );
			results[0] = Errors::kYes;
		}
		else if ((sDialogFlags & Errors::kYesNoMask) == Errors::kNo)
		{
			buttons[0] = Platform::GetString( kStr_No );
			results[0] = Errors::kNo;
		}
		else
		{
			buttons[0] = Platform::GetString( kStr_Yes );
			buttons[1] = Platform::GetString( kStr_No );
			results[0] = Errors::kYes;
			results[1] = Errors::kNo;
		}
	}
	else if (sDialogFlags & Errors::kContinueDebugResetMask)
	{
		int idx = 0;

		if (sDialogFlags & Errors::kContinue)
		{
			buttons[idx] = Platform::GetString( kStr_Continue );
			results[idx++] = Errors::kContinue;
		}

		if (sDialogFlags & Errors::kDebug)
		{
			buttons[idx] = Platform::GetString( kStr_Debug );
			results[idx++] = Errors::kDebug;
		}

		if (sDialogFlags & Errors::kReset)
		{
			buttons[idx] = Platform::GetString( kStr_Reset );
			results[idx++] = Errors::kReset;
		}
	}

	// post dialog:
	Fl_Window *dlg = makeCommonDialog( string(sDialogMsg), label, buttons, results );
	sDialogResult = postModalDialog( dlg );
	delete dlg;

	sDialogDone.post();
}


// -----------------------------------------------------------------------------
// tell the UI thread to post this dialog

int fltkDialog(const char* title, const char* msg, int flags)
{
	// remember the dialog parameters:
	sDialogFlags = flags;
	sDialogMsg = strdup( msg );
	sDialogTitle = strdup( title );
	
	if ( currentlyUIThread() )
	{
		fltkPostDialog();
	}
	else
	{
		// signal the UI thread to put up the given dialog:

		// acquire the dialog mutex. This will insure that only one thread is posting
		// a dialog request to the UI thread.
		sDialogMutex.lock();

		// Tell the UI thread that there is a dialog waiting to be posted:
		fltkExecuteUI( fltkPostDialog );

		// wait for the result
		sDialogDone.wait();

		// release the dialog mutex
		sDialogMutex.unlock();
	}

	// Free the string memory:
	if ( sDialogMsg )
	{
		free( sDialogMsg );
		sDialogMsg = NULL;
	}

	if ( sDialogTitle )
	{
		free( sDialogTitle );
		sDialogTitle = NULL;
	}

	return sDialogResult;  
}


// -----------------------------------------------------------------------------
// URL handler callback. Note that for this to work, netscape must currently
// be running.
//!TODO: work with other browsers, work even if netscape is note running

static void openURL( Fl_Widget* box, void *)
{
	char buffer[PATH_MAX];
	char url[PATH_MAX];

	// get the label, trim the opening '<' and closing '>'
	strcpy( url, &(box->label()[1]) );
	url[ strlen(url) -1 ] = '\0';
	
#ifdef __QNXNTO__
	sprintf( buffer, "voyager -u %s &", url);
#else
	sprintf( buffer, "netscape -remote 'openURL(%s)' || netscape '%s' &", url, url );
#endif

	system( buffer );
}


// -----------------------------------------------------------------------------
// construct the About Box window

Fl_Window* makeAboutWindow()
{
	Fl_Window* w;
	{
		Fl_Window* o = new Fl_Window(362, 231, "About PalmOS(tm) Emulator");
		w = o;
		o->box(FL_EMBOSSED_BOX);
		{
			Fl_Tabs* o = new Fl_Tabs(70, 10, 285, 180);
			{
				Fl_Group* o = new Fl_Group(95, 35, 260, 155, "Palm");
				o->labelsize(10);
				{
					Fl_Box* o = new Fl_Box(100, 45, 240, 20, "Palm OS Emulator Version 3.0a3");
					o->labelfont(FL_HELVETICA_BOLD);
					o->labelsize(12);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 70, 240, 15, "Copyright (c) 1998-1999 Palm Computing, Inc.,");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 85, 240, 15, "3Com Corporation or its subsidiaries");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 105, 115, 15, "<http://www.palm.com>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 120, 140, 15, "<mailto:devsupp@palm.com>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 155, 230, 15, "<http://www.palm.com/devzone/pose/seed.html>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 140, 165, 15, "Download the latest version from:");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				o->end();
			}
			{
				Fl_Group* o = new Fl_Group(95, 35, 260, 155, "Windows");
				o->labelsize(12);
				o->hide();
				{
					Fl_Box* o = new Fl_Box(100, 55, 240, 15, "Original Windows Copilot by Greg Hewgill");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 70, 240, 15, "Portions copyright (c) 1996-1997");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 100, 125, 15, "<http://www.hewgill.com>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 115, 130, 15, "<mailto:greg@hewgill.com>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				o->end();
			}
			{
				Fl_Group* o = new Fl_Group(95, 35, 260, 155, "Macintosh");
				o->labelsize(12);
				o->hide();
				{
					Fl_Box* o = new Fl_Box(100, 55, 240, 15, "Original Mac port by Craig Schofield");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 70, 240, 15, "Portions copyright (c) 1995-1997");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 100, 220, 15, "<http://members.aol.com/illumesoft/illume.html>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 115, 220, 15, "<mailto:ILLUMESoft@aol.com>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				o->end();
			}
			{
				Fl_Group* o = new Fl_Group(95, 35, 260, 155, "UAE");
				o->labelsize(12);
				o->hide();
				{
					Fl_Box* o = new Fl_Box(100, 55, 240, 15, "UAE Amiga Emulator by Bernd Schmidt");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Box* o = new Fl_Box(100, 70, 240, 15, "Portions copyright (c) 1995-1997");
					o->labelsize(10);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 100, 220, 15, "<http://www.freiburg.linux.de/~uae/>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				{
					Fl_Button* o = new Fl_Button(100, 115, 225, 15, "<mailto:crux@pool.informatik.rwth-aachen.de>");
					o->box(FL_NO_BOX);
					o->selection_color(FL_RED);
					o->labelsize(10);
					o->labelcolor(FL_BLUE);
					o->callback((Fl_Callback*)openURL);
					o->align(FL_ALIGN_LEFT + FL_ALIGN_INSIDE);
				}
				o->end();
			}
			o->end();
		}
#ifdef __QNXNTO__
		{
			Fl_Group* o = new Fl_Group(95, 35, 260, 155, "Neutrino");
			o->labelsize(12);
			o->hide();
			{
				Fl_Box* o = new Fl_Box(100, 55, 240, 15, "QNX/Neutrino Port Beta 1.0 by Nexware Corp.");
				o->labelsize(10);
				o->align(20);
			}
			{
				Fl_Box* o = new Fl_Box(100, 70, 240, 15, "Portions copyright (c) 1995-1997");
				o->labelsize(10);
				o->align(20);
			}
			{
				Fl_Button* o = new Fl_Button(100, 100, 220, 15, "<http://www.nexwarecorp.com/>");
				o->box(FL_NO_BOX);
				o->selection_color(1);
				o->labelsize(10);
				o->labelcolor(4);
				o->callback((Fl_Callback*)openURL);
				o->align(20);
			}
			{
				Fl_Button* o = new Fl_Button(100, 115, 225, 15, "<mailto:Nexware@Nexwarecorp.com>");
				o->box(FL_NO_BOX);
				o->selection_color(1);
				o->labelsize(10);
				o->labelcolor(4);
				o->callback((Fl_Callback*)openURL);
				o->align(20);
			}
			o->end();
		}
#endif
		{
			Fl_Box* o = new Fl_Box(10, 15, 50, 60, "");
			Fl_Pixmap* p = new Fl_Pixmap( poser_xpm );
			p->label( o );
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(280, 197, 75, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the about box window

void showAboutBox()
{
	Fl_Window *aboutWin = makeAboutWindow();
	postModalDialog( aboutWin );
	delete aboutWin;
}


// -----------------------------------------------------------------------------
// construct the Gremlins dialog
Fl_Window* makeGremlinsWindow()
{
	Preference<HordeInfo> pref (kPrefKeyHordeInfo);
	HordeInfo info = *pref;

	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(447, 323, "New Gremlin Horde");
		w = o;
		o->box(FL_ENGRAVED_BOX);
		o->color(FL_LIGHT1);
		o->labelcolor(FL_LIGHT1);
		o->align(FL_ALIGN_TOP + FL_ALIGN_WRAP);
		{
			char buffer[16];
			Fl_Input* o = new Fl_Input(165, 20, 60, 25, "Gremlin Start Number (0-999):");
			::PrvSetWidgetID (o, item_ngGremlinStart);
			o->labelsize(10);
			o->textsize(10);
			o->type(FL_INT_INPUT);

			sprintf (buffer, "%ld", info.fStartNumber);
			o->value (buffer);
		}
		{
			char buffer[16];
			Fl_Input* o = new Fl_Input(165, 50, 60, 25, "Gremlin End Number (0-999):");
			::PrvSetWidgetID (o, item_ngGremlinEnd);
			o->labelsize(10);
			o->textsize(10);
			o->type(FL_INT_INPUT);

			sprintf (buffer, "%ld", info.fStopNumber);
			o->value (buffer);
		}
		{
			char buffer[16];
			Fl_Input* o = new Fl_Input(165, 80, 60, 25, "Switching Depth (-1 = none):");
			::PrvSetWidgetID (o, item_ngSwitchDepth);
			o->labelsize(10);
			o->textsize(10);
			o->type(FL_INT_INPUT);

			sprintf (buffer, "%ld", info.fSwitchDepth);
			o->value (buffer);
		}
		{
			char buffer[16];
			Fl_Input* o = new Fl_Input(165, 110, 60, 25, "Maximum Events (-1 = forever):");
			::PrvSetWidgetID (o, item_ngMaxDepth);
			o->labelsize(10);
			o->textsize(10);
			o->type(FL_INT_INPUT);

			sprintf (buffer, "%ld", info.fMaxDepth);
			o->value (buffer);
		}
		{
			char buffer[16];
			Fl_Input* o = new Fl_Input(165, 140, 60, 25, "Snapshot Frequency (0 = never):");
			::PrvSetWidgetID (o, item_ngSnapshotFrequency);
			o->labelsize(10);
			o->textsize(10);
			o->type(FL_INT_INPUT);

			sprintf (buffer, "%ld", info.fSaveFrequency);
			o->value (buffer);
		}
		{
			Fl_Browser* o = new Fl_Browser(235, 20, 200, 256, "Choose application to launch:");
			::PrvSetWidgetID (o, item_ngApplicationList);
			o->type(FL_MULTI_BROWSER);
			o->labelsize(10);
			o->textsize(10);
			o->align(FL_ALIGN_TOP);

			bool selected = false;
			DatabaseInfoList appList;
			::GetDatabases (appList, kApplicationsOnly);
			for (unsigned int ii = 0; ii < appList.size (); ++ii)
			{
				o->add (appList[ii].name);

				// If the item we just added is in our "selected" list,
				// then select it.

				DatabaseInfoList::iterator iter = info.fAppList.begin();
				while (iter != info.fAppList.end())
				{
					if (appList[ii] == *iter++)
					{
						o->select (ii + 1);
						selected = true;
					}
				}
			}

			if (!selected)
				o->select (1);
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(365, 285, 70, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		{
			Fl_Button* o = new Fl_Button(290, 285, 70, 25, "Cancel");
			o->callback( modalCallback, 0 );
			o->labelsize(12);
		}
		o->set_modal();
		o->end();
	}
	return w;
}

// -----------------------------------------------------------------------------
// post the gremlins window

bool showNewGremlinWindow()
{
	Fl_Window* gremlinWin = makeGremlinsWindow();
	bool result = false;

	if (postModalDialog (gremlinWin) == 1)
	{
		// scan the values out of the input widgets:
		Preference<HordeInfo> pref (kPrefKeyHordeInfo);
		HordeInfo info = *pref;
		
		Fl_Input* i = (Fl_Input*) ::PrvFindWidgetByID (item_ngGremlinStart);
		sscanf (i->value(), "%ld", &info.fStartNumber);
		
		i = (Fl_Input*) ::PrvFindWidgetByID (item_ngGremlinEnd);
		sscanf (i->value(), "%ld", &info.fStopNumber);
		
		i = (Fl_Input*) ::PrvFindWidgetByID (item_ngSwitchDepth);
		sscanf (i->value(), "%ld", &info.fSwitchDepth);
		
		i = (Fl_Input*) ::PrvFindWidgetByID (item_ngMaxDepth);
		sscanf (i->value(), "%ld", &info.fMaxDepth);

		i = (Fl_Input*) ::PrvFindWidgetByID (item_ngSnapshotFrequency);
		sscanf (i->value(), "%ld", &info.fSaveFrequency);

		Fl_Browser* br = (Fl_Browser*) ::PrvFindWidgetByID (item_ngApplicationList);
		info.fAppList.clear ();

		DatabaseInfoList appList;
		::GetDatabases (appList, kApplicationsOnly);
		for (int ii = 1; ii <= br->size(); ++ii)
		{
			if (br->selected (ii))
			{
				info.fAppList.push_back (appList [ii - 1]);
			}
		}

		pref = info;
		
		result = true;
	}

	delete gremlinWin;
	return result;
}


// -----------------------------------------------------------------------------
// new emulator dialog

static Configuration gCfg;
static SkinNameList gSkinChoices;

static void setDeviceChoices (Fl_Choice* o, DeviceType dt)
{
	o->clear();

	DeviceTextList devices;
	::GetDeviceTextList (devices);

	bool selected = false;
	int index = 0;
	DeviceTextList::iterator iter = devices.begin();
	while (iter != devices.end())
	{
		// Add the menu items this way in order to get meta-characters
		// inserted (in this case, slashes, which would otherwise add
		// submenus).

		o->add ("Foo");
		o->replace(index, iter->second.c_str());

		if (dt == iter->first)
		{
			o->value (index);
			selected = true;
		}

		++iter;
		++index;
	}

	if (!selected)
		o->value(0);
}


static void setSkinChoices (Fl_Choice* o, DeviceType dt)
{
	o->clear();

	SkinNameList skins;
	::SkinGetSkinNames (dt, skins);

	bool selected = false;
	int index = 0;
	SkinNameList::iterator iter = skins.begin();
	while (iter != skins.end())
	{
		// Add the menu items this way in order to get meta-characters
		// inserted (in this case, slashes, which would otherwise add
		// submenus).

		o->add ("Foo");
		o->replace (index, iter->c_str ());

		if (gSkinChoices[dt] == *iter)
		{
			o->value (index);
			selected = true;
		}

		++iter;
		++index;
	}

	if (!selected)
		o->value(0);
}


static void setSizeChoices (Fl_Choice* o)
{
	o->clear();

	MemoryTextList sizes;
	::GetMemoryTextList (sizes);

	bool selected = false;
	int index = 0;
	MemoryTextList::iterator iter = sizes.begin();
	while (iter != sizes.end())
	{
		// Add the menu items this way in order to get meta-characters
		// inserted (in this case, slashes, which would otherwise add
		// submenus).

		o->add ("Foo");
		o->replace (index, iter->second.c_str ());

		if (gCfg.fRAMSize == iter->first)
		{
			o->value (index);
			selected = true;
		}

		++iter;
		++index;
	}

	if (!selected)
		o->value(0);
}


static void deviceChosenCallback(Fl_Widget* o)
{
	Fl_Choice* menu = (Fl_Choice*) o;

	// Get the currently selected device.

	DeviceTextList devices;
	::GetDeviceTextList(devices);
	gCfg.fDeviceType = devices[menu->value()].first;

	// Rebuild the skin menu.

	Fl_Choice* skin_menu = (Fl_Choice*) ::PrvFindWidgetByID (item_cfgSkinMenu);
	::setSkinChoices (skin_menu, gCfg.fDeviceType);
}


static void skinChosenCallback(Fl_Widget* o)
{
	Fl_Choice* menu = (Fl_Choice*) o;

	// Get the currently selected skin.

	SkinNameList skins;
	::SkinGetSkinNames (gCfg.fDeviceType, skins);
	gSkinChoices[gCfg.fDeviceType] = skins[menu->value()];
}


static void sizeChosenCallback(Fl_Widget* o)
{
	Fl_Choice* menu = (Fl_Choice*) o;

	// Get the currently selected ram size.

	MemoryTextList sizes;
	::GetMemoryTextList (sizes);
	gCfg.fRAMSize = sizes[menu->value()].first;
}


void romChosenCallback(Fl_Widget* o)
{
	Fl_Button* btn = (Fl_Button*) o;

	// ask the user to find a rom file:
	string startingPath = gCfg.fROMFile.GetFilePath ();
	char* fname = fl_file_chooser( "Choose a ROM image",
								   "*.{rom,ROM}",
								   startingPath.c_str () );

	if (fname != NULL)
	{
		gCfg.fROMFile = FileReference (fname);

		// Get the name of the file and store it in a *static*
		// string variable. The label() method copies the pointer,
		// not the string itself, so the memory for the string
		// needs to be persistant.

		static string fileName;
		fileName = gCfg.fROMFile.GetFileName();
		btn->label(fileName.c_str());
	}
}


Fl_Window* makeNewEmulatorWindow (const Configuration& cfg)
{
	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(246, 160, "New Configuration");
		w = o;
		o->box(FL_ENGRAVED_BOX);
		{
			Fl_Choice* o = new Fl_Choice(60, 10, 175, 20, "Device:");
			::PrvSetWidgetID (o, item_cfgDeviceMenu);
			::setDeviceChoices (o, cfg.fDeviceType);
			o->labelsize(10);
			o->textsize(10);
			o->callback(deviceChosenCallback);
		}
		{
			Fl_Choice* o = new Fl_Choice(60, 35, 115, 20, "Skin:");
			::PrvSetWidgetID (o, item_cfgSkinMenu);
			::setSkinChoices (o, cfg.fDeviceType);
			o->labelsize(10);
			o->textsize(10);
			o->callback(skinChosenCallback);
		}
		{
			Fl_Choice* o = new Fl_Choice(60, 60, 115, 20, "RAM Size:");
			::PrvSetWidgetID (o, item_cfgRAMSizeMenu);
			::setSizeChoices (o);
			o->labelsize(10);
			o->textsize(10);
			o->callback(sizeChosenCallback);
		}
		{
			Fl_Button* o = new Fl_Button(60, 85, 175, 20, cfg.fROMFile.GetFileName().c_str() );
			::PrvSetWidgetID (o, item_cfgBrowseButton);
			o->labelsize(10);
			o->callback(romChosenCallback);
		}
		{
			Fl_Box* o = new Fl_Box(10, 85, 50, 25, "ROM File:");
			o->labelsize(10);
		}
		{
			Fl_Button* o = new Fl_Button(100, 125, 65, 25, "Cancel");
			o->labelsize(12);
			o->callback( modalCallback, 0 );
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(170, 125, 65, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}

		o->set_modal();
		o->end();
	}

	return w;
}


// -----------------------------------------------------------------------------
// post the new emulator window

bool showNewEmulatorWindow( Configuration& cfg )
{
	gCfg = cfg; // Save these settings so that they can be altered in our callbacks.

	Preference<SkinNameList> prefSkins (kPrefKeySkins);
	gSkinChoices = *prefSkins;

	Fl_Window *newWin = makeNewEmulatorWindow( cfg );
	bool result = false;

	if (postModalDialog(newWin) == 1)
	{
		cfg = gCfg;
		prefSkins = gSkinChoices;

		result = true;
	}

	delete newWin;
	return result;
}


// -----------------------------------------------------------------------------
// make the "get a database name" dialog

Fl_Window* makeGetDatabaseNameWindow()
{
	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(220, 323, "Export Database");
		w = o;
		o->box(FL_ENGRAVED_BOX);
		o->color(FL_LIGHT1);
		o->labelcolor(FL_LIGHT1);
		o->align(FL_ALIGN_TOP + FL_ALIGN_WRAP);
		{
			Fl_Browser* o = new Fl_Browser(10, 20, 200, 256, "Choose database to export:");
			::PrvSetWidgetID (o, item_exDatabaseList);
			o->type(FL_MULTI_BROWSER);
			o->labelsize(10);
			o->textsize(10);
			o->align(FL_ALIGN_TOP);

			DatabaseInfoList appList;
			::GetDatabases (appList, kAllDatabases);
			for (unsigned int ii = 0; ii < appList.size (); ++ii)
			{
				o->add (appList[ii].name);
			}

			o->select (1);
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(115, 285, 95, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		{
			Fl_Button* o = new Fl_Button(10, 285, 95, 25, "Cancel");
			o->callback( modalCallback, 0 );
			o->labelsize(12);
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// show the "get a database name" dialog

bool getDatabaseName (DatabaseInfo& db)
{
	bool result = false;
	Fl_Window* dlg = makeGetDatabaseNameWindow ();

	if (postModalDialog(dlg) == 1)
	{
		DatabaseInfoList dbList;
		::GetDatabases (dbList, kAllDatabases);
		Fl_Browser* br = (Fl_Browser*) ::PrvFindWidgetByID (item_exDatabaseList);
		int sel = br->value();
		if (sel)
		{
			db = dbList [sel - 1];
			result = true;
		}
	}

	delete dlg;
	return result;
}


// -----------------------------------------------------------------------------
// create the preferences dialog

Fl_Window* makePrefsWindow( void )
{
	Fl_Window* w;
	::PrvClearWidgetIDs();

	{
		Fl_Window* o = new Fl_Window(244, 293, "Preferences");
		w = o;
		{
			Fl_Box* o = new Fl_Box(10, 25, 225, 65, "Communications");
			o->box(FL_THIN_DOWN_FRAME);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(12);
			o->align(FL_ALIGN_TOP_LEFT);
		}
		{
			Fl_Box* o = new Fl_Box(10, 115, 225, 75, "Closing / Quitting");
			o->box(FL_THIN_DOWN_FRAME);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(12);
			o->align(FL_ALIGN_TOP_LEFT);
		}
		{
			Fl_Box* o = new Fl_Box(10, 215, 225, 30, "HotSync User Name");
			o->box(FL_THIN_DOWN_FRAME);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(12);
			o->align(FL_ALIGN_TOP_LEFT);
		}
		{
			Fl_Input* o = new Fl_Input(70, 35, 160, 20, "Serial Port:");
			::PrvSetWidgetID (o, item_prefPort);
			o->value( Preference<string>(kPrefKeyCommPort)->c_str() );
			o->labelsize(10);
			o->textsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 60, 190, 20, "Redirect NetLIb calls to host TCP/IP");
			::PrvSetWidgetID (o, item_prefNetLib);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->value( *(Preference<bool>(kPrefKeyRedirectNetLib)) ? 1 : 0 );
		}
		{
			Preference<CloseActionType> pref (kPrefKeyCloseAction);

			Fl_Group* o = new Fl_Group(15, 115, 120, 70);
			{
				Fl_Round_Button* o = new Fl_Round_Button(15, 120, 120, 25, "Always save session");
				::PrvSetWidgetID (o, item_prefSaveAlways);
				o->type(FL_RADIO_BUTTON);
				o->down_box(FL_ROUND_DOWN_BOX);
				o->labelsize(10);
				if ( *pref == kSaveAlways )
					o->value(1);
			}
			{
				Fl_Round_Button* o = new Fl_Round_Button(15, 140, 115, 25, "Ask to save session");
				::PrvSetWidgetID (o, item_prefSaveAsk);
				o->type(FL_RADIO_BUTTON);
				o->down_box(FL_ROUND_DOWN_BOX);
				o->labelsize(10);
				if ( *pref == kSaveAsk )
					o->value(1);
			}
			{
				Fl_Round_Button* o = new Fl_Round_Button(15, 160, 120, 25, "Never save session");
				::PrvSetWidgetID (o, item_prefSaveNever);
				o->type(FL_RADIO_BUTTON);
				o->down_box(FL_ROUND_DOWN_BOX);
				o->labelsize(10);
				if ( *pref == kSaveNever )
					o->value(1);
			}
			o->end();
		}
		{
			Fl_Input* o = new Fl_Input(15, 220, 215, 20);
			::PrvSetWidgetID (o, item_prefUserName);
			o->labelsize(10);
			o->textsize(10);
			o->value( Preference<string>(kPrefKeyUserName)->c_str() );
		}
		{
			Fl_Button* o = new Fl_Button(100, 260, 65, 25, "Cancel");
			o->labelsize(12);
			o->callback( modalCallback, 0 );
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(170, 260, 65, 25, "OK");
			o->labelsize(12);
			w->hotspot(o);
			o->callback( modalCallback, 1 );
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the prefs window.

bool editPreferences( void )
{
	Fl_Window *prefsWin = makePrefsWindow();
	bool result = false;
	
	if ( postModalDialog( prefsWin ) == 1 )
	{
		Fl_Input* i = (Fl_Input*) ::PrvFindWidgetByID (item_prefPort);
		Preference<string> commPortPref(kPrefKeyCommPort);
		commPortPref = string (i->value ());

		SET_BOOL_FROM_BUTTON( RedirectNetLib, item_prefNetLib);

		Preference<CloseActionType> closeActionPref (kPrefKeyCloseAction);
		Fl_Button* b = (Fl_Button*) ::PrvFindWidgetByID (item_prefSaveAlways);
		if ( b->value() == 1 )
		{
			closeActionPref = kSaveAlways;
		}
		else
		{
			b = (Fl_Button*) ::PrvFindWidgetByID (item_prefSaveAsk);
			closeActionPref = b->value() == 1 ? kSaveAsk : kSaveNever;
		}

		i = (Fl_Input*) ::PrvFindWidgetByID (item_prefUserName);
		Preference<string> userNamePref (kPrefKeyUserName);
		userNamePref = string (i->value());

		// changes
		result = true;
	}

	delete prefsWin;
	return result;
}


// -----------------------------------------------------------------------------
// create the debugging preferences dialog

Fl_Window* makeDebugWindow( void )
{
	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(327, 162, "Debugging Options");
		w = o;
		o->box(FL_EMBOSSED_BOX);
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 10, 120, 20, "Low-Memory Access");
			::PrvSetWidgetID (o, item_dbgLowMem);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportLowMemoryAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 25, 125, 20, "System Globals Access");
			::PrvSetWidgetID (o, item_dbgGlobals);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportSystemGlobalAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 40, 90, 20, "Screen Access");
			::PrvSetWidgetID (o, item_dbgScreen);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportScreenAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 55, 140, 20, "Hardware Register Access");
			::PrvSetWidgetID (o, item_dbgRegisters);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportHardwareRegisterAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 70, 130, 20, "MemMgr Data Structure");
			::PrvSetWidgetID (o, item_dbgMemMgrData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportMemMgrDataAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 85, 120, 20, "Storage Heap Access");
			::PrvSetWidgetID (o, item_dbgStorageHeap);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportStorageHeapAccess) ? 1 : 0 );
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(15, 100, 120, 20, "MemMgr Semaphore");
			::PrvSetWidgetID (o, item_dbgMemMgrSemaphore);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportMemMgrSemaphore) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 10, 105, 20, "Low Stack Access");
			::PrvSetWidgetID (o, item_dbgLowStack);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportLowStackAccess) ? 1 : 0 );
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 25, 110, 20, "Free Chunk Access");
			::PrvSetWidgetID (o, item_dbgFreeChunk);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportFreeChunkAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 40, 130, 20, "Unlocked Chunk Access");
			::PrvSetWidgetID (o, item_dbgUnlockedChunk);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportUnlockedChunkAccess) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 55, 135, 20, "Unitialized Stack Access");
			::PrvSetWidgetID (o, item_dbgUninitStack);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportUninitializedStackAccess) ? 1 : 0 );
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 70, 145, 20, "Uninitialized Chunk Access");
			::PrvSetWidgetID (o, item_dbgUninitChunk);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportUninitializedChunkAccess) ? 1 : 0 );
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(175, 85, 130, 20, "Stack Almost Overflow");
			::PrvSetWidgetID (o, item_dbgStackAlmostOverflow);
			o->down_box(FL_THIN_DOWN_BOX);
			o->value( GET_BOOL_PREF(ReportStackAlmostOverflow) ? 1 : 0 );
			o->labelsize(10);
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(245, 130, 70, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		{
			Fl_Button* o = new Fl_Button(170, 130, 70, 25, "Cancel");
			o->labelsize(12);
			o->callback( modalCallback, 0);
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the debug prefs window.

bool editDebugging()
{
	Fl_Window *debugWin = makeDebugWindow();
	bool result = false;

	if ( postModalDialog( debugWin ) == 1 )
	{
		SET_BOOL_FROM_BUTTON( ReportLowMemoryAccess, item_dbgLowMem );
		SET_BOOL_FROM_BUTTON( ReportSystemGlobalAccess, item_dbgGlobals );
		SET_BOOL_FROM_BUTTON( ReportScreenAccess, item_dbgScreen );
		SET_BOOL_FROM_BUTTON( ReportHardwareRegisterAccess, item_dbgRegisters );
		SET_BOOL_FROM_BUTTON( ReportMemMgrDataAccess, item_dbgMemMgrData );
		SET_BOOL_FROM_BUTTON( ReportStorageHeapAccess, item_dbgStorageHeap );
		SET_BOOL_FROM_BUTTON( ReportMemMgrSemaphore, item_dbgMemMgrSemaphore );
		SET_BOOL_FROM_BUTTON( ReportLowStackAccess, item_dbgLowStack );
		SET_BOOL_FROM_BUTTON( ReportFreeChunkAccess, item_dbgFreeChunk );
		SET_BOOL_FROM_BUTTON( ReportUnlockedChunkAccess, item_dbgUnlockedChunk );
		SET_BOOL_FROM_BUTTON( ReportUninitializedStackAccess, item_dbgUninitStack );
		SET_BOOL_FROM_BUTTON( ReportUninitializedChunkAccess, item_dbgUninitChunk );
		SET_BOOL_FROM_BUTTON( ReportStackAlmostOverflow, item_dbgStackAlmostOverflow );

		// changes
		result = true;
	}

	delete debugWin;
	return result;
}


// -----------------------------------------------------------------------------
// helper data functions for getting & setting prefs & checkboxes

//! TODO: get rid of these globals!
static int curPanel;
static Fl_Window *curLogWin;

void updatePreference( int btnId, const char* key, int bitMask )
{
	Fl_Check_Button *btn = (Fl_Check_Button*) ::PrvFindWidgetByID (btnId);

	Preference<bool> pref (key);

	if ( btn->value() != 0 )
		pref = *pref | bitMask;
	else
		pref = *pref & ~bitMask;
}

void updateCheckButton( int btnId, const char* key, int bitMask )
{
	Fl_Check_Button *btn = (Fl_Check_Button*) ::PrvFindWidgetByID( btnId );

	Preference<bool> pref (key);
	btn->value( ((*pref & bitMask) != 0) ? 1 : 0 );
}

void updateCheckButtons( Fl_Window* w, int panel )
{
	int bitMask = (panel == 1) ? kNormalLogging : kGremlinLogging;

	updateCheckButton( item_logErrors, kPrefKeyLogErrorMessages, bitMask );
	updateCheckButton( item_logWarnings, kPrefKeyLogWarningMessages, bitMask );
	updateCheckButton( item_logGremlins, kPrefKeyLogGremlins, bitMask );
	updateCheckButton( item_logOpcodes, kPrefKeyLogCPUOpcodes, bitMask );
	updateCheckButton( item_logEnqueuedEvents, kPrefKeyLogEnqueuedEvents, bitMask );
	updateCheckButton( item_logDequeuedEvents, kPrefKeyLogDequeuedEvents, bitMask );
	updateCheckButton( item_logSystemCalls, kPrefKeyLogSystemCalls, bitMask );
	updateCheckButton( item_logApplicationCalls, kPrefKeyLogApplicationCalls, bitMask );
	updateCheckButton( item_logSerial, kPrefKeyLogSerial, bitMask );
	updateCheckButton( item_logNetLib, kPrefKeyLogNetLib, bitMask );
	updateCheckButton( item_logExgMgr, kPrefKeyLogExgMgr, bitMask );
	updateCheckButton( item_logHighLevelDebugger, kPrefKeyLogHLDebugger, bitMask );
	updateCheckButton( item_logLowLevelDebugger, kPrefKeyLogLLDebugger, bitMask );
	updateCheckButton( item_logSerialData, kPrefKeyLogSerialData, bitMask );
	updateCheckButton( item_logNetLibData, kPrefKeyLogNetLibData, bitMask );
	updateCheckButton( item_logExgMgrData, kPrefKeyLogExgMgrData, bitMask );
	updateCheckButton( item_logHighLevelDebuggerData, kPrefKeyLogHLDebuggerData, bitMask );
	updateCheckButton( item_logLowLevelDebuggerData, kPrefKeyLogLLDebuggerData, bitMask );
}

void updatePreferences( Fl_Window* w, int panel )
{
	int bitMask = (panel == 1) ? kNormalLogging : kGremlinLogging;

	updatePreference( item_logErrors, kPrefKeyLogErrorMessages, bitMask );
	updatePreference( item_logWarnings, kPrefKeyLogWarningMessages, bitMask );
	updatePreference( item_logGremlins, kPrefKeyLogGremlins, bitMask );
	updatePreference( item_logOpcodes, kPrefKeyLogCPUOpcodes, bitMask );
	updatePreference( item_logEnqueuedEvents, kPrefKeyLogEnqueuedEvents, bitMask );
	updatePreference( item_logDequeuedEvents, kPrefKeyLogDequeuedEvents, bitMask );
	updatePreference( item_logSystemCalls, kPrefKeyLogSystemCalls, bitMask );
	updatePreference( item_logApplicationCalls, kPrefKeyLogApplicationCalls, bitMask );
	updatePreference( item_logSerial,  kPrefKeyLogSerial, bitMask );
	updatePreference( item_logNetLib, kPrefKeyLogNetLib, bitMask );
	updatePreference( item_logExgMgr, kPrefKeyLogExgMgr, bitMask );
	updatePreference( item_logHighLevelDebugger, kPrefKeyLogHLDebugger, bitMask );
	updatePreference( item_logLowLevelDebugger, kPrefKeyLogLLDebugger, bitMask );
	updatePreference( item_logSerialData, kPrefKeyLogSerialData, bitMask );
	updatePreference( item_logNetLibData, kPrefKeyLogNetLibData, bitMask );
	updatePreference( item_logExgMgrData, kPrefKeyLogExgMgrData, bitMask );
	updatePreference( item_logHighLevelDebuggerData, kPrefKeyLogHLDebuggerData, bitMask );
	updatePreference( item_logLowLevelDebuggerData, kPrefKeyLogLLDebuggerData, bitMask );
}

void panelCallback( Fl_Widget* w, long panel )
{
	if ( curPanel == panel )
		return;

	// save current panel:
	updatePreferences( curLogWin, curPanel );

	// switch to new panel
	curPanel = (int) panel;
	updateCheckButtons( curLogWin, curPanel );
}


// -----------------------------------------------------------------------------
// construct the logging preferences window:

Fl_Window* makeLoggingWindow()
{
	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(356, 242, "Logging Options");
		w = o;
		{
			Fl_Group* o = new Fl_Group(220, 5, 130, 25);
			{
				Fl_Round_Button* o = new Fl_Round_Button(225, 5, 55, 25, "Normal");
				o->type(102);
				o->down_box(FL_ROUND_DOWN_BOX);
				o->value(1);
				o->labelsize(10);
				o->callback( panelCallback, 1 );
			}
			{
				Fl_Round_Button* o = new Fl_Round_Button(285, 5, 65, 25, "Gremlins");
				o->type(102);
				o->down_box(FL_ROUND_DOWN_BOX);
				o->labelsize(10);
				o->callback( panelCallback, 2 );
			}
			o->end();
		}
		{
			Fl_Box* o = new Fl_Box(15, 30, 335, 170, "Log:");
			o->box(FL_THIN_DOWN_FRAME);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(12);
			o->align(FL_ALIGN_TOP_LEFT);
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(280, 210, 70, 25, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		{
			Fl_Button* o = new Fl_Button(205, 210, 70, 25, "Cancel");
			o->labelsize(12);
			o->callback( modalCallback, 0 );
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 35, 90, 20, "Error Messages");
			::PrvSetWidgetID (o, item_logErrors);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 50, 105, 20, "Warning Messages");
			::PrvSetWidgetID (o, item_logWarnings);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 65, 100, 20, "Misc Gremlin Info");
			::PrvSetWidgetID (o, item_logGremlins);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 80, 110, 20, "Assembley Opcodes");
			::PrvSetWidgetID (o, item_logOpcodes);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 35, 90, 20, "Posted Events");
			::PrvSetWidgetID (o, item_logEnqueuedEvents);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 50, 105, 20, "Received Events");
			::PrvSetWidgetID (o, item_logDequeuedEvents);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 65, 75, 20, "System Calls");
			::PrvSetWidgetID (o, item_logSystemCalls);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 80, 95, 20, "Application Calls");
			::PrvSetWidgetID (o, item_logApplicationCalls);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 110, 85, 20, "Serial Activity");
			::PrvSetWidgetID (o, item_logSerial);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 125, 90, 20, "NetLib Activity");
			::PrvSetWidgetID (o, item_logNetLib);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 140, 95, 20, "ExgMgr Activity");
			::PrvSetWidgetID (o, item_logExgMgr);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 155, 160, 20, "High-Level Debugger Activity");
			::PrvSetWidgetID (o, item_logHighLevelDebugger);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(20, 170, 160, 20, "Low-Level Debugger Activity");
			::PrvSetWidgetID (o, item_logLowLevelDebugger);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 110, 70, 20, "Serial Data");
			::PrvSetWidgetID (o, item_logSerialData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 125, 75, 20, "NetLib Data");
			::PrvSetWidgetID (o, item_logNetLibData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 140, 95, 20, "ExgMgr Data");
			::PrvSetWidgetID (o, item_logExgMgrData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
			o->deactivate();
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 155, 145, 20, "High-Level Debugger Data");
			::PrvSetWidgetID (o, item_logHighLevelDebuggerData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(195, 170, 145, 20, "Low-Level Debugger Data");
			::PrvSetWidgetID (o, item_logLowLevelDebuggerData);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the logging prefs window.

bool editLogging()
{
	Fl_Window *loggingWin = makeLoggingWindow();

	// set up globals (yuck)
	curLogWin = loggingWin;
	curPanel = 1;

	// get the info into the window the first time:
	updateCheckButtons( curLogWin, 1 );
	
	bool result = false;
	if ( postModalDialog( loggingWin ) == 1 )
	{
		// save current panel into master prefs object
		updatePreferences( curLogWin, curPanel );

		// switch to other panel
		curPanel = (curPanel == 1) ? 2 : 1;
		updateCheckButtons( curLogWin, curPanel );

		// save this panel into master prefs
		updatePreferences( curLogWin, curPanel );
		
		// changes
		result = true;
	}

	curLogWin = NULL;

	delete loggingWin;
	return result;
}


// -----------------------------------------------------------------------------
// construct the skins preferences window:

DeviceType gLastSelectedDevice;

DeviceType PrvGetSelectedDevice(void)
{
	Fl_Browser* b = (Fl_Browser*) ::PrvFindWidgetByID (item_skDevices);
	DeviceType type = kDeviceUnspecified;

	if (b->value() > 0)
	{
		DeviceTextList devices;
		::GetDeviceTextList (devices);

		int index = b->value() - 1;
		type = devices[index].first;
	}

	return type;
}

SkinName PrvGetSelectedSkin(void)
{
	Fl_Browser* b = (Fl_Browser*) ::PrvFindWidgetByID (item_skSkins);
	SkinName name;

	if (b->value() > 0)
	{
		SkinNameList skins;
		::SkinGetSkinNames (gLastSelectedDevice, skins);

		int index = b->value() - 1;
		name = skins[index];
	}

	return name;
}

void PrvSetDeviceList (void)
{
	Fl_Browser* b = (Fl_Browser*) ::PrvFindWidgetByID (item_skDevices);
	Preference<Configuration> cfg(kPrefKeyLastConfiguration);

	DeviceTextList devices;
	::GetDeviceTextList (devices);

	DeviceTextList::iterator iter = devices.begin();
	while (iter != devices.end())
	{
		b->add(iter->second.c_str());

		if (iter->first == cfg->fDeviceType)
			b->value(b->size());

		++iter;
	}

	if (b->value() == 0)
		b->value(1);

	gLastSelectedDevice = ::PrvGetSelectedDevice();
}

void PrvRememberSelectedSkin(void)
{
	SkinName name = ::PrvGetSelectedSkin();
	gSkinChoices[gLastSelectedDevice] = name;
}

void PrvResetSkinList (void)
{
	DeviceType type = ::PrvGetSelectedDevice();

	Fl_Browser* b = (Fl_Browser*) ::PrvFindWidgetByID (item_skSkins);
	b->clear();

	// Get the list of skins for the selected device.

	SkinNameList skins;
	::SkinGetSkinNames (type, skins);

	// Add each skin in the list to the UI element.

	SkinNameList::iterator iter = skins.begin();
	while (iter != skins.end())
	{
		b->add(iter->c_str());

		if (*iter == gSkinChoices[type])
			b->value(b->size());

		++iter;
	}

	if (b->value() == 0)
		b->value(1);
}

void deviceChosenCallback2(Fl_Widget* b)
{
	::PrvRememberSelectedSkin();
	::PrvResetSkinList();
	gLastSelectedDevice = ::PrvGetSelectedDevice();
}

Fl_Window* makeSkinsWindow()
{
	Fl_Window* w;
	::PrvClearWidgetIDs ();

	{
		Fl_Window* o = new Fl_Window(304, 288, "Skins");
		w = o;
		{
			Fl_Browser* o = new Fl_Browser(8, 16, 136, 168, "Devices:");
			::PrvSetWidgetID (o, item_skDevices);
			o->type(FL_HOLD_BROWSER);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(10);
			o->textsize(10);
			o->align(FL_ALIGN_TOP_LEFT);
			o->callback(deviceChosenCallback2);
			::PrvSetDeviceList ();
		}
		{
			Fl_Browser* o = new Fl_Browser(160, 16, 136, 168, "Skins for Device:");
			::PrvSetWidgetID (o, item_skSkins);
			o->type(FL_HOLD_BROWSER);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(10);
			o->textsize(10);
			o->align(FL_ALIGN_TOP_LEFT);
			::PrvResetSkinList ();
		}
		{
			Fl_Box* o = new Fl_Box(8, 200, 288, 48, "Other Options:");
			o->box(FL_THIN_DOWN_FRAME);
			o->labelfont(FL_HELVETICA_BOLD);
			o->labelsize(10);
			o->align(FL_ALIGN_TOP_LEFT);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(16, 208, 112, 16, "Double Scale");
			::PrvSetWidgetID (o, item_skDoubleScale);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);

			Preference<ScaleType> scale(kPrefKeyScale);
			if (*scale == 2)
				o->value(1);
		}
		{
			Fl_Check_Button* o = new Fl_Check_Button(16, 224, 112, 16, "White Background");
			::PrvSetWidgetID (o, item_skWhiteBackground);
			o->down_box(FL_THIN_DOWN_BOX);
			o->labelsize(10);

			Preference<RGBType> bg(kPrefKeyBackgroundColor);
			if (bg.Loaded())
				o->value(1);
		}
		{
			Fl_Return_Button* o = new Fl_Return_Button(218, 256, 80, 20, "OK");
			o->labelsize(12);
			o->callback( modalCallback, 1 );
			w->hotspot(o);
		}
		{
			Fl_Button* o = new Fl_Button(128, 256, 80, 20, "Cancel");
			o->labelsize(12);
			o->callback( modalCallback, 0 );
		}
		o->set_modal();
		o->end();
	}
	return w;
}


// -----------------------------------------------------------------------------
// post the skin prefs window.

bool editSkins()
{
	Preference<SkinNameList> prefSkins(kPrefKeySkins);
	gSkinChoices = *prefSkins;

	Fl_Window* w = makeSkinsWindow();
	bool result = false;

	if (postModalDialog (w) == 1)
	{
		::PrvRememberSelectedSkin();
		prefSkins = gSkinChoices;

		Fl_Check_Button* o = (Fl_Check_Button*) ::PrvFindWidgetByID(item_skDoubleScale);
		Preference<ScaleType> scale(kPrefKeyScale);
		scale = o->value() != 0 ? 2 : 1;

		o = (Fl_Check_Button*) ::PrvFindWidgetByID(item_skWhiteBackground);
		if (o->value())
		{
			RGBType color(0xFF, 0xFF, 0xFF);
			Preference<RGBType> bg(kPrefKeyBackgroundColor);
			bg = color;
		}
		else
		{
			gPrefs->DeletePref (kPrefKeyBackgroundColor);
		}

		result = true;
	}

	delete w;
	return result;
}

