/*
 * ULHAND.C - handlers for Ultra
 *
 * Source Version: 4.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "ultra.h"

static int
 SC_DECLARE(_UL_bc_operate,
         (PFInt basicf, REAL *xa, REAL *ya,
          REAL *xp1, REAL *xp2, REAL *yp1, REAL *yp2,
          int n1, int n2));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_LMT - find the min and max of a REAL array */

void UL_lmt(pt, n, mymin, mymax)
   REAL *pt;
   int n;
   REAL *mymin, *mymax;
   {Register REAL *p;
    Register int i;

    *mymin =  HUGE;
    *mymax = -HUGE;
    for (p = pt, i = 0; i < n; p++, i++)
        {*mymin = min(*mymin, *p);
         *mymax = max(*mymax, *p);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CURVE_STRP - return TRUE iff the given object is a string which
 *               - designates an active curve
 *               - used in building curve based procedures
 */

int UL_curve_strp(obj)
   object *obj;
   {if (SS_stringp(obj))
       {if (_SX_curvep(SS_STRING_TEXT(obj)))
           return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_CHECK_ORDER - check that the array is strictly increasing
 *                - or do a full sort
 */

void UL_check_order(p, n, i)
   REAL *p;
   int n, i;
   {REAL *p1, *p2;
    int j;

    for (p1 = p, p2 = p+1, j = 1; j < n; p1++, p2++, j++)
        {if (*p1 > *p2)                             /* sort if out of order */
            {if (SS_interactive == ON)
                PRINT(stdout, "\nSorting curve %c\n", SX_dataset[i].id);
             UL_sort(i);
             break;};};

    return;}
                        
/*--------------------------------------------------------------------------*/

/*                            FUNCTION HANDLERS                             */

/*--------------------------------------------------------------------------*/

/* UL_US - handler for unary operation acting on a set of curves
 *       - this handler passes a Scheme object to its functions and
 *       - expects Scheme objects back from them
 */

object *UL_us(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *ret, *val, *t;
    PFPObject fun;

    fun = (PFPObject) basicf;
    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (select (menu)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl; !SS_nullobjp(t); t = SS_cdr(t))
        {val = (*fun)(SS_car(t));
         SS_Assign(ret, SS_mk_cons(val, ret));};

    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UC - handler for unary operation acting on a set of curves
 *       - these funtions expect an integer curve index and
 *       - return a valid Scheme object
 */

object *UL_uc(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *ret, *t;
    int j;

    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (copy (lst)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; !SS_nullobjp(t); t = SS_cdr(t))
        {j = SX_get_curve(SS_car(t));
         if (j >= 0)
            {SS_Assign(ret, SS_mk_cons((*((PFPObject) basicf))(j), ret));};};
         
    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_OPXC - performs binary operations on x values of curves
 *         - using a single scalar
 */
    
object *UL_opxc(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *tok, *ret, *t;
    REAL a;
    int i, j, n;
    REAL *xp;
        
    SX_last_arg(tok, argl);
    argl = _SX_prep_arg(argl);

    a = HUGE;
    SS_args(tok,
            SC_REAL_I, &a,
            0);
    if (a == HUGE)
       SS_error("BAD NUMBER - UL_OPXC ", tok);

    SS_Assign(tok, SS_null);

/* set plot flag on so that for example (dx (lst)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {j = SX_get_curve(SS_car(t));
         if (j != -1)
            {n = SX_dataset[j].n;
             for (xp = SX_dataset[j].xp, i = 0; i < n; xp++, i++)
                 *xp = (REAL) (*(PFDouble) basicf)(*xp, a);
             SX_dataset[j].modified = TRUE;

             SS_Assign(ret, SS_mk_cons(SX_dataset[j].obj, ret));
             UL_lmt(SX_dataset[j].xp, n,
                    &SX_dataset[j].xmin,
                    &SX_dataset[j].xmax);};};
         
    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_OPYC - performs binary operations on y values of curves
 *         - using a single scalar
 */

object *UL_opyc(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *tok, *ret, *t;
    REAL a;
    int i, j, n;
    REAL *yp;
        
    SX_last_arg(tok, argl);
    argl = _SX_prep_arg(argl);

    a = HUGE;
    SS_args(tok,
            SC_REAL_I, &a,
            0);
    if (a == HUGE)
       SS_error("BAD NUMBER - UL_OPYC ", tok);

    SS_Assign(tok, SS_null);

/* set plot flag on so that for example (dy (lst)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {j = SX_get_curve(SS_car(t));
         if (j != -1)
            {n = SX_dataset[j].n;
             for (yp = SX_dataset[j].yp, i = 0; i < n; yp++, i++)
                 *yp = (REAL) (*(PFDouble) basicf)(*yp, a);
             SX_dataset[j].modified = TRUE;

             SS_Assign(ret, SS_mk_cons(SX_dataset[j].obj, ret));
             UL_lmt(SX_dataset[j].yp, n,
                    &SX_dataset[j].ymin,
                    &SX_dataset[j].ymax);};};

    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UL2TOC - unary applies last 2 args to curve */

object *UL_ul2toc(basicf, argl)
   PFInt basicf;
   object *argl;
   {return(_UL_ul2toc(basicf, argl, TRUE));}
   
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UL2TOCNP - unary applies last 2 args to curve
 *             - and does not replot
 */

object *UL_ul2tocnp(basicf, argl)
   PFInt basicf;
   object *argl;
   {return(_UL_ul2toc(basicf, argl, FALSE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_UL2TOC - worker routine for UL_UL2TOC and UL_UL2TOCNP */

object *_UL_ul2toc(basicf, argl, replot_flag)
   PFInt basicf;
   object *argl;
   int replot_flag;
   {object *s, *t, *tok1, *tok2, *ret;
    double d1, d2;
    int j;

    tok1 = NULL;
    tok2 = NULL;

    SS_Assign(argl, SS_reverse(argl));
    tok2 = SS_car(argl);
    if (_SS_numberp(tok2))
       {SS_Assign(argl, SS_cdr(argl));
	if (SS_consp(argl))
	   {tok1 = SS_car(argl);
	    if (_SS_numberp(tok1))
	       {SS_Assign(argl, SS_cdr(argl));}
	    else
	       {tok1 = tok2;
		tok2 = NULL;};}
	else
	   {tok1 = tok2;
	    tok2 = NULL;};}
    else
        tok2 = NULL;
    SS_Assign(argl, SS_reverse(argl));

    argl = _SX_prep_arg(argl);

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);
         if (SX_curvep_a(s))
            {j = SX_get_curve(s);
	     if (tok1 != NULL)
	        SS_args(tok1,
			SC_DOUBLE_I, &d1,
			0);
	     else
	        d1 = SX_dataset[j].xmin;
	     if (tok2 != NULL)
	        SS_args(tok2,
			SC_DOUBLE_I, &d2,
			0);
	     else
	        d2 = SX_dataset[j].xmax;
             SS_Assign(ret,
                       SS_mk_cons((*(PFPObject) basicf)(j, d1, d2), ret));};};

    SX_plot_flag = replot_flag;

    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_ULNTOC - unary applies last n args to curve */

object *UL_ulntoc(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *tok, *crvs, *ret, *t, *u;
    int j;

    argl = _SX_prep_arg(argl);

    for (t = argl, crvs = SS_null; SS_consp(t); t = SS_cdr(t))
        {tok = SS_car(t);
         if (SX_curvep_a(tok))
            {SS_Assign(crvs, SS_mk_cons(tok, crvs));}
         else
            break;};

/* set plot flag on so that for example (filter (lst) dp rp) causes replot */
    SX_plot_flag = TRUE;

    for (ret = SS_null, u = crvs; SS_consp(u); u = SS_cdr(u))
        {tok = SS_car(u);
         if (SX_curvep_a(tok))
            {j = SX_get_curve(tok);
             SS_Assign(ret,
                       SS_mk_cons((*(PFPObject) basicf)(j, t), ret));};};

    SS_Assign(crvs, SS_null);
    SS_Assign(argl, SS_null);
    SS_Assign(ret, SS_reverse(ret));

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UOPXC - math handler applies the given unary function to the x values */
    
object *UL_uopxc(basicf, argl)
   PFInt basicf;
   object *argl;
   {int i, j, n;
    REAL *xp;
    object *ret, *t;

    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (expx (lst)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {j = SX_get_curve(SS_car(t));
         if (j != -1)
            {n = SX_dataset[j].n;
             for (xp = SX_dataset[j].xp, i = 0; i < n; xp++, i++)
                 *xp = (REAL) (*(PFDouble) basicf)(*xp);
             SX_dataset[j].modified = TRUE;

             SS_Assign(ret, SS_mk_cons(SX_dataset[j].obj, ret));
             UL_lmt(SX_dataset[j].xp, n,
                    &SX_dataset[j].xmin,
                    &SX_dataset[j].xmax);};};
         
    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_UOPYC - math handler applies the given unary function to the y values */

object *UL_uopyc(basicf, argl)
   PFInt basicf;
   object *argl;
   {int i, j, n;
    REAL *yp, f;
    object *s, *ret, *tmp, *t;

    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (expx (lst)) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);
         j = SX_get_curve(s);
         if (j != -1)
            {n = SX_dataset[j].n;
             for (yp = SX_dataset[j].yp, i = 0; i < n; yp++, i++)
                 *yp = (REAL) (*(PFDouble) basicf)(*yp);
             SX_dataset[j].modified = TRUE;

             tmp = SS_mk_cons(SX_dataset[j].obj, ret);
             UL_lmt(SX_dataset[j].yp, n,
                    &SX_dataset[j].ymin, &SX_dataset[j].ymax);}

         else if (SS_integerp(s))
            {f   = (REAL) SS_INTEGER_VALUE(s);
             tmp = SS_mk_cons(SS_mk_float((*(PFDouble) basicf)(f)), ret);}

         else if (SS_floatp(s))
            {f   = SS_FLOAT_VALUE(s);
             tmp = SS_mk_cons(SS_mk_float((*(PFDouble) basicf)(f)), ret);};

         SS_Assign(ret, tmp);};

    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_BFTOC - unary operation on curves using first argument as
 *          - a parameter
 */

object *UL_bftoc(basicf, argl)
   PFInt basicf;
   object *argl;
   {int j;
    char *s1;
    object *s, *tok, *t;

    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (op param (lst)) causes replot */
    SX_plot_flag = TRUE;

    tok  = SS_car(argl);
    argl = SS_cdr(argl);

    s1 = SC_strsavef(SS_get_string(tok),
         "char*:UL_BFTOC:s1");

    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);

         if (SX_curvep_a(s))
            {j = SX_get_curve(s);
             (*basicf)(j, s1);};};

    SFREE(s1);

    SS_Assign(argl, SS_null);

    return(tok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_BLTOC - unary operation applied to each curve using the last argument
 *          - as a parameter
 */

object *UL_bltoc(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *s, *ret, *tok, *t;

    SX_last_arg(tok, argl);
    if (!_SS_numberp(tok))
       SS_error("BAD LAST ARGUMENT - UL_BLTOC", tok);

    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (color (lst) red) causes replot */
    SX_plot_flag = TRUE;

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);

         if (SX_curvep_a(s))
            SS_Assign(ret, SS_mk_cons((*(PFPObject) basicf)(s, tok), ret));};

    SS_Assign(tok, SS_null);
    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_BLTOCNP - unary operation applied to each curve using the last argument
 *            - as a parameter
 *            - clears screen, prints name of function and
 *            - surpresses plotting
 */

object *UL_bltocnp(basicf, argl)
   PFInt basicf;
   object *argl;
   {object *s, *tok, *ret, *t;
    char *r;

    SX_last_arg(tok, argl);
    if (!_SS_numberp(tok))
       SS_error("BAD LAST ARGUMENT - BLTOCNP", tok);

    if (SS_interactive == ON)
       {r  = SS_get_string(SS_Fun);
	*r = (char) toupper((int) *r);
	PRINT(stdout, "\n     %s\n", r);};

    argl = _SX_prep_arg(argl);

    ret = SS_null;
    for (t = argl ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);

         if (SX_curvep_a(s))
            SS_Assign(ret, SS_mk_cons((*(PFPObject) basicf)(s, tok), ret));};
         
    UL_pause(FALSE);

    SS_Assign(tok, SS_null);
    SS_Assign(argl, SS_null);

    SX_prep_ret(ret);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_BC - does binary operations on curves two at a time and
 *       - accumulates the result
 */

object *UL_bc(basicf, argl)
   PFInt basicf;
   object *argl;
   {REAL *xp1, *xp2, *yp1, *yp2;
    REAL *xa, *af, *ya;                         /* pointers for accumulator */
    REAL value, gxmin, gxmax, xt;
    Register int i, j;
    int ic;
    int n1, n2, na;     /* number of elements in each array and accumulator */
    int temp_flag;                                /* flags for the pointers */
    object *ch, *s, *tmp, *t;

/* set up initial pointers */
    temp_flag = FALSE;
    argl = _SX_prep_arg(argl);

/* set plot flag on so that for example (+ (lst) red) causes replot */
    SX_plot_flag = TRUE;

    if (SS_nullobjp(argl))
       return(argl);

    else if (SS_nullobjp(SS_cdr(argl)))
       return(SS_car(argl));

    else

/* search all curves in the list for the domain
 * this way it is not necessary that the screen be active
 */
       {gxmin =  HUGE;
        gxmax = -HUGE;
        for (ch = argl; !SS_nullobjp(ch); ch = SS_cdr(ch))
            {s = SS_car(ch);
             if (SX_curvep_a(s))
                {i     = SX_get_curve(s);
                 xt    = SX_dataset[i].xmin;
                 gxmin = min(gxmin, xt);
                 xt    = SX_dataset[i].xmax;
                 gxmax = max(gxmax, xt);};};
            
        ch = SS_null;
        t  = argl;
        while (TRUE)
           {s = SS_car(t);
            if (_SS_numberp(s))
               {ch = SS_mk_cons(s, ch);
                t  = SS_cdr(t);}

/* the first non-number in the arg list */
            else
               {if (!SS_nullobjp(ch))
                   {s = SS_binary_flt((PFPObject) basicf,
                                      SS_reverse(ch));
                    if (SS_integerp(s))
                       value = (REAL) SS_INTEGER_VALUE(s);
                    else if (SS_floatp(s))
                       value = (REAL) SS_FLOAT_VALUE(s);
                    temp_flag = TRUE;
                    tmp = _UL_make_ln(0.0, value,
                                      gxmin,
                                      gxmax,
                                      SX_default_npts);

                    i = SX_get_curve(tmp);
                    sprintf(pbuffer, "%s %g",
                            SS_get_string(SS_Fun),
                            value);}

                else if (SX_curvep_a(s))
                   {i = SX_get_curve(s);
                    sprintf(pbuffer, "%s %c",
                            SS_get_string(SS_Fun), SX_dataset[i].id);}

                else
                   SS_error("BAD ARGUMENT - BC", s);

                if (SS_nullobjp(ch))
                   t = SS_cdr(t);

                break;};

/* check for end of arg list */
            if (SS_nullobjp(t))
               {if (SS_nullobjp(ch))
                   return(SS_null);
                else
                   return(SS_binary_flt((PFPObject) basicf,
                                        SS_reverse(ch)));};};

        n1 = SX_dataset[i].n;
        UL_buf1x = FMAKE_N(REAL, n1, "UL_BC:buf1x");
        UL_buf1y = FMAKE_N(REAL, n1, "UL_BC:buf1y");

/* copy the curve data into the buffer
 * this protects against hacking the curve
 */
        xp1 = UL_buf1x;
        yp1 = UL_buf1y;
        xp2 = SX_dataset[i].xp;
        yp2 = SX_dataset[i].yp;
        UL_check_order(xp2, n1, i);
        for (j = 0; j < n1; j++)
            {*xp1++ = *xp2++;
             *yp1++ = *yp2++;};

        xp1 = UL_buf1x;
        yp1 = UL_buf1y;};
        
    UL_buf2x = FMAKE_N(REAL, 1, "UL_BC:buf2x");
    UL_buf2y = FMAKE_N(REAL, 1, "UL_BC:buf2y");

    na = n1;
    xa = xp1;
    ya = yp1;
    af = NULL;
        
    for ( ; SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);

/* combine a number with the accumulator */
         if (_SS_numberp(s))
            {if (SS_integerp(s))
                value = (REAL) SS_INTEGER_VALUE(s);
             else if (SS_floatp(s))
                value = (REAL) SS_FLOAT_VALUE(s);
             for (yp1 = ya, ic = 0; ic < na; yp1++, ic++)
                 *yp1 = (REAL) (*(PFDouble) basicf)(*yp1, value);
             sprintf(pbuffer, "%s %g", pbuffer, value);}

/* combine a curve with the accumulator */
         else if (SX_curvep_a(s))
            {j = SX_get_curve(s);
             sprintf(pbuffer, "%s %c", pbuffer, SX_dataset[j].id);

             xp2 = SX_dataset[j].xp;
             yp2 = SX_dataset[j].yp;
             n2  = SX_dataset[j].n;
             UL_check_order(xp2, n2, j);

/* set/reset the accumulator */
             n1 = na;
             if (af == UL_buf2x)
                {xp1 = UL_buf2x;
                 yp1 = UL_buf2y;
                 REMAKE_N(UL_buf1x, REAL, n1+2+n2);
                 REMAKE_N(UL_buf1y, REAL, n1+2+n2);
                 xa = af = UL_buf1x;
                 ya = UL_buf1y;}
             else
                {xp1 = UL_buf1x;

                 yp1 = UL_buf1y;
                 REMAKE_N(UL_buf2x, REAL, n1+2+n2);
                 REMAKE_N(UL_buf2y, REAL, n1+2+n2);
                 xa = af = UL_buf2x;
                 ya = UL_buf2y;};

             na = _UL_bc_operate(basicf, xa, ya, xp1, xp2, yp1, yp2, n1, n2);
             if (na == -1)
                return(SS_f);};};

    SS_Assign(argl, SS_null);

/* UL_delete the temporary curve if allocated */
    if (temp_flag)
       UL_delete(tmp);

/* create new curve with data in the accumulator */
    ch = _SX_mk_curve(na, xa, ya, pbuffer, NULL,
		      (PFVoid) UL_plot);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);
    SFREE(UL_buf2x);
    SFREE(UL_buf2y);

    return(ch);}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _UL_BC_OPERATE - do the work of operating on two curves now that
 *                - they have been chosen and setup
 */

static int _UL_bc_operate(basicf, xa, ya, xp1, xp2, yp1, yp2, n1, n2)
   PFInt basicf;
   REAL *xa, *ya, *xp1, *xp2, *yp1, *yp2;
   int n1, n2;
   {int na, i, ic1, ic2;
    REAL ida, xv, d1, d2, yva, yvb;
    REAL xta, xtb, xtc, yta, ytb;
    REAL *sx1, *sx2;

/* find starting point or bail out if no overlap */
    ic1 = 0;
    ic2 = 0;
    if (*xp1 < *xp2)
       {xv = *xp2;
        if (xv > xp1[n1-1])
           return(-1);

        for (; (*xp1 < xv) && (ic1 < n1); xp1++, yp1++, ic1++);}
    else
       {xv = *xp1;
        if (xv > xp2[n2-1])
           return(-1);

        for (; (*xp2 < xv) && (ic2 < n2); xp2++, yp2++, ic2++);};

/* select the x values for the accumulator */
    na  = 0;
    sx1 = xp1;
    sx2 = xp2;
    while ((ic1++ < n1) && (ic2++ < n2))
       {d1 = *xp1++ - xv;
        d2 = *xp2++ - xv;
        xv += (d2 < d1) ? d2 : d1;

        xa[na++] = xv;

        ida = 2.0*ABS(d2 - d1)/(d1 + d2 + SMALL);
        if (ida >= 0.2)
           {if (d2 < d1)
               {xp1--;
                ic1--;}
            else
               {xp2--;
                ic2--;};};};

/* get y values for the selected x values */
    xp1 = sx1;
    xp2 = sx2;
    for (i = 0; i < na; i++)
        {xv = xa[i];

         if (xv == *xp1)
            yva = *yp1;
         else
            {for (; xv > *xp1; xp1++, yp1++);
             PM_interp(yva, xv, xp1[-1], yp1[-1], *xp1, *yp1);};

         if (xv == *xp2)
            yvb = *yp2;
         else
            {for (; xv > *xp2; xp2++, yp2++);
             PM_interp(yvb, xv, xp2[-1], yp2[-1], *xp2, *yp2);};

         ya[i] = (REAL) (*(PFDouble) basicf)(yva, yvb);};

    return(na);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* UL_BCXL - binary operation on curves
 *         - uses x values from last curve in string and finds y
 *         - values for these x's.  This is primarily intended for
 *         - use in functional composition, but it may have other
 *         - applications.  cfm 2/18/87
 */

object *UL_bcxl(basicf, argl)
   PFInt basicf;
   object *argl;
   {REAL *xp, *yp;
    int i, j, n;
    object *s, *ch, *t;
    char local[MAXLINE];
        
    argl = _SX_prep_arg(argl);
    argl = SS_reverse(argl);
    i    = -1;
    SS_args(argl,
            UL_CURVE_INDEX_I, &i,
            0);

    if (i < 0)
       SS_error("BAD LAST ARGUMENT - UL_BCXL", SS_reverse(argl));

/* set plot flag on so that for example (compose (lst)) causes replot */
    SX_plot_flag = TRUE;

    xp = SX_dataset[i].xp;
    yp = SX_dataset[i].yp;
    n  = SX_dataset[i].n;

    sprintf(local, "%c", SX_dataset[i].id);
    UL_buf1x = FMAKE_N(REAL, n, "UL_BCXL:buf1x");
    UL_buf1y = FMAKE_N(REAL, n, "UL_BCXL:buf1y");
    for (j = 0; j < n; j++)                          /* copy to accumulator */
        {UL_buf1x[j] = xp[j];
         UL_buf1y[j] = yp[j];};
    xp = UL_buf1x;
    yp = UL_buf1y;

    for (t = SS_cdr(argl); SS_consp(t); t = SS_cdr(t))
        {s = SS_car(t);
         if (!SX_curvep_a(s))
            SS_error("BAD ARGUMENT - UL_BCXL", s);
         i = SX_get_curve(s);
         sprintf(local, "%s %c", local, SX_dataset[i].id);

         for (j = 0; j < n; j++)
             {yp[j] = (REAL) (*(PFDouble) basicf)(yp[j], i);};};

    sprintf(pbuffer, "%s %s", SS_get_string(SS_Fun), SC_strrev(local));
    ch = _SX_mk_curve(n, xp, yp, pbuffer, NULL,
		      (PFVoid) UL_plot);

    SS_Assign(argl, SS_null);

    SFREE(UL_buf1x);
    SFREE(UL_buf1y);           

    return(ch);}
                
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

