## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import math, os, re, string, sys, time, types
import traceback                                                    #bundle#
try:                                                                #bundle#
    from cPickle import Pickler, Unpickler, UnpicklingError         #bundle#
except ImportError:                                                 #bundle#
    from pickle import Pickler, Unpickler, UnpicklingError          #bundle#

# PySol imports
from mfxutil import destruct, Struct, SubclassResponsibility        #bundle#
from mfxutil import merge_dict                                      #bundle#
from util import PACKAGE, VERSION, Timer                            #bundle#
from random import PysolRandom, LCRandom64, LCRandom31, WHRandom    #bundle#
from pysoltk import EVENT_HANDLED, EVENT_PROPAGATE                  #bundle#
from pysoltk import CURSOR_WATCH, ANCHOR_SW, ANCHOR_SE              #bundle#
from pysoltk import tkname, bind, after_idle, deiconify             #bundle#
from pysoltk import MfxDialog, MfxExceptionDialog                   #bundle#
from pysoltk import MfxCanvasText, MfxCanvasImage                   #bundle#
from pysoltk import MfxCanvasLine, MfxCanvasRectangle               #bundle#
from move import AMoveMove, AFlipMove, ATurnStackMove               #bundle#
from move import ANextRoundMove, ASaveSeedMove, AShuffleStackMove   #bundle#
from move import AUpdateStackViewMove                               #bundle#
from hint import DefaultHint                                        #bundle#
from help import helpAbout                                          #bundle#


# /***********************************************************************
# // Base class for all Solitaire games
# //
# // Handles:
# //   load/save
# //   undo/redo (using a move history)
# //   hints/demo
# ************************************************************************/

class Game:
    # for self.gstats.updated
    U_PLAY       =  0
    U_NOT_WON    = -1
    U_WON        = -2
    U_LOST       = -3
    U_STRICT_WON = -4

    # for self.moves.state
    S_INIT = 0x00
    S_DEAL = 0x10
    S_FILL = 0x20
    S_PLAY = 0x30
    S_UNDO = 0x40
    S_REDO = 0x50

    #
    # game construction
    #

    # only basic initialization here
    def __init__(self, game_info, random=None):
        self.random = random
        self.game_info = game_info
        self.id = game_info.id
        assert self.id > 0
        self.busy = 0
        self.version = VERSION
        self.cards = []
        self.stackmap = {}              # dict with (x,y) tuples as key
        self.allstacks = []
        self.s = Struct(                # stacks
            talon = None,
            waste = None,
            foundations = [],
            rows = [],
            reserves = [],
            internals = [],
        )
        self.sg = Struct(               # stack-groups
            openstacks = [],            #   for getClosestStack(): only on these stacks the player can place a card
            talonstacks = [],           #   for Hint
            dropstacks = [],            #   for Hint & getAutoStacks()
            reservestacks = [],         #   for Hint
##            hint = Struct(),            #   extra info for class Hint
            hp_stacks = [],             #   for getHightlightPilesStacks()
        )
        self.regions = Struct(          # for getClosestStack()
            info = [],                  #   list of tuples(stacks, rect)
            remaining = [],             #   list of stacks in no region
        )
        self.reset()

    # main constructor
    def create(self, app):
        old_busy = self.busy
        self.busy = 1
        self.app = app
        self.top = app.top
        self.canvas = app.canvas
        self.filename = ""
        self.drag = Struct(
            stack = None,               #
            cards = [],                 #
            shadows = [],               # list of canvas images
            shade_stack = None,         # stack currently shaded
            shade_img = None,           # canvas image
            canshade_stacks = [],       # list of stacks already tested
            noshade_stacks = [],        #   for this drag
        )
        if self.gstats.start_player is None:
            self.gstats.start_player = self.app.opt.player

        self.setCursor(cursor=CURSOR_WATCH)
        self.top.wm_title("PySol - " + self.getTitleName())
        self.top.wm_iconname("PySol - " + self.getTitleName())

        # create the game
        if self.app.intro.progress: self.app.intro.progress.update(step=1)
        self.createGame()
        # set some defaults
        self.sg.openstacks = filter(lambda s: s.cap.max_accept >= s.cap.min_accept, self.sg.openstacks)
        self.sg.hp_stacks = filter(lambda s: s.cap.max_move >= 2, self.sg.dropstacks)
        # convert stackgroups to tuples (speed)
        self.allstacks = tuple(self.allstacks)
        self.s.foundations = tuple(self.s.foundations)
        self.s.rows = tuple(self.s.rows)
        self.s.reserves = tuple(self.s.reserves)
        self.s.internals = tuple(self.s.internals)
        self.sg.openstacks = tuple(self.sg.openstacks)
        self.sg.talonstacks = tuple(self.sg.talonstacks)
        self.sg.dropstacks = tuple(self.sg.dropstacks)
        self.sg.reservestacks = tuple(self.sg.reservestacks)
        self.sg.hp_stacks = tuple(self.sg.hp_stacks)
        # init the stack bottom
        for stack in self.allstacks:
            stack.prepareStack()
            stack.assertStack()
        if self.s.talon:
            assert hasattr(self.s.talon, "round")
            assert hasattr(self.s.talon, "max_rounds")
        # optimize regions
        self.optimizeRegions()
        # create cards
        if not self.cards:
            self.cards = self.s.talon.createCards(self.game_info.decks, self.app.intro.progress)

        # init keybindings
        # note: a Game is only allowed to bind self.canvas !
        bind(self.canvas, "<1>", self.clickHandler)
        bind(self.canvas, "<2>", self.clickHandler)
        bind(self.canvas, "<3>", self.clickHandler)

        # update display properties
        self.top.wm_geometry("")        # cancel user-specified geometry
        self.canvas.config(width=self.width, height=self.height)
        self.stats.start_time = time.time()
        self.busy = old_busy


    # Do not destroy game structure (like stacks and cards) here !
    def reset(self, restart=0):
        self.filename = ""
        self.demo = None
        self.stats = Struct(
            hints = 0,                  # number of hints consumed
            highlight_piles = 0,        # number of highlight piles consumed
            highlight_cards = 0,        # number of highlight matching cards consumed
            highlight_samerank = 0,     # number of highlight same rank consumed
            undo_moves = 0,             # number of undos
            redo_moves = 0,             # number of redos
            total_moves = 0,            # number of total moves in this game
            player_moves = 0,           # number of moves
            demo_moves = 0,             # number of moves while in demo mode
            autoplay_moves = 0,         # number of moves
            quickplay_moves = 0,        # number of quickplay moves
            start_time = time.time(),
            elapsed_time = 0.0,
        )
        self.startMoves()
        self.hints = Struct(
            list = None,                # list of hints for the current move
            index = -1,
            level = -1,
        )
        self.changed_index = 0
        self.loadinfo = Struct(         # used when loading a game
            stacks = None,
            talon_round = 1,
        )
        if restart:
            return
        self.gstats = Struct(
            loaded = 0,                 # number of times this game was loaded
            saved = 0,                  # number of times this game was saved
            restarted = 0,              # number of times this game was restarted
            updated = self.U_PLAY,      # did this game already update the stats ?
            total_elapsed_time = 0.0,
            start_time = time.time(),
            start_player = None,
        )

    def getTitleName(self):
        return self.app.getGameTitleName(self.id)

    # this is called from within createGame()
    def setSize(self, w, h):
        self.width, self.height = int(round(w)), int(round(h))

    def setCursor(self, cursor):
        if self.canvas:
            self.canvas.config(cursor=cursor)
            ##self.canvas.update_idletasks()
        if self.app and self.app.toolbar:
            self.app.toolbar.setCursor(cursor=cursor)


    #
    # menu support
    #

    def _finishDrag(self):
        self.demo = None        # stop demo
        if self.busy: return 1
        if self.drag.stack:
            self.drag.stack.finishDrag()
        return 0

    def _cancelDrag(self):
        self.demo = None        # stop demo
        if self.busy: return 1
        if self.drag.stack:
            self.drag.stack.cancelDrag()
        return 0

    def updateMenus(self):
        self.app.menubar.updateMenus()


    #
    # UI & graphics support
    #

    def clickHandler(self, *args):
        if self.demo:
            # stop the demo
            self.demo = None
            self.updateMenus()
        return EVENT_PROPAGATE

    def updateStatus(self, **kw):
        tb, sb = self.app.toolbar, self.app.statusbar
        for k in kw.keys():
            v = kw[k]
            if k == "gameid":
                if v is None:
                    if sb: sb.updateText(gameid = "")
                    continue
                if type(v) == types.StringType:
                    if sb: sb.updateText(gameid = v)
                    continue
            if k == "moves":
                if v is None:
                    ##if tb: tb.updateText(moves = "Moves\n")
                    if sb: sb.updateText(moves = "")
                    continue
                if type(v) == types.IntType:
                    ##if tb: tb.updateText(moves = "Moves\n%d" % v)
                    if sb: sb.updateText(moves = "Moves %d" % v)
                    continue
                if type(v) == types.StringType:
                    ##if tb: tb.updateText(moves = v)
                    if sb: sb.updateText(moves = v)
                    continue
            if k == "player":
                if v is None:
                    if tb: tb.updateText(player = "Player\n")
                    continue
                if type(v) == types.StringType:
                    if tb: tb.updateText(player = "Player\n" + v)
                    ##if tb: tb.updateText(player = v)
                    continue
            if k == "stats":
                if v is None:
                    if sb: sb.updateText(stats = "")
                    continue
                if type(v) == types.TupleType:
                    t = "%d: %d/%d" % (v[0]+v[1], v[0], v[1])
                    if sb: sb.updateText(stats = t)
                    continue
            if k == "time":
                ## TODO: what about a timer ?
                continue
            raise AttributeError, k


    #
    # misc. methods
    #

    def areYouSure(self, title=None, text=None, confirm=-1, default=0):
        if confirm < 0:
            confirm = self.app.opt.confirm
        if confirm:
            if not title: title = "PySol"
            if not text: text = "Discard current game ?"
            d = MfxDialog(self.top, title=title, text=text,
                          bitmap="questhead",
                          default=default, strings=("Ok", "Cancel"))
            if d.status != 0 or d.num != 0:
                return 0
        return 1

    def notYetImplemented(self):
        d = MfxDialog(self.top, title="Not yet implemented",
                      text="This function is\nnot yet implemented.",
                      bitmap="error")

    def animatedMoveTo(self, from_stack, to_stack, cards, x, y, tkraise=1, frames=-1, shadow=-1):
        if self.app.opt.animations == 0 or frames == 0:
            return
        if self.app.debug and not self.top.winfo_ismapped():
            return
        # init timer - need a high resolution for this to work
        clock, delay, skip = None, 1, 1
        if self.app.opt.animations >= 2:
            if os.name == "posix":  clock = time.time
            elif os.name == "nt":   clock = time.clock
            elif os.name == "mac":  clock = time.clock  # ???
        SPF = 0.02              # animation speed - seconds per frame
        if frames < 0:
            frames = 8
##            if self.app.opt.animations >= 2:
##                PPS = 10000.0   # animation speed - pixels per second
##                c = cards[0]
##                dist = math.sqrt((x - c.x)**2 + (y - c.y)**2)
        assert frames >= 2
        if self.app.opt.animations == 3:
            frames = frames * 8
            SPF = SPF / 2
        elif self.app.opt.animations == 4:
            frames = frames * 16
            SPF = SPF / 2
        if shadow < 0: shadow = self.app.opt.shadow
        shadows = []
        # animation
        if tkraise:
            for card in cards:
                card.tkraise()
        c = cards[0]
        dx, dy = (x - c.x) / float(frames), (y - c.y) / float(frames)
        tx, ty = 0, 0
        i = 1
        if clock: starttime = clock()
        while i < frames:
            mx, my = int(round(dx * i)) - tx, int(round(dy * i)) - ty
            tx, ty = tx + mx, ty + my
            for s in shadows:
                s.move(mx, my)
            for card in cards:
                card.moveBy(mx, my)
            if i == 1 and shadow:
                # create shadows in the first frame
                images = self.app.images
                img0, img1 = images.getShadow(0), images.getShadow(len(cards))
                if img0 and img1:
                    c = cards[-1]
                    if from_stack and from_stack.CARD_YOFFSET[0] < 0: c = cards[0]
                    cx, cy = c.x + images.CARDW, c.y + images.CARDH
                    cx, cy = cx + images.SHADOW_XOFFSET, cy + images.SHADOW_YOFFSET
                    shadows.append(MfxCanvasImage(self.canvas, cx, cy - img0.height(),
                                                  image=img1, anchor=ANCHOR_SE))
                    shadows.append(MfxCanvasImage(self.canvas, cx, cy,
                                                  image=img0, anchor=ANCHOR_SE))
                    for s in shadows:
                        s.lower(c.item)
            self.canvas.update_idletasks()
            step = 1
            if clock:
                endtime = starttime + i*SPF
                sleep = endtime - clock()
                if delay and sleep >= 0.005:
                    # we're fast - delay
                    ##print "Delay frame", i, sleep
                    time.sleep(sleep)
                elif skip and sleep <= -0.75*SPF:
                    # we're slow - skip 1 or 2 frames
                    ##print "Skip frame", i, sleep
                    step = step + 1
                    if frames > 4 and sleep < -1.5*SPF: step = step + 1
                ##print i, step, mx, my; time.sleep(0.5)
            i = i + step
        # last frame: delete shadows, move card to final position
        for s in shadows:
            s.delete()
        for card in cards:
            card.moveTo(x, y)
            if to_stack:
                y = y + to_stack.CARD_YOFFSET[0]
        self.canvas.update_idletasks()

    def winAnimation(self):
        # Stupid animation when you win.
        # FIXME: make this interruptible by a key- or mousepress
        if self.app.opt.animations == 0:
            return
        if self.app.debug and not self.top.winfo_ismapped():
            return
        cards = []
        for s in self.allstacks:
            if s is not self.s.talon:
                for c in s.cards:
                    cards.append((c,s))
        # select some random cards
        acards = []
        for i in range(16):
            c, s = self.app.miscrandom.choice(cards)
            if not c in acards:
                acards.append(c)
        # animate
        sx, sy = self.s.talon.x, self.s.talon.y
        w, h = self.width, self.height
        while cards:
            # get and un-tuple a random card
            t = self.app.miscrandom.choice(cards)
            c, s = t
            s.removeCard(c, update=0)
            # animation
            if c in acards or len(cards) <= 2:
                self.animatedMoveTo(None, None, [c], w/2, h/2, tkraise=0, shadow=0)
                self.animatedMoveTo(None, None, [c], sx, sy, tkraise=0, shadow=0)
            else:
                c.moveTo(sx, sy)
            cards.remove(t)

    def getFullId(self, format):
        s = str(self.random)
        if format == 2: return "#" + s
        return s

    def getNumberOfFreeReserves(self):
        return len(filter(lambda s: not s.cards, self.s.reserves))
        ##return getNumberOfFreeStacks(self.s.reserves)


    #
    # Game methods
    #

    # quit to app, possibly restarting with another game
    def _quitGame(self, id=0, random=None, loadedgame=None, startdemo=0):
        if id > 0:
            self.setCursor(cursor=CURSOR_WATCH)
        self.updateStats()
        self.app.nextgame.id = id
        self.app.nextgame.random = random
        self.app.nextgame.loadedgame = loadedgame
        self.app.nextgame.startdemo = startdemo
        self.updateStatus(gameid=None, moves=None, stats=None)
        self.top.mainquit()

    # start a new name
    def restartGame(self):
        self.newGame(random=self.random, restart=1)

    def newGame(self, random=None, restart=0, autoplay=1):
        old_busy = self.busy
        self.busy = 1
        self.setCursor(cursor=CURSOR_WATCH)
        if restart:
            if self.moves.index > 0 and self.getPlayerMoves() > 0:
                self.gstats.restarted = self.gstats.restarted + 1
            self.reset(restart=1)
        else:
            self.updateStats()
            self.reset()
        self.s.talon.removeAllCards()
        self.createRandom(random)
        ##print self.random, self.random.__dict__
        self.shuffle()
        assert len(self.s.talon.cards) == self.game_info.cards
        self.s.talon.round = 1
        self.s.talon.updateText()
        ##print self.app.starttimer
        self.updateStatus(player=self.app.opt.player)
        self.updateStatus(gameid=self.getFullId(format=2), moves=0)
        self.updateStatus(stats=self.app.stats.getStats(self.app.opt.player, self.id))
        # unhide toplevel when we use a progress bar
        if self.app.intro.progress:
            deiconify(self.top)
        self.top.busyUpdate()
        # let's go
        self.moves.state = self.S_INIT
        self.startGame()
        for stack in self.allstacks:
            stack.updateText()
        self.startMoves()
        self.updateStatus(moves=0)
        self.updateMenus()
        self.setCursor(cursor=self.app.top_cursor)
        self.stats.start_time = time.time()
        if autoplay:
            self.autoPlay()
            self.stats.player_moves = 0
        self.busy = old_busy

    # restore a loaded game (see load/save below)
    def restoreGame(self, game):
        assert self.id == game.id
        self.updateStats()
        self.reset()
        self.s.talon.removeAllCards()
        self.filename = game.filename
        # 1) copy loaded variables
        self.version = game.version
        self.random = game.random
        self.moves = game.moves
        self.gstats = game.gstats
        self.stats = game.stats
        # 2) copy extra loadinfo
        self.s.talon.round = game.loadinfo.talon_round
        # 3) move cards to stacks
        assert len(self.allstacks) == len(game.loadinfo.stacks)
        for i in range(len(self.allstacks)):
            for t in game.loadinfo.stacks[i]:
                card_id, face_up = t
                card = self.cards[card_id]
                if face_up:
                    card.showFace()
                else:
                    card.showBack()
                self.allstacks[i].addCard(card)
            self.allstacks[i].updateText()
        # 4) subclass settings
        self._restoreGameHook(game)
        # 5) other settings
        self.updateStatus(gameid=self.getFullId(format=2), moves=self.moves.index)
        self.updateStatus(stats=self.app.stats.getStats(self.app.opt.player, self.id))
        self.updateMenus()
        if self.app.intro.progress or not self.top.winfo_ismapped():
            deiconify(self.top)
        self.setCursor(cursor=self.app.top_cursor)
        self.stats.start_time = time.time()

    def createRandom(self, random):
        if random is None:
            if isinstance(self.random, LCRandom64):
                seed = self.random.getSeed()
                self.app.gamerandom.setSeed(seed)
            while 1:
                dummy = self.app.gamerandom.random()
                seed = self.app.gamerandom.getSeed()
                if seed >= 10000000000000000L:  # we want at least 17 digits
                    break
            self.random = LCRandom64(seed)
        else:
            self.random = random
            self.random.reset()

    def enterState(self, state):
        old_state = self.moves.state
        if state < old_state:
            self.moves.state = state
        return old_state

    def leaveState(self, old_state):
        self.moves.state = old_state


    #
    # shuffling
    #

    #
    def shuffle(self):
        # get a fresh copy of the original game-cards
        cards = list(self.cards)[:]
        if 1 and self.app.debug:
            for i in range(len(cards)):
                c = cards[i]
                assert i == c.id == 52*c.deck + 13*c.suit + c.rank
        # init random generator
        if isinstance(self.random, LCRandom31) and len(cards) == 52:
            # FreeCell compat. mode
            ncards = []
            for i in range(13):
                for j in (0, 39, 26, 13):
                    ncards.append(cards[i + j])
            cards = ncards
        self.random.reset()         # reset to initial seed
        # shuffle
        self.random.shuffle(cards)
        # subclass
        cards = self._shuffleHook(cards)
        # finally add the shuffled cards to the Talon
        for card in cards:
            self.s.talon.addCard(card, update=0)
            card.showBack(unhide=0)

    # subclass overrideable (must use self.random)
    def _shuffleHook(self, cards):
        return cards

    # utility for use by subclasses
    def _shuffleHookMoveToTop(self, cards, func, ncards=999999):
        # move cards to top of the Talon (i.e. first cards to be dealt)
        cards, scards = self._shuffleHookMoveSorter(cards, func, ncards)
        return cards + scards

    def _shuffleHookMoveToBottom(self, cards, func, ncards=999999):
        # move cards to bottom of the Talon (i.e. last cards to be dealt)
        cards, scards = self._shuffleHookMoveSorter(cards, func, ncards)
        return scards + cards

    def _shuffleHookMoveSorter(self, cards, func, ncards):
        # note that we reverse the cards, so that smaller sort_orders
        # will be nearer to the top of the Talon
        sitems, i = [], len(cards)
        for c in cards[:]:
            select, sort_order = func(c)
            if select:
                sitems.append((sort_order, i, c))
                cards.remove(c)
                if len(sitems) >= ncards:
                    break
            i = i - 1
        sitems.sort()
        sitems.reverse()
        scards = []
        for x in sitems:
            scards.append(x[2])
        return cards, scards


    #
    # layout support
    #

    def __getClosestStack(self, card, stacks):
        closest = None
        cdist = 999999999
        cx, cy = card.x, card.y
        # Since we only compare distances,
        # we don't bother to take the square root.
        for stack in stacks:
            dist = (stack.x - cx)**2 + (stack.y - cy)**2
            if dist < cdist:
                closest = stack
                cdist = dist
        return closest

    def getClosestStack(self, card):
        cx, cy = card.x, card.y
        for stacks, rect in self.regions.info:
            if cx >= rect[0] and cx < rect[2] and cy >= rect[1] and cy < rect[3]:
                return self.__getClosestStack(card, stacks)
        return self.__getClosestStack(card, self.regions.remaining)

    # define a region for use in getClosestStack()
    def setRegion(self, stacks, rect):
        assert len(stacks) > 0
        assert len(rect) == 4 and rect[0] < rect[2] and rect[1] < rect[3]
        for s in stacks:
            assert s and s in self.allstacks
            # verify that the stack lies within the rectangle
            x, y, r = s.x, s.y, rect
            assert x >= r[0] and x < r[2] and y >= r[1] and y < r[3]
            # verify that the stack is not already in another region
            for rs, re in self.regions.info:
                assert not s in rs
        # add to regions
        self.regions.info.append((tuple(stacks), tuple(rect)))

    # as getClosestStack() is called within the mouse motion handler
    # event it is worth optimizing a little bit
    def optimizeRegions(self):
        self.regions.info = tuple(self.regions.info)
        remaining = list(self.sg.openstacks)[:]
        for stacks, rect in self.regions.info:
            for stack in stacks:
                if stack in remaining:
                    remaining.remove(stack)
        self.regions.remaining = tuple(remaining)


    #
    # Game - subclass overridable actions - IMPORTANT FOR GAME LOGIC
    #

    # create the game (create stacks, cards, etc.)
    def createGame(self):
        raise SubclassResponsibility

    # start the game (i.e. deal initial cards)
    def startGame(self):
        raise SubclassResponsibility

    # can we deal cards ?
    def canDealCards(self):
        # default: ask the Talon
        return self.s.talon and self.s.talon.canDealCards()

    # deal cards - return number of cards dealt
    def dealCards(self):
        # default: set state to deal and pass dealing to Talon
        if self.s.talon and self.canDealCards():
            self.finishMove()
            old_state = self.enterState(self.S_DEAL)
            n = self.s.talon.dealCards()
            self.leaveState(old_state)
            self.finishMove()
            self.autoPlay()
            return n
        return 0

    # fill a stack if rules require it (e.g. Picture Gallery)
    def fillStack(self, stack):
        pass

    # the actual hint class (or None)
    Hint_Class = DefaultHint

    def getHintClass(self):
        return self.Hint_Class

    def getStrictness(self):
        return 0

    # can we save outself ?
    def canSaveGame(self):
        return 1


    #
    # Game - stats handlers
    #

    # game changed - i.e. should we ask the player to discard the game
    def changed(self):
        if self.gstats.updated < 0:
            return 0                    # already won or lost
        if self.gstats.loaded > 0:
            return 0                    # loaded games account for no stats
        if self.gstats.restarted > 0:
            return 1                    # game was restarted - always ask
        if self.moves.index == 0 or self.getPlayerMoves() == 0:
            return 0
        return 2

    def getWinStatus(self, won):
        if not won or self.stats.hints > 0 or self.stats.demo_moves > 0:
            # sorry, you lose
            return (0, self.U_LOST)
        if (self.stats.undo_moves == 0 and
              self.stats.quickplay_moves == 0 and
              self.stats.highlight_piles == 0 and
              self.stats.highlight_cards == 0):
            # very good !
            return (1, self.U_STRICT_WON)
        return (1, self.U_WON)

    # update statistics when a game was won/ended/canceled/...
    def updateStats(self, was_demo=0):
        won = self.isGameWon()
        if was_demo and self.getPlayerMoves() == 0:
            # a pure demo game - update demo stats
            self.app.stats.updateStats(None, self, won, 1-won)
        elif self.changed():
            # must update player stats
            won, self.gstats.updated = self.getWinStatus(won=won)
            self.app.stats.updateStats(self.app.opt.player, self, won, 1-won)
            self.updateStatus(stats=self.app.stats.getStats(self.app.opt.player, self.id))

    def checkForWin(self):
        if not self.isGameWon():
            return 0
        self.finishMove()       # just in case
        won, u = self.getWinStatus(won=1)
        if u == self.U_STRICT_WON:
            self.updateStats()
            d = MfxDialog(self.top, title="Game won",
                          text="\nCongratulations, this\nwas a truly perfect game !\n\n" +
                          "Your playing time is " + self.getTime() + "\n" +
                          "for " + str(self.moves.index) + " moves.\n",
                          strings=("New game", "Cancel"),
                          image=self.app.jokers[5], separatorwidth=2)
        elif u == self.U_WON:
            self.updateStats()
            d = MfxDialog(self.top, title="Game won",
                          text="\nCongratulations, you did it !\n\n" +
                          "Your playing time is " + self.getTime() + "\n" +
                          "for " + str(self.moves.index) + " moves.\n",
                          strings=("New game", "Cancel"),
                          image=self.app.jokers[4], separatorwidth=2)
        elif self.gstats.updated < 0:
            d = MfxDialog(self.top, title="Game finished",
                          text="\nGame finished\n", bitmap="",
                          strings=("New game", "Cancel"))
        else:
            d = MfxDialog(self.top, title="Game finished",
                          text="\nGame finished, but you needed my help...\n",
                          bitmap="", strings=("New game", "Restart", "Cancel"))
            if d.num == 1:
                self.restartGame()
            elif d.num != 2:
                # Don't update if the player clicks on Cancel. He
                # may want to restart the game later.
                self.updateStats()
        if d.num == 0:
            if u == self.U_WON:
                self.winAnimation()
            elif u == self.U_STRICT_WON:
                self.winAnimation()
            self.newGame()
        return 1


    #
    # Game - subclass overridable methods (but usually not)
    #

    def isGameWon(self):
        # default: all Foundations must be filled
        c = 0
        for s in self.s.foundations:
            c = c + len(s.cards)
        return c == len(self.cards)

    # determine the real number of player_moves
    def getPlayerMoves(self):
        player_moves = self.stats.player_moves
##        if self.moves.index > 0 and self.stats.demo_moves == self.moves.index:
##            player_moves = 0
        return player_moves

    def updateTime(self):
        t = time.time()
        d = t - self.stats.start_time
        ##print t, self.stats.start_time, d
        if d > 0:
            self.stats.elapsed_time = self.stats.elapsed_time + d
            self.gstats.total_elapsed_time = self.gstats.total_elapsed_time + d
        self.stats.start_time = t

    def getTime(self):
        self.updateTime()
        t = int(round(self.stats.elapsed_time))
        if t <= 0: return "0:00"
        if t < 3600: return "%d:%02d" % (t / 60, t % 60)
        return "%d:%02d:%02d" % (t / 3600, (t % 3600) / 60, t % 60)


    #
    # Game - subclass overridable intelligence
    #

    def getAutoStacks(self, event=None):
        # returns (flipstacks, dropstacks, quickplaystacks)
        # default: sg.dropstacks
        return (self.sg.dropstacks, self.sg.dropstacks, self.sg.dropstacks)

    # handles autofaceup, autodrop and autodeal
    def autoPlay(self, autofaceup=-1, autodrop=-1, autodeal=-1):
        if self.demo:
            return 0
        old_busy = self.busy
        self.busy = 1
        if autofaceup < 0: autofaceup = self.app.opt.autofaceup
        if autodrop < 0: autodrop = self.app.opt.autodrop
        if autodeal < 0: autodeal = self.app.opt.autodeal
        moves = self.stats.total_moves
        n = self._autoPlay(autofaceup, autodrop, autodeal)
        self.finishMove()
        self.stats.autoplay_moves = self.stats.autoplay_moves + (self.stats.total_moves - moves)
        self.busy = old_busy
        return n

    def _autoPlay(self, autofaceup, autodrop, autodeal):
        flipstacks, dropstacks, quickstacks = self.getAutoStacks()
        done_something = 1
        while done_something:
            done_something = 0
            # a) flip top cards face-up
            if autofaceup and flipstacks:
                for s in flipstacks:
                    if s.canFlipCard():
                        s.flipMove()
                        done_something = 1
                        # each single flip is undo-able unless opt.autofaceup
                        self.finishMove()
                        if self.checkForWin():
                            return 1
            # b) drop cards
            if autodrop and dropstacks:
                for s in dropstacks:
                    to_stack, ncards = s.canDropCards(self.s.foundations)
                    if to_stack:
                        # each single drop is undo-able (note that this call
                        # is before the acutal move)
                        self.finishMove()
                        s.moveMove(ncards, to_stack)
                        done_something = 1
                        if self.checkForWin():
                            return 1
            # c) deal
            if autodeal:
                if self._autoDeal():
                    done_something = 1
                    self.finishMove()
                    if self.checkForWin():
                        return 1
        return 0

    def _autoDeal(self):
        # default: deal if the waste is empty
        w = self.s.waste
        if w and len(w.cards) == 0 and self.canDealCards():
            return self.dealCards()
        return 0


    ### highlight all moveable piles
    def getHighlightPilesStacks(self):
        # default: dropstacks with min pile length = 2
        if self.sg.hp_stacks:
            return ( (self.sg.hp_stacks, 2), )
        return ()

    def _highlightCards(self, info, sleep=1.5):
        if not info:
            return 0
        rects = []
        for x in info:
            s, c1, c2, fill, outline = x
            width = 1
            if outline: width = 4
            assert c1 in s.cards and c2 in s.cards
            sy0 = s.CARD_YOFFSET[0]
            if sy0 >= 0:
                x1, y1 = s.getPositionFor(c1)
                x2, y2 = s.getPositionFor(c2)
                if c2 is not s.cards[-1] and sy0 > 0:
                    y2 = y2 + sy0
                else:
                    y2 = y2 + self.app.images.CARDH
            else:
                x1, y1 = s.getPositionFor(c2)
                x2, y2 = s.getPositionFor(c1)
                y2 = y2 + self.app.images.CARDH
                if c2 is not s.cards[-1]:
                    y1 = y1 + (self.app.images.CARDH + sy0)
            x2 = x2 + self.app.images.CARDW
            ##print c1, c2, x1, y1, x2, y2
            r = MfxCanvasRectangle(self.canvas, x1-1, y1-1, x2+1, y2+1,
                                   width=width, fill=fill, outline=outline)
            r.tkraise(c2.item)
            rects.append(r)
        if not rects:
            return 0
        self.canvas.update_idletasks()
        self.top.sleep(sleep)
        rects.reverse()
        for r in rects:
            r.delete()
        self.canvas.update_idletasks()
        return EVENT_HANDLED

    def highlightPiles(self, stackinfo, sleep=1.5):
        stackinfo = self.getHighlightPilesStacks()
        if not stackinfo:
            return 0
        col = self.app.opt.highlight_piles_colors
        hi = []
        for si in stackinfo:
            for s in si[0]:
                pile = s.getPile()
                if pile and len(pile) >= si[1]:
                    hi.append((s, pile[0], pile[-1], col[0], col[1]))
        return self._highlightCards(hi, sleep)


    ### highlight matching cards
    def shallHighlightMatch(self, stack1, card1, stack2, card2):
        return 0


    #
    # Hint - uses self.getHintClass()
    #

    # compute all hints for the current position
    # this is the only method that actually uses class Hint
    def getHints(self, level, taken_hint=None):
        hint_class = self.getHintClass()
        if hint_class is None:
            return None
        hint = hint_class(self, level)      # call constructor
        return hint.getHints(taken_hint)    # and return all hints

    # give a hint
    def showHint(self, level=0, sleep=1.5, taken_hint=None):
        if self.getHintClass() is None:
            return None
        # reset list if level has changed
        if level != self.hints.level:
            self.hints.level = level
            self.hints.list = None
        # compute all hints
        if self.hints.list is None:
            self.hints.list = self.getHints(level, taken_hint)
            ###print self.hints.list
            self.hints.index = 0
        # get next hint from list
        if not self.hints.list:
            return None
        h = self.hints.list[self.hints.index]
        self.hints.index = self.hints.index + 1
        if self.hints.index >= len(self.hints.list):
            self.hints.index = 0
        # paranoia - verify hint
        score, pos, ncards, from_stack, to_stack, text_color, forced_move = h
        assert from_stack and len(from_stack.cards) >= ncards
        if ncards == 0:
            # a deal move, should not happen with level=0/1
            assert level >= 2
            assert from_stack is self.s.talon
            return h
        elif from_stack == to_stack:
            # a flip move, should not happen with level=0/1
            assert level >= 2
            assert ncards == 1 and len(from_stack.cards) >= ncards
            return h
        else:
            # a move move
            assert to_stack
            assert 1 <= ncards <= len(from_stack.cards)
            assert to_stack.acceptsPile(from_stack, from_stack.cards[-ncards:])
        # compute position for arrow
        if sleep <= 0.0:
            return h
        x1, y1 = from_stack.getPositionFor(from_stack.cards[-ncards])
        x2, y2 = to_stack.getPositionFor(to_stack.getCard())
        if ncards == 1:
            x1 = x1 + self.app.images.CARDW / 2
            y1 = y1 + self.app.images.CARDH / 2
        elif from_stack.CARD_XOFFSET[0]:
            x1 = x1 + from_stack.CARD_XOFFSET[0] / 2
            y1 = y1 + self.app.images.CARDH / 2
        else:
            x1 = x1 + self.app.images.CARDW / 2
            y1 = y1 + from_stack.CARD_YOFFSET[0] / 2
        x2 = x2 + self.app.images.CARDW / 2
        y2 = y2 + self.app.images.CARDH / 2
        # draw the hint
        arrow = MfxCanvasLine(self.canvas, x1, y1, x2, y2, width=7,
                              fill=self.app.opt.hintarrow_color,
                              arrow="last", arrowshape=(30,30,10))
        text = None
        if level == 1 or (level > 1 and self.app.debug):
            text = MfxCanvasText(self.canvas, 10, self.height - 10,
                                 anchor=ANCHOR_SW, fill=text_color,
                                 text="Score %6d" % (score))
        self.canvas.update_idletasks()
        # wait
        self.top.sleep(sleep)
        # delete the hint from the canvas
        if text:
            text.delete()
        arrow.delete()
        self.canvas.update_idletasks()
        return h


    #
    # Demo - uses showHint()
    #

    # start a demo
    def startDemo(self, mixed=1, level=9):
        assert level >= 2               # needed for flip/deal hints
        self.demo = Struct(
            level = level,
            mixed = mixed,
            sleep = self.app.opt.demo_sleep,
            last_deal = [],
            hint = None,
            keypress = None,
            start_demo_moves = self.stats.demo_moves,
        )
        self.hints.list = None
        after_idle(self.top, self.demoEvent) # schedule first move

    # demo event - play one demo move
    def demoEvent(self):
        # note: other events are allowed to stop self.demo at any time
        if self.demo is None or self.demo.keypress:
            self.demo = None
            self.updateMenus()
            return
        finished = self.playOneDemoMove(self.demo)
        self.finishMove()
        self.top.update_idletasks()
        self.hints.list = None
        player_moves = self.getPlayerMoves()
        d, status = None, 0
        bitmap = "info"
        timeout = 10000
        if player_moves == 0:
            timeout = 5000
        if self.isGameWon():
            finished = 1
            if not self.top.winfo_ismapped():
                status = 2
            elif player_moves == 0:
                s = self.app.miscrandom.choice(("Great", "Cool", "Yeah", "Wow"))
                d = MfxDialog(self.top, title="PySol Autopilot",
                              text="\nGame solved in " + str(self.moves.index) + " moves.\n",
                              image=self.app.jokers[4], strings=(s,),
                              separatorwidth=2, timeout=timeout)
                status = d.status
            else:
                s = self.app.miscrandom.choice(("Ok", "Ok"))
                text = "\n   Game finished   \n"
                if self.app.debug:
                    text = text + "\n%d %d\n" % (self.stats.player_moves, self.stats.demo_moves)
                d = MfxDialog(self.top, title="PySol Autopilot",
                              text=text, bitmap=bitmap, strings=(s,),
                              padx=30, timeout=timeout)
                status = d.status
        elif finished:
            if not self.top.winfo_ismapped():
                status = 2
            else:
                s = self.app.miscrandom.choice(("Oh well", "That's life", "Hmm"))
                d = MfxDialog(self.top, title="PySol Autopilot",
                              text="\nThis won't come out...\n",
                              bitmap=bitmap, strings=(s,),
                              padx=30, timeout=timeout)
                status = d.status
        if finished:
            self.updateStats(was_demo=1)
            if self.demo and status == 2 and not self.app.debug:
                # timeout in dialog
                if self.stats.demo_moves > self.demo.start_demo_moves:
                    # only increase the splash-screen counter if the last
                    # demo did do anything
                    self.app.demo_counter =  self.app.demo_counter + 1
                    if self.app.demo_counter % 3 == 0:
                        if self.top.winfo_ismapped():
                            status = helpAbout(self.app, timeout=10000)
            if self.demo and status == 2:
                # timeout in dialog - start another demo
                demo = self.demo
                id = self.id
                if 1 and demo.mixed and self.app.debug:
                    # debug - advance game id to make sure we hit all games
                    gl = self.app.getGamesIdSortedById()
                    gl = self.app.getGamesIdSortedByName()
                    index = (gl.index(self.id) + 1) % len(gl)
                    id = gl[index]
                elif demo.mixed:
                    # choose a random game
                    g = self.app.getGamesIdSortedById()
                    while len(g) > 1:
                        id = self.app.getRandomGameId()
                        if 0 or id != self.id:      # force change of game
                            break
                if id == self.id and self.app.nextgame.cardset_index == self.app.cardset.index:
                    self.newGame(autoplay=0)
                    self.startDemo(mixed=demo.mixed)
                else:
                    self._quitGame(id, startdemo=1)
            else:
                if 0 and self.app.debug:
                    # debug - only for testing winAnimation()
                    self.winAnimation()
                    self.newGame()
                pass
        else:
            # game not finished yet
            self.top.busyUpdate()
            if self.demo:
                after_idle(self.top, self.demoEvent) # schedule next move

    # play one demo move while in the demo event
    def playOneDemoMove(self, demo):
        if self.moves.index > 2000:
            # we're probably looping because of some bug in the hint code
            print "loop", self.id, self.moves.index                 #bundle#
            return 1
        if self.moves.index > 500:                                  #bundle#
            print "loop", self.id, self.moves.index                 #bundle#
            return 1                                                #bundle#
        sleep = demo.sleep
        if self.app.debug:
            if 0 and self.moves.index > 10: return 1                #bundle#
            if not self.top.winfo_ismapped():
                sleep = -1.0
        h = self.showHint(demo.level, sleep, taken_hint=demo.hint)
        demo.hint = h
        if not h:
            return 1
        score, pos, ncards, from_stack, to_stack, text_color, forced_move = h
        if ncards == 0:
            # a deal-move
            if self.dealCards() == 0:
                return 1
            # do not let games like Klondike and Canfield deal forever
            c = self.s.talon.getCard()
            if c in demo.last_deal:
                # We went through the whole Talon. Give up.
                return 1
            # Note that `None' is a valid entry in last_deal[]
            # (this means that all cards are on the Waste).
            demo.last_deal.append(c)
        elif from_stack == to_stack:
            # a flip-move
            from_stack.flipMove()
            demo.last_deal = []
        else:
            # a move-move
            from_stack.moveMove(ncards, to_stack, frames=-1)
            demo.last_deal = []
        ##print self.moves.index
        return 0


    #
    # Handle moves (with move history for undo/redo)
    # Actual move is handled in a subclass of AtomicMove.
    #
    # Note:
    #   User actions should get routed to Stack.flipMove() etc,
    #   but methods like Game.startGame() and dealCards()
    #   may call these directly.
    #

    def startMoves(self):
        self.moves = Struct(
            state = self.S_PLAY,
            history = [],        # list of lists of atomic moves
            index = 0,
            current = [],        # atomic moves for the current move
        )
        # reset statistics
        self.stats.undo_moves = 0
        self.stats.redo_moves = 0
        self.stats.player_moves = 0
        self.stats.demo_moves = 0
        self.stats.total_moves = 0
        self.stats.quickplay_moves = 0

    def __storeMove(self, am):
        if self.S_DEAL <= self.moves.state <= self.S_PLAY:
            self.moves.current.append(am)

    # move type 1
    def moveMove(self, ncards, from_stack, to_stack, frames=-1, shadow=-1):
        assert from_stack and to_stack
        assert 0 < ncards <= len(from_stack.cards)
        am = AMoveMove(ncards, from_stack, to_stack, frames, shadow)
        self.__storeMove(am)
        am.do(self)
        self.hints.list = None

    # move type 2
    def flipMove(self, stack):
        assert stack
        am = AFlipMove(stack)
        self.__storeMove(am)
        am.do(self)
        self.hints.list = None

    # move type 3
    def turnStackMove(self, from_stack, to_stack, update_flags=1):
        assert from_stack and to_stack and (from_stack is not to_stack)
        assert len(to_stack.cards) == 0
        am = ATurnStackMove(from_stack, to_stack, update_flags=update_flags)
        self.__storeMove(am)
        am.do(self)
        self.hints.list = None

    # move type 4
    def nextRoundMove(self, stack):
        assert stack
        am = ANextRoundMove(stack)
        self.__storeMove(am)
        am.do(self)
        self.hints.list = None

    # move type 5
    def saveSeedMove(self, old_seed):
        am = ASaveSeedMove(old_seed, self)
        self.__storeMove(am)
        am.do(self)
        ##self.hints.list = None

    # move type 6
    def shuffleStackMove(self, stack):
        assert stack
        assert stack.can_hide_cards or not stack.is_visible
        am = AShuffleStackMove(stack, self)
        self.__storeMove(am)
        am.do(self)
        self.hints.list = None

    # move type 7
    def updateStackViewMove(self, stack, flags):
        assert stack
        am = AUpdateStackViewMove(stack, flags)
        self.__storeMove(am)
        am.do(self)
        ##self.hints.list = None

    # Finish the current move.
    def finishMove(self):
        moves, stats = self.moves, self.stats
        if not moves.current:
            return 0
        # invalidate hints
        self.hints.list = None
        # resize (i.e. possibly shorten list from previous undos)
        del moves.history[moves.index : ]
        # update stats
        if self.demo:
            stats.demo_moves = stats.demo_moves + 1
            if moves.index == 0:
                stats.player_moves = 0  # clear all player moves
        else:
            stats.player_moves = stats.player_moves + 1
            if moves.index == 0:
                stats.demo_moves = 0    # clear all demo moves
        stats.total_moves = stats.total_moves + 1
        # add current move to history (which is a list of lists)
        moves.history.append(moves.current)
        moves.index = moves.index + 1
        assert moves.index == len(moves.history)
        moves.current = []
        self.updateStatus(moves=moves.index)
        self.updateMenus()
        return 1


    #
    # undo/redo layer
    #

    def undo(self):
        assert self.moves.state == self.S_PLAY and self.moves.current == []
        assert self.moves.index >= 0 and self.moves.index <= len(self.moves.history)
        if self.moves.index == 0:
            return
        self.moves.index = self.moves.index - 1
        m = self.moves.history[self.moves.index]
        m = m[:]
        m.reverse()
        self.moves.state = self.S_UNDO
        for atomic_move in m:
            atomic_move.undo(self)
        self.moves.state = self.S_PLAY
        self.stats.undo_moves = self.stats.undo_moves + 1
        self.stats.total_moves = self.stats.total_moves + 1
        self.hints.list = None
        self.updateStatus(moves=self.moves.index)
        self.updateMenus()

    def redo(self):
        assert self.moves.state == self.S_PLAY and self.moves.current == []
        assert self.moves.index >= 0 and self.moves.index <= len(self.moves.history)
        if self.moves.index == len(self.moves.history):
            return
        m = self.moves.history[self.moves.index]
        self.moves.index = self.moves.index + 1
        self.moves.state = self.S_REDO
        for atomic_move in m:
            atomic_move.redo(self)
        self.moves.state = self.S_PLAY
        self.stats.redo_moves = self.stats.redo_moves + 1
        self.stats.total_moves = self.stats.total_moves + 1
        self.hints.list = None
        self.updateStatus(moves=self.moves.index)
        self.updateMenus()


    #
    # load/save
    #

    def loadGame(self, filename):
        if self.changed():
            if not self.areYouSure("Open game"): return
        game = None
        self.setCursor(cursor=CURSOR_WATCH)
        try:
            game = self._loadGame(filename, self.app)
        except Exception, ex:
            if 0 or self.app.debug: traceback.print_exc()          #bundle#
            self.setCursor(cursor=self.app.top_cursor)
            d = MfxExceptionDialog(self.top, ex, title="Load game error",
                                   text="Error while loading game")
        else:
            self.filename = filename
            game.filename = filename
            # now start the new game
            ##print game.__dict__
            if game.id == self.id and self.app.nextgame.cardset_index == self.app.cardset.index:
                self.restoreGame(game)
            else:
                self._quitGame(game.id, loadedgame=game)


    def saveGame(self, filename, binmode=1):
        self.setCursor(cursor=CURSOR_WATCH)
        try:
            self._saveGame(filename, binmode)
        except Exception, ex:
            if 0 or self.app.debug: traceback.print_exc()          #bundle#
            self.setCursor(cursor=self.app.top_cursor)
            d = MfxExceptionDialog(self.top, ex, title="Save game error",
                                   text="Error while saving game")
        else:
            self.filename = filename
            self.setCursor(cursor=self.app.top_cursor)


    #
    # low level load/save
    #

    def _loadGame(self, filename, app):
        game = None
        f = None
        try:
            f = open(filename, "rb")
            p = Unpickler(f)
            package = p.load()
            version = p.load()
            id = p.load()
            if (type(package) != types.StringType or package != PACKAGE
                    or type(version) != types.StringType
                    or type(id) != types.IntType or id <= 0):
                raise Exception, "Not a PySol file"
            if version < "2.99" or version > VERSION:
                raise Exception, "Cannot load games saved with\nPySol version " + version
            game = app.constructGame(id)
            game.version = version
            game.random = p.load()
            if type(game.random) == types.TupleType:
                # this is an old WHRandom seed
                game.random = WHRandom(game.random)
            assert type(game.random) == types.InstanceType
            assert isinstance(game.random, PysolRandom)
            game.loadinfo.stacks = []
            for i in range(p.load()):
                stack = []
                for j in range(p.load()):
                    card_id = p.load()
                    face_up = p.load()
                    assert type(card_id) == types.IntType and type(face_up) == types.IntType
                    stack.append( (card_id, face_up) )
                game.loadinfo.stacks.append(stack)
            game.loadinfo.talon_round = 1
            game.loadinfo.talon_round = p.load()
            moves = p.load()
            merge_dict(game.moves.__dict__, moves.__dict__)
            gstats = p.load()
            merge_dict(game.gstats.__dict__, gstats.__dict__)
            stats = p.load()
            merge_dict(game.stats.__dict__, stats.__dict__)
            game._loadGameHook(p)
            game.gstats.loaded = game.gstats.loaded + 1
        finally:
            if f: f.close()
        return game


    def _saveGame(self, filename, binmode=1):
        self.updateTime()
        f = None
        try:
            f = open(filename, "wb")
            p = Pickler(f, binmode)
            p.dump(PACKAGE)
            p.dump(VERSION)
            p.dump(self.id)
            p.dump(self.random)
            p.dump(len(self.allstacks))
            for stack in self.allstacks:
                p.dump(len(stack.cards))
                for card in stack.cards:
                    p.dump(card.id)
                    p.dump(card.face_up)
            p.dump(self.s.talon.round)
            p.dump(self.moves)
            self.gstats.saved = self.gstats.saved + 1
            p.dump(self.gstats)
            p.dump(self.stats)
            self._saveGameHook(p)
        finally:
            if f: f.close()


    #
    # subclass hooks
    #

    def _restoreGameHook(self, game):
        pass

    def _loadGameHook(self, p):
        pass

    def _saveGameHook(self, p):
        pass

