## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys, imp, re, os, types

# PySol imports
from mfxutil import Struct                                          #bundle#


# /***********************************************************************
# // constants
# ************************************************************************/

# GameInfo constants
class GI:
    # game type
    GT_1DECK_TYPE    = 0
    GT_2DECK_TYPE    = 1
    GT_3DECK_TYPE    = 2
    GT_4DECK_TYPE    = 3
    GT_BAKERS_DOZEN  = 4
    GT_CANFIELD      = 5
    GT_FAN_TYPE      = 6
    GT_FORTY_THIEVES = 7
    GT_FREECELL      = 8
    GT_GOLF          = 9
    GT_GYPSY         = 10
    GT_HANAFUDA      = 11
    GT_KLONDIKE      = 12
    GT_PAIRING_TYPE  = 13
    GT_ROW_TYPE      = 14
    GT_SIMPLE_TYPE   = 15
    GT_SPIDER        = 16
    GT_YUKON         = 17

    GT_RELAXED       = 1024
    GT_ORIGINAL      = 2048

    # game time
    TIME_QUICK  = 0
    TIME_MEDIUM = 1
    TIME_LONG   = 2

    # game difficulty
    ##GD_EASY   = 0
    ##GD_MEDIUM = 1
    ##GD_HARD   = 2

    # skill or chance
    ##SOC_CHANCE_ONLY   = 0
    ##SOC_MOSTLY_CHANCE = 1
    ##SOC_MODERATE      = 2
    ##SOC_MOSTLY_SKILL  = 3
    ##SOC_DIFFICULT     = 4

    # chance of winning
    ##COW_VERY_LOW  = 0
    ##COW_LOW       = 1
    ##COW_MEDIUM    = 2
    ##COW_HIGH      = 3
    ##COW_VERY_HIGH = 4


    SELECT_GAME_BY_TYPE = (
        ("&Baker's Dozen type", lambda g, gt=GT_BAKERS_DOZEN: g.game_type == gt),
        ("&Canfield type",      lambda g, gt=GT_CANFIELD: g.game_type == gt),
        ("F&an type",           lambda g, gt=GT_FAN_TYPE: g.game_type == gt),
        ("Forty &Thieves type", lambda g, gt=GT_FORTY_THIEVES: g.game_type == gt),
        ("&FreeCell type",      lambda g, gt=GT_FREECELL: g.game_type == gt),
        ("&Golf type",          lambda g, gt=GT_GOLF: g.game_type == gt),
        ("G&ypsy type",         lambda g, gt=GT_GYPSY: g.game_type == gt),
        ("&Klondike type",      lambda g, gt=GT_KLONDIKE: g.game_type == gt),
        ("&Hanafuda type",      lambda g, gt=GT_HANAFUDA: g.game_type == gt),
        ("&Pairing type",       lambda g, gt=GT_PAIRING_TYPE: g.game_type == gt),
###     ("S&imple games",       lambda g, gt=GT_SIMPLE_TYPE: g.game_type == gt),
        ("&Spider type",        lambda g, gt=GT_SPIDER: g.game_type == gt),
        ("&Yukon type",         lambda g, gt=GT_YUKON: g.game_type == gt),
        ("&One-Deck games",     lambda g, gt=GT_1DECK_TYPE: g.game_type == gt),
        ("&Two-Deck games",     lambda g, gt=GT_2DECK_TYPE: g.game_type == gt),
###     ("&Original games",     lambda g, gt=GT_ORIGINAL: g.game_type == gt),
###     ("&Relaxed variants",   lambda g, gt=GT_RELAXED: g.game_type & gt),
    )

    POPULAR_GAMES = (1, 2, 8, 11, 13, 19, 31, 36, 38, 52, 59, 105,)

    # the 10 games from the Atari ST Patience game v2.13
    COMPAT_ATARI_GAMES = (1, 3, 4, 7, 12, 14, 15, 16, 17, 39,)
    # Gnome AisleRiot 1.0.2
    COMPAT_GNOME_GAMES = (
        2, 8, 11, 19, 27, 29, 33, 34, 35, 40, 41, 42, 43, 44, 52, 58,
        59, 92, 93, 94, 95, 96, 105,
    )
    # KDE Patience 1.1.2
    COMPAT_KPAT_GAMES = (2, 8)
    # xpat2 1.06
    COMPAT_XPAT2_GAMES = (1, 2, 8, 9, 11, 31, 52, 54, 63, 82, 89, 105, 256,)

    CHILDREN_GAMES = (16, 33, 90, 91, 96, 97,)

    OPEN_GAMES = (
        5, 6, 8, 9, 26, 31, 34, 45, 46, 50, 53, 56, 57, 63, 64,
        77, 82, 85, 86, 87, 88, 96, 98, 258,
    )

    # These obsolete gameids have been used in previous versions of
    # PySol and are no longer supported because of internal changes.
    # The game has been assigned a new id.
    PROTECTED_GAMES = (22, 32, 262,)


# /***********************************************************************
# // core games database
# ************************************************************************/

class GameInfoException(Exception):
    pass

class GameInfo(Struct):
    def __init__(self, id, gameclass, name,
                 game_type, decks, redeals):
        if not (1 <= id <= 32767):
            raise GameInfoException, name + ": invalid game ID " + str(id)
        if not (1 <= decks <= 4):
            raise GameInfoException, name + ": invalid number of decks " + str(id)
        if not name or not (3 <= len(name) <= 25):
            raise GameInfoException, name + ": invalid game name"
        if id in GI.PROTECTED_GAMES:
            raise GameInfoException, name + ": protected game ID " + str(id)
        game_type = game_type & ~GI.GT_RELAXED
        Struct.__init__(self, id=id, gameclass=gameclass, name=name,
                        game_type=game_type, decks=decks, redeals=redeals,
                        ## additional
                        cards=52*decks, rules_filename=None)


class GameDb:
    def __init__(self):
        self.__games = {}
        self.__games_id = []

    def __getitem__(self, key):
        return self.__games[key]

    def get(self, key):
        return self.__games.get(key)

    def getAll(self):
        return self.__games_id

    def add(self, game_info):
        if not isinstance(game_info, GameInfo):
            raise GameInfoException, "wrong class"
        if self.__games.has_key(game_info.id):
            raise GameInfoException, "duplicate game ID " + str(game_info.id)
        if 1:
            for id in self.__games_id:
                if game_info.gameclass is self.__games[id].gameclass:
                    raise GameInfoException, "duplicate game class " + str(game_info.id)
                if game_info.name == self.__games[id].name:
                    raise GameInfoException, "duplicate game name " + str(game_info.id)
        self.__games[game_info.id] = game_info
        self.__games_id = self.__games.keys()
        self.__games_id.sort()


    #
    # access games database
    #

    def getGamesIdSortedById(self):
        return self.__games_id[:]

    def getGamesIdSortedByName(self):
        list1 = []
        for id in self.__games_id:
            list1.append((self.__games[id].name, id))
        list1.sort()
        list2 = []
        for item in list1:
            list2.append(item[1])
        return list2


# /***********************************************************************
# //
# ************************************************************************/

# the global database
GAME_DB = GameDb()


def registerGame(game_info):
    GAME_DB.add(game_info)
    return game_info


def loadGame(modname, filename):
    ##print "load game", modname, filename
#%ifdef BUNDLE
    #$unbundle#execfile(filename, globals(), globals())
#%else
    if sys.modules.has_key("gamedb"):
        # unbundled
        module = imp.load_source(modname, filename)
    else:
        # bundled
        execfile(filename, globals(), globals())
#%endif

